/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2025 Univ. Grenoble Alpes, CNRS, Grenoble INP - UGA, TIMC, 38000 Grenoble, France
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// -- Core image component stuff
#include "SingleImageComponent.h"
#include "ImageComponent.h"

// -- Core stuff
#include "TransformationManager.h"

// -- VTK stuff
// disable warning generated by clang about the surrounded headers
#include "CamiTKDisableWarnings"
#include <vtkProperty.h>
#include "CamiTKReEnableWarnings"

#include <vtkUnstructuredGrid.h>
#include <vtkImageClip.h>
#include <vtkImageChangeInformation.h>
#include <vtkMatrix4x4.h>
#include <vtkDataSetMapper.h>

#include <cmath>
#include "InteractiveViewer.h"
#include "Log.h"

namespace camitk {
// -------------------- constructor  --------------------
SingleImageComponent::SingleImageComponent(Component* parentComponent, Slice::SliceOrientation sliceOrientation, const QString& name, vtkSmartPointer<vtkWindowLevelLookupTable> lut)
    : Component(parentComponent, name, Component::SLICE, false) {

    this->sliceOrientation = sliceOrientation;

    this->lut = lut;

    // set my frame as the data frame of my parent ImageComponent
    setFrame(TransformationManager::getFrameOfReferenceOwnership(dynamic_cast<ImageComponent*>(parentComponent)->getDataFrame()));

    // build the slice 3D
    initRepresentation();
}

// -------------------- setSelected --------------------
void SingleImageComponent::setSelected(const bool b, const bool r) {
    if (dynamic_cast<ImageComponent*>(getParent()) != nullptr) {
        dynamic_cast<ImageComponent*>(getParent())->setSelected(b, false);
    }
    else {
        CAMITK_WARNING("A single image component has no parent !");
    }
}

// -------------------- singleImageSelected --------------------
void SingleImageComponent::singleImageSelected(const bool b) {
    if (cursorActor != nullptr) {
        cursorActor->VisibilityOn();
    }
    Component::setSelected(b, false);
}

// -------------------- initRepresentation --------------------
void SingleImageComponent::initRepresentation() {
    // initialize Slice
    mySlice = new Slice(dynamic_cast<ImageComponent*>(getParentComponent())->getImageData(), sliceOrientation, lut);

    switch (sliceOrientation) {
        case Slice::AXIAL:
        case Slice::AXIAL_NEURO:
            setVisibility("Axial Viewer", true);
            break;
        case Slice::CORONAL:
            setVisibility("Coronal Viewer", true);
            break;
        case Slice::SAGITTAL:
            setVisibility("Sagittal Viewer", true);
            break;
        case Slice::ARBITRARY:
            setVisibility("Arbitrary Viewer", true);
            break;
    }
    setVisibility("3D Viewer", false);
}

// ---------------------- pixelPicked  ----------------------------
void SingleImageComponent::pixelPicked(double i, double j, double k) {
    ((ImageComponent*)getParent())->pixelPicked(i, j, k);
}

// -------------------- getSliceOrientation --------------------
Slice::SliceOrientation SingleImageComponent::getSliceOrientation() {
    return sliceOrientation;
}

vtkSmartPointer<vtkActor> SingleImageComponent::get3DCursor() {
    // Because we cannot add the same actor to multiple viewers
    // We duplicate the 3D cursor Actor of the parent ImageComponent or we return our copy if it was already built
    if (getParentComponent() != nullptr && cursorActor == nullptr) {
        vtkSmartPointer<vtkActor> parentCursor = getParentComponent()->get3DCursor();
        if (parentCursor != nullptr) {
            cursorActor = vtkSmartPointer<vtkActor>::New();
            vtkSmartPointer<vtkDataSetMapper> cursorMapper = vtkSmartPointer<vtkDataSetMapper>::New();
            cursorMapper->SetInputData(parentCursor->GetMapper()->GetInput());
            cursorActor->SetMapper(cursorMapper);
            cursorActor->SetProperty(parentCursor->GetProperty());
            // The cursor cannot be picked
            cursorActor->PickableOff();
            // The cursor is not visible unless there is a picked pixel
            cursorActor->SetVisibility(parentCursor->GetVisibility());
        }
    }
    if (cursorActor != nullptr) {
        cursorActor->SetVisibility(getParentComponent()->get3DCursor()->GetVisibility());
    }
    return cursorActor;
}

}

