/*
 *  $Id: bdep_synth.c 22317 2019-07-22 11:25:23Z yeti-dn $
 *  Copyright (C) 2015-2019 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwyrandgenset.h>
#include <libprocess/stats.h>
#include <libprocess/arithmetic.h>
#include <libprocess/filters.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include "dimensions.h"
#include "preview.h"

#define BDEP_SYNTH_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

#define WORK_UPDATE_CHECK 1000000

enum {
    PAGE_DIMENSIONS = 0,
    PAGE_GENERATOR  = 1,
    PAGE_GRAPHS     = 2,
    PAGE_NPAGES
};

typedef enum {
    GRAPH_MEAN   = (1 << 0),
    GRAPH_RMS    = (1 << 1),
    GRAPH_ALL    = (1 << 2) - 1,
} GraphFlags;

typedef struct _BDepSynthControls BDepSynthControls;

typedef struct {
    gint active_page;
    gint seed;
    gboolean randomize;
    gboolean animated;
    gdouble coverage;
    gdouble height;
    gdouble height_noise;
    guint graph_flags;
} BDepSynthArgs;

struct _BDepSynthControls {
    BDepSynthArgs *args;
    GwyDimensions *dims;
    GtkWidget *dialog;
    GtkWidget *view;
    GtkWidget *update_now;
    GtkWidget *animated;
    GtkObject *seed;
    GtkWidget *randomize;
    GtkTable *table;
    GtkObject *coverage;
    GtkObject *height;
    GtkObject *height_noise;
    GtkWidget *height_units;
    GtkWidget *height_init;
    GSList *graph_flags;
    GwyContainer *mydata;
    GwyDataField *surface;
    gboolean in_init;
    gdouble zscale;
    gdouble pxsize;
};

static gboolean           module_register          (void);
static void               bdep_synth               (GwyContainer *data,
                                                    GwyRunType run);
static void               run_noninteractive       (BDepSynthArgs *args,
                                                    const GwyDimensionArgs *dimsargs,
                                                    GwyContainer *data,
                                                    GwyDataField *dfield,
                                                    gint oldid,
                                                    GQuark quark);
static gboolean           bdep_synth_dialog        (BDepSynthArgs *args,
                                                    GwyDimensionArgs *dimsargs,
                                                    GwyContainer *data,
                                                    GwyDataField *dfield,
                                                    gint id);
static void               update_controls          (BDepSynthControls *controls,
                                                    BDepSynthArgs *args);
static void               page_switched            (BDepSynthControls *controls,
                                                    GtkNotebookPage *page,
                                                    gint pagenum);
static void               update_values            (BDepSynthControls *controls);
static void               height_init_clicked      (BDepSynthControls *controls);
static void               bdep_synth_invalidate    (BDepSynthControls *controls);
static void               preview                  (BDepSynthControls *controls);
static gboolean           bdep_synth_do            (BDepSynthArgs *args,
                                                    GwyDataField *dfield,
                                                    GwyGraphCurveModel **gcmodels,
                                                    gdouble preview_time,
                                                    gdouble zscale);
static void               bdep_synth_load_args     (GwyContainer *container,
                                                    BDepSynthArgs *args,
                                                    GwyDimensionArgs *dimsargs);
static void               bdep_synth_save_args     (GwyContainer *container,
                                                    const BDepSynthArgs *args,
                                                    const GwyDimensionArgs *dimsargs);

#define GWY_SYNTH_CONTROLS BDepSynthControls
#define GWY_SYNTH_INVALIDATE(controls) bdep_synth_invalidate(controls)

#include "synth.h"

static const GwyEnum graph_flags[] = {
    { N_("Mean value"), GRAPH_MEAN },
    { N_("RMS"),        GRAPH_RMS  },
};

enum {
    GRAPH_NFLAGS = G_N_ELEMENTS(graph_flags)
};

static const BDepSynthArgs bdep_synth_defaults = {
    PAGE_DIMENSIONS,
    42, TRUE, TRUE,
    10.0,
    1.0, 0.0,
    0,
};

static const GwyDimensionArgs dims_defaults = GWY_DIMENSION_ARGS_INIT;

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Generates surfaces by ballistic deposition."),
    "Yeti <yeti@gwyddion.net>",
    "1.4",
    "David Nečas (Yeti)",
    "2015",
};

GWY_MODULE_QUERY2(module_info, bdep_synth)

static gboolean
module_register(void)
{
    gwy_process_func_register("bdep_synth",
                              (GwyProcessFunc)&bdep_synth,
                              N_("/S_ynthetic/_Deposition/_Ballistic..."),
                              GWY_STOCK_SYNTHETIC_BALLISTIC_DEPOSITION,
                              BDEP_SYNTH_RUN_MODES,
                              0,
                              N_("Generate surface by ballistic deposition"));

    return TRUE;
}

static void
bdep_synth(GwyContainer *data, GwyRunType run)
{
    BDepSynthArgs args;
    GwyDimensionArgs dimsargs;
    GwyDataField *dfield;
    GQuark quark;
    gint id;

    g_return_if_fail(run & BDEP_SYNTH_RUN_MODES);
    bdep_synth_load_args(gwy_app_settings_get(), &args, &dimsargs);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     GWY_APP_DATA_FIELD_KEY, &quark,
                                     0);

    if (run == GWY_RUN_IMMEDIATE
        || bdep_synth_dialog(&args, &dimsargs, data, dfield, id)) {
        run_noninteractive(&args, &dimsargs, data, dfield, id, quark);
    }

    gwy_dimensions_free_args(&dimsargs);
}

static void
run_noninteractive(BDepSynthArgs *args,
                   const GwyDimensionArgs *dimsargs,
                   GwyContainer *data,
                   GwyDataField *dfield,
                   gint oldid,
                   GQuark quark)
{
    GwyDataField *newfield;
    GwySIUnit *siunit;
    GwyGraphCurveModel *gcmodels[GRAPH_NFLAGS];
    gboolean replace = dimsargs->replace && dfield;
    gboolean add = dimsargs->add && dfield;
    gdouble zscale;
    gint newid;
    guint i;
    gboolean ok, wait_enabled;

    if (args->randomize)
        args->seed = g_random_int() & 0x7fffffff;

    if (add || replace) {
        if (add)
            newfield = gwy_data_field_duplicate(dfield);
        else
            newfield = gwy_data_field_new_alike(dfield, TRUE);
    }
    else {
        gdouble mag = pow10(dimsargs->xypow10) * dimsargs->measure;
        newfield = gwy_data_field_new(dimsargs->xres, dimsargs->yres,
                                      mag*dimsargs->xres, mag*dimsargs->yres,
                                      TRUE);

        siunit = gwy_data_field_get_si_unit_xy(newfield);
        gwy_si_unit_set_from_string(siunit, dimsargs->xyunits);

        siunit = gwy_data_field_get_si_unit_z(newfield);
        gwy_si_unit_set_from_string(siunit, dimsargs->zunits);
    }

    zscale = pow10(dimsargs->zpow10) * args->height;
    wait_enabled = gwy_app_wait_get_enabled();
    if (wait_enabled && args->animated)
        set_up_wait_data_field_preview(newfield, data, oldid);
    gwy_app_wait_start(gwy_app_find_window_for_channel(data, oldid),
                       _("Initializing..."));
    ok = bdep_synth_do(args, newfield, gcmodels,
                       wait_enabled ? 1.25 : 0.0, zscale);
    gwy_app_wait_finish();

    if (!ok) {
        g_object_unref(newfield);
        return;
    }

    if (replace) {
        gwy_app_undo_qcheckpointv(data, 1, &quark);
        gwy_container_set_object(data, gwy_app_get_data_key_for_id(oldid),
                                 newfield);
        gwy_app_channel_log_add_proc(data, oldid, oldid);
        g_object_unref(newfield);
        newid = oldid;
    }
    else {
        if (data) {
            newid = gwy_app_data_browser_add_data_field(newfield, data, TRUE);
            if (oldid != -1)
                gwy_app_sync_data_items(data, data, oldid, newid, FALSE,
                                        GWY_DATA_ITEM_GRADIENT,
                                        0);
        }
        else {
            newid = 0;
            data = gwy_container_new();
            gwy_container_set_object(data, gwy_app_get_data_key_for_id(newid),
                                     newfield);
            gwy_app_data_browser_add(data);
            gwy_app_data_browser_reset_visibility(data,
                                                  GWY_VISIBILITY_RESET_SHOW_ALL);
            g_object_unref(data);
        }

        gwy_app_set_data_field_title(data, newid, _("Generated"));
        gwy_app_channel_log_add_proc(data, add ? oldid : -1, newid);
        g_object_unref(newfield);
    }

    for (i = 0; i < GRAPH_NFLAGS; i++) {
        GwyGraphModel *gmodel;
        gchar *s, *title;

        if (!gcmodels[i])
            continue;

        gmodel = gwy_graph_model_new();
        gwy_graph_model_add_curve(gmodel, gcmodels[i]);
        g_object_unref(gcmodels[i]);

        s = gwy_app_get_data_field_title(data, newid);
        title = g_strdup_printf("%s (%s)", _(graph_flags[i].name), s);
        g_free(s);
        g_object_set(gmodel,
                     "title", title,
                     "x-logarithmic", TRUE,
                     "y-logarithmic", TRUE,
                     "axis-label-bottom", _("Mean deposited thickness"),
                     "axis-label-left", _(graph_flags[i].name),
                     NULL);
        g_free(title);

        if (graph_flags[i].value == GRAPH_MEAN
            || graph_flags[i].value == GRAPH_RMS) {
            gwy_graph_model_set_units_from_data_field(gmodel, newfield,
                                                      0, 1, 0, 1);
        }
        else {
            gwy_graph_model_set_units_from_data_field(gmodel, newfield,
                                                      0, 1, 0, 0);
        }

        gwy_app_data_browser_add_graph_model(gmodel, data, TRUE);
    }
}

static gboolean
bdep_synth_dialog(BDepSynthArgs *args,
                  GwyDimensionArgs *dimsargs,
                  GwyContainer *data,
                  GwyDataField *dfield_template,
                  gint id)
{
    GtkWidget *dialog, *table, *vbox, *hbox, *notebook, *label;
    BDepSynthControls controls;
    GwyDataField *dfield;
    gboolean finished;
    gint response, row;

    gwy_clear(&controls, 1);
    controls.in_init = TRUE;
    controls.args = args;
    controls.pxsize = 1.0;
    dialog = gtk_dialog_new_with_buttons(_("Ballistic Deposition"),
                                         NULL, 0,
                                         _("_Reset"), RESPONSE_RESET,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                         NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);
    controls.dialog = dialog;

    hbox = gtk_hbox_new(FALSE, 2);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox,
                       FALSE, FALSE, 4);

    vbox = gtk_vbox_new(FALSE, 4);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 4);

    controls.mydata = gwy_container_new();
    dfield = gwy_data_field_new(PREVIEW_SMALL_SIZE, PREVIEW_SMALL_SIZE,
                                dimsargs->measure*PREVIEW_SMALL_SIZE,
                                dimsargs->measure*PREVIEW_SMALL_SIZE,
                                TRUE);
    gwy_container_set_object_by_name(controls.mydata, "/0/data", dfield);
    if (dfield_template) {
        gwy_app_sync_data_items(data, controls.mydata, id, 0, FALSE,
                                GWY_DATA_ITEM_PALETTE,
                                0);
        controls.surface = gwy_synth_surface_for_preview(dfield_template,
                                                         PREVIEW_SMALL_SIZE);
        controls.zscale = 3.0*gwy_data_field_get_rms(dfield_template);
    }
    controls.view = create_preview(controls.mydata, 0, PREVIEW_SMALL_SIZE,
                                   FALSE);
    gtk_box_pack_start(GTK_BOX(vbox), controls.view, FALSE, FALSE, 0);

    gtk_box_pack_start(GTK_BOX(vbox),
                       gwy_synth_progressive_preview_new(&controls,
                                                         &controls.update_now,
                                                         &controls.animated,
                                                         &args->animated),
                       FALSE, FALSE, 0);
    g_signal_connect_swapped(controls.update_now, "clicked",
                             G_CALLBACK(preview), &controls);

    gtk_box_pack_start(GTK_BOX(vbox),
                       gwy_synth_random_seed_new(&controls,
                                                 &controls.seed, &args->seed),
                       FALSE, FALSE, 0);

    controls.randomize = gwy_synth_randomize_new(&args->randomize);
    gtk_box_pack_start(GTK_BOX(vbox), controls.randomize, FALSE, FALSE, 0);

    notebook = gtk_notebook_new();
    gtk_box_pack_start(GTK_BOX(hbox), notebook, TRUE, TRUE, 4);
    g_signal_connect_swapped(notebook, "switch-page",
                             G_CALLBACK(page_switched), &controls);

    controls.dims = gwy_dimensions_new(dimsargs, dfield_template);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
                             gwy_dimensions_get_widget(controls.dims),
                             gtk_label_new(_("Dimensions")));

    table = gtk_table_new(4 + (dfield_template ? 1 : 0), 3, FALSE);
    /* This is used only for synt.h helpers. */
    controls.table = GTK_TABLE(table);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table,
                             gtk_label_new(_("Generator")));
    row = 0;

    controls.coverage = gtk_adjustment_new(args->coverage,
                                           0.01, 10000.0, 0.01, 1.0, 0);
    g_object_set_data(G_OBJECT(controls.coverage), "target", &args->coverage);
    gwy_table_attach_adjbar(table, row, _("Co_verage:"), NULL,
                            controls.coverage, GWY_HSCALE_LOG);
    g_signal_connect_swapped(controls.coverage, "value-changed",
                             G_CALLBACK(gwy_synth_double_changed), &controls);
    row++;

    row = gwy_synth_attach_height(&controls, row,
                                  &controls.height, &args->height,
                                  _("_Height:"), NULL, &controls.height_units);

    if (dfield_template) {
        controls.height_init
            = gtk_button_new_with_mnemonic(_("_Like Current Image"));
        g_signal_connect_swapped(controls.height_init, "clicked",
                                 G_CALLBACK(height_init_clicked), &controls);
        gtk_table_attach(GTK_TABLE(table), controls.height_init,
                         0, 2, row, row+1, GTK_FILL, 0, 0, 0);
        row++;
    }

    row = gwy_synth_attach_variance(&controls, row,
                                    &controls.height_noise,
                                    &args->height_noise);

    table = gtk_table_new(1 + GRAPH_NFLAGS, 3, FALSE);
    /* This is used only for synt.h helpers. */
    controls.table = GTK_TABLE(table);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table,
                             gtk_label_new(_("Evolution")));
    row = 0;

    label = gtk_label_new(_("Plot graphs:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.graph_flags
        = gwy_check_boxes_create(graph_flags, GRAPH_NFLAGS,
                                 G_CALLBACK(gwy_synth_flag_changed_silent),
                                 &args->graph_flags, args->graph_flags);
    row = gwy_check_boxes_attach_to_table(controls.graph_flags,
                                          GTK_TABLE(table), 3, row);

    gtk_widget_show_all(dialog);
    controls.in_init = FALSE;
    /* Must be done when widgets are shown, see GtkNotebook docs */
    gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), args->active_page);
    update_values(&controls);

    finished = FALSE;
    while (!finished) {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            case GTK_RESPONSE_OK:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            finished = TRUE;
            break;

            case RESPONSE_RESET:
            {
                gint temp2 = args->active_page;
                *args = bdep_synth_defaults;
                args->active_page = temp2;
            }
            controls.in_init = TRUE;
            update_controls(&controls, args);
            controls.in_init = FALSE;
            break;

            default:
            g_assert_not_reached();
            break;
        }
    }

    bdep_synth_save_args(gwy_app_settings_get(), args, dimsargs);

    g_object_unref(controls.mydata);
    GWY_OBJECT_UNREF(controls.surface);
    gwy_dimensions_free(controls.dims);

    return response == GTK_RESPONSE_OK;
}

static void
update_controls(BDepSynthControls *controls,
                BDepSynthArgs *args)
{
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->seed), args->seed);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->randomize),
                                 args->randomize);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->animated),
                                 args->animated);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->coverage),
                             args->coverage);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->height), args->height);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->height_noise),
                             args->height_noise);
    gwy_check_boxes_set_selected(controls->graph_flags, args->graph_flags);
}

static void
page_switched(BDepSynthControls *controls,
              G_GNUC_UNUSED GtkNotebookPage *page,
              gint pagenum)
{
    if (controls->in_init)
        return;

    controls->args->active_page = pagenum;
    if (pagenum == PAGE_GENERATOR)
        update_values(controls);
}

static void
update_values(BDepSynthControls *controls)
{
    GwyDimensions *dims = controls->dims;

    controls->pxsize = dims->args->measure * pow10(dims->args->xypow10);
    if (controls->height_units)
        gtk_label_set_markup(GTK_LABEL(controls->height_units),
                             dims->zvf->units);
}

static void
height_init_clicked(BDepSynthControls *controls)
{
    gdouble mag = pow10(controls->dims->args->zpow10);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->height),
                             controls->zscale/mag);
}

static void
bdep_synth_invalidate(G_GNUC_UNUSED BDepSynthControls *controls)
{
}

static void
preview(BDepSynthControls *controls)
{
    BDepSynthArgs *args = controls->args;
    GwyDimensionArgs *dimsargs = controls->dims->args;
    GwyDataField *dfield;
    gdouble zscale;

    dfield = GWY_DATA_FIELD(gwy_container_get_object_by_name(controls->mydata,
                                                             "/0/data"));
    zscale = pow10(dimsargs->zpow10) * args->height;

    if (controls->dims->args->add && controls->surface)
        gwy_data_field_copy(controls->surface, dfield, TRUE);
    else
        gwy_data_field_clear(dfield);

    gwy_app_wait_start(GTK_WINDOW(controls->dialog), _("Initializing..."));
    bdep_synth_do(args, dfield, NULL, 1.25, zscale);
    gwy_app_wait_finish();

    gwy_data_field_data_changed(dfield);
}

static gboolean
bdep_synth_do(BDepSynthArgs *args,
              GwyDataField *basefield,
              GwyGraphCurveModel **gcmodels,
              gdouble preview_time,
              gdouble zscale)
{
    GwyDataField *dfield;
    gint xres, yres, xext, yext, n;
    gdouble nextgraphx;
    gdouble flux, height, hnoise;
    GwySynthUpdateType update;
    GTimer *timer;
    guint64 workdone, niter, iter;
    GwyRandGenSet *rngset;
    GRand *rng_k, *rng_height;
    GArray **evolution = NULL, *ev;
    gdouble *d;
    gboolean finished = FALSE;

    xext = gwy_data_field_get_xres(basefield)/12;
    yext = gwy_data_field_get_yres(basefield)/12;
    dfield = gwy_data_field_extend(basefield, xext, xext, yext, yext,
                                   GWY_EXTERIOR_MIRROR_EXTEND, 0.0, FALSE);

    timer = g_timer_new();
    if (!args->animated)
        preview_time = 0.0;

    if (gcmodels) {
        evolution = gwy_synth_make_evolution_arrays(graph_flags, GRAPH_NFLAGS,
                                                    args->graph_flags);
    }

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    n = xres*yres;
    flux = 1.0/n;

    rngset = gwy_rand_gen_set_new(2);
    gwy_rand_gen_set_init(rngset, args->seed);
    rng_k = gwy_rand_gen_set_rng(rngset, 0);
    rng_height = gwy_rand_gen_set_rng(rngset, 1);

    gwy_synth_update_progress(NULL, 0, 0, 0);
    if (!gwy_app_wait_set_message(_("Depositing particles...")))
        goto fail;

    d = dfield->data;
    hnoise = args->height_noise;
    niter = (guint64)(args->coverage/flux + 0.5);
    nextgraphx = 0.0;
    workdone = 0.0;
    iter = 0;

    while (iter < niter) {
        guint k = g_rand_int_range(rng_k, 0, n);
        gdouble v = (hnoise
                     ? hnoise*g_rand_double(rng_height) + 1.0 - hnoise
                     : 1.0);
        guint ii = (k/xres)*xres, j = k % xres;
        gdouble h = d[k] + v*zscale;
        guint iim = G_LIKELY(ii) ? ii - xres : 0;
        guint iip = G_LIKELY(ii != n - xres) ? ii + xres : n - xres;
        guint jl = G_LIKELY(j) ? j-1 : 0;
        guint jr = G_LIKELY(j != xres-1) ? j+1 : xres-1;
        gdouble h1 = MAX(d[iim + j], d[ii + jl]);
        gdouble h2 = MAX(d[ii + jr], d[iip + j]);
        d[k] = MAX(h, MAX(h1, h2));

        iter++;
        workdone++;

        if (workdone >= WORK_UPDATE_CHECK) {
            update = gwy_synth_update_progress(timer, preview_time,
                                               iter, niter);
            if (update == GWY_SYNTH_UPDATE_CANCELLED)
                goto fail;
            if (update == GWY_SYNTH_UPDATE_DO_PREVIEW) {
                gwy_data_field_area_copy(dfield, basefield,
                                         xext, yext,
                                         xres - 2*xext, yres - 2*yext,
                                         0, 0);
                gwy_data_field_data_changed(basefield);
            }
            workdone -= WORK_UPDATE_CHECK;
        }

        if (evolution && iter >= nextgraphx) {
            gwy_data_field_invalidate(dfield);
            height = iter*flux*zscale;
            g_array_append_val(evolution[GRAPH_NFLAGS], height);
            if ((ev = gwy_synth_find_evolution_array(evolution,
                                                     graph_flags, GRAPH_NFLAGS,
                                                     GRAPH_MEAN))) {
                gdouble avg = gwy_data_field_get_avg(dfield);
                g_array_append_val(ev, avg);
            }
            if ((ev = gwy_synth_find_evolution_array(evolution,
                                                     graph_flags, GRAPH_NFLAGS,
                                                     GRAPH_RMS))) {
                gdouble rms = gwy_data_field_get_rms(dfield);
                g_array_append_val(ev, rms);
            }

            nextgraphx += 0.0001/flux + MIN(0.2*nextgraphx, 0.08/flux);
        }
    }

    gwy_synth_make_evolution_plots(gcmodels, evolution,
                                   graph_flags, GRAPH_NFLAGS);

    gwy_data_field_area_copy(dfield, basefield,
                             xext, yext, xres - 2*xext, yres - 2*yext, 0, 0);
    gwy_data_field_data_changed(basefield);
    finished = TRUE;

fail:
    g_object_unref(dfield);
    g_timer_destroy(timer);
    gwy_rand_gen_set_free(rngset);
    gwy_synth_free_evolution_arrays(evolution, GRAPH_NFLAGS);

    return finished;
}

static const gchar prefix[]           = "/module/bdep_synth";
static const gchar active_page_key[]  = "/module/bdep_synth/active_page";
static const gchar animated_key[]     = "/module/bdep_synth/animated";
static const gchar coverage_key[]     = "/module/bdep_synth/coverage";
static const gchar graph_flags_key[]  = "/module/bdep_synth/graph_flags";
static const gchar height_key[]       = "/module/bdep_synth/height";
static const gchar height_noise_key[] = "/module/bdep_synth/height_noise";
static const gchar randomize_key[]    = "/module/bdep_synth/randomize";
static const gchar seed_key[]         = "/module/bdep_synth/seed";

static void
bdep_synth_sanitize_args(BDepSynthArgs *args)
{
    args->active_page = CLAMP(args->active_page,
                              PAGE_DIMENSIONS, PAGE_NPAGES-1);
    args->seed = MAX(0, args->seed);
    args->randomize = !!args->randomize;
    args->animated = !!args->animated;
    args->height = CLAMP(args->height, 0.001, 1000.0);
    args->height_noise = CLAMP(args->height_noise, 0.0, 1.0);
    args->coverage = CLAMP(args->coverage, 0.01, 10000.0);
    args->graph_flags &= GRAPH_ALL;
}

static void
bdep_synth_load_args(GwyContainer *container,
                     BDepSynthArgs *args,
                     GwyDimensionArgs *dimsargs)
{
    *args = bdep_synth_defaults;

    gwy_container_gis_int32_by_name(container, active_page_key,
                                    &args->active_page);
    gwy_container_gis_int32_by_name(container, seed_key, &args->seed);
    gwy_container_gis_boolean_by_name(container, randomize_key,
                                      &args->randomize);
    gwy_container_gis_boolean_by_name(container, animated_key,
                                      &args->animated);
    gwy_container_gis_double_by_name(container, height_key, &args->height);
    gwy_container_gis_double_by_name(container, height_noise_key,
                                     &args->height_noise);
    gwy_container_gis_double_by_name(container, coverage_key, &args->coverage);
    gwy_container_gis_int32_by_name(container, graph_flags_key,
                                    &args->graph_flags);
    bdep_synth_sanitize_args(args);

    gwy_clear(dimsargs, 1);
    gwy_dimensions_copy_args(&dims_defaults, dimsargs);
    gwy_dimensions_load_args(dimsargs, container, prefix);
}

static void
bdep_synth_save_args(GwyContainer *container,
                    const BDepSynthArgs *args,
                    const GwyDimensionArgs *dimsargs)
{
    gwy_container_set_int32_by_name(container, active_page_key,
                                    args->active_page);
    gwy_container_set_int32_by_name(container, seed_key, args->seed);
    gwy_container_set_boolean_by_name(container, randomize_key,
                                      args->randomize);
    gwy_container_set_boolean_by_name(container, animated_key,
                                      args->animated);
    gwy_container_set_double_by_name(container, height_key, args->height);
    gwy_container_set_double_by_name(container, height_noise_key,
                                     args->height_noise);
    gwy_container_set_double_by_name(container, coverage_key, args->coverage);
    gwy_container_set_int32_by_name(container, graph_flags_key,
                                    args->graph_flags);

    gwy_dimensions_save_args(dimsargs, container, prefix);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
