/* PSPP - a program for statistical analysis.
   Copyright (C) 2013 Free Software Foundation, Inc.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#include <config.h>

#include <string.h>

#include "libpspp/cmac-aes256.h"

#undef NDEBUG
#include <assert.h>

static void
test_cmac (const uint8_t key[32], size_t key_size,
           const uint8_t *data, size_t data_size,
           const uint8_t *exp_cmac, size_t exp_cmac_size)
{
  uint8_t cmac[16];

  assert (key_size == 32);
  assert (exp_cmac_size <= 16);

  cmac_aes256 (key, data, data_size, cmac);
  assert (!memcmp (cmac, exp_cmac, exp_cmac_size));
}

/* From NIST CMAC test vectors. */
static void
test_cmac1 (void)
{
  static const uint8_t key[] = {
    0x0b,0x12,0x2a,0xc8, 0xf3,0x4e,0xd1,0xfe,
    0x08,0x2a,0x36,0x25, 0xd1,0x57,0x56,0x14,
    0x54,0x16,0x7a,0xc1, 0x45,0xa1,0x0b,0xbf,
    0x77,0xc6,0xa7,0x05, 0x96,0xd5,0x74,0xf1
  };
  static const uint8_t data[] = {
    0x49,0x8b,0x53,0xfd, 0xec,0x87,0xed,0xcb,
    0xf0,0x70,0x97,0xdc, 0xcd,0xe9,0x3a,0x08,
    0x4b,0xad,0x75,0x01, 0xa2,0x24,0xe3,0x88,
    0xdf,0x34,0x9c,0xe1, 0x89,0x59,0xfe,0x84,
    0x85,0xf8,0xad,0x15, 0x37,0xf0,0xd8,0x96,
    0xea,0x73,0xbe,0xdc, 0x72,0x14,0x71,0x3f,
  };
  static const uint8_t exp_cmac[] = { 0xf6,0x2c,0x46,0x32, 0x9b };

  test_cmac (key, sizeof key,
             data, sizeof data,
             exp_cmac, sizeof exp_cmac);
}

/* CMAC-AES-256 test vectors from NIST's updated SP800-38B examples.  */
static void
test_cmac2 (void)
{
  static const uint8_t key[] = {
    0x60,0x3d,0xeb,0x10, 0x15,0xca,0x71,0xbe,
    0x2b,0x73,0xae,0xf0, 0x85,0x7d,0x77,0x81,
    0x1f,0x35,0x2c,0x07, 0x3b,0x61,0x08,0xd7,
    0x2d,0x98,0x10,0xa3, 0x09,0x14,0xdf,0xf4,
  };
  static const uint8_t data[] = {
    0x6b,0xc1,0xbe,0xe2, 0x2e,0x40,0x9f,0x96,
    0xe9,0x3d,0x7e,0x11, 0x73,0x93,0x17,0x2a,
    0xae,0x2d,0x8a,0x57, 0x1e,0x03,0xac,0x9c,
    0x9e,0xb7,0x6f,0xac, 0x45,0xaf,0x8e,0x51,
    0x30,0xc8,0x1c,0x46, 0xa3,0x5c,0xe4,0x11,
    0xe5,0xfb,0xc1,0x19, 0x1a,0x0a,0x52,0xef,
    0xf6,0x9f,0x24,0x45, 0xdf,0x4f,0x9b,0x17,
    0xad,0x2b,0x41,0x7b, 0xe6,0x6c,0x37,0x10,
  };
  static const uint8_t exp_cmac0[] = {
    0x02,0x89,0x62,0xf6, 0x1b,0x7b,0xf8,0x9e,
    0xfc,0x6b,0x55,0x1f, 0x46,0x67,0xd9,0x83,
  };
  static const uint8_t exp_cmac16[] = {
    0x28,0xa7,0x02,0x3f, 0x45,0x2e,0x8f,0x82,
    0xbd,0x4b,0xf2,0x8d, 0x8c,0x37,0xc3,0x5c,
  };
  static const uint8_t exp_cmac40[] = {
    0xaa,0xf3,0xd8,0xf1, 0xde,0x56,0x40,0xc2,
    0x32,0xf5,0xb1,0x69, 0xb9,0xc9,0x11,0xe6,
  };
  static const uint8_t exp_cmac64[] = {
    0xe1,0x99,0x21,0x90, 0x54,0x9f,0x6e,0xd5,
    0x69,0x6a,0x2c,0x05, 0x6c,0x31,0x54,0x10,
  };

  test_cmac (key, sizeof key, data, 0, exp_cmac0, sizeof exp_cmac0);
  test_cmac (key, sizeof key, data, 16, exp_cmac16, sizeof exp_cmac16);
  test_cmac (key, sizeof key, data, 40, exp_cmac40, sizeof exp_cmac40);
  test_cmac (key, sizeof key, data, 64, exp_cmac64, sizeof exp_cmac64);
}

int
main(void)
{
  test_cmac1 ();
  test_cmac2 ();
  return 0;
}
