% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/availableCores.R
\name{availableCores}
\alias{availableCores}
\title{Get Number of Available Cores on The Current Machine}
\usage{
availableCores(
  constraints = NULL,
  methods = getOption2("future.availableCores.methods", c("system", "nproc",
    "mc.cores", "_R_CHECK_LIMIT_CORES_", "PBS", "SGE", "Slurm", "LSF", "fallback",
    "custom")),
  na.rm = TRUE,
  logical = getOption2("future.availableCores.logical", TRUE),
  default = c(current = 1L),
  which = c("min", "max", "all")
)
}
\arguments{
\item{constraints}{An optional character specifying under what
constraints ("purposes") we are requesting the values.
For instance, on systems where multicore processing is not supported
(i.e. Windows), using \code{constrains = "multicore"} will force a
single core to be reported.}

\item{methods}{A character vector specifying how to infer the number
of available cores.}

\item{na.rm}{If TRUE, only non-missing settings are considered/returned.}

\item{logical}{Passed to
\code{\link[parallel]{detectCores}(logical = logical)}, which, if supported,
returns the number of logical CPUs (TRUE) or physical CPUs/cores (FALSE).
This argument is only if argument \code{methods} includes \code{"system"}.}

\item{default}{The default number of cores to return if no non-missing
settings are available.}

\item{which}{A character specifying which settings to return.
If \code{"min"}, the minimum value is returned.
If \code{"max"}, the maximum value is returned (be careful!)
If \code{"all"}, all values are returned.}
}
\value{
Return a positive (>= 1) integer.
If \code{which = "all"}, then more than one value may be returned.
Together with \code{na.rm = FALSE} missing values may also be returned.
}
\description{
The current/main \R session counts as one, meaning the minimum
number of cores available is always at least one.
}
\details{
The following settings ("methods") for inferring the number of cores
are supported:
\itemize{
\item \code{"system"} -
Query \code{\link[parallel]{detectCores}(logical = logical)}.
\item \code{"nproc"} -
On Unix, query system command \code{nproc}.
\item \code{"mc.cores"} -
If available, returns the value of option
\code{\link[base:options]{mc.cores}}.
Note that \option{mc.cores} is defined as the number of
\emph{additional} \R processes that can be used in addition to the
main \R process.  This means that with \code{mc.cores = 0} all
calculations should be done in the main \R process, i.e. we have
exactly one core available for our calculations.
The \option{mc.cores} option defaults to environment variable
\env{MC_CORES} (and is set accordingly when the \pkg{parallel}
package is loaded).  The \option{mc.cores} option is used by for
instance \code{\link[=mclapply]{mclapply}()} of the \pkg{parallel}
package.
\item \code{"PBS"} -
Query TORQUE/PBS environment variables \env{PBS_NUM_PPN} and \env{NCPUS}.
Depending on PBS system configuration, these \emph{resource}
parameter may or may not default to one.
An example of a job submission that results in this is
\verb{qsub -l nodes=1:ppn=2}, which requests one node with two cores.
\item \code{"SGE"} -
Query Sun/Oracle Grid Engine (SGE) environment variable
\env{NSLOTS}.
An example of a job submission that results in this is
\verb{qsub -pe smp 2} (or \verb{qsub -pe by_node 2}), which
requests two cores on a single machine.
\item \code{"Slurm"} -
Query Simple Linux Utility for Resource Management (Slurm)
environment variable \env{SLURM_CPUS_PER_TASK}.
This may or may not be set.  It can be set when submitting a job,
e.g. \verb{sbatch --cpus-per-task=2 hello.sh} or by adding
\verb{#SBATCH --cpus-per-task=2} to the \file{hello.sh} script.
If \env{SLURM_CPUS_PER_TASK} is not set, then it will fall back to
use \env{SLURM_CPUS_ON_NODE} if the job is a single-node job
(\env{SLURM_JOB_NUM_NODES} is 1), e.g. \verb{sbatch --ntasks=2 hello.sh}.
\item \code{"LSF"} -
Query Platform Load Sharing Facility (LSF) environment variable
\env{LSB_DJOB_NUMPROC}.
Jobs with multiple (CPU) slots can be submitted on LSF using
\verb{bsub -n 2 -R "span[hosts=1]" < hello.sh}.
\item \code{"custom"} -
If option \option{future.availableCores.custom} is set and a function,
then this function will be called (without arguments) and it's value
will be coerced to an integer, which will be interpreted as a number
of available cores.  If the value is NA, then it will be ignored.
}
For any other value of a \code{methods} element, the \R option with the
same name is queried.  If that is not set, the system environment
variable is queried.  If neither is set, a missing value is returned.
}
\section{Avoid ending up with zero cores}{

Note that \code{availableCores()} may return a single core.  Because of this,
using something like:\if{html}{\out{<div class="r">}}\preformatted{ncores <- availableCores() - 1
}\if{html}{\out{</div>}}

may return zero, which is often not intended.  Instead, use:\if{html}{\out{<div class="r">}}\preformatted{ncores <- max(1, availableCores() - 1)
}\if{html}{\out{</div>}}
}

\section{Advanced usage}{

It is possible to override the maximum number of cores on the machine
as reported by \code{availableCores(methods = "system")}.  This can be
done by first specifying
\code{options(future.availableCores.methods = "mc.cores")} and
then the number of cores to use, e.g. \code{options(mc.cores = 8)}.
}

\examples{
message(paste("Number of cores available:", availableCores()))

\dontrun{
options(mc.cores = 2L)
message(paste("Number of cores available:", availableCores()))
}

\dontrun{
## IMPORTANT: availableCores() may return 1L
options(mc.cores = 1L)
ncores <- max(1, availableCores() - 1)
message(paste("Number of cores to use:", ncores))
}

\dontrun{
## Use 75\% of the cores on the system but never more than four
options(parallelly.availableCores.custom = function() {
  ncores <- max(parallel::detectCores(), 1L, na.rm = TRUE)
  min(0.75 * ncores, 4L)
})
message(paste("Number of cores available:", availableCores()))
}

}
\seealso{
To get the set of available workers regardless of machine,
see \code{\link[=availableWorkers]{availableWorkers()}}.
}
