/** Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 * xine panel related stuff
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <unistd.h>
#include <time.h>
#include <sys/time.h>

#include "common.h"
#include "config_wrapper.h"
#include "xine-toolkit/backend.h"
#include "xine-toolkit/label.h"
#include "xine-toolkit/slider.h"
#include "xine-toolkit/button.h"
#include "xine-toolkit/button_list.h"
#include "xine-toolkit/skin.h"
#include "panel.h"
#include "acontrol.h"
#include "control.h"
#include "event_sender.h"
#include "mediamark.h"
#include "menus.h"
#include "mrl_browser.h"
#include "osd.h"
#include "playlist.h"
#include "setup.h"
#include "snapshot.h"
#include "stream_infos.h"
#include "videowin.h"
#include "actions.h"
#include "event.h"
#include "errors.h"

typedef enum {
  /* keep order */
  _W_prev = 0,
  _W_stop,
  _W_play,
  _W_pause,
  _W_next,
  _W_slider_play,
  /* /keep order */
  _W_title,
  _W_runtime,
  _W_eject,
  _W_slider_mixer,
  _W_mute,
  _W_audio,
  _W_spu,
  _W_autoplay,
  _W_LAST
} _W_t;

static const uint8_t _panel_mixer_max[LAST_MIXER] = {
  [SOUND_CARD_MIXER] = 100,
  [SOFTWARE_MIXER]   = 200
};

static const int _xine_volume_param[LAST_MIXER] = {
  [SOUND_CARD_MIXER] = XINE_PARAM_AUDIO_VOLUME,
  [SOFTWARE_MIXER] = XINE_PARAM_AUDIO_AMP_LEVEL
};

struct xui_panel_st {
  gui_new_window_t      nw;

  xitk_widget_t        *w[_W_LAST];

  struct {
    unsigned int        slider_play_time;
    int                 enabled;
  } playback_widgets;

  struct {
    int                 enable;
    int                 timeout;
  } tips;

  int                   visible;

  struct {
    pthread_mutex_t     mutex;
    pthread_cond_t      wake;
    pthread_t           thread;
    stream_infos_ident_t *ident;
    xitk_vers_string_t  istr;
    int                 run;
    unsigned int        message_time;
    int                 message_hold;
  }                     sl;

  /* private vars to avoid useless updates */
  unsigned int          shown_time;
  unsigned int          shown_length;
  int                   new_mrl;

  int                   logo_synthetic;
};

void panel_update_nextprev_tips (xui_panel_t *panel) {

  if (panel->nw.gui->skip_by_chapter) {
    xitk_set_widget_tips(panel->w[_W_prev], _("Play previous chapter or mrl"));
    xitk_set_widget_tips(panel->w[_W_next], _("Play next chapter or mrl"));
  }
  else {
    xitk_set_widget_tips(panel->w[_W_prev], _("Play previous entry from playlist"));
    xitk_set_widget_tips(panel->w[_W_next], _("Play next entry from playlist"));
  }
}

int panel_playback_ctrl (xui_panel_t *panel, int enable) {
  int res;

  if (!panel)
    return 0;
  if (XITK_0_TO_MAX_MINUS_1 (enable, 2) && (panel->playback_widgets.enabled != enable)) {
    panel->playback_widgets.enabled = enable;
    /* prev, stop, play, pause, next, slider_play */
    xitk_widgets_state (panel->w + _W_prev, 6, XITK_WIDGET_STATE_ENABLE, enable ? ~0u : 0);
  }
  res = panel->playback_widgets.enabled;
  return res;
}

/* Somewhat paranoia conditionals (Hans, YOU're paranoid ;-) )*/
/*
int panel_get_tips_enable (xui_panel_t *panel) {
  return (panel) ? panel->tips.enable : 0;
}
unsigned long panel_get_tips_timeout (xui_panel_t *panel) {
  return (panel) ? panel->tips.timeout : 0;
}
*/

/*
 * Config callbacks.
 */
static void panel_enable_tips_cb(void *data, xine_cfg_entry_t *cfg) {
  xui_panel_t *panel = data;
  panel->tips.enable = cfg->num_value;
  xitk_set_tips_timeout (panel->nw.gui->xitk, panel->tips.enable ? panel->tips.timeout : XITK_TIPS_TIMEOUT_OFF);
}
static void panel_timeout_tips_cb(void *data, xine_cfg_entry_t *cfg) {
  xui_panel_t *panel = data;
  panel->tips.timeout = cfg->num_value >= 0 ? cfg->num_value : 0;
  xitk_set_tips_timeout (panel->nw.gui->xitk, panel->tips.enable ? panel->tips.timeout : XITK_TIPS_TIMEOUT_OFF);
}

/*
 * Change displayed logo, if selected skin want to customize it.
 */
static void _update_logo (xui_panel_t *panel) {
  xine_cfg_entry_t     cfg_entry;
  const char          *skin_logo;
  int                  cfg_err_result;

  cfg_err_result = xine_config_lookup_entry (panel->nw.gui->xine, "gui.logo_mrl", &cfg_entry);
  skin_logo = xitk_skin_get_meta (panel->nw.gui->skin_config, XITK_SKIN_logo);
#ifdef XINE_LOGO2_MRL
#  define USE_XINE_LOGO_MRL XINE_LOGO2_MRL
#else
#  define USE_XINE_LOGO_MRL XINE_LOGO_MRL
#endif
  do {
    if (!skin_logo) {
      /* Skin don't use logo feature, set to xine's default */
      /* Back to default logo only on a skin change, not at the first skin loading. */
      if (!panel->logo_synthetic)
        break;
      skin_logo = USE_XINE_LOGO_MRL;
    }
    if (cfg_err_result && cfg_entry.str_value) {
      if (!strcmp (cfg_entry.str_value, skin_logo))
        /* Old and new logo are same, don't reload */
        break;
    }
    config_update_string (panel->nw.gui->xine, "gui.logo_mrl", skin_logo);
    panel->nw.gui->logo_mrl = skin_logo;
    gui_display_logo (panel->nw.gui);
  } while (0);
  panel->nw.gui->logo_has_changed--;
}

/*
 * Change the current skin.
 */
void panel_change_skins (xui_panel_t *panel, int synthetic) {
  int res;

  panel->logo_synthetic = !!synthetic;
  panel->nw.gui->logo_has_changed++;

  res = xitk_window_change_skin (panel->nw.xwin, panel->nw.gui->skin_config, "BackGround");
  if (res) {
    if (res == 2) {
      gui_msg (panel->nw.gui, XUI_MSG_ERROR, _("%s(): couldn't find image for background\n"), __XINE_FUNCTION__);
      exit (-1);
    }
    return;
  }
  xitk_window_get_window_position (panel->nw.xwin, &panel->nw.wr);
  /* no need to apply the same setting again.
   * even worse: doing so tends to cut off a pending or running EXPOSE/redraw sequence.
  video_window_set_transient_for (panel->nw.gui->vwin, panel->nw.xwin);
  if (panel_is_visible (panel) > 1)
    raise_window (panel->nw.gui, panel->nw.xwin, 1, 1);
   */
  /* panel_playback_ctrl (panel, panel->playback_widgets.enabled); */
  event_sender_move (panel->nw.gui);
}

/*
 * Update displayed MRL according to the current one.
 */
static int _panel_test_message (xui_panel_t *panel) {
  int hold;
  pthread_mutex_lock (&panel->sl.mutex);
  hold = panel->sl.message_hold;
  pthread_mutex_unlock (&panel->sl.mutex);
  return hold <= 0;
}
static void _panel_show_mrl (xui_panel_t *panel) {
  gui_playlist_lock (panel->nw.gui);
  xitk_label_change_label (panel->w[_W_title], (panel->nw.gui->is_display_mrl) ? panel->nw.gui->mmk.mrl : panel->nw.gui->mmk.ident);
  gui_playlist_unlock (panel->nw.gui);
}

/*
 * Update runtime displayed informations.
 */
static void secs2str (char *s, unsigned int secs, unsigned int length) {
  /* Avoid slow snprintf (). Yes this does not support south asian */
  /* alternative digits, but neither does the widget bitmap font.  */
  /* But wait -- what if you modify letters.png now ?              */
  if (secs >= 24 * 60 * 60) { /* [  dd::hh] */
    unsigned int days;
    secs /= 3600u;
    days = secs / 24u;
    secs %= 24u;
    memcpy (s, "    ::  ", 8);
    /* s[8] = 0; */
    s[7] = (secs % 10u) + '0';
    s[6] = (secs / 10u) + '0';
    s[3] = (days % 10u) + '0';
    days /= 10u;
    if (days)
      s[2] = days + '0';
  } else { /* [hh:mm:ss] */
    length /= 1000u;
    if ((length == (unsigned int)-1 / 1000u) || (secs > length))
      length = secs;
    do {
      memset (s, ' ', 4);
      /* s[8] = 0; */
      s[7] = (secs % 10u) + '0';
      secs /= 10u;
      s[6] = (secs % 6u) + '0';
      secs /= 6u;
      s[5] = ':';
      s[4] = (secs % 10u) + '0';
      secs /= 10u;
      if (length < 10 * 60)
        break;
      s[3] = (secs % 6u) + '0';
      secs /= 6u;
      if (length < 60 * 60)
        break;
      s[2] = ':';
      s[1] = (secs % 10u) + '0';
      secs /= 10u;
      if (length < 10 * 60 * 60)
        break;
      s[0] = secs + '0';
    } while (0);
  }
}

static int _panel_get_visibility (xui_panel_t *panel) {
  uint32_t flags = xitk_window_flags (panel->nw.xwin, 0, 0);

  panel->visible = (flags & XITK_WINF_VISIBLE) ? 2 : (flags & XITK_WINF_ICONIFIED) ? 1 : 0;
  return panel->visible;
}

/*
#define _PURD_LOCK 1
#define _PURD_SLIDER 2
#define _PURD_TITLE 8
*/
#define _PURD_TIME 4
#define _PURD_PLAYLIST 16
#define _PURD_TIPS 32
#define _PURD_NEW_STREAM 64
#define _PURD_LENGTH 128

int panel_update_runtime_display (xui_panel_t *panel, uint32_t flags) {
  int v[3], res;
  unsigned int step = 500;
  char timestr[16], tips[256];

  flags &= (_PURD_LOCK | _PURD_SLIDER | _PURD_TITLE | _PURD_NEW_STREAM);
  tips[0] = 0;

  if ((_panel_get_visibility (panel) >= 2) &&
    ((res = gui_xine_get_pos_length (panel->nw.gui, panel->nw.gui->stream, v)) != 0)) {

    do {
      int speed;
      unsigned int msecs = v[1], secs;
      if (!(v[0] || msecs)) { /* are you sure there is something to play? */
        panel->shown_time = ~3;
        strcpy (timestr, "--:--:--");
        break;
      }
      if (panel->nw.gui->runtime_mode) { /* show remaining time */
        if (msecs > (unsigned int)v[2]) {
          panel->shown_time = ~3;
          strcpy (timestr, "  :  :  ");
          break;
        }
        msecs = v[2] - msecs;
      }
      if (msecs == panel->shown_time)
        break;
      panel->shown_time = msecs;
      flags |= _PURD_TIME;
      speed = xine_get_param (panel->nw.gui->stream, XINE_PARAM_FINE_SPEED);
      if (speed > XINE_FINE_SPEED_NORMAL * 3 / 4) {
        /* find a stream step that is a multiple of, and plays roughly, 0.5s. */
        step = 500u * (((unsigned int)speed + (XINE_FINE_SPEED_NORMAL >> 1)) / XINE_FINE_SPEED_NORMAL);
        /* allow up to 4 updates/s on fast play. */
        if (step > 1000)
          step >>= 1;
        /* round display value. */
        secs = (msecs + 250u) / 1000u;
        /* preferred next stop stream time. */
        step = panel->nw.gui->runtime_mode
              ? msecs + step - (msecs + (step >> 1)) / step * step
              : step + (msecs + (step >> 1)) / step * step - msecs;
        /* and in real time. */
        step = step * XINE_FINE_SPEED_NORMAL / speed;
        timestr[8] = 0;
        /* printf ("xine-ui: msecs=%u, step=%u.\n", msecs, step); */
      } else {
        secs   = msecs / 1000u;
        msecs %= 1000u;
        /* millisecond information is useful when editing. since we only update */
        /* twice a second, show this in slow mode only. */
        timestr[12] = 0;
        timestr[11] = (msecs % 10u) + '0';
        msecs /= 10u;
        timestr[10] = (msecs % 10u) + '0';
        timestr[9] = (msecs / 10u) + '0';
        timestr[8] = '.';
      }
      secs2str (timestr, secs, v[2]);
    } while (0);

    if ((unsigned int)v[2] != panel->shown_length) {
      const char *ts = _("Total time: ");
      size_t tl = strlen (ts);
      if (tl > sizeof (tips) - 9)
        tl = sizeof (tips) - 9;
      memcpy (tips, ts, tl);
      panel->shown_length = v[2];
      flags |= _PURD_TIPS;
      if (v[2] >= 0) {
        tips[tl + 8] = 0;
        secs2str (tips + tl, v[2] / 1000, v[2]);
        flags |= _PURD_LENGTH;
      } else {
        strcpy (tips + tl, " ?:??:??");
      }
    }
    if (flags & (_PURD_NEW_STREAM | _PURD_LENGTH)) {
      /* including the rare case: new mrl with same length
       * (most likely, the same item again).
       * avoid always bothering the playlist. */
      if ((v[2] >= 0) && (res <= 2)) {
        if (gui_playlist_set_length (panel->nw.gui, v[2]) > 1)
          flags |= _PURD_PLAYLIST;
        if (flags & _PURD_NEW_STREAM) {
          pthread_mutex_lock (&panel->sl.mutex);
          panel->new_mrl = 0;
          pthread_mutex_unlock (&panel->sl.mutex);
        }
      }
    }
    if (v[0] < 0)
      flags &= ~_PURD_SLIDER;

  } else {
    /* panel hidden and/or no stream info. */
    flags &= ~(_PURD_SLIDER | _PURD_TIME);
  }

  if (flags & (_PURD_SLIDER | _PURD_TIME | _PURD_TITLE | _PURD_PLAYLIST | _PURD_TIPS)) {
    if (flags & _PURD_LOCK)
      xitk_lock (panel->nw.gui->xitk, 1);
    if (flags & _PURD_TIME)
      xitk_label_change_label (panel->w[_W_runtime], timestr);
    if (flags & _PURD_SLIDER)
      xitk_slider_set_pos (panel->w[_W_slider_play], v[0]);
    if (flags & _PURD_TITLE)
      _panel_show_mrl (panel);
    if (flags & _PURD_TIPS)
      xitk_set_widget_tips (panel->w[_W_runtime], tips);
    if (flags & _PURD_PLAYLIST)
      playlist_update_playlist (panel->nw.gui);
    if (flags & _PURD_LOCK)
      xitk_lock (panel->nw.gui->xitk, 0);
  }

  return step;
}

void panel_message (xui_panel_t *panel, const char *message) {
  if (!panel)
    return;
  if (message) {
    unsigned int stime;
    struct timespec ts = {0, 0};
    /* Update ui gfx smoothly, at most 20 times a second. */
    xitk_gettime_ts (&ts);
    stime = ts.tv_nsec / (1000000000 / 20) + (ts.tv_sec & 0x00ffffff) * 20;
    pthread_mutex_lock (&panel->sl.mutex);
    if (stime == panel->sl.message_time) {
      pthread_mutex_unlock (&panel->sl.mutex);
      return;
    }
    panel->sl.message_time = stime;
    panel->sl.message_hold = panel->tips.timeout / 500;
    pthread_mutex_unlock (&panel->sl.mutex);
    xitk_label_change_label (panel->w[_W_title], message);
    return;
  } else {
    pthread_mutex_lock (&panel->sl.mutex);
    panel->new_mrl = _PURD_NEW_STREAM;
    if (panel->sl.message_hold > 0) {
      pthread_mutex_unlock (&panel->sl.mutex);
      return;
    }
    pthread_mutex_unlock (&panel->sl.mutex);
  }
  _panel_show_mrl (panel);
}

/*
 * Update slider thread.
 */
static void *slider_loop (void *data) {
  xui_panel_t *panel = data;
  int screensaver_timer = 0, lastmsecs = -1, it = 19, step = 0;
  struct timespec wtime = {0, 0};

  xitk_vers_string_init (&panel->sl.istr, NULL, 0);
  panel->sl.ident = stream_infos_ident_new ();

  while (1) {
    uint32_t flags = _PURD_LOCK | _PURD_SLIDER;
    int hold;

    if (--it < 0)
      it = 19;

    if (step > 0) {
      wtime.tv_nsec += step * 1000000;
      if (wtime.tv_nsec >= 1000000000) {
        wtime.tv_nsec -= 1000000000;
        wtime.tv_sec += 1;
      }
      pthread_mutex_lock (&panel->sl.mutex);
      if (!panel->sl.run) {
        pthread_mutex_unlock (&panel->sl.mutex);
        break;
      }
      pthread_cond_timedwait (&panel->sl.wake, &panel->sl.mutex, &wtime);
      flags |= panel->new_mrl;
      if (panel->sl.message_hold > 1) {
        /* hold message. */
        panel->sl.message_hold--;
      } else if ((panel->sl.message_hold == 1) && panel->nw.gui->stream) {
        /* switch back to stream info if we got 1. */
        panel->sl.message_hold = 0;
        flags |= _PURD_TITLE;
      }
    } else {
      pthread_mutex_lock (&panel->sl.mutex);
    }
    hold = panel->sl.message_hold;
    pthread_mutex_unlock (&panel->sl.mutex);

    xitk_gettime_ts (&wtime);
    step = 500;
    if (panel->nw.gui->stream) {
      int status, speed;

      if (!panel->nw.gui->logo_mode && (_panel_get_visibility (panel) > 1)) {
        step = panel_update_runtime_display (panel, flags);
        if (it == 0) {
          int level = panel->nw.gui->mixer.level[panel->nw.gui->mixer.type_volume] =
            xine_get_param (panel->nw.gui->stream, _xine_volume_param[panel->nw.gui->mixer.type_volume]);

          xitk_lock (panel->nw.gui->xitk, 1);
          xitk_slider_set_pos (panel->w[_W_slider_mixer], level);
          panel_update_channel_display (panel);
          xitk_button_set_state (panel->w[_W_mute], panel->nw.gui->mixer.mute[panel->nw.gui->mixer.type_mute]);
          if (panel->nw.gui->flags & XUI_FLAG_sinfo_update)
            stream_infos_update_infos (panel->nw.gui->streaminfo);
          xitk_lock (panel->nw.gui->xitk, 0);
        }
      } else if (flags & _PURD_TITLE) {
        xitk_lock (panel->nw.gui->xitk, 1);
        _panel_show_mrl (panel);
        xitk_lock (panel->nw.gui->xitk, 0);
      }

      status = xine_get_status (panel->nw.gui->stream);
      speed  = xine_get_param (panel->nw.gui->stream, XINE_PARAM_SPEED);
      if (status == XINE_STATUS_PLAY) {
        int v[3];
        /* the partial play feature. */
        v[1] = 0;
        if ((panel->nw.gui->mmk.end != -1)
          && (gui_xine_get_pos_length (panel->nw.gui, panel->nw.gui->stream, v) == 1)) {
          /* pthread_mutex_lock (&panel->nw.gui->xe_mutex); */
          if (panel->nw.gui->playlist.num && panel->nw.gui->playlist.cur >= 0 && panel->nw.gui->playlist.mmk &&
            panel->nw.gui->playlist.mmk[panel->nw.gui->playlist.cur]) {
            if ((v[1] / 1000) >= panel->nw.gui->playlist.mmk[panel->nw.gui->playlist.cur]->end) {
              panel->nw.gui->ignore_next = 0;
              gui_playlist_start_next (panel->nw.gui, 0);
              /* pthread_mutex_unlock (&panel->nw.gui->xe_mutex); */
              continue;
	    }
	  }
          /* pthread_mutex_unlock (&panel->nw.gui->xe_mutex); */
        }
        /* seekability */
        if (!(it & 1)) {
          int seekable = xine_get_stream_info (panel->nw.gui->stream, XINE_STREAM_INFO_SEEKABLE);
          if (!xitk_lock (panel->nw.gui->xitk, 2)) {
            if (seekable) {
              xitk_widgets_state (panel->w + _W_slider_play, 1, XITK_WIDGET_STATE_ENABLE, ~0u);
            } else if (xitk_is_widget_enabled (panel->w[_W_slider_play])) {
              xitk_slider_reset (panel->w[_W_slider_play]);
              xitk_widgets_state (panel->w + _W_slider_play, 1, XITK_WIDGET_STATE_ENABLE, 0);
            }
            xitk_lock (panel->nw.gui->xitk, 0);
          }
        }
      }
      /* timeout ticks */
      if (it & 1)
        osd_tick (panel->nw.gui);

      if ((status == XINE_STATUS_PLAY) &&
        ((speed != XINE_SPEED_PAUSE) || ((int)panel->shown_time != lastmsecs))) {

        if (stream_infos_ident_get (panel->sl.ident, &panel->sl.istr, panel->nw.gui->stream)) {
          if (!gui_playlist_trylock (panel->nw.gui)) {
            if (panel->sl.istr.s && panel->sl.istr.s[0]) {
              if (gui_playlist_set_str_val (panel->nw.gui, panel->sl.istr.s, MMK_VAL_IDENT, GUI_MMK_CURRENT)) {
                mediamark_t *m = &panel->nw.gui->mmk;
                mediamark_set_str_val (&m, panel->sl.istr.s, MMK_VAL_IDENT);
                gui_playlist_unlock (panel->nw.gui);
                xitk_lock (panel->nw.gui->xitk, 1);
                video_window_set_mrl (panel->nw.gui->vwin, panel->sl.istr.s);
                playlist_update_playlist (panel->nw.gui);
                if (hold <= 0)
                  _panel_show_mrl (panel);
                xitk_lock (panel->nw.gui->xitk, 0);
              } else {
                gui_playlist_unlock (panel->nw.gui);
              }
            } else {
              char buf[2048];
              strlcpy (buf, panel->nw.gui->mmk.mrl, sizeof (buf));
              gui_playlist_unlock (panel->nw.gui);
              xitk_lock (panel->nw.gui->xitk, 1);
              video_window_set_mrl (panel->nw.gui->vwin, buf);
              xitk_lock (panel->nw.gui->xitk, 0);
            }
          }
        }

        {
          if (video_window_is_visible (panel->nw.gui->vwin) > 1) {
            if (panel->nw.gui->ssaver_timeout > 1) {
              if (!(it & 1))
                screensaver_timer++;
              if (screensaver_timer >= panel->nw.gui->ssaver_timeout) {
                xitk_lock (panel->nw.gui->xitk, 1);
                screensaver_timer = video_window_reset_ssaver (panel->nw.gui->vwin, 1);
                xitk_lock (panel->nw.gui->xitk, 0);
              }
            }
          }
        }

      }
      lastmsecs = panel->shown_time;
    }

    if ((panel->nw.gui->cursor_visible > 0) && (it & 1)) {
      xitk_lock (panel->nw.gui->xitk, 1);
      if (panel->nw.gui->cursor_visible > 0) {
        if (--panel->nw.gui->cursor_visible == 0)
          video_window_set_cursor_visibility (panel->nw.gui->vwin, 0);
      }
      xitk_lock (panel->nw.gui->xitk, 0);
    }

    if (panel->nw.gui->logo_has_changed)
      _update_logo (panel);
  }

  stream_infos_ident_delete (&panel->sl.ident);
  xitk_vers_string_deinit (&panel->sl.istr);

  return NULL;
}

/*
 * Boolean about panel visibility.
 */
int panel_is_visible (xui_panel_t *panel) {
  return panel ? _panel_get_visibility (panel) : 0;
}

/*
 * Show/Hide panel window.
 */
void panel_toggle_visibility (xitk_widget_t *w, void *data) {
  xui_panel_t *panel = data;

  (void)w;
  if (!panel)
    return;
  if (_panel_get_visibility (panel) > 1) {
    panel->visible = panel->nw.gui->use_root_window
                  || (video_window_is_visible (panel->nw.gui->vwin) <= 0)
                  || video_window_is_separate_display (panel->nw.gui->vwin);
    xitk_window_flags (panel->nw.xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, panel->visible * XITK_WINF_ICONIFIED);
    video_window_grab_input_focus(panel->nw.gui->vwin);
  } else {
    panel->visible = 2;
    panel->nw.gui->nongui_error_msg = NULL;
    xitk_window_flags (panel->nw.xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE);
  }
  config_update_num (panel->nw.gui->xine, "gui.panel_visible", panel->visible > 1);
}

void panel_raise_window (xui_panel_t *panel) {
  if (panel_is_visible (panel) > 1)
    xitk_window_raise_window (panel->nw.xwin);
}

void panel_get_window_position (xui_panel_t *panel, xitk_rect_t *r) {
  if (panel && r)
    *r = panel->nw.wr;
}

/*
 * Check and set the correct state of pause button.
 */
void panel_check_pause (xui_panel_t *panel) {

  if (!panel)
    return;

  xitk_button_set_state (panel->w[_W_pause],
    (((xine_get_status (panel->nw.gui->stream) == XINE_STATUS_PLAY) &&
      (xine_get_param (panel->nw.gui->stream, XINE_PARAM_SPEED) == XINE_SPEED_PAUSE)) ? 1 : 0));
}

static void _panel_change_display_mode(xitk_widget_t *w, void *data) {
  xui_panel_t *panel = data;

  (void)w;
  panel->nw.gui->is_display_mrl = !panel->nw.gui->is_display_mrl;
  _panel_show_mrl (panel);
  playlist_update_playlist (panel->nw.gui);
}

static void _panel_change_time_label(xitk_widget_t *w, void *data) {
  xui_panel_t *panel = data;

  (void)w;
  panel->nw.gui->runtime_mode = !panel->nw.gui->runtime_mode;
  panel_update_runtime_display (panel, 0);
}

/*
 * Reset the slider of panel window (set to 0).
 */
void panel_reset_slider (xui_panel_t *panel) {
  if (!panel)
    return;
  xitk_slider_reset(panel->w[_W_slider_play]);
  xitk_label_change_label (panel->w[_W_runtime], panel->nw.gui->runtime_mode == 0 ? " 0:00:00" : " .:..:..");
}

/*
 * Update audio/spu channel displayed informations.
 */
void panel_update_channel_display (xui_panel_t *panel) {
  char buf[XINE_LANG_MAX];

  if (!panel)
    return;

  gui_lang_str (panel->nw.gui, buf, 0);
  xitk_label_change_label (panel->w[_W_audio], buf);

  gui_lang_str (panel->nw.gui, buf, 1);
  xitk_label_change_label (panel->w[_W_spu], buf);
}

static void panel_lang_list (xitk_widget_t *w, void *data) {
  xui_panel_t *panel = data;
  xitk_rect_t wr = {0, 0, 0, 0};

  xitk_widget_get_pos (w, &wr);
  lang_menu (panel->nw.gui, panel->nw.wl, w == panel->w[_W_spu], wr.x, wr.y + wr.height);
}

void panel_update_mixer_display (xui_panel_t *panel) {
  if (panel) {
    gGui_t *gui = panel->nw.gui;
    int max = _panel_mixer_max[gui->mixer.type_volume], vol = gui->mixer.level[gui->mixer.type_volume];

    xitk_slider_set_range (panel->w[_W_slider_mixer], 0, max, max / 100);
    xitk_slider_set_pos (panel->w[_W_slider_mixer], vol);
    xitk_button_set_state (panel->w[_W_mute], gui->mixer.mute[gui->mixer.type_mute]);
  }
}

/*
 *
 */
void panel_toggle_audio_mute(xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  static const int _param[LAST_MIXER] = {
    [SOUND_CARD_MIXER] = XINE_PARAM_AUDIO_MUTE,
    [SOFTWARE_MIXER] = XINE_PARAM_AUDIO_AMP_MUTE
  };
  xui_panel_t *panel = data;
  xui_mixer_type_t type = panel->nw.gui->mixer.type_mute;

  (void)w;
  (void)modifier;
  if (state < 0)
    state = !panel->nw.gui->mixer.mute[type];
  panel->nw.gui->mixer.mute[type] = state;
  xine_set_param (panel->nw.gui->stream, _param[type], state);
  osd_message (panel->nw.gui, type == SOFTWARE_MIXER ? _("Amp: %s") : _("Audio: %s"),
    state ? _("Muted") : _("Unmuted"));
  xitk_button_set_state (panel->w[_W_mute], panel->nw.gui->mixer.mute[panel->nw.gui->mixer.type_mute]);
}

/*
 *  Use internal xine-lib framegrabber function
 *  to snapshot current frame.
 */
static void panel_snapshot_msg (gGui_t *gui, const char *text, int info0_warn1_err2) {
  static const uint8_t mode[3] = { XUI_MSG_INFO, XUI_MSG_WARN, XUI_MSG_ERROR };
  gui_msg (gui, mode[info0_warn1_err2], "%s", text);
}
void panel_snapshot (xitk_widget_t *w, void *data) {
  xui_panel_t *panel = data;

  (void)w;
  gui_playlist_lock (panel->nw.gui);
  create_snapshot (panel->nw.gui, panel->nw.gui->mmk.mrl, panel_snapshot_msg);
  gui_playlist_unlock (panel->nw.gui);
}

/*
 * Handle paddle moving of slider.
 */
static void panel_slider_play_cb (xitk_widget_t *w, void *data, int pos, unsigned int modifier) {
  xui_panel_t *panel = data;
  unsigned int stime;
  struct timespec ts = {0, 0};

  (void)w;
  (void)modifier;
  /* Update ui gfx smoothly, at most 20 times a second (~ 5 times less than before).
   * There is hardly a visible difference, but a lot less X action - especially
   * with KDE window thumbnails on. Slider thread will fix the final position. */
  xitk_gettime_ts (&ts);
  stime = ts.tv_nsec / (1000000000 / 20) + (ts.tv_sec & 0x00ffffff) * 20;
  /* printf ("panel_slider_play_cb: timeslot #%u.\n", stime); */
  /* always true inside slider callback.
  if (xitk_is_widget_enabled (panel->w[_W_slider_play])) */
  {
    gui_set_current_position (panel->nw.gui, pos);
    if (stime != panel->playback_widgets.slider_play_time) {
      panel->playback_widgets.slider_play_time = stime;
      if (xine_get_status (panel->nw.gui->stream) != XINE_STATUS_PLAY) {
        panel_reset_slider (panel);
        xitk_button_set_state (panel->w[_W_pause], 0);
      } else {
        /*
        int v[3];
        if (gui_xine_get_pos_length (panel->nw.gui, panel->nw.gui->stream, v))
          xitk_slider_set_pos (panel->w[_W_slider_play], v[0]); */
        panel_update_runtime_display (panel, 0);
        xitk_button_set_state (panel->w[_W_pause],
          (xine_get_param (panel->nw.gui->stream, XINE_PARAM_SPEED) == XINE_SPEED_PAUSE) ? 1 : 0);
      }
    }
  }
}

static void panel_slider_mixer_cb (xitk_widget_t *w, void *data, int pos, unsigned int modifier) {
  xui_panel_t *panel = data;

  (void)w;
  (void)modifier;
  gui_set_audio_vol (panel->nw.gui, pos);
}

/*
 * Handle X events here.
 */
static int panel_event (void *data, const xitk_be_event_t *e) {
  xui_panel_t *panel = data;

  switch (e->type) {
    case XITK_EV_DEL_WIN:
      gui_exit (NULL, panel->nw.gui);
      return 1;
    case XITK_EV_SHOW:
      if (panel->nw.gui->cursor_grabbed)
        video_window_grab_pointer (panel->nw.gui->vwin, 0);
      /* paranoia: if panel is completely off screen, center it. */
      {
        uint32_t flags = video_window_mode (panel->nw.gui->vwin, TOGGLE_MODE);
        if ((flags & (WINDOWED_MODE | FULLSCR_MODE)) && !(flags & FULLSCR_XI_MODE)) {
          xitk_rect_t wr = {0, 0, 0, 0};
          int desktopw, desktoph;
          xitk_window_get_window_position (panel->nw.xwin, &wr);
          xitk_get_display_size (panel->nw.gui->xitk, &desktopw, &desktoph);
          if (((wr.x + wr.width) <= 0) || ((wr.y + wr.height) <= 0) || (wr.x >= desktopw) || (wr.y >= desktoph)) {
            wr.x = (desktopw - wr.width) >> 1;
            wr.y = (desktoph - wr.height) >> 1;
            wr.width = wr.height = XITK_INT_KEEP;
            xitk_window_move_resize (panel->nw.xwin, &wr);
          }
        }
      }
      if (panel->nw.gui->logo_mode == 0) {
        if (panel->playback_widgets.enabled)
          panel_update_runtime_display (panel, _PURD_SLIDER);
      }
      /* also: restart label anim. */
      if (_panel_test_message (panel))
        _panel_show_mrl (panel);
      return 1;
    case XITK_EV_POS_SIZE:
      panel->nw.wr.x = e->x;
      panel->nw.wr.y = e->y;
      panel->nw.wr.width = e->w;
      panel->nw.wr.height = e->h;
      event_sender_move (panel->nw.gui);
      return 1;
    case XITK_EV_BUTTON_DOWN:
      if (e->code == 3) {
        xitk_widget_t *w = xitk_get_focused_widget (panel->nw.wl);

        if (w == panel->w[_W_slider_mixer]) {
          volume_menu (panel->nw.gui, panel->nw.wl, e->w, e->h);
          return 1;
        }
      }
      break;
    case XITK_EV_KEY_DOWN:
      if (e->utf8[0] == XITK_CTRL_KEY_PREFIX) {
        switch (e->utf8[1]) {
          case XITK_KEY_MENU:
            {
              xitk_widget_t *w = xitk_get_focused_widget (panel->nw.wl);

              if (w == panel->w[_W_slider_mixer]) {
                volume_menu (panel->nw.gui, panel->nw.wl, e->w, e->h);
                return 1;
              }
            }
          default: ;
        }
      }
    default: ;
  }
  return gui_handle_be_event (panel->nw.gui, e);
}

void panel_init2 (xui_panel_t *panel) {
  const char * const *autoplay_plugins;
  const char *name[64], *tips[64];
  char buf[2][XINE_LANG_MAX];
  int vol, mute;
  xitk_button_list_widget_t nbl;

  if (!panel)
    return;

  /* Check if there a mixer control available, */
  if (panel->nw.gui->ao_port) {
    mute = xine_get_param (panel->nw.gui->stream, XINE_PARAM_AUDIO_MUTE);
    vol = xine_get_param (panel->nw.gui->stream, XINE_PARAM_AUDIO_VOLUME);
  } else {
    vol = mute = -1;
  }
  panel->nw.gui->mixer.mute[SOUND_CARD_MIXER] = mute;
  panel->nw.gui->mixer.level[SOUND_CARD_MIXER] = vol;
  if (mute < 0)
    panel->nw.gui->mixer.type_mute = SOFTWARE_MIXER;
  panel->nw.gui->mixer.mute[SOFTWARE_MIXER] = xine_get_param (panel->nw.gui->stream, XINE_PARAM_AUDIO_AMP_MUTE);
  panel->nw.gui->mixer.level[SOFTWARE_MIXER] = xine_get_param (panel->nw.gui->stream, XINE_PARAM_AUDIO_AMP_LEVEL);

  gui_lang_str (panel->nw.gui, buf[0], 0);
  gui_lang_str (panel->nw.gui, buf[1], 1);

  /* Create buttons from information if input plugins autoplay featured.
   * We couldn't do this into panel_init(), this function is called before xine engine initialization. */
  autoplay_plugins = xine_get_autoplay_input_plugin_ids (panel->nw.gui->xine);
  if (autoplay_plugins) {
    unsigned int i, n;

    memset (&nbl, 0, sizeof (nbl));
    nbl.nw.wl = panel->nw.wl;
    nbl.nw.skin_element_name = "AutoPlayGUI";
    nbl.nw.tips = _("More sources...");
    nbl.nw.userdata = panel->nw.gui;
    nbl.nw.add_state = XITK_WIDGET_STATE_KEEP;
    nbl.callback = playlist_scan_input_w;
    nbl.names = name;
    nbl.tips = tips;

    for (n = 0; autoplay_plugins[n]; n++) ;
    if (n > sizeof (tips) / sizeof (tips[0]) - 1)
      n = sizeof (tips) / sizeof (tips[0]) - 1;
    for (i = 0; i < n; i++)
      tips[i] = xine_get_input_plugin_description (panel->nw.gui->xine, name[i] = autoplay_plugins[i]);
    name[n] = tips[n] = NULL;
  }

  {
    int visible = xine_config_register_bool (panel->nw.gui->xine, "gui.panel_visible", 1,
      _("gui panel visibility"),
      CONFIG_NO_HELP, CONFIG_LEVEL_DEB, CONFIG_NO_CB, CONFIG_NO_DATA);

    panel->visible = (panel->nw.gui->flags & XUI_FLAG_no_gui) ? 0 : visible;
  }
  /*  The user don't want panel on startup */
  if (panel->visible && (panel->nw.gui->flags & XUI_FLAG_start_vis)) {
    panel->visible = 0;
    config_update_num (panel->nw.gui->xine, "gui.panel_visible", 0);
  }

  /* beware of xitk event bridge thead. */
  xitk_lock (panel->nw.gui->xitk, 1);

  xitk_widgets_state (panel->w + _W_mute, 1, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_ON,
    XITK_WIDGET_STATE_ENABLE | (panel->nw.gui->mixer.mute[panel->nw.gui->mixer.type_mute] ? XITK_WIDGET_STATE_ON : 0));
  xitk_widgets_state (panel->w + _W_slider_mixer, 1, XITK_WIDGET_STATE_ENABLE, ~0u);
  xitk_slider_set_pos (panel->w[_W_slider_mixer], panel->nw.gui->mixer.level[panel->nw.gui->mixer.type_volume]);

  xitk_label_change_label (panel->w[_W_audio], buf[0]);
  xitk_label_change_label (panel->w[_W_spu], buf[1]);

  panel_message (panel, NULL);
  /* panel still is hidden, so this does nothing.
  panel_update_runtime_display (panel, 0); */

  if (autoplay_plugins)
    panel->w[_W_autoplay] = xitk_button_list_new (&nbl, panel->nw.gui->skin_config);

  /* this finally shows the panel (see panel_init ()). */
  if (panel->visible) {
    panel->visible = 2;
    gui_raise_window (panel->nw.gui, panel->nw.xwin);
  } else {
    panel->visible = panel->nw.gui->use_root_window
                  || (video_window_is_visible (panel->nw.gui->vwin) <= 0)
                  || video_window_is_separate_display (panel->nw.gui->vwin);
    xitk_window_flags (panel->nw.xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, panel->visible * XITK_WINF_ICONIFIED);
  }

  xitk_lock (panel->nw.gui->xitk, 0);

  panel->nw.gui->cursor_visible = 1;
  video_window_set_cursor_visibility (panel->nw.gui->vwin, 1);
}

void panel_deinit (xui_panel_t *panel) {
  if (!panel)
    return;

  config_update_num (panel->nw.gui->xine, "gui.panel_visible", _panel_get_visibility (panel) > 1);
#ifdef HAVE_XINE_CONFIG_UNREGISTER_CALLBACKS
  xine_config_unregister_callbacks (panel->nw.gui->xine, NULL, NULL, panel, sizeof (*panel));
#endif
  pthread_mutex_lock (&panel->sl.mutex);
  panel->sl.run = 0;
  pthread_cond_signal (&panel->sl.wake);
  pthread_mutex_unlock (&panel->sl.mutex);
  pthread_join (panel->sl.thread, NULL);
  pthread_cond_destroy (&panel->sl.wake);
  pthread_mutex_destroy (&panel->sl.mutex);

  gui_window_delete (&panel->nw);
  panel->nw.gui->panel = NULL;
  free (panel);
}

void panel_paint (xui_panel_t *panel) {
  if (panel_is_visible (panel) > 1)
    xitk_partial_paint_widget_list (panel->nw.wl, NULL);
}


/*
 * Create the panel window, and fit all widgets in.
 */
xui_panel_t *panel_init (gGui_t *gui) {
  xui_panel_t              *panel;

  if (!gui)
    return NULL;

  panel = calloc (1, sizeof (*panel));
  if (!panel)
    return NULL;

  panel->nw.gui = gui;

  panel->shown_length = ~3;
  panel->shown_time = ~3;

  panel->playback_widgets.enabled = -1;

  /* open the panel window */
  panel->nw.title = _("xine Panel");
  panel->nw.id = "panel";
  panel->nw.skin = "BackGround";
  panel->nw.wfskin = "WinFocus";
  panel->nw.adjust = NULL;
  panel->nw.wr.x = 200;
  panel->nw.wr.y = 100;
  {
    int r = gui_window_new (&panel->nw);
    if (r < 0) {
      gui_msg (panel->nw.gui, XUI_MSG_ERROR,
        r == -3 ? _("panel: couldn't find image for background\n")
                : _("panel: couldn't create window\n"));
      free (panel);
      return NULL;
    }
  }

  {
    xitk_button_widget_t b = {
      .nw = {
        .wl = panel->nw.wl,
        .userdata = GUI_PREV (panel->nw.gui),
        .add_state = XITK_WIDGET_STATE_KEEP,
        .skin_element_name = "Prev"
      },
      .callback = gui_nextprev_mrl
    };

    panel->w[_W_prev] = xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "AudioPrev";
    b.nw.tips = _("Previous audio channel");
    b.callback = gui_nextprev_audio_channel;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "SpuPrev";
    b.nw.tips = _("Previous SPU channel");
    b.callback = gui_nextprev_spu_channel;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "PlaySlow";
    b.nw.tips = _("Slow motion");
    b.callback = gui_nextprev_speed;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.userdata = GUI_NEXT (panel->nw.gui);

    b.nw.skin_element_name = "Next";
    b.nw.tips = NULL;
    b.callback = gui_nextprev_mrl;
    panel->w[_W_next] = xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "AudioNext";
    b.nw.tips = _("Next audio channel");
    b.callback = gui_nextprev_audio_channel;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "SpuNext";
    b.nw.tips = _("Next SPU channel");
    b.callback = gui_nextprev_spu_channel;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "PlayFast";
    b.nw.tips = _("Fast motion");
    b.callback = gui_nextprev_speed;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.userdata = panel->nw.gui;

    b.nw.skin_element_name = "Stop";
    b.nw.tips = _("Stop playback");
    b.callback = gui_stop;
    panel->w[_W_stop] =  xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "Play";
    b.nw.tips = _("Play selected entry");
    b.callback = gui_play;
    panel->w[_W_play] = xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "Eject";
    b.nw.tips = _("Eject current medium");
    b.callback = gui_eject;
    panel->w[_W_eject] = xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "Exit";
    b.nw.tips = _("Quit");
    b.callback = gui_exit;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "Setup";
    b.nw.tips = _("Setup window");
    b.callback = setup_main;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "Nav";
    b.nw.tips = _("Navigator");
    b.callback = event_sender_main;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "FullScreen";
    b.nw.tips = _("Fullscreen/Window mode");
    b.callback = gui_set_fullscreen_mode;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "CtlBtn";
    b.nw.tips = _("Control");
    b.callback = control_main;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "AudioCtlBtn";
    b.nw.tips = _("Audio control");
    b.callback = acontrol_main;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "MrlBtn";
    b.nw.tips = _("Open Location");
    b.callback = mrl_browser_main;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "PlBtn";
    b.nw.tips = _("Playlist");
    b.callback = playlist_main;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.userdata = panel;

    b.nw.skin_element_name = "Close";
    b.nw.tips = _("Hide GUI");
    b.callback = panel_toggle_visibility;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "Snapshot";
    b.nw.tips = _("Take a snapshot");
    b.callback = panel_snapshot;
    xitk_button_create (&b, panel->nw.gui->skin_config);

    b.callback = NULL;

    b.nw.skin_element_name = "Pause";
    b.nw.tips = _("Pause/Resume playback");
    b.nw.userdata          = panel->nw.gui;
    b.nw.add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_IMMEDIATE;
    b.state_callback = gui_pause;
    panel->w[_W_pause] = xitk_button_create (&b, panel->nw.gui->skin_config);

    b.nw.skin_element_name = "Mute";
    b.nw.tips = _("Mute toggle");
    b.nw.userdata          = panel;
    b.nw.add_state = XITK_WIDGET_STATE_VISIBLE;
    b.state_callback = panel_toggle_audio_mute;
    panel->w[_W_mute] = xitk_button_create (&b, panel->nw.gui->skin_config);
  }

  /*
   * Init to default, otherwise if panel is hide
   * at startup, label is empty 'till it's updated
   */
  panel->nw.gui->runtime_mode   = 0;

  {
    xitk_label_widget_t lbl = {
      .nw = {
        .wl        = panel->nw.wl,
        .userdata  = panel,
        .add_state = XITK_WIDGET_STATE_KEEP,
        .skin_element_name = "TimeLabel",
        .tips = _("Total time: --:--:--")
      },
      .label = "00:00:00",
      .callback = _panel_change_time_label
    };

    panel->w[_W_runtime] = xitk_label_create (&lbl, panel->nw.gui->skin_config);

    lbl.label = "\x07";
    panel->sl.message_time = 0;
    panel->sl.message_hold = 3;

    lbl.nw.skin_element_name = "TitleLabel";
    lbl.nw.tips = _("show MRL or ident");
    lbl.callback = _panel_change_display_mode;
    panel->w[_W_title] =  xitk_label_create (&lbl, panel->nw.gui->skin_config);

    lbl.nw.skin_element_name = "AudioLabel";
    lbl.nw.tips = _("Audio/Channel"); /** << same as in menus.c/video_window_menu (). */
    lbl.callback = panel_lang_list;
    panel->w[_W_audio] = xitk_label_create (&lbl, panel->nw.gui->skin_config);

    lbl.nw.skin_element_name = "SpuLabel";
    lbl.nw.tips = _("Subtitle/Channel"); /** << same as in menus.c/video_window_menu (). */
    panel->w[_W_spu] =  xitk_label_create (&lbl, panel->nw.gui->skin_config);
  }

  {
    xitk_slider_widget_t sl = {
      .nw = {
        .wl = panel->nw.wl,
        .skin_element_name = "SliderPlay",
        .add_state = panel->nw.gui->playlist.num > 0
                   ? (XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ENABLE)
                   : XITK_WIDGET_STATE_VISIBLE,
        .mode_mask = WIDGET_KEEP_FOCUS,
        .mode_value = 0,
        .tips = _("Stream playback position slider"),
        .userdata = panel
      },
      .max = 65535,
      .step = 65535 / 50,
      .type = XITK_HSLIDER,
      .motion_callback = panel_slider_play_cb
    };

    panel->w[_W_slider_play] = xitk_slider_create (&sl, panel->nw.gui->skin_config);

    sl.nw.skin_element_name = "SliderVol";
    sl.nw.tips = _("Volume control");
    sl.nw.add_state = XITK_WIDGET_STATE_VISIBLE;
    sl.nw.mode_mask = 0;
    sl.max             = _panel_mixer_max[panel->nw.gui->mixer.type_volume];
    sl.step            = sl.max / 100;
    sl.type            = XITK_VSLIDER;
    sl.motion_callback = panel_slider_mixer_cb;
    panel->w[_W_slider_mixer] = xitk_slider_create (&sl, panel->nw.gui->skin_config);
  }

  xitk_window_set_role (panel->nw.xwin, XITK_WR_VICE);

  panel->tips.enable = xine_config_register_bool (panel->nw.gui->xine, "gui.tips_visible", 1,
    _("gui tips visibility"),
    _("If disabled, no tooltips are shown."),
    CONFIG_LEVEL_ADV, panel_enable_tips_cb, panel);
  panel->tips.timeout = xine_config_register_num (panel->nw.gui->xine, "gui.tips_timeout", 5000,
    _("Tips timeout (ms)"),
    _("Persistence time of tooltips, in milliseconds."),
    CONFIG_LEVEL_ADV, panel_timeout_tips_cb, panel);
  if (panel->tips.timeout < 0)
    panel->tips.timeout = 0;

  pthread_mutex_init (&panel->sl.mutex, NULL);
  xitk_set_tips_timeout (panel->nw.gui->xitk, panel->tips.enable ? panel->tips.timeout : XITK_TIPS_TIMEOUT_OFF);
  /* _panel_show_mrl (panel); */
  panel_update_nextprev_tips (panel);

  /* NOTE: panel and video window follow a complex visibility logic.
   * defer this to panel_add_autoplay_buttons () when video win is stable. */
  panel->visible = 0;
  /* NOTE: do this _before_ xitk_window_flags (, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE),
   * and make sure to get that initial expose event handled in xitk.c -
   * otherwise some widgets may not show until focused. */
  xitk_window_flags (panel->nw.xwin,
    XITK_WINF_TASKBAR | XITK_WINF_PAGER | XITK_WINF_DND, XITK_WINF_TASKBAR | XITK_WINF_PAGER | XITK_WINF_DND);
  panel->nw.key = xitk_be_register_event_handler ("panel", panel->nw.xwin, panel_event, panel, NULL, NULL);

  pthread_cond_init (&panel->sl.wake, NULL);
  pthread_mutex_lock (&panel->sl.mutex);
  panel->sl.run = 1;
  pthread_mutex_unlock (&panel->sl.mutex);
  {
    pthread_attr_t       pth_attrs;
#if defined(_POSIX_THREAD_PRIORITY_SCHEDULING) && (_POSIX_THREAD_PRIORITY_SCHEDULING > 0)
    struct sched_param   pth_params;
#endif
    pthread_attr_init (&pth_attrs);
    /* this won't work on linux, freebsd 5.0 */
#if defined(_POSIX_THREAD_PRIORITY_SCHEDULING) && (_POSIX_THREAD_PRIORITY_SCHEDULING > 0)
    pthread_attr_getschedparam(&pth_attrs, &pth_params);
    pth_params.sched_priority = sched_get_priority_min(SCHED_OTHER);
    pthread_attr_setschedparam(&pth_attrs, &pth_params);
#endif
    pthread_create (&panel->sl.thread, &pth_attrs, slider_loop, panel);
    pthread_attr_destroy (&pth_attrs);
  }

  panel->nw.gui->panel = panel;
  return panel;
}

