/* XRACER (C) 1999-2000 Richard W.M. Jones <rich@annexia.org> and other AUTHORS
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: start_menu.c,v 1.9 2000/01/15 16:14:35 rich Exp $
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>

#include <GL/glut.h>

#include "xracer.h"
#include "xracer-log.h"
#include "xracer-mode.h"
#include "xracer-menu.h"
#include "xracer-game.h"
#include "xracer-craft.h"

/* Which craft is being displayed in the background. */
static const struct xrCraft *craft;

/* When did we start displaying this craft? */
static double craft_start_time;

#define CRAFT_TIME 12		/* Time per craft, seconds. */
#define CRAFT_ROTATE_TIME 3	/* Time craft takes to rotate, seconds. */

/* Program-level menu initialization. */
void
xrStartMenuInit ()
{
}

/* This function is called when we enter the menu mode. */
static void
start_menu (const void *args)
{
  craft = xrCraftGetDefault ();
  craft_start_time = xrCurrentTime;
}

/* This function is called when we leave the menu mode, to do clean up. */
static void
end_menu ()
{
}

static const char *items[] = {
  "Start game",
  "Load track",
  "Configuration",
  "Exit"
};

static void
start_game ()
{
  struct xrGameModeArgs game_args;

  game_args.mode = XR_GAME_MODE_TIME_TRIAL;
  game_args.nr_laps = 3;
  xrEnterMode (&xrGameMode, &game_args);
}

static void
load_track ()
{
  xrEnterMode (&xrLoadTrackMode, 0);
}

static void
configuration ()
{
  xrEnterMenu (&xrConfigurationMenu, 0);
}

static void
exit_program ()
{
  xrEnterMenu (&xrQuitMenu, 0);
}

static void (*actions[])(void) = {
  start_game,
  load_track,
  configuration,
  exit_program
};

static void
display_background ()
{
  GLfloat a;
  static GLfloat light[4] = { 3, 3, 0, 0 };
  static GLfloat ambient[4] = { 0.2, 0.2, 0.2, 0 };

  xrLogAssert (craft != 0);

  /* Time to start displaying a different craft? */
  if (xrCurrentTime > craft_start_time + CRAFT_TIME)
    {
      const struct xrCraft *new_craft = xrCraftGetNext (craft);

      if (craft != new_craft)
	{
	  craft_start_time = xrCurrentTime;
	  craft = new_craft;
	}
    }

  /* Create modelview transformation. */
  glMatrixMode (GL_PROJECTION);
  glPushMatrix ();
  glLoadIdentity ();
  gluLookAt (0, 0, 1,		/* eye */
	     0, 0, 0,		/* centre */
	     0, 1, 0);		/* up */

  /* Work out how many degrees to rotate it in the axis. Do this so that
   * it rotates once every CRAFT_ROTATE_TIME seconds.
   */
  a = 360 * (xrCurrentTime - craft_start_time) / CRAFT_ROTATE_TIME;

  /* Create projection. */
  glMatrixMode (GL_MODELVIEW);
  glPushMatrix ();
  glLoadIdentity ();

  /* Shrink it. */
  glScalef (0.25, 0.25, 0.25);

  /* Apply rotation around Y axis. */
  glRotatef (a, 0, 1, 0);

  /* Rotate craft 110 degrees in X axis so it is flat, nose slightly down. */
  /*glRotatef (110, 1, 0, 0);*/

  /* Set things up. */
  glShadeModel (GL_FLAT);
  /*glShadeModel (GL_SMOOTH);*/
  glEnable (GL_DEPTH_TEST);
  glEnable (GL_LIGHTING);
  glEnable (GL_LIGHT0);
  glLightfv (GL_LIGHT0, GL_POSITION, light);
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, ambient);

  /* Draw it. */
  xrCraftDisplay (craft);

  /* Clean up state. */
  glDisable (GL_LIGHT0);
  glDisable (GL_LIGHTING);
  glDisable (GL_DEPTH_TEST);
  /*glShadeModel (GL_FLAT);*/

  /* Restore matrices. */
  glMatrixMode (GL_MODELVIEW);
  glPopMatrix ();
  glMatrixMode (GL_PROJECTION);
  glPopMatrix ();
}

struct xrMenu xrStartMenu = {
  name: "start menu",
  title: "XRacer",
  footer: "Copyright (C) 1999-2000 Richard W.M. Jones and other AUTHORS",
  /* Or, if you prefer ...
   * footer: "Use Up and Down arrow keys to navigate, and Return to select",
   */
  nr_items: 4,
  items: items,
  actions: actions,
  display_background: display_background,
  start_menu: start_menu,
  end_menu: end_menu
};
