/**
   File : xsldbginspector.cpp
   Author : Keith Isdale
   Date : 30th March 2002
   Description : Dialog to inspect stylesheet using xsldbg. Based on
                  file created by uic
 */

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ************************************************************************* */

#include "xsldbginspector.h"

#include <klocale.h>

#include <QVariant>
#include <QPushButton>
#include <QTabWidget>
#include <QWidget>
#include <QLayout>
#include <QToolTip>

#include <QImage>
#include <QPixmap>
#include <KIcon>
#include <QGridLayout>
#include <QBoxLayout>

#include <kpushbutton.h>
#include <kstandardguiitem.h>
#include "xsldbgdebugger.h"
#include "xsldbgbreakpointsimpl.h"
#include "xsldbgvariablesimpl.h"
#include "xsldbgcallstackimpl.h"
#include "xsldbgtemplatesimpl.h"
#include "xsldbgsourcesimpl.h"
#include "xsldbgentitiesimpl.h"

/*
 *  Constructs a XsldbgInspector which is a child of 'parent', with the
 *  name 'name' and widget flags set to 'f'.
 *
 *  The dialog will by default be modeless, unless you set 'modal' to
 *  true to construct a modal dialog.
 */
XsldbgInspector::XsldbgInspector( XsldbgDebugger *debugger, QWidget* parent, bool modal)
    : QDialog( parent )
{
    setModal(modal);
    Q_CHECK_PTR(debugger);
    this->debugger = debugger;
    breakpointWidget = 0L;
    variablesWidget = 0L;
    callStackWidget = 0L;
    templateWidget = 0L;
    sourceWidget = 0L;
    entityWidget = 0L;
    resize( 597, 364 );
    setWindowTitle( i18n( "Xsldbg Inspector" ) );
    setSizeGripEnabled( true );
    XsldbgInspectorLayout = new QGridLayout(this);

    tabWidget = new QTabWidget( this );
    Q_CHECK_PTR( tabWidget );
    breakpointWidget = new XsldbgBreakpointsImpl( debugger, tabWidget );
    Q_CHECK_PTR( breakpointWidget );
    tabWidget->addTab( breakpointWidget, i18n( "Breakpoints" ) );

    variablesWidget = new  XsldbgVariablesImpl( debugger, tabWidget );
    Q_CHECK_PTR( variablesWidget );
    tabWidget->addTab( variablesWidget,
	  KIcon( "xsldbg_source" ),
	  i18n( "Variables" ) );

    callStackWidget = new  XsldbgCallStackImpl( debugger, tabWidget );
    Q_CHECK_PTR( callStackWidget );
    tabWidget->addTab( callStackWidget,
	  KIcon( "xsldbg_source" ),
	  i18n( "CallStack" ));

    templateWidget = new  XsldbgTemplatesImpl( debugger, tabWidget );
    Q_CHECK_PTR( templateWidget );
    tabWidget->addTab( templateWidget,
	  KIcon( "xsldbg_source" ),
          i18n( "Templates" ));

    sourceWidget = new  XsldbgSourcesImpl( debugger, tabWidget );
    Q_CHECK_PTR( sourceWidget );
    tabWidget->addTab( sourceWidget,
	  KIcon( "xsldbg_source" ),
	  i18n( "Sources" ));

    entityWidget = new  XsldbgEntitiesImpl( debugger, tabWidget );
    Q_CHECK_PTR( entityWidget );
    tabWidget->addTab( entityWidget,
	  KIcon( "xsldbg_data" ),
	  i18n( "Entities" ));

    XsldbgInspectorLayout->addWidget( tabWidget, 0, 1 );
    Layout1 = new QHBoxLayout;


    buttonOk = new KPushButton( KStandardGuiItem::ok(), this);
	buttonOk->setObjectName("buttonOk" );
    buttonOk->setShortcut( 0 );
    buttonOk->setAutoDefault( true );
    buttonOk->setDefault( true );
    Layout1->addWidget( buttonOk );

    buttonCancel = new KPushButton( KStandardGuiItem::cancel(), this);
	buttonCancel->setObjectName("buttonCancel" );
    buttonCancel->setShortcut( 0 );
    buttonCancel->setAutoDefault( true );
    Layout1->addWidget( buttonCancel );

    XsldbgInspectorLayout->addLayout( Layout1, 1, 1 );

    // signals and slots connections
    connect( buttonOk, SIGNAL( clicked() ), this, SLOT( accept() ) );
    connect( buttonCancel, SIGNAL( clicked() ), this, SLOT( reject() ) );

    hide();
}

/*
 *  Destroys the object and frees any allocated resources
 */
XsldbgInspector::~XsldbgInspector()
{
  debugger = 0L;
  // no need to delete child widgets, Qt does it all for us
}

void XsldbgInspector::show()
{
    QWidget::show();
    refresh();
}

void XsldbgInspector::accept()
{
  QDialog::accept();
}

void XsldbgInspector::reject()
{
  QDialog::reject();
}


void XsldbgInspector::refresh()
{

  refreshBreakpoints();
  refreshVariables();

  if ( templateWidget != 0L)
    templateWidget->refresh();

  if ( sourceWidget != 0L)
    sourceWidget->refresh();

  if ( entityWidget != 0L)
    entityWidget->refresh();

}

void XsldbgInspector::refreshBreakpoints()
{
    if ( breakpointWidget != 0L )
      breakpointWidget->refresh();
}


void XsldbgInspector::refreshVariables()
{
    if ( variablesWidget != 0L )
      variablesWidget->refresh();

    if (callStackWidget != 0L)
      callStackWidget->refresh();
}

#include "xsldbginspector.moc"
