/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2004-12-21
 * Description : USB Mass Storage camera interface
 *
 * Copyright (C) 2004-2005 by Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Copyright (C) 2005-2009 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include "umscamera.h"

// C ANSI includes

extern "C"
{
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <utime.h>
}

// Qt includes

#include <QDir>
#include <QFile>
#include <QFileInfo>
#include <QMatrix>
#include <QStringList>

// KDE includes

#include <kcodecs.h>
#include <kdebug.h>
#include <kio/global.h>
#include <klocale.h>
#include <kmimetype.h>
#include <solid/device.h>
#include <solid/storageaccess.h>
#include <solid/storagedrive.h>
#include <solid/storagevolume.h>

// LibKDcraw includes

#include <libkdcraw/kdcraw.h>

// Local includes

#include "dimg.h"
#include "dmetadata.h"

namespace Digikam
{

UMSCamera::UMSCamera(const QString& title, const QString& model,
                     const QString& port, const QString& path)
         : DKCamera(title, model, port, path)
{
    m_cancel = false;
    getUUIDFromSolid();
}

UMSCamera::~UMSCamera()
{
}

QByteArray UMSCamera::cameraMD5ID()
{
    QString camData;
    // Camera media UUID is used (if available) to improve fingerprint value
    // registration to database. We want to be sure that MD5 sum is really unique.
    // We don't use camera information here. media UUID is enough because it came be
    // mounted by a card reader or a camera. In this case, "already downloaded" flag will
    // be independent of the device used to mount memory card.
    camData.append(uuid());
    KMD5 md5(camData.toUtf8());
    return md5.hexDigest();
}

bool UMSCamera::getFreeSpace(unsigned long& /*kBSize*/, unsigned long& /*kBAvail*/)
{
    return false; // NOTE: implemented in gui, outside the camera thread.
}

bool UMSCamera::doConnect()
{
    QFileInfo dir(m_path);
    if (!dir.exists() || !dir.isReadable() || !dir.isDir())
        return false;

    if (dir.isWritable())
    {
        m_deleteSupport = true;
        m_uploadSupport = true;
        m_mkDirSupport  = true;
        m_delDirSupport = true;
    }
    else
    {
        m_deleteSupport = false;
        m_uploadSupport = false;
        m_mkDirSupport  = false;
        m_delDirSupport = false;
    }

    m_thumbnailSupport    = true;   // UMS camera always support thumbnails.
    m_captureImageSupport = false;  // UMS camera never support capture mode.

    return true;
}

void UMSCamera::cancel()
{
    // set the cancel flag
    m_cancel = true;
}

void UMSCamera::getAllFolders(const QString& folder, QStringList& subFolderList)
{
    m_cancel = false;
    subFolderList.clear();
    subFolderList.append(folder);
    listFolders(folder, subFolderList);
}

bool UMSCamera::getItemsInfoList(const QString& folder, GPItemInfoList& infoList, bool getImageDimensions)
{
    m_cancel = false;
    infoList.clear();

    QDir dir(folder);
    dir.setFilter(QDir::Files);
    if (!dir.exists())
        return false;

    const QFileInfoList list = dir.entryInfoList();
    if (list.isEmpty())
        return true;        // Nothing to do.

    QFileInfoList::const_iterator fi;
    QFileInfo          thmlo, thmup;
    DMetadata          meta;
    PhotoInfoContainer pInfo;

    for (fi = list.constBegin() ; !m_cancel && (fi != list.constEnd()) ; ++fi)
    {
        QString mime = mimeType(fi->suffix().toLower());

        if (!mime.isEmpty())
        {
            QSize      dims;
            QDateTime  dt;
            GPItemInfo info;
            thmlo.setFile(folder + QString("/") + fi->baseName() + QString(".thm"));
            thmup.setFile(folder + QString("/") + fi->baseName() + QString(".THM"));

            if (thmlo.exists())
            {
                // Try thumbnail sidecar files with lowercase extension.
                meta.load(thmlo.filePath());
                dt    = meta.getImageDateTime();
                dims  = meta.getImageDimensions();
                pInfo = meta.getPhotographInformation();
            }
            else if (thmup.exists())
            {
                // Try thumbnail sidecar files with uppercase extension.
                meta.load(thmup.filePath());
                dt    = meta.getImageDateTime();
                dims  = meta.getImageDimensions();
                pInfo = meta.getPhotographInformation();
            }
            else
            {
                // If no thumbnail sidecar file available , try to load image metadata for files.
                meta.load(fi->filePath());
                dt    = meta.getImageDateTime();
                dims  = meta.getImageDimensions();
                pInfo = meta.getPhotographInformation();
            }

            if (dt.isNull())
            {
                // If date is not found in metadata, use file time stamp
                dt = fi->created();
            }

            info.name             = fi->fileName();
            info.folder           = !folder.endsWith('/') ? folder + QString('/') : folder;
            info.mime             = mime;
            info.mtime            = dt;
            info.size             = fi->size();
            info.width            = getImageDimensions ? dims.width()  : -1;
            info.height           = getImageDimensions ? dims.height() : -1;
            info.downloaded       = GPItemInfo::DownloadUnknow;
            info.readPermissions  = fi->isReadable();
            info.writePermissions = fi->isWritable();
            info.photoInfo        = pInfo;

            infoList.append(info);
        }
    }

    return true;
}

bool UMSCamera::getThumbnail(const QString& folder, const QString& itemName, QImage& thumbnail)
{
    m_cancel = false;

    // JPEG files: try to get thumbnail from Exif data.

    DMetadata metadata(QFile::encodeName(folder + QString("/") + itemName));
    thumbnail = metadata.getExifThumbnail(true);
    if (!thumbnail.isNull())
        return true;

    // RAW files : try to extract embedded thumbnail using dcraw

    KDcrawIface::KDcraw::loadDcrawPreview(thumbnail, QString(folder + QString("/") + itemName));
    if (!thumbnail.isNull())
        return true;

    // THM files: try to get thumbnail from '.thm' files if we didn't manage to get
    // thumbnail from Exif. Any cameras provides *.thm files like JPEG files with RAW files.
    // Using this way is always speed up than ultimate loading using DImg.
    // Note: the thumbnail extracted with this method can be in poor quality.
    // 2006/27/01 - Gilles - Tested with my Minolta Dynax 5D USM camera.

    QFileInfo fi(folder + QString("/") + itemName);

    if (thumbnail.load(folder + QString("/") + fi.baseName() + QString(".thm")))        // Lowercase
    {
        if (!thumbnail.isNull())
           return true;
    }
    else if (thumbnail.load(folder + QString("/") + fi.baseName() + QString(".THM")))   // Uppercase
    {
        if (!thumbnail.isNull())
           return true;
    }

    // Finally, we trying to get thumbnail using DImg API (slow).

    DImg dimgThumb(QFile::encodeName(folder + QString("/") + itemName));

    if (!dimgThumb.isNull())
    {
        thumbnail = dimgThumb.copyQImage();
        return true;
    }
  return false;
}

bool UMSCamera::getExif(const QString&, const QString&, char **, int&)
{
    // Not necessary to implement this. read data directly from the file
    // (done in camera controller)
    kWarning(50003) << "Exif implemented yet in camera controller" << endl;
    return false;
}

bool UMSCamera::downloadItem(const QString& folder, const QString& itemName, const QString& saveFile)
{
    m_cancel     = false;
    QString src  = folder + QString("/") + itemName;
    QString dest = saveFile;

    QFile sFile(src);
    QFile dFile(dest);

    if ( !sFile.open(QIODevice::ReadOnly) )
    {
        kWarning(50003) << "Failed to open source file for reading: "
                        << src << endl;
        return false;
    }

    if ( !dFile.open(QIODevice::WriteOnly) )
    {
        sFile.close();
        kWarning(50003) << "Failed to open dest file for writing: "
                        << dest << endl;
        return false;
    }

    const int MAX_IPC_SIZE = (1024*32);
    char buffer[MAX_IPC_SIZE];

    Q_LONG len;
    while ((len = sFile.read(buffer, MAX_IPC_SIZE)) != 0 && !m_cancel)
    {
        if (len == -1 || dFile.write(buffer, (Q_ULONG)len) != len)
        {
            sFile.close();
            dFile.close();
            return false;
        }
    }

    sFile.close();
    dFile.close();

    // set the file modification time of the downloaded file to that
    // of the original file
    struct stat st;
    ::stat(QFile::encodeName(src), &st);

    struct utimbuf ut;
    ut.modtime = st.st_mtime;
    ut.actime  = st.st_atime;

    ::utime(QFile::encodeName(dest), &ut);

    return true;
}

bool UMSCamera::setLockItem(const QString& folder, const QString& itemName, bool lock)
{
    QString src = folder + QString("/") + itemName;

    if (lock)
    {
        // Lock the file to set read only flag
        if (::chmod(QFile::encodeName(src), S_IREAD) == -1)
            return false;
    }
    else
    {
        // Unlock the file to set read/write flag
        if (::chmod(QFile::encodeName(src), S_IREAD | S_IWRITE) == -1)
            return false;
    }

    return true;
}

bool UMSCamera::deleteItem(const QString& folder, const QString& itemName)
{
    m_cancel = false;

    // Any camera provide THM (thumbnail) file with real image. We need to remove it also.

    QFileInfo fi(folder + QString("/") + itemName);

    QFileInfo thmLo(folder + QString("/") + fi.baseName() + ".thm");          // Lowercase

    if (thmLo.exists())
        ::unlink(QFile::encodeName(thmLo.filePath()));

    QFileInfo thmUp(folder + QString("/") + fi.baseName() + ".THM");          // Uppercase

    if (thmUp.exists())
        ::unlink(QFile::encodeName(thmUp.filePath()));

    // Remove the real image.
    return (::unlink(QFile::encodeName(folder + QString("/") + itemName)) == 0);
}

bool UMSCamera::uploadItem(const QString& folder, const QString& itemName, const QString& localFile,
                           GPItemInfo& info, bool getImageDimensions)
{
    m_cancel     = false;
    QString dest = folder + QString("/") + itemName;
    QString src  = localFile;

    QFile sFile(src);
    QFile dFile(dest);

    if ( !sFile.open(QIODevice::ReadOnly) )
    {
        kWarning(50003) << "Failed to open source file for reading: "
                        << src << endl;
        return false;
    }

    if ( !dFile.open(QIODevice::WriteOnly) )
    {
        sFile.close();
        kWarning(50003) << "Failed to open dest file for writing: "
                        << dest << endl;
        return false;
    }

    const int MAX_IPC_SIZE = (1024*32);
    char buffer[MAX_IPC_SIZE];

    Q_LONG len;
    while ((len = sFile.read(buffer, MAX_IPC_SIZE)) != 0 && !m_cancel)
    {
        if (len == -1 || dFile.write(buffer, (Q_ULONG)len) == -1)
        {
            sFile.close();
            dFile.close();
            return false;
        }
    }

    sFile.close();
    dFile.close();

    // set the file modification time of the uploaded file to that
    // of the original file
    struct stat st;
    ::stat(QFile::encodeName(src), &st);

    struct utimbuf ut;
    ut.modtime = st.st_mtime;
    ut.actime  = st.st_atime;

    ::utime(QFile::encodeName(dest), &ut);

    // Get new camera item information.

    PhotoInfoContainer pInfo;
    DMetadata          meta;
    QFileInfo          fi(dest);
    QString            mime = mimeType(fi.suffix().toLower());

    if (!mime.isEmpty())
    {
        QSize     dims;
        QDateTime dt;

        // Try to load image metadata.
        meta.load(fi.filePath());
        dt    = meta.getImageDateTime();
        dims  = meta.getImageDimensions();
        pInfo = meta.getPhotographInformation();

        if (dt.isNull())
        {
            // If date is not found in metadata, use file time stamp
            dt = fi.created();
        }

        info.name             = fi.fileName();
        info.folder           = !folder.endsWith("/") ? folder + QString("/") : folder;
        info.mime             = mime;
        info.mtime            = dt;
        info.size             = fi.size();
        info.width            = getImageDimensions ? dims.width()  : -1;
        info.height           = getImageDimensions ? dims.height() : -1;
        info.downloaded       = GPItemInfo::DownloadUnknow;
        info.readPermissions  = fi.isReadable();
        info.writePermissions = fi.isWritable();
        info.photoInfo        = pInfo;
    }

    return true;
}

void UMSCamera::listFolders(const QString& folder, QStringList& subFolderList)
{
    if (m_cancel)
        return;

    QDir dir(folder);
    dir.setFilter(QDir::Dirs|QDir::Executable);

    const QFileInfoList list = dir.entryInfoList();
    if (list.isEmpty())
        return;

    QFileInfoList::const_iterator fi;

    for (fi = list.constBegin() ; !m_cancel && (fi != list.constEnd()) ; ++fi)
    {
        if (fi->fileName() == "." || fi->fileName() == "..")
            continue;

        QString subfolder = folder + QString(folder.endsWith('/') ? "" : "/") + fi->fileName();
        subFolderList.append(subfolder);
        listFolders(subfolder, subFolderList);
    }
}

bool UMSCamera::cameraSummary(QString& summary)
{
    summary =  QString(i18n("<b>Mounted Camera</b> driver for USB/IEEE1394 mass storage cameras and "
                            "Flash disk card readers.<br/><br/>"));

    summary += i18n("Title: <b>%1</b><br/>"
                    "Model: <b>%2</b><br/>"
                    "Port: <b>%3</b><br/>"
                    "Path: <b>%4</b><br/>"
                    "UUID: <b>%5</b><br/><br/>",
                    title(),
                    model(),
                    port(),
                    path(),
                    uuid());

    summary += i18n("Thumbnails: <b>%1</b><br/>"
                    "Capture image: <b>%2</b><br/>"
                    "Delete items: <b>%3</b><br/>"
                    "Upload items: <b>%4</b><br/>"
                    "Create directories: <b>%5</b><br/>"
                    "Delete directories: <b>%6</b><br/><br/>",
                    thumbnailSupport()    ? i18n("yes") : i18n("no"),
                    captureImageSupport() ? i18n("yes") : i18n("no"),
                    deleteSupport()       ? i18n("yes") : i18n("no"),
                    uploadSupport()       ? i18n("yes") : i18n("no"),
                    mkDirSupport()        ? i18n("yes") : i18n("no"),
                    delDirSupport()       ? i18n("yes") : i18n("no"));
    return true;
}

bool UMSCamera::cameraManual(QString& manual)
{
    manual = QString(i18n("For more information about the <b>Mounted Camera</b> driver, "
                          "please read the <b>Supported Digital Still "
                          "Cameras</b> section in the digiKam manual."));
    return true;
}

bool UMSCamera::cameraAbout(QString& about)
{
    about = QString(i18n("The <b>Mounted Camera</b> driver is a simple interface to a camera disk "
                         "mounted locally on your system.<br/><br/>"
                         "It does not use libgphoto2 drivers.<br/><br/>"
                         "To report any problems with this driver, please contact the digiKam team at:<br/><br/>"
                         "http://www.digikam.org/?q=contact"));
    return true;
}

void UMSCamera::getUUIDFromSolid()
{
    QList<Solid::Device> devices = Solid::Device::listFromType(Solid::DeviceInterface::StorageAccess);

    foreach(const Solid::Device& accessDevice, devices)
    {
        // check for StorageAccess
        if (!accessDevice.is<Solid::StorageAccess>())
            continue;

        const Solid::StorageAccess *access = accessDevice.as<Solid::StorageAccess>();

        if (!access->isAccessible())
            continue;

        // check for StorageDrive
        Solid::Device driveDevice;
        for (Solid::Device currentDevice = accessDevice; currentDevice.isValid();
             currentDevice = currentDevice.parent())
        {
            if (currentDevice.is<Solid::StorageDrive>())
            {
                driveDevice = currentDevice;
                break;
            }
        }
        if (!driveDevice.isValid())
            continue;

        // check for StorageVolume
        Solid::Device volumeDevice;
        for (Solid::Device currentDevice = accessDevice; currentDevice.isValid(); currentDevice = currentDevice.parent())
        {
            if (currentDevice.is<Solid::StorageVolume>())
            {
                volumeDevice = currentDevice;
                break;
            }
        }
        if (!volumeDevice.isValid())
            continue;

        Solid::StorageVolume *volume = volumeDevice.as<Solid::StorageVolume>();

        if (m_path.startsWith(access->filePath()))
            m_uuid = volume->uuid();
    }
}

}  // namespace Digikam
