/* -------------------------------------------------------------------------- */
/*                                                                            */
/* [te_widget.cpp]         Terminal Emulation Widget                          */
/*                                                                            */
/* -------------------------------------------------------------------------- */
/*                                                                            */
/* Copyright (c) 1997 by Lars Doelle                                          */
/*                                                                            */
/* This file is part of Kom - a serial line terminal for KDE                  */
/*                                                                            */
/* The whole program is available under the GNU Public Software Licence.      */
/* See COPYING, the documenation, or <http://www.gnu.org> for details.        */
/*                                                                            */
/* -------------------------------------------------------------------------- */

/*! \class
 
   This class is responsible to map the `image' of a terminal emulation to the
   display. All the dependency of the emulation to a specific GUI or toolkit is
   localized here. Further, this widget has no knowledge about being part of an
   emulation, it simply work within the terminal emulation framework by exposing
   size and key events and by being ordered to show a new image.

   FIXME: give a better outline here.
   - The internal image has the size of the widget (evtl. rounded up)
   - The external image used in setImage can have any size.
   - (internally) the external image is simply copied to the internal
     during if a setImage happens during a resizeEvent expecting a
     paintEvent following anyway. This refreshDisplay is not used anymore.
   - Hmm, it also knows how to ring a bell...

   /sa TEScreen /sa AnsiEmulation
*/

/* TODO
   - add font and default color changing material.
   - find means to have a neat border.
   - add destructor
   - set different 'rounding' styles.
   - refine speedup.
     check thru `refreshDisplay' and `paintEvent'. See if we can optimize or
     clearify there a bit.
   - evtl. be sensitive to `paletteChange' while using default colors.
*/


#include "TEWidget.h"

#include <qpainter.h>
#include <qkeycode.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <assert.h>

#include "TEWidget.moc"
#include <kapp.h>
#include <X11/Xlib.h>

#define MIN(a,b) ((a)<(b)?(a):(b))
#define MAX(a,b) ((a)>(b)?(a):(b))

#define HERE printf("%s(%d): here\n",__FILE__,__LINE__)

#define loc(X,Y) ((Y)*columns+(X))

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                                Colors                                     */
/*                                                                           */
/* ------------------------------------------------------------------------- */

static const QColor base_color_table[TABLE_COLORS] =
// The following are almost IBM standard color codes, with some slight
// gamma correction for the dim colors to compensate for bright X screens.
// It contains the 8 ansiterm/xterm colors in 2 intensities.
{
  // Fixme: could add faint colors here, also.
  // normal
  QColor(0x00,0x00,0x00), QColor(0xB2,0x18,0x18), // Black, Red
  QColor(0x18,0xB2,0x18), QColor(0xB2,0x68,0x18), // Green, Yellow
  QColor(0x18,0x18,0xB2), QColor(0xB2,0x18,0xB2), // Blue,  Magenta
  QColor(0x18,0xB2,0xB2), QColor(0xB2,0xB2,0xB2), // Cyan,  White
  // intensive
  QColor(0x68,0x68,0x68), QColor(0xFF,0x54,0x54), 
  QColor(0x54,0xFF,0x54), QColor(0xFF,0xFF,0x54), 
  QColor(0x54,0x54,0xFF), QColor(0xFF,0x54,0xFF),
  QColor(0x54,0xFF,0xFF), QColor(0xFF,0xFF,0xFF)
};

/* Note that we use ANSI color order (bgr), while IBMPC color order is (rgb)
   
   Code        0       1       2       3       4       5       6       7
   ----------- ------- ------- ------- ------- ------- ------- ------- -------
   ANSI  (bgr) Black   Red     Green   Yellow  Blue    Magenta Cyan    White
   IBMPC (rgb) Black   Blue    Green   Cyan    Red     Magenta Yellow  White
*/

void TEWidget::setDefaultForeColor(QColor c)
{
  color_table[DEFAULT_FORE_COLOR] = c;
  update();
}

void TEWidget::setDefaultBackColor(QColor c)
{
  color_table[DEFAULT_BACK_COLOR] = c;
  setBackgroundColor( color_table[DEFAULT_BACK_COLOR] );
  update();
}

QColor TEWidget::getDefaultForeColor()
{
  return color_table[DEFAULT_FORE_COLOR];
}

QColor TEWidget::getDefaultBackColor()
{
  return color_table[DEFAULT_BACK_COLOR];
}

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/* ------------------------------------------------------------------------- */

TEWidget::TEWidget(QWidget *parent, const char *name) : QWidget(parent,name)
{

  // Set Font ////////////////////////////////////////////////////////////

#if 0
#if defined(_WS_X11_)
  // we only have a poor 8x16 raster font
  // that works for a Linux console and for ANSI graphics
  f.setFamily( "vga" );
  f.setRawMode( TRUE );
  if ( !f.exactMatch() )
    debug( "Sorry, could not find the X specific font 'vga'" );
#endif
#else
//FIXME: font selection problem in QT 1.33?

  f.setFamily( "9x15" );
  f.setRawMode( TRUE );

//f.setFamily("fixed");
//f.setPointSize(12);
//f.setFixedPitch(TRUE);
//f.setCharSet(QFont::Latin1);
#endif

  QFontMetrics metric(f);
  QFontInfo info(f);

  font_h = metric.height();
  font_w = metric.maxWidth();
  font_a = metric.ascent();


//columns = 80;
//lines = 35;
//setFixedSize( columns*font_w, lines*font_h );

  resizing = FALSE;
  makeImage();

  for (int i = 0; i < TABLE_COLORS; i++) color_table[2+i] = base_color_table[i];
  color_table[DEFAULT_FORE_COLOR] = kapp->textColor;
  color_table[DEFAULT_BACK_COLOR] = kapp->backgroundColor;

  if ( parent ) parent->installEventFilter( this ); //FIXME: see below
  setBackgroundColor( color_table[DEFAULT_BACK_COLOR] );

}

//FIXME: make proper destructor

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                             Display Operations                            */
/*                                                                           */
/* ------------------------------------------------------------------------- */

/*!
    The image can only be set completely.

    The size of the new image may or may not match the size of the widget.
*/

void TEWidget::setImage(const ca* const newimg, int lines, int columns)
{ int lin,col,j;
//printf("setImage(_,%d,%d); resizing == %d.\n",lines,columns,resizing);

  QPainter paint;
  setUpdatesEnabled(FALSE);
  paint.begin( this );
  paint.setFont( f );
  paint.setBackgroundMode( OpaqueMode );

  int cf  = -1; // undefined
  int cb  = -1; // undefined
  
  int lins = MIN(this->lines,  MAX(0,lines  ));
  int cols = MIN(this->columns,MAX(0,columns));
  char disstr[cols];

  for (lin = 0; lin < lins; lin++)
  {
    const ca*       lcl = &image[lin*this->columns];
    const ca* const ext = &newimg[lin*columns];
    if (!resizing) // not while resizing, we're expecting a paintEvent
    for (col = 0; col < cols; col++)
    {
#assert sizeof(ca)==sizeof(int) //TRICK: only used in the line below
      if (((int*)ext)[col] != ((int*)lcl)[col])
      {
        if (ext[col].f != cf) 
        {
          cf = ext[col].f; paint.setPen(color_table[cf]); 
        }
        if (ext[col].b != cb) 
        {
          cb = ext[col].b; paint.setBackgroundColor(color_table[cb]); 
        }
        int lln = cols - col;
        disstr[0] = ext[col+0].c;
        for (j = 1; j < lln; j++)
        {
          if (ext[col+j].f != cf || ext[col+j].b != cb ||
              ext[col+j].c == lcl[col+j].c )
            break;
          disstr[j] = ext[col+j].c;
        }
//printf("draw: %d %d %d\n",lin,col,j);
        paint.drawText(font_w*col,font_a+font_h*lin, disstr,j);
        col += j - 1;
      }
    }
    // finally, make `image' == `newimg'.
    memcpy((void*)lcl,(const void*)ext,cols*sizeof(ca));
  }
  paint.end();				// painting done
  setUpdatesEnabled(TRUE);
}

// paint Event ////////////////////////////////////////////////////

/*!
    The difference of this routine vs. the `setImage' is,
    that the drawing does not include a difference analysis
    between the old and the new image. Instead, the internal
    image is used and the painting bound by the PaintEvent box.
*/

void TEWidget::paintEvent( QPaintEvent* pe )
{
  QPainter paint;
  setUpdatesEnabled(FALSE);
  paint.begin( this );
  paint.setFont( f );
  paint.setBackgroundMode( OpaqueMode );

  // Note that the actual widget size can be slightly larger
  // that the image (the size is truncated towards the smaller
  // number of characters in `resizeEvent'. The paint rectangle
  // can thus be larger than the image, but less then the size
  // of one character.

  // FIXME: explain/change 'rounding' style
 
  int lux = MIN(columns-1, MAX(0,pe->rect().left()   / font_w));
  int luy = MIN(lines-1,   MAX(0,pe->rect().top()    / font_h));
  int rlx = MIN(columns-1, MAX(0,pe->rect().right()  / font_w));
  int rly = MIN(lines-1,   MAX(0,pe->rect().bottom() / font_h));

//printf("paintEvent: %d..%d, %d..%d (%d..%d, %d..%d)\n",lux,rlx,luy,rly,
//  pe->rect().left(), pe->rect().right(), pe->rect().top(), pe->rect().bottom());

  for (int y = luy; y <= rly; y++)
  for (int x = lux; x <= rlx; x++)
  { char c[columns]; int len = 1;
    c[0] = image[loc(x,y)].c; 
    paint.setPen( color_table[image[loc(x,y)].f] ); 
    paint.setBackgroundColor( color_table[image[loc(x,y)].b] );
    while (x+len <= rlx && 
           image[loc(x,y)].b == image[loc(x+len,y)].b && 
           image[loc(x,y)].f == image[loc(x+len,y)].f )
    {
      c[len] = image[loc(x+len,y)].c; len += 1;
    }
    paint.drawText(font_w*x,font_a+font_h*y, c,len);
    x += len - 1;
  }
  paint.end();
  setUpdatesEnabled(TRUE);
}

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/* ------------------------------------------------------------------------- */

void TEWidget::resizeEvent(QResizeEvent* ev)
{
//printf("resize: %d,%d\n",ev->size().width(),ev->size().height());
//printf("approx: %d,%d\n",ev->size().width()/font_w,ev->size().height()/font_h);
//printf("leaves: %d,%d\n",ev->size().width()%font_w,ev->size().height()%font_h);
//printf("curren: %d,%d\n",width(),height());

  // see comment in `paintEvent' concerning the rounding.
  //FIXME: could make a routine here; check width(),height()
  assert(ev->size().width() == width());
  assert(ev->size().height() == height());

  {
    ca* oldimg = image;
    int oldlin = lines;
    int oldcol = columns;
    makeImage();
    // we copy the old image to reduce flicker
    int lins = MIN(oldlin,lines);
    int cols = MIN(oldcol,columns);
    for (int lin = 0; lin < lins; lin++)
      memcpy((void*)&image[columns*lin],(void*)&oldimg[oldcol*lin],cols*sizeof(ca));
    free(oldimg); //FIXME: try new,delete
  }

  //NOTE: control flows from the back through the chest right into the eye.
  //      `emu' will call back via `setImage'.

  resizing = TRUE;
  emit changedImageSizeSignal(lines, columns); // expose resizeEvent
  resizing = FALSE;
}

void TEWidget::mousePressEvent(QMouseEvent* ev)
{
  printf("press [%d,%d] %d\n",ev->x()/font_w,ev->y()/font_h,ev->button());
//FIXME: only if mode1000
  if ( ev->button() == LeftButton)
  emit mouseSignal( 0, // left button
                    ev->x()/font_w + 1,
                    ev->y()/font_h + 1 );
  if ( ev->button() == RightButton )
  emit configureRequest( this, ev->x(), ev->y() );
}

void TEWidget::mouseReleaseEvent(QMouseEvent* ev)
{
  printf("release [%d,%d] %d\n",ev->x()/font_w,ev->y()/font_h,ev->button());
//FIXME: only if mode1000
  if ( ev->button() == LeftButton)
  emit mouseSignal( 3, // release
                    ev->x()/font_w + 1,
                    ev->y()/font_h + 1 );
}

//FIXME: an `eventFilter' has been installed instead of a `keyPressEvent'
//       due to a bug in `QT' or the ignorance of the author to prevent
//       repaintevents being emitted to the screen whenever one leaves
//       or reenters the screen to/from another application.
//       I've reported this to troll, but was not able to look it up
//       in the qt source at this time. The bug is still present in QT v1.31. 
//       Think i shall have a look to the qt source when i find time
//       to make a patch.

bool TEWidget::eventFilter( QObject *, QEvent *e )
{
  if ( e->type() == Event_KeyPress )
  {
    emit keyPressedSignal((QKeyEvent*)e); // expose keypress event
    return TRUE;                          // accept event
  }
  return FALSE; // standard event processing
}

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/* ------------------------------------------------------------------------- */

void TEWidget::Bell()
{
  XBell(qt_xdisplay(),0);
}

/* ------------------------------------------------------------------------- */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/* ------------------------------------------------------------------------- */

void TEWidget::clearImage()
// initialize the image
// for internal use only
{
  for (int y = 0; y < lines; y++)
  for (int x = 0; x < columns; x++)
  {
    image[loc(x,y)].c = ' ';
    image[loc(x,y)].f = DEFAULT_FORE_COLOR ;
    image[loc(x,y)].b = DEFAULT_BACK_COLOR ;
  }
}

// Create Image ///////////////////////////////////////////////////////

void TEWidget::makeImage()
{
  //FIXME: support 'rounding' styles
  columns = width()  / font_w;
  lines   = height() / font_h;
  image = (ca*) malloc(lines*columns*sizeof(ca));
  clearImage();
}
