/*
 *   kmamerun - a interface for M.A.M.E
 *   Copyright (C) 1998  Juergen Vigna
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <unistd.h>

#include <qframe.h>
#include <qlayout.h>
#include <qpushbt.h>
#include <qlabel.h>
#include <qlined.h>
#include <qregexp.h>

#include <kapp.h>
#include <ktabctl.h>
#include <kfontdialog.h>
#include <ksimpleconfig.h>

#include "preferences.h"

extern QString debugFlag, globalGameName, globalGameFullName;

Preferences::Preferences()
{
    setCaption(i18n("M.A.M.E Preferences"));

    tabbar = new KTabCtl(this);
    tabbar->setGeometry(10,10, 300, 200);
    
    prefgen = new PrefGeneral(tabbar,"General");
    tabbar->addTab(prefgen, i18n("General"));
    
    prefvid = new PrefVideo(tabbar,"Video");
    tabbar->addTab(prefvid, i18n("Video"));
    
    prefaud = new PrefAudio(tabbar,"Audio");
    tabbar->addTab(prefaud, i18n("Audio"));
    
    prefinp = new PrefInput(tabbar,"Input");
    tabbar->addTab(prefinp, i18n("Input"));
/*
    prefinp = new PrefInput(tabbar,"Input");
    tabbar->addTab(prefinp, i18n("Input"));
*/
    prefgam = new PrefGame(tabbar,"Game");
    tabbar->addTab(prefgam, i18n("Game"));

    b_ok = new QPushButton(i18n("Ok"), this);
    b_ok->setFixedSize(80, 24);
    b_apply = new QPushButton(i18n("Apply"), this);
    b_apply->setFixedSize(80, 24);
    b_cancel = new QPushButton(i18n("Cancel"), this);
    b_cancel->setFixedSize(80, 24);

    QGridLayout *l_top_down = new QGridLayout(this, 2, 1, 4);
    QBoxLayout *l_buttons = new QBoxLayout(QBoxLayout::LeftToRight);
    l_top_down->addWidget(tabbar, 0, 0);
    l_top_down->setRowStretch(0, 1);
    l_top_down->addLayout(l_buttons, 1, 0);
    l_buttons->addStretch(1);
    l_buttons->addWidget(b_ok);
    l_buttons->addStretch(1);
    l_buttons->addWidget(b_apply);
    l_buttons->addStretch(1);
    l_buttons->addWidget(b_cancel);
    l_buttons->addStretch(1);
    l_top_down->activate();

    readConfig();
    
    connect(b_ok, SIGNAL(clicked()), this, SLOT(ok()) );
    connect(b_apply, SIGNAL(clicked()), this, SLOT(apply()) );
    connect(b_cancel, SIGNAL(clicked()), this, SLOT(cancel()) );

    setMinimumSize(370, 200);
    resize(370, 200);
}

void Preferences::readConfig()
{
    QString
        opt = "Global";
    
    setDefaultConfig();
    prefGame.game_name = globalGameName;
    prefGame.game_full_name = globalGameFullName;
    prefgam->readConfig(&prefGame,opt);
    if (!globalGameName.isEmpty()) {
        prefgam->readConfig(&prefGame,globalGameName);
        if (!prefGame.default_opt)
            opt = globalGameName;
    }
    prefgam->setPrefs(&prefGame);
    prefgen->readConfig(&prefGeneral,opt);
    prefgen->setPrefs(&prefGeneral);
    prefvid->readConfig(&prefVideo,opt);
    prefvid->setPrefs(&prefVideo);
    prefaud->readConfig(&prefAudio,opt);
    prefaud->setPrefs(&prefAudio);
    prefinp->readConfig(&prefInput,opt);
    prefinp->setPrefs(&prefInput);
}

void Preferences::writeConfig()
{
    QString
        opt;
    
    setDefaultConfig();
    if (prefgam->useDefaultOpt()) {
        opt = globalGameName;
        QString
            cgroup,
            aConfigName = KApplication::localconfigdir() + "/kmamerunrc";
        KSimpleConfig
            config(aConfigName);

        cgroup = opt + "General";
        config.deleteGroup(cgroup);
        cgroup = opt + "Video";
        config.deleteGroup(cgroup);
        cgroup = opt + "Audio";
        config.deleteGroup(cgroup);
        cgroup = opt + "Input";
        config.deleteGroup(cgroup);
        cgroup = opt + "Game";
        config.deleteGroup(cgroup);
        opt = "Global";
    } else {
        opt = globalGameName;
    }
    prefgen->writeConfig(&prefGeneral,opt);
    prefvid->writeConfig(&prefVideo,opt);
    prefaud->writeConfig(&prefAudio,opt);
    prefinp->writeConfig(&prefInput,opt);
    if (!globalGameName.isEmpty()) {
        if (prefgam->writeConfig(&prefGame,globalGameName))
            emit prefGameChanged();
    }
}

void Preferences::ok()
{
    writeConfig();
    emit commitChanges();
    emit prefHide();
}

void Preferences::apply()
{
    writeConfig();
    emit commitChanges();
}

void Preferences::cancel()
{
    emit prefHide();
}

void Preferences::closeEvent(QCloseEvent *)
{
    emit prefHide();
}

void Preferences::setPos(QLabel *lab, QLineEdit *le)
{
    int pos, diff;

    pos = lab->x() + lab->width() + 5;
    diff = pos - le->x();
    if (diff > 0)
        le->setGeometry(pos, le->y(), le->width() - diff, le->height());
}

void Preferences::setDefaultConfig()
{
    QString
        mamecmd,
        logdir,
        message = i18n("Error in call to M.A.M.E -showconfig:\n"),
        filename,
        cmd;
    QFile
        ff;

    //
    // preferences General
    //
    prefGeneral.mamecmd = "";
    prefGeneral.logdir = "";
    prefgen->readConfig(&prefGeneral,"Global");
    if (prefGeneral.mamecmd.isEmpty())
        prefGeneral.mamecmd = "xmame @@Game";
    if (prefGeneral.logdir.isEmpty())
        prefGeneral.logdir = "/tmp";
    mamecmd = prefGeneral.mamecmd;
    logdir = prefGeneral.logdir;
    prefGeneral.mamedir = "";
    prefGeneral.spooldir = "";
    prefGeneral.cheat = 0;
    prefGeneral.version = 33;
    //
    // preferences Video
    //
    prefVideo.display = getenv("DISPLAY");
    prefVideo.heightscale = 1;
    prefVideo.widthscale = 1;
    prefVideo.frameskip = 0;
    prefVideo.use_xsync = 1;
    prefVideo.use_dirty = 1;
    prefVideo.use_mitshm = 0;
    prefVideo.private_colormap = 0;
    prefVideo.force_truecolor = 0;
    prefVideo.throttle = 1;
    prefVideo.rotation = 0;
    prefVideo.ror90 = 0;
    prefVideo.rol90 = 0;
    prefVideo.flip_Xaxis = 0;
    prefVideo.flip_Yaxis = 0;
    //
    // preferences Audio
    //
    prefAudio.play_sound = 0;
    prefAudio.audiodevice = 0;
    prefAudio.samplefreq = 22050;
    prefAudio.timerfreq = 50;
    prefAudio.fake_sound = 1;
    //
    // preferences Input
    //
    prefInput.use_mouse = 0;
    prefInput.use_joystick = 0;
    prefInput.x11joyname = "";
    prefInput.swapjoyaxis = 0;
    prefInput.analogstick = 0;
    prefInput.towns_pad_name = "/dev/pad00";
    prefInput.use_pad = 0;
    //
    // preferences Game
    //
    prefGame.priority = 9;
    prefGame.type = 5;
    prefGame.default_opt = true;

    if (mamecmd.isEmpty()) {
        QString str;
        str = i18n("M.A.M.E program command not specified!");
        QMessageBox::critical(this,"KMameRun",str.data());
        return;
    }
    filename.sprintf("%s/kmamerun%d.tmp",logdir.data(),getpid());
    cmd.sprintf("%s >%s 2>%s",mamecmd.data(),filename.data(),filename.data());
    cmd.replace(QRegExp("@@Game"),"-showconfig");
    if (!debugFlag.isEmpty())
        fprintf(stderr,"%s\n",cmd.data());
    ff.setName(filename);
    system(cmd);
    if (ff.size() <= 0) {
        QMessageBox::warning(this,i18n("Error in calling MAME -showconfig"),message);
        remove(filename);
        return;
    }
    if (!ff.open(IO_ReadOnly))
        return;
    QTextStream
        t(&ff);
    QString
        line,
        str;
    char
        buf[256];
    int
        b;

    line = t.readLine(); // read first line
    while (!t.eof()) {
        if (line.left(1) == "#") {
            line = t.readLine();
            continue;
        }
        sscanf(line.data(),"%s",buf);
        str = buf;
        *buf = 0;
        if (str == "mamedir") {
            sscanf(line.data(),"%*s%s",buf);
            prefGeneral.mamedir = buf;
        } else if (str == "rompath") { // for xmame>0.34.1
            sscanf(line.data(),"%*s%s",buf);
            prefGeneral.mamedir = buf;
            prefGeneral.version = 34;
        } else if (str == "spooldir") {
            sscanf(line.data(),"%*s%s",buf);
            prefGeneral.spooldir = buf;
        } else if (str == "mamecmd") {
            sscanf(line.data(),"%*s%s",buf);
            prefGeneral.mamecmd = buf;
        } else if (str == "display") {
            sscanf(line.data(),"%*s%s",buf);
            prefVideo.display = buf;
        } else if (str == "heightscale") {
            sscanf(line.data(),"%*s%d",&(prefVideo.heightscale));
            prefVideo.heightscale--;
        } else if (str == "widthscale") {
            sscanf(line.data(),"%*s%d",&(prefVideo.widthscale));
            prefVideo.widthscale--;
        } else if (str == "frameskip") {
            sscanf(line.data(),"%*s%d",&(prefVideo.frameskip));
        } else if (str == "use_xsync") {
            sscanf(line.data(),"%*s%d",&b);
            prefVideo.use_xsync = b;
        } else if (str == "use_dirty") {
            sscanf(line.data(),"%*s%d",&b);
            prefVideo.use_dirty = b;
        } else if (str == "use_mitshm") {
            sscanf(line.data(),"%*s%d",&b);
            prefVideo.use_mitshm = b;
        } else if (str == "private_colormap") {
            sscanf(line.data(),"%*s%d",&b);
            prefVideo.private_colormap = b;
        } else if (str == "force_truecolor") {
            sscanf(line.data(),"%*s%d",&b);
            prefVideo.force_truecolor = b;
        } else if (str == "play_sound") {
            sscanf(line.data(),"%*s%d",&(prefAudio.play_sound));
        } else if (str == "audiodevice") {
            sscanf(line.data(),"%*s%s",buf);
            str = buf;
            if (str == "/dev/dsp")
                prefAudio.audiodevice = 0;
            else
                prefAudio.audiodevice = 1;
        } else if (str == "samplefreq") {
            sscanf(line.data(),"%*s%ld",&(prefAudio.samplefreq));
        } else if (str == "timerfreq") {
            sscanf(line.data(),"%*s%d",&(prefAudio.timerfreq));
        } else if (str == "use_mouse") {
            sscanf(line.data(),"%*s%d",&(prefInput.use_mouse));
        } else if (str == "use_joystick") {
            sscanf(line.data(),"%*s%d",&(prefInput.use_joystick));
        } else if (str == "analogstick") {
            sscanf(line.data(),"%*s%d",&(prefInput.use_pad));
        } else if (str == "x11joyname") {
            sscanf(line.data(),"%*s%s",buf);
            prefInput.x11joyname = buf;
        } else if (str == "towns_pad_name") {
            sscanf(line.data(),"%*s%s",buf);
            prefInput.towns_pad_name = buf;
        }
        line = t.readLine();
    }
    ff.close();
    remove(filename);
}

const QString Preferences::optionString(const char *cname = 0)
{
    QString
        opt = "";

    if (!cname)
        cname = "Global";
    
    opt += prefgen->optionString(&prefGeneral,cname);
    opt += " ";
    opt += prefvid->optionString(&prefVideo,cname);
    opt += " ";
    opt += prefinp->optionString(&prefInput,cname);
    opt += " ";
    opt += prefaud->optionString(&prefAudio,cname);

    return opt;
}
