/*****************************************g**********************************
                          cmapwindow.cpp
                      -------------------
    description          : Auto Mapper Window Class
    begin                : Fri Oct 22 1999
    copyright            : (C) 1999 by John-Paul Stanford
    email                : john-paul.stanford@virgin.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cmapwindow.h"

#include <qmessagebox.h>
#include <kfiledialog.h>

/** Used to add a map to the map view */
void CMapView ::addMap(CMapWidget *mapWidget)
{
	map = mapWidget;
	addChild(map);
}

/** the resize event which has been over riddent to risze the map correctly */
void CMapView::resizeEvent(QResizeEvent *e)
{
	int w,h;

	QScrollView::resizeEvent(e);

	map->checkSize(map->xMax,map->yMax);
}


CMapWindow::CMapWindow(QString mudTitle,QString mudLogin,KmudDoc *document)
{
	setCaption(i18n("Kmud Mapper"));

	doc = document;
	mud_Title = mudTitle;
	mud_Login = mudLogin;

	///////////////////////////////////////////////////////////////////
	// read the config file options
	// read them _before_ the inits so we init the right
	readOptions();

	this->resize(mapperWindowWidth,mapperWindowHeight);

	initMenubar();
	initStatusBar();
	initToolBars();

	mapView = new CMapView(this,"mapview");
	CharDBEntry* entry=doc->getEntry(mud_Login,mud_Title);
	if (entry!=NULL)
	{
		CMudProfile* mudPro = doc->getMudProfile(entry->mudfile);
	
		map = new CMapWidget(doc,mud_Title,mudPro,mapView,mapView->viewport());
	}
	else
	{
		map = new CMapWidget(doc,mud_Title,NULL,mapView,mapView->viewport());	
	}
	
	mapView->addMap(map);				

	map->setUpdatesEnabled(false);

	map->setTool(ID_MAPPER_TOOL_SELECT);
	map->setGridVisable(bGridShow);
	map->setViewHigherMap(bViewHigherMap);
	map->setViewLowerMap(bViewLowerMap);
	map->setViewOverview(bViewOverview);
	map->setGridColour(gridColour);
	map->setBackgroundColour(backgroundColour);	
	map->setElementsColour(lowerPathColour,defaultPathColour,higherPathColour,
                         lowerRoomColour,defaultRoomColour,higherRoomColour,loginColour,selectedColour,defaultTextColour);
 	map->setDefaultPathTwoWay(bTwoWay);

	map->loadMap();
	map->checkSize(map->xMax,map->yMax);
	map->setSpeedWalkLimit(speedWalkLimit);
	map->setSpeedWalkAbortActive(bLimitSpeedWalk);

	map->setUpdatesEnabled(true);
	// map->createDummyMap();
	// map->saveMap();
	
	CMapRoom *room=map->getCurrentRoom();									

	mapView->ensureVisible(room->getLowX()+HALF_ROOM_SIZE,room->getLowY()+HALF_ROOM_SIZE);

	if (!bViewStatusbar)
		enableStatusBar(KStatusBar::Hide);

	if (entry!=NULL)
	{
		CMudProfile* mudPro = doc->getMudProfile(entry->mudfile);

		if (mudPro->getFollowMode())
		{
		        options_menu->setItemChecked(ID_MAPPER_OPTIONS_FOLLOW,mudPro->getFollowMode());
						modesToolbar->setButton(ID_MAPPER_OPTIONS_FOLLOW,mudPro->getFollowMode());
		}
	
		if (mudPro->getCreateMode())
		{
		        options_menu->setItemChecked(ID_MAPPER_OPTIONS_CREATE,mudPro->getCreateMode());;
						modesToolbar->setButton(ID_MAPPER_OPTIONS_CREATE,mudPro->getCreateMode());
		}
	}

	menuBar()->setItemChecked(ID_MAPPER_TOGGLE_GRID, bGridShow);	
	menuBar()->setItemChecked(ID_MAPPER_VIEW_TOOLS, bViewToolsToolbar);	
	menuBar()->setItemChecked(ID_MAPPER_VIEW_MODES, bViewModesToolbar);	
	menuBar()->setItemChecked(ID_MAPPER_VIEW_MAIN, bViewMainToolbar);	
	menuBar()->setItemChecked(ID_MAPPER_VIEW_HIGHER, bViewHigherMap);	
	menuBar()->setItemChecked(ID_MAPPER_VIEW_LOWER, bViewLowerMap);	
	menuBar()->setItemChecked(ID_MAPPER_VIEW_OVERVIEW, bViewOverview);

	toolsToolbar->setButton(ID_MAPPER_TOGGLE_GRID,bGridShow);

	setView(mapView,false);
	show();

	connect(map,SIGNAL(movePlayer(direction)),SLOT(slotMovePlayer(direction)));
	connect(map,SIGNAL(levelChange(signed int)),SLOT(slotLevelChanged(signed int)));
	connect(map,SIGNAL(statusMsg(const char *)),SLOT(slotStatusMsg(const char *)));
	connect(map,SIGNAL(statusHelpMsg(const char *)),SLOT(slotStatusHelpMsg(const char *)));

}

CMapWindow::~CMapWindow()
{
	delete file_menu;
	delete view_menu;
	delete options_menu;
	delete tools_menu;				

	delete mainToolbar;
	delete toolsToolbar;
	delete modesToolbar;
		
	delete map;
	delete mapView;	
}

void CMapWindow::readOptions(void)
{                           	
	///////////////////////////////////////////////////////////////////
	// read the config file entries
	KConfig *config = kapp->getConfig();

	config->setGroup("MAPPER APPEARANCE");
	mapperWindowWidth = config->readNumEntry("WindowWidth",600);
	mapperWindowHeight = config->readNumEntry("WindowHeight",400);
	bViewStatusbar = config->readBoolEntry("ShowStatusbar", true);
	bViewToolsToolbar = config->readBoolEntry("ShowToolsToolbar",true);
	bViewModesToolbar = config->readBoolEntry("ShowModesToolbar",true);
	bViewMainToolbar = config->readBoolEntry("ShowMainToolbar",true);
	bViewHigherMap = config->readBoolEntry("ShowHigherMap",true);
	bViewLowerMap = config->readBoolEntry("ShowLowerMap",false);
	bViewOverview = config->readBoolEntry("ShowOverview",false);
	bGridShow = config->readBoolEntry("GridShow",true);
	bTwoWay = config->readBoolEntry("DefaltPathTwoWay",true);
	
	tools_toolbar_pos = (KToolBar::BarPosition)config->readNumEntry("Mapper_ToolBar_Tools_Pos", KToolBar::Left);
	modes_toolbar_pos = (KToolBar::BarPosition)config->readNumEntry("Mapper_ToolBar_Modes_Pos", KToolBar::Top);
	main_toolbar_pos = (KToolBar::BarPosition)config->readNumEntry("Mapper_ToolBar_Main_Pos", KToolBar::Top);

	config->setGroup("MAPPER COLOUR");
//	backgroundColour=QColor(214,187,255);	
	backgroundColour=QColor(201,167,55);	
	backgroundColour=config->readColorEntry("Background", &backgroundColour);
//	gridColour=lightGray;
	gridColour=QColor(213,178,88);	
	gridColour=config->readColorEntry("Grid", &gridColour);
	lowerRoomColour=darkGray;
	lowerRoomColour=config->readColorEntry("LowerRoom", &lowerRoomColour);
	defaultRoomColour=QColor(192,192,192);
	defaultRoomColour=config->readColorEntry("DefaultRoom", &defaultRoomColour);
	higherRoomColour=white;
	higherRoomColour=config->readColorEntry("HigherRoom", &higherRoomColour);
	lowerPathColour=darkGray;
	lowerPathColour=config->readColorEntry("LowerPath", &lowerPathColour);
	defaultPathColour=black;
	defaultPathColour=config->readColorEntry("DefaultPath", &defaultPathColour);
	higherPathColour=white;
	higherPathColour=config->readColorEntry("HigherPath", &higherPathColour);
	defaultTextColour=black;
	defaultTextColour=config->readColorEntry("DefaultText", &defaultTextColour);
	selectedColour=blue;
	selectedColour=config->readColorEntry("Selected", &selectedColour);
	loginColour=blue;
	loginColour=config->readColorEntry("Login", &loginColour);
	
	config->setGroup("GENERAL");	
	bLimitSpeedWalk = config->readBoolEntry("SpeedWalkLimitActive",true);
	speedWalkLimit = config->readNumEntry("SpeedWalkLimit",1000);

}

void CMapWindow::saveOptions(void)
{
	///////////////////////////////////////////////////////////////////
	// save the global map options to the config file.
	KConfig *config = kapp->getConfig();

	config->setGroup("MAPPER APPEARANCE");
	config->writeEntry("WindowWidth",this->width());
	config->writeEntry("WindowHeight",this->height());
	config->writeEntry("ShowStatusbar",statusBar()->isVisible());
	config->writeEntry("ShowToolsToolbar",bViewToolsToolbar);
	config->writeEntry("ShowModesToolbar",bViewModesToolbar);	
	config->writeEntry("ShowMainToolbar",bViewMainToolbar);	
	config->writeEntry("ShowHigherMap",bViewHigherMap);
	config->writeEntry("ShowLowerMap",bViewLowerMap);
	config->writeEntry("GridShow",bGridShow);
	config->writeEntry("Mapper_ToolBar_Tools_Pos",(int)toolsToolbar->barPos());
	config->writeEntry("Mapper_ToolBar_Modes_Pos",(int)modesToolbar->barPos());
	config->writeEntry("Mapper_ToolBar_Main_Pos",(int)mainToolbar->barPos());
	config->writeEntry("ShowOverview",	bViewOverview);
	config->writeEntry("DefaltPathTwoWay", map->getDefaultPathTwoWay());

	config->setGroup("GENERAL");	
	config->writeEntry("SpeedWalkLimitActive", map->getSpeedWalkAbortActive());
	config->writeEntry("SpeedWalkLimit",map->getSpeedWalkLimit());
	
	config->setGroup("MAPPER COLOUR");	
	config->writeEntry("Background", map->getBackgroundColour());
	config->writeEntry("Grid", map->getGridColour());
	config->writeEntry("LowerRoom", map->getLowerRoomColour());
	config->writeEntry("DefaultRoom", map->getDefaultRoomColour());
	config->writeEntry("DefaultText", map->getDefaultTextColour());
	config->writeEntry("HigherRoom", map->getHigherRoomColour());
	config->writeEntry("LowerPath", map->getLowerPathColour());
	config->writeEntry("DefaultPath", map->getDefaultPathColour());
	config->writeEntry("HigherPath", map->getHigherPathColour());
	config->writeEntry("Selected", map->getLoginColour());	
	config->writeEntry("Login", map->getSelectedColour());	
	

	CharDBEntry* entry=doc->getEntry(mud_Login,mud_Title);
	if (entry!=NULL)
	{
		doc->getMudProfile(entry->mudfile)->writeData();
	}
}

void CMapWindow::initMenubar(void)
{
	///////////////////////////////////////////////////////////////////
	// menuBar entry file_menu
	file_menu = new QPopupMenu();
	file_menu->insertItem(Icon("filenew.xpm"),i18n("&New Map"), ID_MAPPER_FILE_NEW );
	file_menu->insertSeparator();
	file_menu->insertItem(Icon("fileopen.xpm"),i18n("&Import Map..."), ID_MAPPER_FILE_IMPORT );
	file_menu->insertItem(Icon("filefloppy.xpm"), i18n("&Export Map..."), ID_MAPPER_FILE_EXPORT );
	file_menu->insertSeparator();
	file_menu->insertItem(i18n("In&formation..."), ID_MAPPER_FILE_INFO );
	file_menu->insertSeparator();	
	file_menu->insertItem(i18n("C&lose Window"), ID_MAPPER_FILE_CLOSE);

	// file_menu key accelerators
	file_menu->setAccel(CTRL+Key_W, ID_MAPPER_FILE_CLOSE);

	///////////////////////////////////////////////////////////////////
	// menuBar entry file_menu
	edit_menu = new QPopupMenu();
	edit_menu->insertItem(Icon("editcut.xpm"),i18n("C&ut"),ID_MAPPER_EDIT_CUT);
	edit_menu->insertItem(Icon("editcopy.xpm"),i18n("&Copy"),ID_MAPPER_EDIT_COPY);
	edit_menu->insertItem(Icon("editpaste.xpm"),i18n("&Paste"),ID_MAPPER_EDIT_PASTE);
	edit_menu->insertItem(i18n("&Delete"),ID_MAPPER_EDIT_DELETE);
	edit_menu->insertSeparator();
	edit_menu->insertItem(i18n("&Select All"),ID_MAPPER_EDIT_SELECTALL);
	edit_menu->insertItem(i18n("D&eselect All"),ID_MAPPER_EDIT_DESELECTALL);
	edit_menu->insertItem(i18n("&Invert Selection"),ID_MAPPER_EDIT_INVERTSEL);

	// edit_menu key accelerators
	edit_menu->setAccel(CTRL+Key_X,ID_MAPPER_EDIT_CUT);
	edit_menu->setAccel(CTRL+Key_C,ID_MAPPER_EDIT_COPY);
	edit_menu->setAccel(CTRL+Key_V,ID_MAPPER_EDIT_PASTE);
	edit_menu->setAccel(Key_Delete,ID_MAPPER_EDIT_DELETE);
	edit_menu->setAccel(CTRL+Key_A,ID_MAPPER_EDIT_SELECTALL);	

	///////////////////////////////////////////////////////////////////
	// menuBar entry view_menu
	view_menu = new QPopupMenu();
	view_menu->setCheckable(true);
	view_menu->insertItem(i18n("Show &Lower Map"), ID_MAPPER_VIEW_LOWER );
	view_menu->insertItem(i18n("Show &Higher Map"), ID_MAPPER_VIEW_HIGHER );
	view_menu->insertItem(Icon("kmud_grid.xpm"), i18n("Show &Grid"), ID_MAPPER_TOGGLE_GRID );	
	view_menu->insertSeparator();
	view_menu->insertItem(i18n("&Edit toolbar"), ID_MAPPER_VIEW_MAIN );
	view_menu->insertItem(i18n("&Tools toolbar"), ID_MAPPER_VIEW_TOOLS );
	view_menu->insertItem(i18n("&Modes toolbar"), ID_MAPPER_VIEW_MODES );
	view_menu->insertSeparator();
	view_menu->insertItem(i18n("Show &Overview"), ID_MAPPER_VIEW_OVERVIEW );
	view_menu->insertSeparator();
	view_menu->insertItem(i18n("&Statusbar"), ID_MAPPER_VIEW_STATUSBAR );

	view_menu->setItemChecked(ID_MAPPER_VIEW_STATUSBAR, bViewStatusbar);
	view_menu->setItemChecked(ID_MAPPER_VIEW_MODES, bViewModesToolbar);
	view_menu->setItemChecked(ID_MAPPER_VIEW_TOOLS, bViewToolsToolbar);
	view_menu->setItemChecked(ID_MAPPER_VIEW_MAIN, bViewMainToolbar);
	view_menu->setItemChecked(ID_MAPPER_TOGGLE_GRID, bGridShow);
	view_menu->setItemChecked(ID_MAPPER_VIEW_HIGHER, bViewHigherMap);
	view_menu->setItemChecked(ID_MAPPER_VIEW_LOWER, bViewLowerMap);

	///////////////////////////////////////////////////////////////////
	// menuBar entry tools_menu
	tools_menu = new QPopupMenu();
	tools_menu->setCheckable(true);
	tools_menu->insertItem(Icon("kmud_pointer.xpm"),i18n("Select"),ID_MAPPER_TOOL_SELECT);
	tools_menu->insertItem(Icon("kmud_room.xpm"), i18n("Create Room"),ID_MAPPER_TOOL_CREATE_ROOM);
	tools_menu->insertItem(Icon("kmud_path.xpm"),i18n("Create Path"),ID_MAPPER_TOOL_CREATE_PATH);
	tools_menu->insertItem(Icon("kmud_text.xpm"),i18n("Create Text"),ID_MAPPER_TOOL_CREATE_TEXT);
	tools_menu->insertItem(Icon("kmud_eraser.xpm"),i18n("Delete"),ID_MAPPER_TOOL_DELETE);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_SELECT,true);
	tools_menu->insertSeparator();
	tools_menu->insertItem(Icon("kmud_lvlup.xpm"), i18n("Level Up"), ID_MAPPER_LEVEL_UP);
	tools_menu->insertItem(Icon("kmud_lvldown.xpm"),i18n("Level Down"), ID_MAPPER_LEVEL_DOWN );
	tools_menu->insertSeparator();
	tools_menu->insertItem(i18n("Create New Top Level"), ID_MAPPER_TOOL_NEW_TOP );		
	tools_menu->insertItem(i18n("Create New Bottom Level"), ID_MAPPER_TOOL_NEW_BOTTOM );

	///////////////////////////////////////////////////////////////////
	// menuBar entry options_menu
	options_menu = new QPopupMenu();
	options_menu->setCheckable(true);
	options_menu->insertItem(Icon("kmud_follow.xpm"),i18n("&Follow Player"),ID_MAPPER_OPTIONS_FOLLOW);
	options_menu->insertItem(Icon("kmud_create.xpm"),i18n("&Create New Rooms"),ID_MAPPER_OPTIONS_CREATE);
	options_menu->insertSeparator();
	options_menu->insertItem(i18n("&Preferences..."),ID_MAPPER_OPTIONS_PREFS);

	///////////////////////////////////////////////////////////////////
	// MENUBAR CONFIGURATION
	// set menuBar() the current menuBar and the position due to config file
	KMenuBar *menu = new KMenuBar(this);
	menu->insertItem(i18n("&File"), file_menu,-1,-1);
	menu->insertItem(i18n("&Edit"), edit_menu);
	menu->insertItem(i18n("&View"), view_menu);
	menu->insertItem(i18n("&Tools"), tools_menu);
	menu->insertItem(i18n("&Options"), options_menu);
	setMenu(menu);

	CONNECT_CMD(file_menu);
	CONNECT_CMD(view_menu);
	CONNECT_CMD(options_menu);
	CONNECT_CMD(tools_menu);
	CONNECT_CMD(edit_menu);
}

void CMapWindow::initStatusBar(void)
{
	statusBar()->setInsertOrder(KStatusBar::RightToLeft);
	statusBar()->insertItem(i18n("Level : 1"), ID_MAPPER_LEVEL_NUM);
	statusBar()->insertItem(i18n("Ready."), ID_MAPPER_STATUS_MSG );
	
}

void CMapWindow::initToolBars(void)
{


	mainToolbar = new KToolBar(this);
	addToolBar(mainToolbar);	

	mainToolbar->insertButton(Icon("filenew.xpm"), ID_MAPPER_FILE_NEW, true, i18n("New Map"));
	mainToolbar->insertSeparator();
	mainToolbar->insertButton(Icon("editcut.xpm"), ID_MAPPER_EDIT_CUT, true, i18n("Cut"));
	mainToolbar->insertButton(Icon("editcopy.xpm"), ID_MAPPER_EDIT_COPY, true, i18n("Copy"));
	mainToolbar->insertButton(Icon("editpaste.xpm"), ID_MAPPER_EDIT_PASTE, true, i18n("Paste"));
	mainToolbar->insertSeparator();

	mainToolbar->setFullWidth(false);

	CONNECT_TOOLBAR(mainToolbar);

	if (bViewMainToolbar)
	{

		enableToolBar(KToolBar::Show,ID_MAPPER_TOOLBAR_MAIN);
	}
	else
	{
		enableToolBar(KToolBar::Hide,ID_MAPPER_TOOLBAR_MAIN);
	}

  mainToolbar->setBarPos(main_toolbar_pos);

	toolsToolbar = new KToolBar(this);
	addToolBar(toolsToolbar);

	toolsToolbar->insertButton(Icon("kmud_pointer.xpm"), ID_MAPPER_TOOL_SELECT, true, i18n("Select"));
	toolsToolbar->insertButton(Icon("kmud_room.xpm"), ID_MAPPER_TOOL_CREATE_ROOM, true, i18n("Create Room"));
	toolsToolbar->insertButton(Icon("kmud_path.xpm"), ID_MAPPER_TOOL_CREATE_PATH, true, i18n("Create Path"));
	toolsToolbar->insertButton(Icon("kmud_text.xpm"), ID_MAPPER_TOOL_CREATE_TEXT, true, i18n("Add text"));
	toolsToolbar->insertButton(Icon("kmud_eraser.xpm"), ID_MAPPER_TOOL_DELETE, true, i18n("Erase"));
	toolsToolbar->setToggle(ID_MAPPER_TOOL_SELECT,true);
	toolsToolbar->setToggle(ID_MAPPER_TOOL_CREATE_ROOM,true);
	toolsToolbar->setToggle(ID_MAPPER_TOOL_CREATE_PATH,true);
	toolsToolbar->setToggle(ID_MAPPER_TOOL_CREATE_TEXT,true);
	toolsToolbar->setToggle(ID_MAPPER_TOOL_DELETE,true);
	toolsToolbar->setButton(ID_MAPPER_TOOL_SELECT,true);
	toolsToolbar->insertSeparator();
	toolsToolbar->insertButton(Icon("kmud_grid.xpm"), ID_MAPPER_TOGGLE_GRID, true, i18n("Erase"));
	toolsToolbar->setToggle(ID_MAPPER_TOGGLE_GRID,true);
	toolsToolbar->insertSeparator();
	toolsToolbar->insertButton(Icon("kmud_lvlup.xpm"), ID_MAPPER_LEVEL_UP, true, i18n("Level Up"));
	toolsToolbar->insertButton(Icon("kmud_lvldown.xpm"), ID_MAPPER_LEVEL_DOWN, true, i18n("Level Down"));

	toolsToolbar->setFullWidth(false);

	CONNECT_TOOLBAR(toolsToolbar);

	if (bViewToolsToolbar)
	{

		enableToolBar(KToolBar::Show,ID_MAPPER_TOOLBAR_TOOLS);
	}
	else
	{
		enableToolBar(KToolBar::Hide,ID_MAPPER_TOOLBAR_TOOLS);
	}

  toolsToolbar->setBarPos(tools_toolbar_pos);

	modesToolbar = new KToolBar(this);
	addToolBar(modesToolbar);

	modesToolbar->insertButton(Icon("kmud_follow.xpm"), ID_MAPPER_OPTIONS_FOLLOW, true, i18n("Follow Mode"));
	modesToolbar->insertButton(Icon("kmud_create.xpm"), ID_MAPPER_OPTIONS_CREATE, true, i18n("Create Mode"));
	modesToolbar->setToggle(ID_MAPPER_OPTIONS_FOLLOW,true);
  modesToolbar->setToggle(ID_MAPPER_OPTIONS_CREATE,true);

	modesToolbar->setFullWidth(false);

	CONNECT_TOOLBAR(modesToolbar);

	if (bViewModesToolbar)
	{

		enableToolBar(KToolBar::Show,ID_MAPPER_TOOLBAR_MODES);
	}
	else
	{
		enableToolBar(KToolBar::Hide,ID_MAPPER_TOOLBAR_MODES);
	}
}

void CMapWindow::movePlayerBy(direction dir)
{
	CharDBEntry* entry=doc->getEntry(mud_Login,mud_Title);
	if (entry!=NULL)
	{		
		CMudProfile* mudPro = doc->getMudProfile(entry->mudfile);

		if (mudPro->getFollowMode())
		{
			map->movePlayerBy(dir,mudPro->getCreateMode());
			CMapRoom *room=map->getCurrentRoom();									
			mapView->ensureVisible(room->getLowX()+HALF_ROOM_SIZE,room->getLowY()+HALF_ROOM_SIZE);
		}
	}
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION

void CMapWindow::closeEvent(QCloseEvent *e)
{
	map->saveMap();
	emit (slotMapperClosed());
}

void CMapWindow::commandCallback(int id_){
	switch (id_)
	{
		ON_CMD(ID_MAPPER_FILE_EXPORT,          slotFileExport())
		ON_CMD(ID_MAPPER_FILE_IMPORT,          slotFileImport())
		ON_CMD(ID_MAPPER_FILE_NEW,             slotFileNew())
		ON_CMD(ID_MAPPER_FILE_CLOSE,           slotFileCloseWindow())
		ON_CMD(ID_MAPPER_FILE_INFO,            slotFileInfo())

		ON_CMD(ID_MAPPER_OPTIONS_PREFS,        slotOptionsPrefs())
		ON_CMD(ID_MAPPER_OPTIONS_FOLLOW,       slotOptionsFollow())
		ON_CMD(ID_MAPPER_OPTIONS_CREATE,       slotOptionsCreate())

		ON_CMD(ID_MAPPER_EDIT_CUT,             slotEditCut())
		ON_CMD(ID_MAPPER_EDIT_COPY,            slotEditCopy())
		ON_CMD(ID_MAPPER_EDIT_PASTE,           slotEditPaste())
		ON_CMD(ID_MAPPER_EDIT_DELETE,          slotEditDelete())
		ON_CMD(ID_MAPPER_EDIT_SELECTALL,       slotSelectAll())
		ON_CMD(ID_MAPPER_EDIT_DESELECTALL,     slotDeselectAll())
		ON_CMD(ID_MAPPER_EDIT_INVERTSEL,       slotInvertSelection())

		ON_CMD(ID_MAPPER_VIEW_STATUSBAR,       slotViewStatusBar())
		ON_CMD(ID_MAPPER_VIEW_TOOLS,		   slotViewTools())
		ON_CMD(ID_MAPPER_VIEW_MODES,		   slotViewModesToolBar())
		ON_CMD(ID_MAPPER_VIEW_MAIN, 		   slotViewMainToolBar())
		ON_CMD(ID_MAPPER_VIEW_LOWER,           slotViewLowerMap())
		ON_CMD(ID_MAPPER_VIEW_HIGHER,		   slotViewHigherMap())
		ON_CMD(ID_MAPPER_VIEW_OVERVIEW,    slotViewOverview())

		ON_CMD(ID_MAPPER_TOOL_SELECT,		   slotToolSelect())
		ON_CMD(ID_MAPPER_TOOL_CREATE_ROOM,	   slotToolCreateRoom())
		ON_CMD(ID_MAPPER_TOOL_CREATE_PATH,	   slotToolCreatePath())
		ON_CMD(ID_MAPPER_TOOL_CREATE_TEXT,	   slotToolCreateText())
		ON_CMD(ID_MAPPER_TOOL_DELETE,		   slotToolErase())

		ON_CMD(ID_MAPPER_LEVEL_UP,             slotLevelUp())
		ON_CMD(ID_MAPPER_LEVEL_DOWN,           slotLevelDown())
		
		ON_CMD(ID_MAPPER_TOOL_NEW_TOP,           slotCreateTopLevel())
		ON_CMD(ID_MAPPER_TOOL_NEW_BOTTOM,        slotCreateBottomLevel())

		ON_CMD(ID_MAPPER_TOGGLE_GRID,		   slotShowGrid())
	}
}

void CMapWindow::statusCallback(int id_)
{
	switch (id_)
	{
    	ON_STATUS_MSG(ID_MAPPER_FILE_NEW,	 i18n("Erase current map and create a new map"))
		ON_STATUS_MSG(ID_MAPPER_FILE_EXPORT,     i18n("Export the map to a new file (Not yet implemented)"))
		ON_STATUS_MSG(ID_MAPPER_FILE_IMPORT,     i18n("Import a map from a file (Not yet implemented)"))
		ON_STATUS_MSG(ID_MAPPER_FILE_CLOSE,      i18n("Close the map window"))
		ON_STATUS_MSG(ID_MAPPER_FILE_INFO,       i18n("Display information about the map"))

		ON_STATUS_MSG(ID_MAPPER_OPTIONS_PREFS,   i18n("Configure mapper settings"))
		ON_STATUS_MSG(ID_MAPPER_OPTIONS_FOLLOW,  i18n("Turns the follow player mode on or off"))
		ON_STATUS_MSG(ID_MAPPER_OPTIONS_CREATE,  i18n("Create new rooms whilst following the player"))

		ON_STATUS_MSG(ID_MAPPER_EDIT_CUT,        i18n("Cut map elements"))
		ON_STATUS_MSG(ID_MAPPER_EDIT_COPY,       i18n("Copy map elements"))
		ON_STATUS_MSG(ID_MAPPER_EDIT_PASTE,      i18n("Paste map elements"))
		ON_STATUS_MSG(ID_MAPPER_EDIT_DELETE,     i18n("Delete map elements"))
		ON_STATUS_MSG(ID_MAPPER_EDIT_SELECTALL,  i18n("Select all elements"))		
		ON_STATUS_MSG(ID_MAPPER_EDIT_DESELECTALL,i18n("Deselect all elements"))		
		ON_STATUS_MSG(ID_MAPPER_EDIT_INVERTSEL,  i18n("Invert selected elements"))
		
		ON_STATUS_MSG(ID_MAPPER_VIEW_STATUSBAR,  i18n("Display/hide the status bar"))
	  	ON_STATUS_MSG(ID_MAPPER_VIEW_TOOLS,      i18n("Display/hide tools toolbar"))
		ON_STATUS_MSG(ID_MAPPER_VIEW_MODES,      i18n("Display/hide modes toolbar"))
		ON_STATUS_MSG(ID_MAPPER_VIEW_MAIN,       i18n("Display/hide main toolbar"))

		ON_STATUS_MSG(ID_MAPPER_TOOL_SELECT,	 i18n("The select tool"))
		ON_STATUS_MSG(ID_MAPPER_TOOL_CREATE_ROOM,i18n("Create a new room"))
		ON_STATUS_MSG(ID_MAPPER_TOOL_CREATE_PATH,i18n("Create a new path"))
		ON_STATUS_MSG(ID_MAPPER_TOOL_CREATE_TEXT,i18n("Create a text lable"))
		ON_STATUS_MSG(ID_MAPPER_TOOL_DELETE,	 i18n("Delete a map element"))

		ON_STATUS_MSG(ID_MAPPER_LEVEL_UP,	 i18n("Up a level"))
		ON_STATUS_MSG(ID_MAPPER_LEVEL_DOWN,	 i18n("Down a level"))

		ON_STATUS_MSG(ID_MAPPER_TOOL_NEW_TOP,	 i18n("Create a new top level"))
		ON_STATUS_MSG(ID_MAPPER_TOOL_NEW_BOTTOM, i18n("Create a new bottom level"))

		ON_STATUS_MSG(ID_MAPPER_TOGGLE_GRID,	 i18n("Show/hide the grid"))

	}
}

void CMapWindow::slotOptionsCreate(void)
{
	CharDBEntry* entry=doc->getEntry(mud_Login,mud_Title);
	if (entry!=NULL)
	{
		CMudProfile* mudPro = doc->getMudProfile(entry->mudfile);

		mudPro->setCreateMode(!mudPro->getCreateMode());
		menuBar()->setItemChecked(ID_MAPPER_OPTIONS_CREATE, mudPro->getCreateMode());
		modesToolbar->setButton(ID_MAPPER_OPTIONS_CREATE, mudPro->getCreateMode());
		mudPro->writeData();
	}
}

void CMapWindow::slotOptionsFollow(void)
{
	CharDBEntry* entry=doc->getEntry(mud_Login,mud_Title);
	if (entry!=NULL)
	{
		CMudProfile* mudPro = doc->getMudProfile(entry->mudfile);

		mudPro->setFollowMode(!mudPro->getFollowMode());
		menuBar()->setItemChecked(ID_MAPPER_OPTIONS_FOLLOW, mudPro->getFollowMode());
		modesToolbar->setButton(ID_MAPPER_OPTIONS_FOLLOW, mudPro->getFollowMode());
		mudPro->writeData();
	}
}

void CMapWindow::slotLevelUp(void)
{
	map->slotLevelUp();
}

void CMapWindow::slotLevelDown(void)
{
	map->slotLevelDown();
}

/** This slot is called when the copy menu option is selected */
void CMapWindow::slotEditCopy(void)
{
	map->slotCopy();
}

/** This slot is called when the paste menu option is selected */
void CMapWindow::slotEditCut(void)
{
	map->slotCut();
}

/** This slot is called when the cut menu option is selected */
void CMapWindow::slotEditPaste(void)
{
	map->slotPaste();
}

/** This slot is called when the delete menu option is selected */
void CMapWindow::slotEditDelete(void)
{
	map->slotDelete();
}

void CMapWindow::slotViewStatusBar(void)
{
	///////////////////////////////////////////////////////////////////
	// turn the Statusbar on or off
	bViewStatusbar=!bViewStatusbar;
	menuBar()->setItemChecked(ID_MAPPER_VIEW_STATUSBAR, bViewStatusbar);
	statusBar()->clear();
	enableStatusBar();
	slotStatusMsg(i18n("Ready."));
}

void CMapWindow::slotViewOverview(void)
{
	bViewOverview = !bViewOverview;
	menuBar()->setItemChecked(ID_MAPPER_VIEW_OVERVIEW, bViewOverview);	
	map->setViewOverview(bViewOverview);
}

void CMapWindow::slotViewTools(void)
{
	bViewToolsToolbar=!bViewToolsToolbar;
	menuBar()->setItemChecked(ID_MAPPER_VIEW_TOOLS, bViewToolsToolbar);	
	if (bViewToolsToolbar) enableToolBar(KToolBar::Show,ID_MAPPER_TOOLBAR_TOOLS); else enableToolBar(KToolBar::Hide,ID_MAPPER_TOOLBAR_TOOLS);
}

void CMapWindow::slotStatusMsg(const char *text)
{
	///////////////////////////////////////////////////////////////////
	// change status message permanently
	statusBar()->clear();
	statusBar()->changeItem(text, ID_STATUS_MSG );
}


void CMapWindow::slotStatusHelpMsg(const char *text)
{
	///////////////////////////////////////////////////////////////////
	// change status message of whole statusbar temporary (text, msec)
	statusBar()->message(text, 2000);
}


void CMapWindow::slotFileCloseWindow(void)
{
	emit (slotMapperClosed());
}

void CMapWindow::slotFileExport(void)
{
	QString filename = KFileDialog::getSaveFileName (""	, "*.map", this, "export");
	if (filename !="" && filename != NULL)
		map->exportMap(filename);
}

void CMapWindow::slotFileImport(void)
{
	QString filename = KFileDialog::getOpenFileName ("", "*.map", this, "export");
	if (filename !="" && filename != NULL)
		map->importMap(filename);
}

void CMapWindow::slotFileNew(void)
{
	if ( QMessageBox::information (this,"Kmud Mapper",
                                      "This will erase the current map\n"
                                      "Do you wish to erase the map?",
                                      "&Erase","&Cancel",0,1)==0)
	{
		map->eraseMap();
		map->createNewMap();
		map->repaint(false);
	}
}

void CMapWindow::slotOptionsPrefs(void)
{
	directions *directions;

	DlgMapOptionsPreferences d(this, "Preferences");

	CharDBEntry* entry=doc->getEntry(mud_Login,mud_Title);
	if (entry!=NULL)
	{
		CMudProfile* mudPro = doc->getMudProfile(entry->mudfile);
	 	directions = mudPro->getDirections();
		d.setNorth(directions->north);
		d.setNorthEast(directions->northeast);
		d.setEast(directions->east);
		d.setSouthEast(directions->southeast);
		d.setSouth(directions->south);
		d.setSouthWest(directions->southwest);
		d.setWest(directions->west);
		d.setNorthWest(directions->northwest);
		d.setUp(directions->up);
		d.setDown(directions->down);
		d.setNoMoveString(mudPro->getNoMoveStr());
		d.setCanMoveString(mudPro->getCanMoveStr());
		d.setDoMoveCheck(mudPro->getDoMoveCheck());
	}

	d.setBackgroundColour(map->getBackgroundColour());
	d.setDefaultPathColour(map->getDefaultPathColour());
	d.setDefaultRoomColour(map->getDefaultRoomColour());
	d.setLowerPathColour(map->getLowerPathColour());
	d.setLowerRoomColour(map->getLowerRoomColour());
	d.setHigherPathColour(map->getHigherPathColour());
	d.setHigherRoomColour(map->getHigherRoomColour());
	d.setGridColour(map->getGridColour());
	d.setLoginColour(map->getLoginColour());
	d.setSelectedColour(map->getSelectedColour());
	d.setDefaultTextColour(map->getSelectedColour());
	d.setDefaultTwoWay(map->getDefaultPathTwoWay());
	d.setSpeedWalkLimit(map->getSpeedWalkLimit());
	d.setDoLimitSpeedWalk(map->getSpeedWalkAbortActive());
	
	
	if (d.exec())
	{
		directions->north = d.getNorth();
		directions->northeast = d.getNorthEast();
		directions->east = d.getEast();
		directions->southeast = d.getSouthEast();
		directions->south = d.getSouth();
		directions->southwest = d.getSouthWest();
		directions->west = d.getWest();
		directions->northwest = d.getNorthWest();
		directions->up = d.getUp();
		directions->down = d.getDown();		


		map->setBackgroundColour(d.getBackgroundColour());
		map->setGridColour(d.getGridColour());	
		map->setElementsColour(d.getLowerPathColour(),d.getDefaultPathColour(),d.getHigherPathColour(),
                           d.getLowerRoomColour(),d.getDefaultRoomColour(),d.getHigherRoomColour(),d.getLoginColour(),d.getSelectedColour(),d.getDefaultTextColour());
  		map->setDefaultPathTwoWay(d.getDefaultTwoWay());
		
		map->setSpeedWalkLimit(d.getSpeedWalkLimit());
		map->setSpeedWalkAbortActive(d.getDoLimitSpeedWalk());
  		
  		if (entry!=NULL)
		{
			CMudProfile* mudPro = doc->getMudProfile(entry->mudfile);
			mudPro->setNoMoveStr(d.getNoMoveString());
			mudPro->setCanMoveStr(d.getCanMoveString());
			mudPro->setDoMoveCheck(d.getDoMoveCheck());
		}
	}

	map->repaint(false);
	saveOptions();
}

void CMapWindow::slotSelectAll(void)
{
	map->slotSelectAll();
}

void CMapWindow::slotDeselectAll(void)
{
	map->slotDeselectAll();
}

void CMapWindow::slotInvertSelection(void)
{
	map->slotInvertSelection();
}

/** This slot is used to activate the select tool */
void CMapWindow::slotToolSelect(void)
{
	toolsToolbar->setButton(ID_MAPPER_TOOL_SELECT,true);
	toolsToolbar->setButton(ID_MAPPER_TOOL_CREATE_ROOM,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_CREATE_PATH,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_CREATE_TEXT,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_DELETE,false);
	map->setTool(ID_MAPPER_TOOL_SELECT);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_SELECT,true);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_CREATE_ROOM,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_CREATE_PATH,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_CREATE_TEXT,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_DELETE,false);

	edit_menu->setItemEnabled(ID_MAPPER_EDIT_CUT,true);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_COPY,true);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_PASTE,true);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_DELETE,true);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_SELECTALL,true);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_DESELECTALL,true);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_INVERTSEL,true);

	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_CUT,true);
	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_COPY,true);
	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_PASTE,true);
}

/** This slot is used to activate the create room tool */
void CMapWindow::slotToolCreateRoom(void)
{
	toolsToolbar->setButton(ID_MAPPER_TOOL_SELECT,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_CREATE_ROOM,true);
	toolsToolbar->setButton(ID_MAPPER_TOOL_CREATE_PATH,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_CREATE_TEXT,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_DELETE,false);
	map->setTool(ID_MAPPER_TOOL_CREATE_ROOM);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_SELECT,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_CREATE_ROOM,true);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_CREATE_PATH,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_CREATE_TEXT,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_DELETE,false);

	edit_menu->setItemEnabled(ID_MAPPER_EDIT_CUT,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_COPY,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_PASTE,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_DELETE,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_SELECTALL,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_DESELECTALL,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_INVERTSEL,false);

	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_CUT,false);
	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_COPY,false);
	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_PASTE,false);
}

/** This slot is used to activate the create path tool */
void CMapWindow::slotToolCreatePath(void)
{
	toolsToolbar->setButton(ID_MAPPER_TOOL_SELECT,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_CREATE_ROOM,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_CREATE_PATH,true);
	toolsToolbar->setButton(ID_MAPPER_TOOL_CREATE_TEXT,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_DELETE,false);
	map->setTool(ID_MAPPER_TOOL_CREATE_PATH);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_SELECT,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_CREATE_ROOM,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_CREATE_PATH,true);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_CREATE_TEXT,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_DELETE,false);

	edit_menu->setItemEnabled(ID_MAPPER_EDIT_CUT,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_COPY,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_PASTE,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_DELETE,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_SELECTALL,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_DESELECTALL,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_INVERTSEL,false);

	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_CUT,false);
	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_COPY,false);
	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_PASTE,false);
}

/** This slot is used to activate the create text tool */
void CMapWindow::slotToolCreateText(void)
{
	toolsToolbar->setButton(ID_MAPPER_TOOL_SELECT,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_CREATE_ROOM,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_CREATE_PATH,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_CREATE_TEXT,true);
	toolsToolbar->setButton(ID_MAPPER_TOOL_DELETE,false);
	map->setTool(ID_MAPPER_TOOL_CREATE_TEXT);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_SELECT,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_CREATE_ROOM,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_CREATE_PATH,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_CREATE_TEXT,true);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_DELETE,false);

	edit_menu->setItemEnabled(ID_MAPPER_EDIT_CUT,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_COPY,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_PASTE,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_DELETE,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_SELECTALL,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_DESELECTALL,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_INVERTSEL,false);

	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_CUT,false);
	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_COPY,false);
	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_PASTE,false);
}

/** This slot is used to activate the erase tool */
void CMapWindow::slotToolErase(void)
{
	toolsToolbar->setButton(ID_MAPPER_TOOL_SELECT,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_CREATE_ROOM,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_CREATE_PATH,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_CREATE_TEXT,false);
	toolsToolbar->setButton(ID_MAPPER_TOOL_DELETE,true);
	map->setTool(ID_MAPPER_TOOL_DELETE);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_SELECT,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_CREATE_ROOM,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_CREATE_PATH,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_CREATE_TEXT,false);
	tools_menu->setItemChecked(ID_MAPPER_TOOL_DELETE,true);

	edit_menu->setItemEnabled(ID_MAPPER_EDIT_CUT,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_COPY,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_PASTE,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_DELETE,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_SELECTALL,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_DESELECTALL,false);
	edit_menu->setItemEnabled(ID_MAPPER_EDIT_INVERTSEL,false);

	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_CUT,false);
	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_COPY,false);
	mainToolbar->setItemEnabled(ID_MAPPER_EDIT_PASTE,false);


}

/** This slot is used to create a new top level */
void CMapWindow::slotCreateTopLevel(void)
{
	map->createLevel(UP);
}

/** This slot is used to create a new bottom level */
void CMapWindow::slotCreateBottomLevel(void)
{
	map->createLevel(DOWN);
}


/** This slot is used to show/hide the grid */
void CMapWindow::slotShowGrid(void)
{
	bGridShow = !bGridShow;
	map->setGridVisable(bGridShow);
	toolsToolbar->setButton(ID_MAPPER_TOGGLE_GRID,bGridShow);
	view_menu->setItemChecked(ID_MAPPER_TOGGLE_GRID,bGridShow);
}

/** Turn on or off the higher map */
void CMapWindow::slotViewHigherMap(void)
{
	bViewHigherMap= !bViewHigherMap;	
	view_menu->setItemChecked(ID_MAPPER_VIEW_HIGHER,bViewHigherMap);
	map->setViewHigherMap(bViewHigherMap);
}

/** Turn on or off the lower map */
void CMapWindow::slotViewLowerMap(void)
{
	bViewLowerMap= !bViewLowerMap;
	view_menu->setItemChecked(ID_MAPPER_VIEW_LOWER,bViewLowerMap);
	map->setViewLowerMap(bViewLowerMap);
}

/** turn on or off the main toolbar */
void CMapWindow::slotViewMainToolBar(void)
{
	bViewMainToolbar=!bViewMainToolbar;
	menuBar()->setItemChecked(ID_MAPPER_VIEW_MAIN, bViewMainToolbar);	
	if (bViewMainToolbar) enableToolBar(KToolBar::Show,ID_MAPPER_TOOLBAR_MAIN); else enableToolBar(KToolBar::Hide,ID_MAPPER_TOOLBAR_MAIN);
}

/** turn on or off the modes toolbar */
void CMapWindow::slotViewModesToolBar(void)
{
	bViewModesToolbar=!bViewModesToolbar;
	menuBar()->setItemChecked(ID_MAPPER_VIEW_MODES, bViewModesToolbar);	
	if (bViewModesToolbar) enableToolBar(KToolBar::Show,ID_MAPPER_TOOLBAR_MODES); else enableToolBar(KToolBar::Hide,ID_MAPPER_TOOLBAR_MODES);
}

void CMapWindow::slotMovePlayer(direction dir)
{
	emit movePlayer(dir);
}

void CMapWindow::slotLevelChanged(signed int level)
{
	QString lvlStr;
	
	lvlStr.sprintf(i18n("Level : %d"),level);
	statusBar()->changeItem(lvlStr, ID_MAPPER_LEVEL_NUM );
}

void CMapWindow::slotFileInfo(void)
{
        DlgMapInfo d(this, "Information");
        CMapRoom *loginRoom = map->getLoginRoom();
        int lvl,room,text,path;

        map->getTotals(&lvl,&room,&text,&path);

        d.setStart(loginRoom->getX(),loginRoom->getY(),loginRoom->getLevel());
        d.setGeneral(mud_Title,mud_Login);
        d.setTotals(lvl,room,path,text);

        d.exec();
}       
