/***************************************************************************
                          cmudview.cpp
                      -------------------
    description		 : class providing a colored output view
    begin		       : Wed Sep 8 1999
    copyright		   : (C) 1999 by Stephan Uhlmann,
                                 Andre Alexander Bell
    email	         : suhlmann@gmx.de
                     andre.bell@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#include "cmudview.h"

#include <qtableview.h>
#include <qpixmap.h>
#include <qregexp.h>
#include <qfontmetrics.h>

// for copy&paste
#include <qclipboard.h>
#include <qapplication.h>

#include <kapp.h>


#include <iostream.h>


CMudView::CMudView(KmudDoc* document, QWidget * parent, const char * name)
 : QTableView(parent,name)
{
	doc = document;

	myParent = parent;

	init();

	setMaxLines(5000);

	setForeground(QColor(0,0,0));
	setBackground(QColor(255,255,255));
	setFont(QFont("fixed"));

	// for cut&paste
	markedAreaStartRow = -1;
	markedAreaStartCol = -1;
	markedAreaEndRow = -1;
	markedAreaEndCol = -1;
	markedText = false;

	// dunno why i have to cast here, maybe multiple inheritance in QScrollBar
	QObject::connect((QObject*)this->verticalScrollBar(),SIGNAL(sliderMoved(int)),SLOT(slotVSliderMoved(int)));
	QObject::connect((QObject*)this->verticalScrollBar(),SIGNAL(valueChanged(int)),SLOT(slotVSliderMoved(int)));

}


CMudView::~CMudView()
{
}


void CMudView::init()
{
	MudViewLineData* ld;

	setAutoUpdate(false);

	setTopLeftCell(0,0);
	lineX=0;

	data.setAutoDelete(true);
	data.clear();


	ld = new MudViewLineData;
	ld->chunks.setAutoDelete(true);
	ld->chunks.clear();
	ld->lineHeight=0;
	data.append(ld);

	setNumRows(1);
	setNumCols(1);
	updateTableSize();

	fontMetric=NULL;

	setBackgroundColor(defaultBackground);
	setTableFlags( Tbl_autoVScrollBar );
	setFrameStyle( QFrame::WinPanel | QFrame::Sunken );
	setCellHeight(ld->lineHeight);
	setCellWidth(viewWidth());

	vSliderAtBottom=false;

	setAutoUpdate(true);
	repaint();

}


void CMudView::paintCell(QPainter* painter, int row, int col )
{
	QPainter* p;
	QPixmap* buf;
	MudViewChunkData* chunk;
	QFontMetrics* fm;
	QRect rect = cellUpdateRect();
	int x=0, c=0, col1=0, col2=0;
	uint l;

	buf = new QPixmap;

	buf->resize(cellWidth(col),cellHeight(row));
	buf->fill(defaultBackground);

	p = new QPainter (buf);
	p->setBackgroundMode(OpaqueMode);

	chunk=data.at(row)->chunks.first();

	while (chunk!=NULL) {
		p->setFont(chunk->font);
		fm = new QFontMetrics(chunk->font);
		if (!rowMarked(row)) {
			if (markedAreaStartRow > markedAreaEndRow) {
				if (row == markedAreaStartRow) col2 = 0;
				else col1 = 1000; //row==markedAreaEndRow
			} else {
				col1=0;
				col2=1000;
			}
			if (row == markedAreaStartRow) {
				col1 = markedAreaStartCol;
			}
			if (row == markedAreaEndRow) {
				col2 = markedAreaEndCol;
			}
			if (col1 > col2) { //swap col1 and col2
				col1 ^= col2;
				col2 ^= col1;
				col1 ^= col2;
			}
			for (l = 0;  l < chunk->s.length (); l++) {
				if (((c + (int)l) < col1) || ((c + (int)l) >= col2)){
		   			p->setPen(chunk->fgc);
   					p->setBackgroundColor(chunk->bgc);
				} else {
		   			p->setPen(chunk->bgc);
		   			p->setBackgroundColor(chunk->fgc);
				}
				p->drawText(x,0,buf->width()-x,buf->height(), AlignLeft | AlignTop, chunk->s.mid (l, 1));
				x+=fm->width(chunk->s.mid(l, 1));
			}
		} else {
			p->setPen(chunk->fgc);
			p->setBackgroundColor(chunk->bgc);
			if (rowMarked (row) == 1) {
				p->setPen(chunk->bgc);
				p->setBackgroundColor(chunk->fgc);
			} 
			p->drawText(x,0,buf->width()-x,buf->height(), AlignLeft | AlignTop, chunk->s);
			x+=fm->width(chunk->s);
		}
		c += chunk->s.length ();
		delete fm;

		chunk=data.at(row)->chunks.next();
	}
	delete p;

	painter->drawPixmap(rect.topLeft(), *buf, rect);

	delete buf;
}

void CMudView::resizeEvent(QResizeEvent* e)
{
	QTableView::resizeEvent(e);
	setCellWidth(viewWidth());
}

QString CMudView::append(QString s, QColor foreground, QColor background, QFont font)
{
	MudViewLineData* ld;
	QString ansiAttrib[8];

	// t is important (thats why it has the descibing name "t" ;-)
	// we iterate through s char by char and build up t
	// t is then flushed to the current chunk occasionally
	QString t="";

	QString ansiCode, ansiCodeChunk, ansiCodeAttrib,u,real,tabSpace;

	int tabSpaceI,i=0,l=0,ansiStart=0,ansiI,lastAttrib=0,count=0;
	//int ansiEnd=0;
	char c, ansiCommand = 'm';
	bool useBold=false, useBright=false, colorUsed=false, useNormal=true;
	bool useUnderline=false;
	
	bool useBlack=false, useRed=false, useGreen=false, useYellow=false;
	bool useBlue=false, useMagenta=false, useCyan=false, useWhite=false;
	
	bool scrollDown=(yOffset()==maxYOffset());

	QString newWord = QString("");
	QString oldLine = QString("");

	// the string we return, cleared from ANSI codes and certain special chars
	// used for logging
	QString returnString="";




	setAutoUpdate(false); // avoid flickering
	for (i=0;i<(int)s.size();i++) {
		c=s.at(i);
		switch (c) {
			case '\007' :	
				if (useBeep == true) KApplication::beep();
				break;

			case '\r'   :	break; // skip all \r dos-cr's

			case '\t'   : // expand tabs to spaces
				tabSpace="";
				for (tabSpaceI=0;tabSpaceI<(8-(lineX&7));tabSpaceI++)
					tabSpace+=" ";
				s.replace(i,1,tabSpace);
				i--;lineX--;

				break;

			case '\n'   : // flush the chunk
				returnString+=t+"\n";
				flushChunk(t,foreground,background,font);

				//and make a new line
				ld = new MudViewLineData;
				(ld->chunks).setAutoDelete(true);
				ld->lineHeight=0;
				data.append(ld);
				setNumRows(numRows()+1);
				updateTableSize();

				oldLine.setStr("");
				newWord.setStr("");
				lineX=0;


				break;


			case '\x1B' :   // flush chunk and do the ansi/vt100 escape code interpretation
				returnString+=t;
				flushChunk(t,foreground,background,font);

				ansiStart=i;

				/** Find the end of the ANSI chunk */
				//ansiEnd=s.find('m',ansiStart,true);

/** WARNING: it might be that there is an escape in the telnet buffer but the
 ** appropiate 'm' is in the next buffer.  Fixed that in CTelnet, but one "if"
 ** more does no harm...
 **/
				//if (ansiEnd < ansiStart)
					/*break;*/ /** sanity check */

				/** The code chunk */
				//ansiCode = s.mid(ansiStart+2,ansiEnd-ansiStart-1);
				//ansiCodeChunk = ansiCode;

				/** The end of the code chunk */
				//i = ansiEnd;

				/** detects the ANSI-Sequnces */
				l = 1;
				ansiCodeChunk = "";
				ansiCommand = s.at (ansiStart + 1);
				ansiCode = ansiCommand;
				while (((ansiStart+l+1) < ((int) s.length ())) 
				&& ((ansiCommand<'a') || (ansiCommand>'z')) 
				&& ((ansiCommand<'A') || (ansiCommand>'Z'))){
					l++;
					ansiCommand = s.at (ansiStart + l);
					ansiCode += ansiCommand;
					ansiCodeChunk += ansiCommand;
				} // while
				i = ansiStart + l;

            /** This loop gets the ansi codes into an array */
				if (ansiCommand == 'm')	{
					count = 0;
					while ((ansiCode.isEmpty()==false) 
					&& (count < 8)){
						ansiI = ansiCode.find(QRegExp("[;m]"));
						ansiAttrib[count] = ansiCode.mid(0, ansiI);
						count++;
						ansiCode = ansiCode.mid(ansiI + 1, ansiCode.length() - ansiI - 1);
					}
				} // if

				lastAttrib = count;
				count = 0;

				switch (interpretType) {
					case 0: /** DEFAULT -- detects ANSI/VT100 for you */
					case 1: /** ANSI -- Force ANSI -- Combined 0 and 1 because they're so similar */

                        /** \x1B[2J Erases the screen with the background color and moves the cursor home.
                         ** According to ANSI Standard, that sequence "Clears screen and homes cursor".
                         **/
						if ((s.at(ansiStart + 1) == '[') && (s.at(ansiStart + 2) == '2') && (s.at(ansiStart + 3) == 'J')){
							erase();
							init();
							// i = ansiStart + 4;
							break;
			                        }

						count = 0;

						while (count < lastAttrib) {
				                        /** Loop through and find the values 0-9 */
							if ((ansiAttrib[count].toInt() >= 0) && (ansiAttrib[count].toInt() <= 9)){
								useNormal=false;
								switch (ansiAttrib[count].toInt()) {
									case 0:
										actualBackground = defaultBackground;
										useBold = false;
										useBright = false;
										useUnderline = false;
										colorUsed = false;
										useNormal = true;
                                                	
										useBlack = false;
										useRed = false;
										useGreen = false;
										useYellow = false;
										useBlue = false;
										useMagenta = false;
										useCyan = false;
										useWhite = false;
									break;
									case 1:
										useBright = (colorUsed == true)||(interpretType == 1);
										useBold = !useBright;
									break;
									case 2:
										useBold = false;
										useBright = false;
									break;
									case 4:
										useUnderline = true;
									break;
									case 5:
										/** Insert blinking code here */
									break;
									case 7:
										actualBackground = defaultForeground;
									case 8:
										actualForeground = defaultBackground;
								}
							}
							//forground colors
							else if ((ansiAttrib[count].toInt() >= 30) && (ansiAttrib[count].toInt() <= 37)){
								actualForeground = defaultForeground;
								colorUsed = true;
								useNormal = false;
								actualFont.setBold(false);

								if (useBold == true) {
									useBright = true;
									useBold = false;
								}
								
								useBlack = useRed = useGreen = useYellow = false;
								useBlue = useMagenta = useCyan = useWhite = false;
								
								switch (ansiAttrib[count].toInt()) {
									case 30: useBlack =true;    break;
									case 31: useRed = true;     break;
									case 32: useGreen = true;   break;
									case 33: useYellow = true;  break;
									case 34: useBlue = true;    break;
									case 35: useMagenta = true; break;
									case 36: useCyan = true;    break;
									case 37: useWhite = true;   break;
								}
							}
							
							/** Background colors */
							else if ((ansiAttrib[count].toInt() >= 40) && (ansiAttrib[count].toInt() <= 47)) {
								switch (ansiAttrib[count].toInt()) {
									case 40: actualBackground = FBgColor[BG_BLACK]; break;
									case 41: actualBackground = FBgColor[BG_RED]; break;
									case 42: actualBackground = FBgColor[BG_GREEN]; break;
									case 43: actualBackground = FBgColor[BG_YELLOW]; break;
									case 44: actualBackground = FBgColor[BG_BLUE]; break;
									case 45: actualBackground = FBgColor[BG_MAGENTA]; break;
									case 46: actualBackground = FBgColor[BG_CYAN]; break;
									case 47: actualBackground = FBgColor[BG_WHITE]; break;
								}
							}
							count++;
						}

						actualFont.setBold((!useBright)&&(useBold));
						if (useNormal == true) {
							actualForeground=defaultForeground;
							actualBackground=defaultBackground;
							actualFont=defaultFont;
						}
						if (useUnderline == true)
							actualFont.setUnderline(true);
						
						if (useBright){
							if (useBlack) actualForeground = FANSIColor[GRAY];
							else if (useRed) actualForeground = FANSIColor[LIGHT_RED];
							else if (useGreen) actualForeground = FANSIColor[LIGHT_GREEN];
							else if (useYellow) actualForeground = FANSIColor[LIGHT_YELLOW];
							else if (useBlue) actualForeground = FANSIColor[LIGHT_BLUE];
							else if (useMagenta) actualForeground = FANSIColor[LIGHT_MAGENTA];
							else if (useCyan) actualForeground = FANSIColor[LIGHT_CYAN];
							else if (useWhite) actualForeground = FANSIColor[LIGHT_WHITE];
						} else {
							if (useBlack) actualForeground = FANSIColor[BLACK];
							else if (useRed) actualForeground = FANSIColor[DARK_RED];
							else if (useGreen) actualForeground = FANSIColor[DARK_GREEN];
							else if (useYellow) actualForeground = FANSIColor[DARK_YELLOW];
							else if (useBlue) actualForeground = FANSIColor[DARK_BLUE];
							else if (useMagenta) actualForeground = FANSIColor[DARK_MAGENTA];
							else if (useCyan) actualForeground = FANSIColor[DARK_CYAN];
							else if (useWhite) actualForeground = FANSIColor[DARK_WHITE];
						}
						break;
					case 2: /** VT100 -- forces VT100 interpretation */
						count = 0;
						int a;
						while (count < lastAttrib){
							a=ansiAttrib[count].toInt();
							if (a == 0)
								actualFont.setBold(false);
							else if (a == 1)
								actualFont.setBold(true);
							count++;
						}
						break;
					case 3: /** NONE -- eats ANSI codes */
						break;
					case 4: /** DEBUG -- shows ANSI codes as BOLD*/
						useDebugHighlight = true;
						MudViewChunkData* v = new MudViewChunkData;

						ansiCodeChunk = QString("[").append(ansiCodeChunk);

						/** For choosing whether or not to highlight ANSI codes */
//						if (useDebugHighlight == true)
							v->fgc = DEBUG_HIGHLIGHT;
//commented out because we just set useDebugHiglight=true
//						else 
//							v->fgc = defaultForeground;

						if (background==defaultBackground) 
							v->bgc=actualBackground; 
						else 
							v->bgc=background;

						if (font!=defaultFont)
							actualFont=font;
						v->font=actualFont;

						v->s=ansiCodeChunk;
						(data.last())->chunks.append(v);
						break;
					}
					// An escape character must not be counted for the wrapping
					lineX--;
					break;

			// case '\x1B' ends

			case ' '    : 
				t+=c;
				newWord += c;
				if ( fontMetric && (fontMetric->width(oldLine,-1)+fontMetric->width(newWord,-1)) > (myParent->width()-72) ){
				// remove the last word from t and put in on the next line
					if( t.length() > newWord.length() ) {
						t = t.left( ( t.length() - newWord.length() ) );
						returnString+=t;
 						flushChunk(t,foreground,background,font);
						// new line
						ld = new MudViewLineData;
						(ld->chunks).setAutoDelete(true);
						ld->lineHeight=0;
						data.append(ld);
						setNumRows(numRows()+1);
						updateTableSize();
						t = newWord;
						lineX= t.length() -1;
					} else {
						returnString+=t;
						flushChunk(t,foreground,background,font);
						// new line
						ld = new MudViewLineData;
						(ld->chunks).setAutoDelete(true);
						ld->lineHeight=0;
						data.append(ld);
						setNumRows(numRows()+1);
						updateTableSize();
						lineX=0;
					}
					oldLine.setStr("");
					newWord.setStr("");

				} else {
					oldLine += newWord;
					newWord.setStr("");
				}

				break;
			default:
				t+=c;
				newWord += c;
		}
		lineX++;
	}


	if (!t.isEmpty())
	{
		returnString+=t;
		flushChunk(t,foreground,background,font);
	}

	// shrink view to maxLines
	setMaxLines(maxLines);

	if (scrollDown)
		setYOffset(maxYOffset());
		
	setAutoUpdate(true);
	repaint(false); // no need to erase, we do it in paintCell() anyway

	return returnString;
}

int CMudView::lineLength(int line)
{
	return cellWidth(line);
}

int CMudView::numLines()
{
	return data.count()-1;
}

int CMudView::cellHeight(int row)
{
	if ((unsigned int)row > data.count()-1) 
		return 0;
	return (data.at(row)->lineHeight);
}

void CMudView::setForeground(QColor fgc)
{
	defaultForeground=actualForeground=fgc;
	repaint(false);
}

void CMudView::setBackground(QColor bgc)
{
	defaultBackground=actualBackground=bgc;
	setBackgroundColor(bgc);
	repaint(false);
}

void CMudView::setFont(QFont font)
{
	defaultFont=actualFont=font;
	if (fontMetric!=NULL) delete fontMetric;
	fontMetric = new QFontMetrics( font );
	repaint(false);
}

void CMudView::setInterpretType(int i)
{
	interpretType = i;
}

void CMudView::setDebugHighlightColor(QColor color)
{
	DEBUG_HIGHLIGHT = color;
}

// FOR ANSI COLORS

void CMudView::setOutputANSIColor(int ANSIColor, QColor color)
{
  FANSIColor[ANSIColor]=color;
}

// background colors

void CMudView::setOutputBgColor(int BgColor, QColor color)
{
  FBgColor[BgColor]=color;
}

QColor CMudView::getForeground()
{
	return defaultForeground;
}

QColor CMudView::getBackground()
{
	return defaultBackground;
}

QFont CMudView::getFont()
{
	return defaultFont;
}

int CMudView::getInterpretType()
{
	return interpretType;
}

// For Debug Highlighting
QColor CMudView::getDebugHighlightColor()
{
	return DEBUG_HIGHLIGHT;
}

// FOR ANSI COLORS

QColor CMudView::getOutputANSIColor(int ANSIColor)
{
  return FANSIColor[ANSIColor];
}

// background colors

QColor CMudView::getOutputBgColor(int BgColor)
{
  return FBgColor[BgColor];
}

void CMudView::setMaxLines(int newMaxLines)
{
	int i,pixels=0;
	// calculate number of pixels to be removed if maxlines are smaller now
	if (newMaxLines < numLines()){
		for (i=0;i<(numLines()-newMaxLines);i++)
			pixels+=cellHeight(i);
	}

	maxLines=newMaxLines;
	if (newMaxLines<=0)
		maxLines=1;
	while (numLines()>maxLines) {
		data.removeFirst();
		markedAreaEndRow--;
		markedAreaStartRow--;
	}
	setYOffset(yOffset()-pixels);
	setNumRows(maxLines+1);
	updateTableSize();
}

void CMudView::slotScrollKey(QKeyEvent * k){
	bool needrepaint = false;
	bool sab=false; //slider at bottom

	setAutoUpdate(false);
	switch (k->key()){
		case Key_Up:
			if (yOffset()>0) {
				setOffset(0,yOffset()-actualFont.pointSize());
				needrepaint = true;
			}
			break;
		case Key_Down:
			if (yOffset()<maxYOffset()) {
				setOffset(0,yOffset()+actualFont.pointSize());
				needrepaint = true;
			}
			break;
		case Key_PageUp:
			if (yOffset()>0){
				setOffset(0,yOffset()-(viewHeight()>>1));
				needrepaint = true;
			}
			break;
		case Key_PageDown:
			if (yOffset()<maxYOffset()){
				setOffset(0,yOffset()+(viewHeight()>>1));
				needrepaint = true;
			}
			break;
		case Key_Home:
			if (yOffset()>0){
				setOffset(0,0);
			        needrepaint = true;
			}
			break;
		case Key_End:
			if (yOffset()<maxYOffset()){
				setOffset(0,maxYOffset());
				needrepaint = true;
			}
			break;
	}

	if (yOffset()!=maxYOffset())
		sab=true;
	
	if (sab!=vSliderAtBottom){
		vSliderAtBottom=sab;
		emit(vSliderAtBottomChanged(vSliderAtBottom));
	}

	setAutoUpdate(true);
	if (needrepaint) 
		repaint();
}

/** copies the marked text in the X-Clipboard */
void CMudView::copyMarkedText (){
	QClipboard* cb = QApplication::clipboard();
	if (markedText) 
		cb->setText (getMarkedText());
}

/** overwrites the mousePressEvent-Method from QWidget
implements the start of marking a text-area */
void CMudView::mousePressEvent (QMouseEvent* e){
	int oldStartRow = markedAreaStartRow;
	int oldEndRow = markedAreaEndRow;
	int l;

	markedText = false;
	if (oldStartRow > oldEndRow){//swap
		oldStartRow ^= oldEndRow;
		oldEndRow ^= oldStartRow;
		oldStartRow ^= oldEndRow;
	}
	markedAreaStartRow = findRow (e->y ());
	if (markedAreaStartRow == -1){
		markedAreaStartRow = data.count()-2; //numlines-1
		markedAreaStartCol = 0;
	} else	{
		markedAreaStartCol=getCharacterNumber(markedAreaStartRow, e->x());
	}
	markedAreaEndCol = markedAreaStartCol;
	markedAreaEndRow = markedAreaStartRow;
	for (l = oldStartRow; l <= oldEndRow; l++)
		updateCell (l, 0);
	updateCell (markedAreaStartRow, 0);

	QTableView::mousePressEvent(e);
	
}

/** gets the number of the character in the line row and on the position xPos */
int CMudView::getCharacterNumber (int Row, int xPos){
	MudViewChunkData* chunk;
	int x = 0;
	int n = 0;
	uint l;
	bool gotit = false;
	QFontMetrics* fm;
	QString s;

	chunk = data.at(Row)->chunks.first();
	while (chunk!=NULL){
		fm = new QFontMetrics (chunk->font);
		s = chunk->s;
		for (l = 0; l < s.length(); l++){
			x += fm->width (s.mid (l, 1));
			if (x > xPos) { 
				gotit = true; 
				break; 
			}
			n++;
		} 
		delete fm;
		if (gotit) 
			break;
		chunk=data.at(Row)->chunks.next();
	}
	return n;
}

/** overwrites the mouseMoveEvent from QWidget
implements the marking of a area */
void CMudView::mouseMoveEvent (QMouseEvent* e){
	int oldendRow = markedAreaEndRow;
	int oldendCol = markedAreaEndCol;
	int newRow,newCol;
	int l;

	newRow = findRow (e->y ());
	if (newRow == -1){
		newCol = 0;
		newRow = 0;
		if (e->y() > 10){
			newRow = numLines ();
			newCol = 10000;
		}
	} else
		newCol = getCharacterNumber (markedAreaEndRow, e->x ());

	if ((data.count()> 1) && ((newRow != oldendRow) || (newCol != oldendCol)))	{
		markedText = true;
		markedAreaEndRow = newRow;
		markedAreaEndCol = newCol;
		copyMarkedText ();
		if (oldendRow > newRow)	{ //swap
			oldendRow ^= newRow;
			newRow ^= oldendRow;
			oldendRow ^= newRow;
		}
		for (l = oldendRow; l <= newRow; l++)
			updateCell (l, 0);
	}

	QTableView::mouseMoveEvent(e);
}

/** returns the text in the marked area */
QString CMudView::getMarkedText (){
	
	// No text marked - no need to get or even make variables
	if ((!markedText)
	||  ((markedAreaStartRow==markedAreaEndRow)
	&&   (markedAreaStartCol==markedAreaEndCol)))
		return "";
	
	QString markedText = "";
	MudViewChunkData* chunk;
	MudViewLineData* line;
	int row1 = markedAreaStartRow, row2 = markedAreaEndRow;
	int col1 = markedAreaStartCol, col2 = markedAreaEndCol;
	int l;

	if (row1 == row2){
		if (col1 > col2) { //swap
			col1 ^= col2;
			col2 ^= col1;
			col1 ^= col2;
		}
		chunk = data.at(row1)->chunks.first();
	 	l = 0;
		// let it be much simplier.
		while (chunk!=0)
		{
			markedText += chunk->s;
			chunk = data.at(row1)->chunks.next();
		}
		return markedText.mid(col1,col2-col1);
	}

	else if (row1 > row2) //swap rows and cols
	{
		row1 ^= row2;
		row2 ^= row1;
		row1 ^= row2;
		
		col1 ^= col2;
		col2 ^= col1;
		col1 ^= col2;
	}

        line=data.at(row1);
        if (!line) return markedText;
	chunk = line->chunks.first();
	l = 0;
	int len;
	while (chunk != 0){
		len=chunk->s.length();
		if (len - col1 + l > 0)
			markedText+=chunk->s.right(len - col1 + l);
		l += len;
		chunk = data.at(row1)->chunks.next();
	}
	markedText = markedText + "\n";
	for (l = row1 + 1; l < row2; l++) {
		chunk = data.at(l)->chunks.first();
		while (chunk != 0){
			markedText += chunk->s;
			chunk = data.at(l)->chunks.next();
		} 
		markedText += "\n";
	} 
	chunk = data.at(row2)->chunks.first();
	l = 0;
	while ( (chunk != 0) && (l < col2) ) {
		markedText += chunk->s.left (col2 - l);
		l += chunk->s.length ();
		chunk = data.at(row2)->chunks.next();
	}

	return markedText;
}

/** returns true, if the specified row is marked */
int CMudView::rowMarked (int Row){
	if (markedText){
		if (((Row > markedAreaStartRow) && (Row < markedAreaEndRow)) 
		|| ((Row < markedAreaStartRow) && (Row > markedAreaEndRow)))
			return 1;
		else if ((Row == markedAreaStartRow) || (Row == markedAreaEndRow)) 
			return 0;
	}
	return -1;
}

void CMudView::setUseBeep(bool u) {
	useBeep = u;
}

bool CMudView::getUseBeep() {
	return useBeep;
}

void CMudView::slotVSliderMoved(int i)
{
	bool sab=(yOffset()!=maxYOffset());

	if (sab!=vSliderAtBottom) {
		vSliderAtBottom=sab;
		emit(vSliderAtBottomChanged(vSliderAtBottom));
	}
}

int CMudView::getVScrollBarWidth()
{
	return ((QWidget*)(verticalScrollBar()))->width();
}

void CMudView::setVScrollBarVisible(bool visible)
{
	if (visible) {
		setTableFlags( Tbl_autoVScrollBar );
		return;
	}
	clearTableFlags( Tbl_autoVScrollBar );
}

/** overwrites the mouseDoubleClickEvent inherited from QWidget
marks the word the mouse pints to */
void CMudView::mouseDoubleClickEvent (QMouseEvent* e)
{
	int oldStartRow = markedAreaStartRow;
	int oldEndRow = markedAreaEndRow;
	int l;
	QTableView::mouseDoubleClickEvent(e);
	if (oldStartRow > oldEndRow){//swap
		oldStartRow ^= oldEndRow;
		oldEndRow ^= oldStartRow;
		oldStartRow ^= oldEndRow;
	}
	markedAreaStartRow = findRow (e->y ());
	if (markedAreaStartRow == -1){
		markedAreaStartRow = data.count()-2;//numlines-1
		markedAreaStartCol = 0;
		markedText = false;
	} else {
		markedAreaStartCol = getCharacterNumber (markedAreaStartRow, e->x ());
		markedAreaEndCol = markedAreaStartCol;
		while ((markedAreaStartCol > 0)
			&& (((getCharacterAt (markedAreaStartRow, markedAreaStartCol) >= 'a') && (getCharacterAt (markedAreaStartRow, markedAreaStartCol) <= 'z'))
			|| ((getCharacterAt (markedAreaStartRow, markedAreaStartCol) >= 'A') && (getCharacterAt (markedAreaStartRow, markedAreaStartCol) <= 'Z'))
			|| ((getCharacterAt (markedAreaStartRow, markedAreaStartCol) >= '0') && (getCharacterAt (markedAreaStartRow, markedAreaStartCol) <= '9'))))
				markedAreaStartCol --;
		while ((markedAreaEndCol <= getLineLength (markedAreaStartRow))
			&& (((getCharacterAt (markedAreaStartRow, markedAreaEndCol) >= 'a') && (getCharacterAt (markedAreaStartRow, markedAreaEndCol) <= 'z'))
			|| ((getCharacterAt (markedAreaStartRow, markedAreaEndCol) >= 'A') && (getCharacterAt (markedAreaStartRow, markedAreaEndCol) <= 'Z'))
			|| ((getCharacterAt (markedAreaStartRow, markedAreaEndCol) >= '0') && (getCharacterAt (markedAreaStartRow, markedAreaEndCol) <= '9'))))
				markedAreaEndCol++;
		if (markedAreaStartCol != 0) 
			markedAreaStartCol++;
		markedText = true;
	}
	markedAreaEndRow = markedAreaStartRow;
	for (l = oldStartRow; l <= oldEndRow; l++){
		updateCell (l, 0);
	}
	updateCell (markedAreaStartRow, 0);
	copyMarkedText ();
} 

/** returns the Character at the Position (x, y) */
char CMudView::getCharacterAt (int row, int col)
{
	MudViewLineData* line = data.at (row);
	MudViewChunkData* chunk = line->chunks.first ();
	int l = 0;
	char c = 0;
	while ((chunk != 0) && (c == 0) && (getLineLength (row) > col))	{
		if ( (int)( l + chunk->s.length () ) < col){
			l += chunk->s.length ();
			chunk = line->chunks.next ();
		} else
			c = chunk->s[col - l];
	}
	return c;
}

/** returns the length of the specified row */
int CMudView::getLineLength (int row)
{
	MudViewLineData* line = data.at (row);
	MudViewChunkData* chunk = line->chunks.first ();
	int l = 0;
	while (chunk != 0){
		l += chunk->s.length ();
		chunk = line->chunks.next ();
	} 
	return l;
}

/** Flush a chunk so we can start working on a new one.
	* Warning: resets incoming variable "t" to empty string
	*/
void CMudView::flushChunk(QString &t, QColor foreground, QColor background, QFont font){
	MudViewChunkData *cd;
	QFontMetrics* fm;
	
	cd = new MudViewChunkData;
	if (foreground==defaultForeground)
		cd->fgc=actualForeground;
	else
		cd->fgc=foreground;
	if (background==defaultBackground)
		cd->bgc=actualBackground;
	else
		cd->bgc=background;
	if (font!=defaultFont)
		actualFont=font;
	cd->font=actualFont;
	
	cd->s=t;
	(data.last())->chunks.append(cd);
	fm = new QFontMetrics(font);
	if (fm->height() > data.last()->lineHeight)
		data.last()->lineHeight=fm->height();
	delete fm;
	t="";
}


