/*-
 * masaki.c - The Masaki family for xlock, the X Window System lockscreen.
 *
 * Copyright (c) 1988 by Sun Microsystems, picture (C) by AIC
 *
 * See xlock.c for copying information.
 *
 * Revision History:
 * 07-Oct-98: Masaki, a TenchiLock mode ported to KDE
 *            (David.Banz@smail.inf.fh-rhein-sieg.de)
 * 18-Sep-95: 5 bats now in color (patol@info.isbiel.ch)
 * 20-Sep-94: 5 bats instead of bouncing balls, based on bounce.c
 *            (patol@info.isbiel.ch)
 * 2-Sep-93: bounce version (David Bagley bagleyd@hertz.njit.edu)
 * 1986: Sun Microsystems
 */

/* 
 * original copyright
 * **************************************************************************
 * Copyright 1988 by Sun Microsystems, Inc. Mountain View, CA.
 *
 * All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the names of Sun or MIT not be used in advertising
 * or publicity pertaining to distribution of the software without specific
 * prior written permission. Sun and M.I.T. make no representations about the
 * suitability of this software for any purpose. It is provided "as is"
 * without any express or implied warranty.
 *
 * SUN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * IN NO EVENT SHALL SUN BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 * ***************************************************************************
 */

/* original "bat" mode ported to kscreensave:
   July 1997, Emanuel Pirker <epirker@edu.uni-klu.ac.at>
   Contact me if something doesn't work correctly!
   Last revised: 11-Jul-97
*/

// layout management added 1998/04/19 by Mario Weilguni <mweilguni@kde.org>

#include "xlock.h"
#include <math.h>

#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

#define MINSPEED 0
#define MAXSPEED 100
#define DEFSPEED 50
#define MINBATCH 0
#define MAXBATCH 20
#define DEFBATCH 5

#if HAVE_XPM
#if 1
#include <X11/xpm.h>
#else
#include <xpm.h>
#endif
#include "pixmaps/masaki-0.xpm"
#endif

#include "bitmaps/masaki-0.xbm"

#define MAX_STRENGTH 24
#define FRICTION 15
#define PENETRATION 0.4
#define SLIPAGE 4
#define TIME 32

#define ORIENTS 8
#define ORIENTCYCLE 32
#define CCW 1
#define CW (ORIENTS-1)
#define DIR(x)	(((x)>=0)?CCW:CW)
#define SIGN(x)	(((x)>=0)?1:-1)
#define ABS(x)	(((x)>=0)?x:-(x))

//ModeSpecOpt masaki_opts = {0, NULL, NULL, NULL};

static XImage bimages[] =
{
	{0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1},
	{0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1},
	{0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1},
	{0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1},
	{0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1}
};
static XImage *images[ORIENTS / 2 + 1];

typedef struct {
	int         x, y, xlast, ylast;
	int         spincount, spindelay, spindir, orient;
	int         vx, vy, vang;
	int         mass, size, sizex, sizey;
	unsigned long color;
} masakistruct;

typedef struct {
	int         width, height;
	int         nmasakis;
	int         restartnum;
	masakistruct   masakis[MAXBATCH];
} bouncestruct;

static bouncestruct bounces[MAXSCREENS];

static void checkCollision(int a_masaki);
static void drawamasaki(Window win, masakistruct * masaki);
static void movemasaki(masakistruct * masaki);
static void flapmasaki(masakistruct * masaki, int dir, int *vel);
static int  collide(int a_masaki);
static void XEraseImage(Display * display, Window win, GC gc, int x, int y, int xlast, int ylast, int xsize, int ysize);

static      first = 1;

static unsigned char *bits[] =
{
	masaki0_bits, masaki0_bits, masaki0_bits, masaki0_bits, masaki0_bits
};

#if HAVE_XPM
static char **pixs[] =
{
	masaki0, masaki0, masaki0, masaki0, masaki0
};

#endif

static void
init_images()
{
	int         i;

#if HAVE_XPM
	int         xpm_ret = 0;

	if (!mono && Scr[screen].npixels > 2)
		for (i = 0; i <= ORIENTS / 2; i++)
			xpm_ret += XpmCreateImageFromData(dsp, pixs[i], &(images[i]),
				   (XImage **) NULL, (XpmAttributes *) NULL);
	if (mono || Scr[screen].npixels <= 2 || xpm_ret != 0)
#endif
		for (i = 0; i <= ORIENTS / 2; i++) {
			bimages[i].data = (char *) bits[i];
			bimages[i].width = masaki0_width;
			bimages[i].height = masaki0_height;
			bimages[i].bytes_per_line = (masaki0_width + 7) / 8;
			images[i] = &(bimages[i]);
		}
}

void
initmasaki(Window win)
{
	bouncestruct *bp = &bounces[screen];
	int         i;
	XWindowAttributes xwa;

	if (first) {
		init_images();
		first = 0;
	}
	XGetWindowAttributes(dsp, win, &xwa);
	bp->width = xwa.width;
	bp->height = xwa.height;
	bp->restartnum = TIME;

	bp->nmasakis = batchcount;
	if (bp->nmasakis < 1)
		bp->nmasakis = 1;
	//if (!bp->masakis)
	//	bp->masakis = (masakistruct *) malloc(bp->nmasakis * sizeof (masakistruct));
	i = 0;
	while (i < bp->nmasakis) {
		if (masaki0_width > bp->width / 2 || masaki0_height > bp->height / 2) {
			bp->masakis[i].sizex = 7;
			bp->masakis[i].sizey = 3;
			bp->masakis[i].size = (bp->masakis[i].sizex + bp->masakis[i].sizey) / 2;
		} else {
			bp->masakis[i].sizex = masaki0_width;
			bp->masakis[i].sizey = masaki0_height;
			bp->masakis[i].size = (bp->masakis[i].sizex + bp->masakis[i].sizey) / 2;
		}
		bp->masakis[i].vx = ((LRAND() & 1) ? -1 : 1) * (LRAND() % MAX_STRENGTH + 1);
		bp->masakis[i].x = (bp->masakis[i].vx >= 0) ? 0 : bp->width - bp->masakis[i].sizex;
		bp->masakis[i].y = LRAND() % (bp->height / 2);
		if (i == collide(i)) {
			if (!mono && Scr[screen].npixels > 2)
				bp->masakis[i].color = Scr[screen].pixels[LRAND() % Scr[screen].npixels];
			else
				bp->masakis[i].color = WhitePixel(dsp, screen);
			bp->masakis[i].xlast = -1;
			bp->masakis[i].ylast = 0;
			bp->masakis[i].spincount = 1;
			bp->masakis[i].spindelay = 1;
			bp->masakis[i].vy = ((LRAND() & 1) ? -1 : 1) * (LRAND() % MAX_STRENGTH);
			bp->masakis[i].spindir = 0;
			bp->masakis[i].vang = 0;
			bp->masakis[i].orient = LRAND() % ORIENTS;
			i++;
		} else
			bp->nmasakis--;
	}
	XSetForeground(dsp, Scr[screen].gc, BlackPixel(dsp, screen));
	XFillRectangle(dsp, win, Scr[screen].gc, 0, 0, bp->width, bp->height);
}

static void
checkCollision(int a_masaki)
{
	bouncestruct *bp = &bounces[screen];
	int         i, amount, spin, d, size;
	double      x, y;

	for (i = 0; i < bp->nmasakis; i++) {
		if (i != a_masaki) {
			x = (double) (bp->masakis[i].x - bp->masakis[a_masaki].x);
			y = (double) (bp->masakis[i].y - bp->masakis[a_masaki].y);
			d = (int) sqrt(x * x + y * y);
			size = (bp->masakis[i].size + bp->masakis[a_masaki].size) / 2;
			if (d > 0 && d < size) {
				amount = size - d;
				if (amount > PENETRATION * size)
					amount = (int)(PENETRATION * size);
				bp->masakis[i].vx += (int)(amount * x / d);
				bp->masakis[i].vy += (int)(amount * y / d);
				bp->masakis[i].vx -= bp->masakis[i].vx / FRICTION;
				bp->masakis[i].vy -= bp->masakis[i].vy / FRICTION;
				bp->masakis[a_masaki].vx -= (int)(amount * x / d);
				bp->masakis[a_masaki].vy -= (int)(amount * y / d);
				bp->masakis[a_masaki].vx -= bp->masakis[a_masaki].vx / FRICTION;
				bp->masakis[a_masaki].vy -= bp->masakis[a_masaki].vy / FRICTION;
				spin = (bp->masakis[i].vang - bp->masakis[a_masaki].vang) /
					(2 * size * SLIPAGE);
				bp->masakis[i].vang -= spin;
				bp->masakis[a_masaki].vang += spin;
				bp->masakis[i].spindir = DIR(bp->masakis[i].vang);
				bp->masakis[a_masaki].spindir = DIR(bp->masakis[a_masaki].vang);
				if (!bp->masakis[i].vang) {
					bp->masakis[i].spindelay = 1;
					bp->masakis[i].spindir = 0;
				} else
					bp->masakis[i].spindelay = (int)(M_PI * bp->masakis[i].size /
						(ABS(bp->masakis[i].vang)) + 1);
				if (!bp->masakis[a_masaki].vang) {
					bp->masakis[a_masaki].spindelay = 1;
					bp->masakis[a_masaki].spindir = 0;
				} else
					bp->masakis[a_masaki].spindelay = (int)(M_PI * bp->masakis[a_masaki].size /
						(ABS(bp->masakis[a_masaki].vang)) + 1);
				return;
			}
		}
	}
}

void
drawmasaki(Window win)
{
	bouncestruct *bp = &bounces[screen];
	int         i;

	for (i = 0; i < bp->nmasakis; i++) {
		drawamasaki(win, &bp->masakis[i]);
		movemasaki(&bp->masakis[i]);
	}
	for (i = 0; i < bp->nmasakis; i++)
		checkCollision(i);
	if (!(LRAND() % TIME))	/* Put some randomness into the time */
		bp->restartnum--;
	if (!bp->restartnum)
		initmasaki(win);
}

static void
drawamasaki(Window win, masakistruct * masaki)
{
	if (masaki->sizex < masaki0_width) {
		if (masaki->xlast != -1) {
			XSetForeground(dsp, Scr[screen].gc, BlackPixel(dsp, screen));
			XFillRectangle(dsp, win, Scr[screen].gc,
			     masaki->xlast, masaki->ylast, masaki->sizex, masaki->sizey);
		}
		XSetForeground(dsp, Scr[screen].gc, masaki->color);
		XFillRectangle(dsp, win, Scr[screen].gc,
			       masaki->x, masaki->y, masaki->sizex, masaki->sizey);
	} else {
		XSetForeground(dsp, Scr[screen].gc, masaki->color);
		XPutImage(dsp, win, Scr[screen].gc,
			  images[(masaki->orient > ORIENTS / 2) ? ORIENTS - masaki->orient : masaki->orient],
			  0, 0, masaki->x, masaki->y, masaki->sizex, masaki->sizey);
		if (masaki->xlast != -1) {
			XSetForeground(dsp, Scr[screen].gc, BlackPixel(dsp, screen));
			XEraseImage(dsp, win, Scr[screen].gc,
				    masaki->x, masaki->y, masaki->xlast, masaki->ylast, masaki->sizex, masaki->sizey);
		}
	}
}

static void
movemasaki(masakistruct * masaki)
{
	bouncestruct *bp = &bounces[screen];

	masaki->xlast = masaki->x;
	masaki->ylast = masaki->y;
	masaki->x += masaki->vx;
	if (masaki->x > (bp->width - masaki->sizex)) {
		/* Bounce off the right edge */
		masaki->x = 2 * (bp->width - masaki->sizex) - masaki->x;
		masaki->vx = -masaki->vx + masaki->vx / FRICTION;
		flapmasaki(masaki, 1, &masaki->vy);
	} else if (masaki->x < 0) {
		/* Bounce off the left edge */
		masaki->x = -masaki->x;
		masaki->vx = -masaki->vx + masaki->vx / FRICTION;
		flapmasaki(masaki, -1, &masaki->vy);
	}
	masaki->vy++;
	masaki->y += masaki->vy;
	if (masaki->y >= (bp->height + masaki->sizey)) {	/* Don't see masaki bounce */
		/* Bounce off the bottom edge */
		masaki->y = (bp->height - masaki->sizey);
		masaki->vy = -masaki->vy + masaki->vy / FRICTION;
		flapmasaki(masaki, -1, &masaki->vx);
	}			/* else if (masaki->y < 0) { */
	/* Bounce off the top edge */
	/*masaki->y = -masaki->y;
	   masaki->vy = -masaki->vy + masaki->vy / FRICTION;
	   flapmasaki(masaki, 1, &masaki->vx);
	   } */
	if (masaki->spindir) {
		masaki->spincount--;
		if (!masaki->spincount) {
			masaki->orient = (masaki->spindir + masaki->orient) % ORIENTS;
			masaki->spincount = masaki->spindelay;
		}
	}
}

static void
flapmasaki(masakistruct * masaki, int dir, int *vel)
{
	*vel -= (int)((*vel + SIGN(*vel * dir) * masaki->spindelay * ORIENTCYCLE /
		 (M_PI * masaki->size)) / SLIPAGE);
	if (*vel) {
		masaki->spindir = DIR(*vel * dir);
		masaki->vang = *vel * ORIENTCYCLE;
		masaki->spindelay = (int)(M_PI * masaki->size / (ABS(masaki->vang)) + 1);
	} else
		masaki->spindir = 0;
}

static int
collide(int a_masaki)
{
	bouncestruct *bp = &bounces[screen];
	int         i, d, x, y;

	for (i = 0; i < a_masaki; i++) {
		x = (bp->masakis[i].x - bp->masakis[a_masaki].x);
		y = (bp->masakis[i].y - bp->masakis[a_masaki].y);
		d = (int) sqrt((double) (x * x + y * y));
		if (d < (bp->masakis[i].size + bp->masakis[a_masaki].size) / 2)
			return i;
	}
	return i;
}

/* This stops some flashing, could be more efficient */
static void
XEraseImage(Display * display, Window win, GC gc, int x, int y, int xlast, int ylast, int xsize, int ysize)
{
	if (ylast < y) {
		if (y < ylast + ysize)
			XFillRectangle(display, win, gc, xlast, ylast, xsize, y - ylast);
		else
			XFillRectangle(display, win, gc, xlast, ylast, xsize, ysize);
	} else if (ylast > y) {
		if (y > ylast - ysize)
			XFillRectangle(display, win, gc, xlast, y + ysize, xsize, ylast - y);
		else
			XFillRectangle(display, win, gc, xlast, ylast, xsize, ysize);
	}
	if (xlast < x) {
		if (x < xlast + xsize)
			XFillRectangle(display, win, gc, xlast, ylast, x - xlast, ysize);
		else
			XFillRectangle(display, win, gc, xlast, ylast, xsize, ysize);
	} else if (xlast > x) {
		if (x > xlast - xsize)
			XFillRectangle(display, win, gc, x + xsize, ylast, xlast - x, ysize);
		else
			XFillRectangle(display, win, gc, xlast, ylast, xsize, ysize);
	}
}

// --------------------------------------------------------------------

#include <qpushbt.h>
#include <qchkbox.h>
#include <qcolor.h>
#include <qmsgbox.h>
#include <qlayout.h>
#include <kbuttonbox.h>
#include "kslider.h"
#include "helpers.h"

#include "masaki.h"

#include "masaki.moc"

// this refers to klock.po. If you want an extra dictionary, 
// create an extra KLocale instance here.
extern KLocale *glocale;

static kMasakiSaver *saver = NULL;

void startScreenSaver( Drawable d )
{
	if ( saver )
		return;
	saver = new kMasakiSaver( d );
}

void stopScreenSaver()
{
	if ( saver )
		delete saver;
	saver = NULL;
}

int setupScreenSaver()
{
	kMasakiSetup dlg;

	return dlg.exec();
}

const char *getScreenSaverName()
{
	return glocale->translate("Masaki");
}

//-----------------------------------------------------------------------------

kMasakiSaver::kMasakiSaver( Drawable drawable ) : kScreenSaver( drawable )
{
	readSettings();

	colorContext = QColor::enterAllocContext();

	batchcount = maxLevels;

	initXLock( gc );
	initmasaki( d );

	timer.start( speed );
	connect( &timer, SIGNAL( timeout() ), SLOT( slotTimeout() ) );
}

kMasakiSaver::~kMasakiSaver()
{
	timer.stop();
	QColor::leaveAllocContext();
	QColor::destroyAllocContext( colorContext );
}

void kMasakiSaver::setSpeed( int spd )
{
	timer.stop();
	speed = MAXSPEED - spd;
	timer.start( speed );
}

void kMasakiSaver::setLevels( int l )
{
	batchcount = maxLevels = l;
	initmasaki( d );
}

void kMasakiSaver::readSettings()
{
	KConfig *config = KApplication::getKApplication()->getConfig();
	config->setGroup( "Settings" );

	QString str;

	str = config->readEntry( "Speed" );
	if ( !str.isNull() )
		speed = MAXSPEED - atoi( str );
	else
		speed = DEFSPEED;

	str = config->readEntry( "MaxLevels" );
	if ( !str.isNull() )
		maxLevels = atoi( str );
	else
		maxLevels = DEFBATCH;

}

void kMasakiSaver::slotTimeout()
{
    drawmasaki( d );
}

//-----------------------------------------------------------------------------

kMasakiSetup::kMasakiSetup( QWidget *parent, const char *name )
	: QDialog( parent, name, TRUE )
{
	speed = 50;

	readSettings();

	setCaption( glocale->translate("Setup KMasaki") );

	QLabel *label;
	QPushButton *button;
	KSlider *slider;
	
	QVBoxLayout *tl = new QVBoxLayout(this, 10);
	QHBoxLayout *tl1 = new QHBoxLayout;
	tl->addLayout(tl1);

	QVBoxLayout *tl11 = new QVBoxLayout(5);
	tl1->addLayout(tl11);
	label = new QLabel( glocale->translate("Speed:"), this );
	min_size(label);
	tl11->addWidget(label);

	slider = new KSlider( KSlider::Horizontal, this );
	slider->setFixedHeight(20);
	slider->setMinimumWidth(90);
	slider->setRange( MINSPEED, MAXSPEED );
	slider->setSteps( (MAXSPEED-MINSPEED)/4, (MAXSPEED-MINSPEED)/2 );
	slider->setValue( speed );
	connect( slider, SIGNAL( valueChanged( int ) ), 
		 SLOT( slotSpeed( int ) ) );
	tl11->addWidget(slider);
	tl11->addSpacing(15);

	label = new QLabel( glocale->translate("Number of pics:"), this );
	min_size(label);
	tl11->addWidget(label);

	slider = new KSlider( KSlider::Horizontal, this );
	slider->setFixedHeight(20);
	slider->setMinimumWidth(90);
	slider->setRange( MINBATCH, MAXBATCH );
	slider->setSteps( (MAXBATCH-MINBATCH)/4, (MAXBATCH-MINBATCH)/2 );
	slider->setValue( maxLevels );
	connect( slider, SIGNAL( valueChanged( int ) ), 
		 SLOT( slotLevels( int ) ) );
	tl11->addWidget(slider);
	tl11->addStretch(1);

	preview = new QWidget( this );
	preview->setFixedSize( 220, 170 );
	preview->setBackgroundColor( black );
	preview->show();    // otherwise saver does not get correct size
	saver = new kMasakiSaver( preview->winId() );
	tl1->addWidget(preview);

	KButtonBox *bbox = new KButtonBox(this);	
	button = bbox->addButton( glocale->translate("About"));
	connect( button, SIGNAL( clicked() ), SLOT(slotAbout() ) );
	bbox->addStretch(1);

	button = bbox->addButton( glocale->translate("Ok"));	
	connect( button, SIGNAL( clicked() ), SLOT( slotOkPressed() ) );

	button = bbox->addButton(glocale->translate("Cancel"));
	connect( button, SIGNAL( clicked() ), SLOT( reject() ) );
	bbox->layout();
	tl->addWidget(bbox);

	tl->freeze();
}

void kMasakiSetup::readSettings()
{
	KConfig *config = KApplication::getKApplication()->getConfig();
	config->setGroup( "Settings" );

	QString str;

	str = config->readEntry( "Speed" );
	if ( !str.isNull() )
		speed = atoi( str );

	if ( speed > MAXSPEED )
		speed = MAXSPEED;
	else if ( speed < MINSPEED )
		speed = MINSPEED;

	str = config->readEntry( "MaxLevels" );
	if ( !str.isNull() )
		maxLevels = atoi( str );
	else
		maxLevels = DEFBATCH;

}

void kMasakiSetup::slotSpeed( int num )
{
	speed = num;

	if ( saver )
		saver->setSpeed( speed );
}

void kMasakiSetup::slotLevels( int num )
{
	maxLevels = num;

	if ( saver )
		saver->setLevels( maxLevels );
}

void kMasakiSetup::slotOkPressed()
{
	KConfig *config = KApplication::getKApplication()->getConfig();
	config->setGroup( "Settings" );

	QString sspeed;
	sspeed.setNum( speed );
	config->writeEntry( "Speed", sspeed );

	QString slevels;
	slevels.setNum( maxLevels );
	config->writeEntry( "MaxLevels", slevels );

	config->sync();
	accept();
}

void kMasakiSetup::slotAbout()
{
	QMessageBox::message(glocale->translate("About Masaki"), 
			     glocale->translate("Masaki\n\nCopyright (c) 1986 by Sun Microsystems,\nDavid Banz and AIC\n\nPorted to kscreensave by David Banz."), 
			     glocale->translate("Ok"));
}


