/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** obj.cpp
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "obj.h"
#include "view.h"

obj::obj(base *p, char *n,texture *t):
	body(p,n,t)
{
	numtype = NUM_OBJ;

	children.setErrorValue(0);

	if(parent) addToParent(parent);
}

obj::obj(base *p, char *n,Vector3 &vs,Vector3 &vr,Vector3 &vt):
	body(p,n,vs,vr,vt)
{
	numtype = NUM_OBJ;

	children.setErrorValue(0);

	if(parent) addToParent(parent);
}

obj::obj(base *p, char *n,texture *t,Vector3 &vs,Vector3 &vr,Vector3 &vt):
	body(p,n,t,vs,vr,vt)
{
	numtype = NUM_OBJ;

	children.setErrorValue(0);

	if(parent) addToParent(parent);
}

obj::obj(base *p,obj *bc):
	body(p,(body*)bc)
{
	int	t;

	numtype = NUM_OBJ;
	
	children.setErrorValue(0);
	for(t = 0;t < bc->children.length();t++)
	{
		// create a copy
		(bc->children[t])->copy(this);
	}
	if(parent) addToParent(parent);
}

obj::~obj()
{
	body	*b;

	while((b = children.getFirst()))
	{
		delete b;
		children.deleteCurrent();
		while(children.find(b) >= 0)
			children.deleteCurrent();
	}
	removeFromParent();
}

base	*obj::copy(base *p)
{
	return new obj(p,this);
}

void	obj::dumpNames(int tab,int tabsize)
{
	body		*d;
	int		t;

	for(t = 0;t < tab;t++) printf(" ");

	printf("obj: %s\n",name);

	for(t = 0;t < children.length();t++)
	{
		d = children[t];
		d->dumpNames(tab + tabsize,tabsize);
	}
}

void	obj::dumpNames(int tabsize)
{
	dumpNames(0,tabsize);
}

int	obj::exportPOV(FILE *fp,int tab,int tabsize,int anim)
{
	body	*b;
	int	t;

	if(isFlag(NO_EXPORT)) return 0;

	printTab(fp,tab);
	fprintf(fp,"// Objectname = %s\n",name);
	printTab(fp,tab);
	fprintf(fp,"// Objecttype = obj\n");

	printTab(fp,tab);
	fprintf(fp,"union\n");

	printTab(fp,tab);
	fprintf(fp,"{\n");

	for(t = 0;t < children.length();t++)
	{
		b = children[t];
		b->exportPOV(fp,tab + tabsize,tabsize,anim);
	}

	if(texptr)
		texptr->exportPOV(fp,tab + tabsize,tabsize,anim);

	dim::exportPOV(fp,tab + tabsize,tabsize,anim);

	printTab(fp,tab);
	fprintf(fp,"}\n");

	return 0;
}

int	obj::addToParent(base *p)
{
	if(!p) return -2;

	parent = p;
	return p->addChild(this);  
}

int	obj::removeFromParent()
{
	if(!parent) return -2;

	return parent->removeChild(this);
}

int	obj::addChild(body* child)
{
	if(!child) return -2;
	//if(hasChild(child)) return -3;
	return children += child;
}

int	obj::addChild(refptr* child)
{
	if(!child) return -2;
	//if(hasChild(child)) return -3;
	return children += child;
}

int	obj::addChild(csgobj* child)
{
	if(!child) return -2;
	//if(hasChild(child)) return -3;
	return children += child;
}

int	obj::addChild(blobobj* child)
{
	if(!child) return -2;
	//if(hasChild(child)) return -3;
	return children += child;
}

int	obj::addChild(blob* child)
{
	if(!child) return -2;
	//if(hasChild(child)) return -3;
	return children += child;
}

int	obj::addChild(nonsolid* child)
{
	if(!child) return -2;
	//if(hasChild(child)) return -3;
	return children += child;
}

/*
int	obj::moveChild(dim *b) 
{ 
	if(!b->getParent()) return -1; 
 
	b->getParent()->removeChild(b);
	addChild(b);

	return 0; 
} 
*/

int	obj::removeChild(body *b) 
{ 
	if(!hasChild(b)) return -1; 

	if(children.find(b) < 0) return -2;
	children.deleteCurrent();

	return 0; 
} 

int	obj::removeChild(refptr *b) 
{ 
	if(!hasChild(b)) return -1; 

	if(children.find(b) < 0) return -2;
	children.deleteCurrent();

	return 0; 
} 

int	obj::removeChild(csgobj *b) 
{ 
	if(!hasChild(b)) return -1; 

	if(children.find(b) < 0) return -2;
	children.deleteCurrent();

	return 0; 
} 

int	obj::removeChild(blob *b) 
{ 
	if(!hasChild(b)) return -1; 

	if(children.find(b) < 0) return -2;
	children.deleteCurrent();

	return 0; 
} 

int	obj::removeChild(blobobj *b) 
{ 
	if(!hasChild(b)) return -1; 

	if(children.find(b) < 0) return -2;
	children.deleteCurrent();

	return 0; 
} 

int	obj::removeChild(nonsolid *b) 
{ 
	if(!hasChild(b)) return -1; 

	if(children.find(b) < 0) return -2;
	children.deleteCurrent();

	return 0; 
} 

int	obj::hasChild(body *c)
{
	body	*b;

	for(b = children.getFirst();b != NULL;b = ++children)
	{
		if(b == c) return !0;
		if(b->hasChild(c)) return !0;
	}

	return 0;
}

int	obj::existsName(const char *n)
{
	body	*d;
	int	t;

	if(n == NULL) return -1;

	if(name != NULL && strcmp(name,n) == 0)
		return !0;

	for(t = 0;t < children.length();t++)
	{
		d = children[t];
		if(d->existsName(n)) return !0;
	}

	return 0;
}

base	*obj::searchName(const char *n)
{
	body	*d;
	base	*b;
	int	t;

	if(n == NULL) return NULL;

	if(name != NULL && strcmp(name,n) == 0)
		return this;

	for(t = 0;t < children.length();t++)
	{
		d = children[t];
		if((b = d->searchName(n))) return b;
	}

	return NULL;
}

int	obj::save(media *m)
{
	int	t;

	if(!m) return -1;

	setMedia(m);

	writeChunk("OBJC");
	writeNameChunk(name);

	saveFlags(m);

	body	*b;

	for(t = 0;t < children.length();t++) 
	{ 
		b = children[t];
		b->save(m); 
	} 
	
	anim::save(m);
	dim::save(m);

	saveTexture(m);

	writeChunkLen();

	return 0;
}

int	obj::load(media *m,int l)
{
	int	pos = m->tell();
	base	*b;

	loadFlags(m,l);

	while(m->tell() < pos + l - (int)sizeof(double) * 12)
	{
		b = parse(m);
		if(b)
		{
			b->addToParent(this);
			b->setParent(this);
		}
	}

	anim::load(m,l - (m->tell() - pos));
	dim::load(m,l - (m->tell() - pos));

 	if(l - (m->tell() - pos) > 0)
	{
		loadTexture(m,l - (m->tell() - pos));
	}

	return 0;
}

int	obj::draw(view *v,Matrix44 m,int anim)
{
	body		*d;
	int		t;
	Vector3		vec(0,0,0);

	if(v == 0)
		return -1;

	if(isFlag(HIDE))
		return 0;

	dimMatrix(m,anim);

	if(this == v->getSelected())
		v->setDrawSelected(1);

	v->drawCross(vec,m,anim);

	if(isFlag(DRAW_BOUNDINGBOX))
		drawBB(v,m,anim);
	else
		for(t = 0;t < children.length();t++)
		{
			d = children[t];
			d->draw(v,m,anim);
		}

	if(this == v->getSelected())
		v->setDrawSelected(0);

	if(isFlag(DRAW_AXIS))
		v->drawAxis(m,anim);

	drawDragvectors(v,m,anim);

	return 0;
}

int	obj::calculate(int anim)
{
	body		*d;
	int		t;
	Vector4		smin,smax;
	Matrix44	m;

	vmin = Vector3(1e100,1e100,1e100);
	vmax = Vector3(-1e100,-1e100,-1e100);

	for(t = 0;t < children.length();t++)
	{
		d = children[t];
		m.unify();
		d->dimMatrix(m,anim);
		d->calculate(anim);
		smin = Vector4(d->min(),1);
		smax = Vector4(d->max(),1);
		smin *= m;
		smax *= m;
		vmin[0] = MIN(vmin[0],smin[0]); 
		vmin[1] = MIN(vmin[1],smin[1]); 
		vmin[2] = MIN(vmin[2],smin[2]); 
		vmax[0] = MAX(vmax[0],smax[0]); 
		vmax[1] = MAX(vmax[1],smax[1]); 
		vmax[2] = MAX(vmax[2],smax[2]); 
	}

	return 0;
}

double	obj::volume(int anim)
{
	Vector3		v;

	v = getVScale();

	// aproximate

	return 0.5 * v[0] * v[1] * v[2];
}



