/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** ref.cpp
*/

/*
	Fr referenzierbare Objekte gelten folgende Regeln

	1. ref mu in world-reference angelegt sein
	2. jedes Objekt kann ref nur einmal enthalten
	3. aber verschiedene Objekte knnen das selbe ref enthalten
	4. wird ref aus world entfernt, mu ref auch aus allen Objekten entfernt werden

	zustzlich:

	5. kann ref keine Texture zugewiesen werden

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ref.h"
#include "num.h" 
#include "dim.h"
#include "refptr.h"
#include "view.h"

ref::ref(base *p,const char *n) :
	base(p,n)
{
	numtype = NUM_REF;

	reflist.setErrorValue(0);
	objectlist.setErrorValue(0);

	if(parent)
		addToParent(parent);
}

ref::ref(base *p,ref *rc) :
	base(p,rc->getName())
{
	int	t;

	numtype = NUM_REF;

	reflist.setErrorValue(0);
	objectlist.setErrorValue(0);

	for(t = 0;t < rc->reflist.length();t++)
	{
		rc->reflist[t]->copy(this);
	}
	for(t = 0;t < rc->objectlist.length();t++)
	{
		rc->objectlist[t]->copy(this);
	}

	if(parent)
		addToParent(parent);
}

ref::~ref()
{
	refptr	*r;
	dim	*d;

	for(r = reflist.getFirst();r != NULL;r = ++reflist)
	{
		//r->removeChild(this);
	}
	reflist.empty();
	for(d = objectlist.getFirst();d != NULL;d = ++objectlist)
	{
		delete d;
	}
	removeFromParent();
}

base	*ref::copy(base *p)
{
	return new ref(p,this);
}

int	ref::addToParent(base *b)
{
	if(!b) return -2;

	parent = b;
	return b->addChild(this);
}

int	ref::removeFromParent()
{
	if(!parent) return -2;
	return parent->removeChild(this);
}

int	ref::addChild(body *d)
{
	if(!d) return -2;

	return objectlist += d;
}

int	ref::addChild(refptr *r)
{
	if(!r) return -2;

	return reflist += r;
}

int	ref::addChild(csgobj *r)
{
	if(!r) return -2;

	return objectlist += r;
}

int	ref::addChild(blobobj *r)
{
	if(!r) return -2;

	return objectlist += r;
}

int	ref::addChild(nonsolid *r)
{
	if(!r) return -2;

	return objectlist += r;
}

int	ref::removeChild(body *d)
{
	if(!d) return -2;
	if(objectlist.find(d) < 0) return -3;
	return (objectlist.deleteCurrent() ? 0 : -1);
}

int	ref::removeChild(refptr *r)
{
	if(!r) return -2;
	if(reflist.find(r) < 0) return -3;
	return (reflist.deleteCurrent() ? 0 : -1);
}

int	ref::removeChild(csgobj *r)
{
	if(!r) return -2;
	if(objectlist.find(r) < 0) return -3;
	return (objectlist.deleteCurrent() ? 0 : -1);
}

int	ref::removeChild(blobobj *r)
{
	if(!r) return -2;
	if(objectlist.find(r) < 0) return -3;
	return (objectlist.deleteCurrent() ? 0 : -1);
}

int	ref::removeChild(nonsolid *r)
{
	if(!r) return -2;
	if(objectlist.find(r) < 0) return -3;
	return (objectlist.deleteCurrent() ? 0 : -1);
}

void	ref::getChildren(base ***b)
{
	int	t,i;
	base	**bt;

	*b = NULL;
	bt = (base**)malloc(sizeof(void*) * (reflist.length() + objectlist.length() + 1));
	if(!bt) return;

	for(t = 0;t < objectlist.length();t++)
		bt[t] = objectlist[t];
	for(i = 0;i < objectlist.length();i++)
		bt[t + i] = reflist[i];
	bt[t + i] = NULL;
	*b = bt;
}

int	ref::exportPOV(FILE *fp,int tab,int tabsize,int anim)
{
	body		*d;
	refptr		*r;

	if(isFlag(NO_EXPORT)) return 0;

	printTab(fp,tab);
	fprintf(fp,"// Objectname = %s\n",name);
	printTab(fp,tab);
	fprintf(fp,"// Objecttype = ref\n");

	printTab(fp,tab);
	fprintf(fp,"union\n");
	printTab(fp,tab);
	fprintf(fp,"{\n");

	for(d = objectlist.getFirst();d != NULL;d = ++objectlist)
	{
		d->exportPOV(fp,tab + tabsize,tabsize,anim);
	}
	for(r = reflist.getFirst();r != NULL;r = ++reflist)
	{
		r->exportPOV(fp,tab + tabsize,tabsize,anim);
	}

	printTab(fp,tab);
	fprintf(fp,"}\n");
	
	return 0;
}

void	ref::dumpNames(int tab,int tabsize)
{
	body		*d;
	refptr		*r;
	int		t;

	for(t = 0;t < tab;t++) printf(" ");

	printf("ref: %s\n",name);

	for(d = objectlist.getFirst();d != NULL;d = ++objectlist)
	{
		d->dumpNames(tab + tabsize,tabsize);
	}
	for(r = reflist.getFirst();r != NULL;r = ++reflist)
	{
		r->dumpNames(tab + tabsize,tabsize);
	}
}

int	ref::existsName(const char *n)
{
	body	*d;
	refptr	*r;

	if(n == NULL) return -1;

	if(name != NULL && strcmp(name,n) == 0)
		return !0;

	for(d = objectlist.getFirst();d != NULL;d = ++objectlist)
	{
		if(d->existsName(n)) return !0;
	}
	for(r = reflist.getFirst();r != NULL;r = ++reflist)
	{
		if(r->existsName(n)) return !0;
	}

	return 0;
}

base	*ref::searchName(const char *n)
{
	base	*b;
	int	i;

	if(n == 0) return 0;

	if(name != 0 && strcmp(name,n) == 0)
		return this;

	for(i = 0;i < objectlist.length();i++)
	{
		if((b = objectlist[i]->searchName(n))) return b;
	}
	for(i = 0;i < reflist.length();i++)
	{
		if((b = reflist[i]->searchName(n))) return b;
	}

	return 0;
}

int	ref::save(media *m)
{
	int	i;

	if(!m) return -1;

	setMedia(m);

	writeChunk("REFO");
	writeNameChunk(name);
	saveFlags(m);

	for(i = 0;i < objectlist.length();i++) 
	{ 
		objectlist[i]->save(m); 
	} 
	
	writeChunkLen();

	return 0;
}

int	ref::load(media *m,int l)
{
	int	pos;
	base	*b;

	if(!m) return -1;

	pos = m->tell();

	loadFlags(m,l - (m->tell() - pos));

	while(l - (m->tell() - pos) > 0)
	{ 
		b = parse(m);
		if(b)
		{
			b->addToParent(this);
			b->setParent(this);
		}
	} 
	
	return 0;
}

int	ref::draw(view *v,Matrix44 m,int anim)
{
	body	*d;
	int	t;

	if(v == 0) return -1;

	if(isFlag(HIDE)) return 0;

	if(this == v->getSelected()) v->setDrawSelected(1);

	if(isFlag(DRAW_BOUNDINGBOX))
		drawBB(v,m,anim);
	else
	{
		for(t = 0;t < objectlist.length();t++)
		{
			d = objectlist[t];
			d->draw(v,m,anim);
		}
	}

	if(this == v->getSelected()) v->setDrawSelected(0);

	if(isFlag(DRAW_AXIS))
		v->drawAxis(m,anim);

	drawDragvectors(v,m,anim);

	return 0;
}

int	ref::drawRef(view *v,Matrix44 m,int anim,int bb)
{
	body	*d;
	int	t;

	if(v == 0)
		 return -1;

	if(!isFlag(HIDE))
		return 0;

	if(isFlag(DRAW_BOUNDINGBOX) || bb)
		drawBB(v,m,anim);
	else
	{
		for(t = 0;t < objectlist.length();t++)
		{
			d = objectlist[t];
			d->draw(v,m,anim);
		}
	}

	return 0;
}

int	ref::calculate(int anim)
{
	body		*d;
	int		t;
	Vector4		smin,smax;
	Matrix44	m;

	vmin = Vector3(1e100,1e100,1e100);
	vmax = Vector3(-1e100,-1e100,-1e100);

	for(t = 0;t < objectlist.length();t++)
	{
		d = objectlist[t];
		d->calculate(anim);

		m.unify();
		d->dimMatrix(m,anim);

		smin = Vector4(d->min(),1);
		smax = Vector4(d->max(),1);
		smin *= m;
		smax *= m;
		vmin[0] = MIN(vmin[0],smin[0]); 
		vmin[1] = MIN(vmin[1],smin[1]); 
		vmin[2] = MIN(vmin[2],smin[2]); 
		vmax[0] = MAX(vmax[0],smax[0]); 
		vmax[1] = MAX(vmax[1],smax[1]); 
		vmax[2] = MAX(vmax[2],smax[2]); 
	}

	return 0;
}



