/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** view.cpp
*/
#include "view.h"

#include "camera.h"

view::view(world *wptr,const char *n) :
	dim(wptr,n)
{
	cameraptr = 0;

	fov = 0.3;
	aspect_ratio = 1.33;

	worldptr = 0;
	selectedptr = 0;

	drawselected = 0;

	projmode = PERSPECTIVE;

	// dummy values
	w = h = 100;

	calcVectors(0);
	createView(0);
	createScreen(w,h);
};

view::view(world *wptr,view *v) :
	dim(wptr,v)
{
	cameraptr = 0;

	fov = v->fov;
	aspect_ratio = v->aspect_ratio;

	selectedptr = 0;

	projmode = v->projmode;

	w = v->w;
	h = v->h;

	calcVectors(0);
	createView(0);
	createScreen(w,h);
};

view::~view()
{
}

base	*view::copy(base *p)
{
	if(p && p->getType() == NUM_WORLD)
		return new view((world*)p,this);
		
	return 0;
}

void	view::take(view &v)
{
	fov = v.fov;
	aspect_ratio = v.aspect_ratio;

	calcVectors(0);
	createView(0);
	createScreen(w,h);
};


void	view::take(view *v)
{
	if(v == 0)
		return;

	fov = v->fov;
//	aspect_ratio = v->a//ct_ratio;

	calcVectors(0);
	createView(0);
	createScreen(w,h);
};


void	view::setCamera(camera *c)
{
	cameraptr = c;
}

camera	*view::getCamera()
{
	return cameraptr;
}

int	view::setWorld(world *wp)
{
	worldptr = wp;

	if(wp)
	{
		wp->addView(this);
	}

	return 0;
}

int		view::setFOV(double d)
{
	fov = d;

	if(fov < 0.0005) fov = 0.0005;
	if(fov > 179.9999999 / 180 * PI) fov = 179.9999999 / 180 * PI;

	calcVectors(0);

	return 0;
}

int		view::setAspectRatio(double d)
{
	aspect_ratio = d;

	calcVectors(0);

	return 0;
}

int		view::setScreenSize(int ww,int hh)
{
	w = ww;
	h = hh;

	calcVectors(0);

	return 0;
}

int		view::project(Vector3 v,Vector2 &xy,Matrix44 m,int anim)
{
	switch(projmode)
	{
		case PERSPECTIVE:
		{
			Vector4	v4,vh;

			
			v4 = v;
			v4[1] = v4[1];
			v4[2] = v4[2];

			m = m * mcamera;
			
			v4 *= m;
			v4 *= mview;
			vh = v4;
			v4 *= mscreen;

			if(vh[2] > -EPSILON) return -1;

			xy[0] = v4[0] / vh[2];
			xy[1] = v4[1] / vh[2];

			//xy.print();
		}
		break;
		case PARALLEL_XY:
		{
			Vector4	v4;

			v4 = v;
			v4 *= m;
			v4 /= tan(fov) / 1.4;

			xy[0] = (w / 2) + v4[0];
			xy[1] = (h / 2) - v4[1];

			//xy.print();
		}
		break;
		case PARALLEL_XZ:
		{
			Vector4	v4;

			v4 = v;
			v4 *= m;
			v4 /= tan(fov) / 1.4;

			xy[0] = (w / 2) + v4[0];
			xy[1] = (h / 2) - v4[2];

			//xy.print();
		}
		break;
		case PARALLEL_ZY:
		{
			Vector4	v4;

			v4 = v;
			v4 *= m;
			v4 /= tan(fov) / 1.4;

			xy[0] = (w / 2) + v4[2];
			xy[1] = (h / 2) - v4[1];

			//xy.print();
		}
		break;
	}

	return 0;
}

int		view::reproject(Vector2 xy,Vector3 &v,Matrix44 m,int)
{
	switch(projmode)
	{
		case PERSPECTIVE:
		{
			Vector4	v4;

			v4 = xy;

			v4 *= ~mscreen;
			v4 *= ~mview;

			v = v4;
		}
		break;
		case PARALLEL_XY:
		{
			Vector4	v4;

			v4 = xy;
			v[0] = (w / 2) - v4[0];
			v[1] = (h / 2) + v4[1];
			v4 *= tan(fov) / 1.4;

			v = v4;
		}
		break;
		case PARALLEL_XZ:
		{
			Vector4	v4;

			v4 = v;
			v4 *= m;
			v4 /= tan(fov) / 1.4;

			xy[0] = (w / 2) + v4[0];
			xy[1] = (h / 2) - v4[2];

			//xy.print();
		}
		break;
		case PARALLEL_ZY:
		{
			Vector4	v4;

			v4 = v;
			v4 *= m;
			v4 /= tan(fov) / 1.4;

			xy[0] = (w / 2) + v4[2];
			xy[1] = (h / 2) - v4[1];

			//xy.print();
		}
		break;
	}

	return 0;
}

void		view::calcVectors(int anim)
{
	mcamera.unify();
		
	dimMatrix(mcamera,anim);
	
	createView(anim);
}

void	view::createView(int anim)
{
	Matrix44	mh;
	double		d,a;

	d = tan(fov / 2);
	a = (double)w / (double)h * aspect_ratio;

	mview(0,0) = d * a;
	mview(0,1) = 0;
	mview(0,2) = 0;
	mview(0,3) = 0;
	mview(1,0) = 0;
	mview(1,1) = d;
	mview(1,2) = 0;
	mview(1,3) = 0;
	mview(2,0) = 0;
	mview(2,1) = 0;
	mview(2,2) = 1;
	mview(2,3) = 0;
	mview(3,0) = 0;
	mview(3,1) = 0;
	mview(3,2) = 0;
	mview(3,3) = 1;
	mh = mview;
	mview = ~mh;
}

void	view::createScreen(int w,int h)
{
	mscreen.unify();
	mscreen(0,0) = ((double)w - 1) / -2;
	mscreen(0,2) = ((double)w - 1) / 2;
	mscreen(1,1) = ((double)h - 1) / -2;
	mscreen(1,2) = ((double)h - 1) / 2;
}

int	view::exportPOV(FILE*,int,int,int)
{
	return 0;
}

int	view::exportPOV(FILE *fp,int tabsize,int anim)
{
	int		tab = tabsize;
	double		d,a;
        Vector3		vr,vt;

	d = tan(fov / 2);
	a = (double)w / (double)h * aspect_ratio;

	fprintf(fp,"// Objectname = %s\n",name);
	fprintf(fp,"// Objecttype = view\n\n"); 
	fprintf(fp,"// This camera is build for size %i:%i\n\n\n",w,h);

	fprintf(fp,"camera\n");
	fprintf(fp,"{\n");

	printTab(fp,tab);

	fprintf(fp,"#ifdef (LEFTEYE)\n");

	printTab(fp,tab);
	fprintf(fp,"location  <0 - %g * f,0,0>\n",d * a);

	fprintf(fp,"#else\n");
	printTab(fp,tab);

	fprintf(fp,"#ifdef (RIGHTEYE)\n");

	printTab(fp,tab); 
	fprintf(fp,"location  <0 + %g * f,0,0>\n",d * a);

	fprintf(fp,"#else\n");

	printTab(fp,tab); 
	fprintf(fp,"location  <0,0,0>\n");

	fprintf(fp,"#end\n");
	printTab(fp,tab);
	fprintf(fp,"#end\n");

	printTab(fp,tab);
	fprintf(fp,"up        <0,%g,0>\n",-d * 2);

	printTab(fp,tab);
	fprintf(fp,"right     <%g,0,0>\n",d * a * 2);

	printTab(fp,tab);
	fprintf(fp,"direction <0,0,-1>\n\n");

	//dim::exportPOV(fp,tab + tabsize,tabsize,anim);

	vr = getVRotate();
	vt = getVTranslate();
		
	printTab(fp,tab);
	fprintf(fp,"translate <%g,%g,%g>\n",-vt[0],-vt[1],-vt[2]);
	printTab(fp,tab);
	fprintf(fp,"rotate    <%g,%g,%g>\n",-vr[0] * 180 / PI,-vr[1] * 180 / PI,-vr[2] * 180 / PI);
	
	fprintf(fp,"}\n\n");

	return 0;
}

void		view::setProjectionMode(int pm)
{
	projmode = pm;
}

int		view::projectionMode()
{
	return projmode;
}

void		view::setSelected(base *bp)
{
	selectedptr = bp;
}

base		*view::getSelected()
{
	return selectedptr;
}

void		view::setDrawSelected(int sel)
{
	drawselected = sel;
}

int	view::save(media *m/*,int ver*/)
{
	chunk	c(m);
        int	ver = 0;

	if(!m) return -1;

	switch(ver)
	{
		case 0:
		case -1:
			c.writeChunk("VIEW");
			//dim::save(m,ver);
			c.writeDouble(fov);
			c.writeInt(w);
			c.writeInt(h);
			c.writeChunkLen();
		break;
		default:
			return -2;
	}

	return 0;
}

int	view::load(media *m,int/*,int ver*/)
{
        char		chnk[4];
        int		len;
        chunk		c;
        int		ver = 0;

	if(!m) return -1;

	switch(ver)
	{
		case 0:
		case -1:
			c.setMedia(m);
			c.readChunk(chnk,len);
			if(strncmp(chnk,"VIEW",4) == 0)
			{
				//dim::load(m,len,ver);
				fov = c.readDouble();
				w = c.readInt();
				h = c.readInt();
			}
			else
			{
				c.rejectChunk();
				return -3;
			}			
		break;
		default:
			return -2;
	}

	return 0;
}



int	view::drawLine(Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawCross(Vector3,Matrix44,int) {return -1;};
int	view::drawTriangle(Vector3,Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawTrapezium(Vector3,Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawCircle(Vector3,Vector3,Vector3,Matrix44,int) {return -1;};

int	view::drawSphere(Vector3,Vector3,Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawSemiSphere(Vector3,Vector3,Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawBox(Vector3,Vector3,Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawCone(Vector3,Vector3,Vector3,Vector3,double,Matrix44,int) {return -1;};
int	view::drawCylinder(Vector3,Vector3,Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawTorus(Vector3,Vector3,Vector3,Vector3,double,Matrix44,int) {return -1;};

int	view::drawBlobSphere(Vector3,Vector3,Vector3,Vector3,Matrix44,int) {return -1;};
int	view::drawBlobCylinder(Vector3,Vector3,Vector3,Vector3,Matrix44,int) {return -1;};

int	view::drawSymbol(Vector3,const char*,Matrix44,int) {return -1;};

int	view::drawAxis(Matrix44,int) {return -1;};
int	view::drawDragvector(Matrix44,Vector3,Vector3,int) {return -1;};

int	view::calculate(int)
{
	vmin = Vector3(-0.5,-0.5,-0.5);
	vmax = Vector3(0.5,0.5,0.5);

	return 0;
}

double	view::volume(int)
{
	return 0;
}

void	view::dumpNames(int tab,int)
{
	printTab(stdout,tab);
	printf("view: %s\n",name);
}

int	view::addToParent(base *p)
{
	if(!p) return -2;

	parent = p;

	return p->addChild(this);
}

int	view::removeFromParent()
{
	if(!parent) return -2;

	return parent->removeChild(this);
}

int	view::draw(view*,Matrix44,int)
{
	return 0;
}



