/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** texture.cpp
*/
#include <misc.h>
#include "texture.h"

const char *TextYellowFolderPic[]={
"15 13 6 1",
"d c #000000",
"b c #ffff00",
". c None",
"# c #999999",
"a c #cccccc",
"c c #ffffff",
"..#####........",
".#ababa#.......",
"#abababa######.",
"#cccccccccccc#d",
"#cbababababab#d",
"#cabababababa#d",
"#cbababababab#d",
"#cabababababa#d",
"#cbababababab#d",
"#cabababababa#d",
"#cbababababab#d",
"##############d",
".dddddddddddddd"};

texture::texture(textureBase *p) :
	  textureBase(TEXTURE,p)
{
	setName("Texture");
	setInfo( (char*)"Texture");
	setSubItems(true);

}

const char **texture::getXPM()
{
	return TextYellowFolderPic;
}

textureBase *texture::getFirstElement()
{
	elementnr=0;
	if (elements.length()==0) return NULL;
	return elements[0];
}

textureBase *texture::getNextElement()
{
	elementnr++;
	if (elementnr>=elements.length()) return NULL;
	return elements[elementnr];
}

void texture::remove(textureBase *tb)
{
int i;
	for (i=0;i<elements.length();i++) {
		if (tb->isEqual(elements[i])) {
			elements.deleteAt(i);			
			i=elements.length()+1;
		}
	}
}

bool texture::canAdd(textureBase *tb)
{
int t;
	if (!tb) return false;
	t=tb->getType();
	if (
		(t==PIGMENT)||
		(t==NORMAL)||
		(t==FINISH)||
		(t==INTERIOR)||
		(t==TRANSFORMATIONS)||
		(t==COLOR)||
		(t==IMAGEMAP)||
		(t==COLORMAP)||
		(t==COLORLIST)||
		(t==BUMPMAP)||
		(t==DENSITY)||
		(t==SLOPEMAP)		
	) return true;
	return false;
}

void	texture::setActive(bool a,textureBase *tb)
{
int i;

	if (tb==NULL) {
		textureBase::setActive(a,NULL);
		return;
	}

        switch(tb->getType() ) {
		case PIGMENT:
			for (i=0;i<elements.length();i++) {
				if (elements[i]->getType()==PIGMENT) {
					elements[i]->setActive(false,NULL);
				}
			}
			tb->setActive(a);
		break;
		case NORMAL:
			for (i=0;i<elements.length();i++) {
				if (elements[i]->getType()==NORMAL) {
					elements[i]->setActive(false,NULL);
				}
			}
			tb->setActive(a);
		break;
		case FINISH:
			for (i=0;i<elements.length();i++) {
				if (elements[i]->getType()==FINISH) {
					elements[i]->setActive(false,NULL);
				}
			}
			tb->setActive(a);
		break;
		case INTERIOR:
			for (i=0;i<elements.length();i++) {
				if (elements[i]->getType()==INTERIOR) {
					elements[i]->setActive(false,NULL);
				}
			}
			tb->setActive(a,NULL);
		break;
		case TRANSFORMATIONS:
			for (i=0;i<elements.length();i++) {
				if (elements[i]->getType()==TRANSFORMATIONS) {
					elements[i]->setActive(false,NULL);
				}
			}
			tb->setActive(a,NULL);
		break;
		default:
		break;
	}
}

void texture::addElement(textureBase* what,textureBase* where)
{
	if (!what) return;
	if (canAdd(what)==false) return;
	what->setParent(this);	
	setActive(true,what);
	switch (what->getType()) {
		case COLOR:
		case COLORLIST:
		case COLORMAP:
		case IMAGEMAP:
		{
			pigment *pi=(pigment*)getElementByType(PIGMENT);
			if (!pi) {
				pi=new pigment(this);
				addElement(pi,NULL);
			}
			what->setParent(pi);
			pi->add(what);
		}						
		break;
		case BUMPMAP:
		case SLOPEMAP:
		{
			normal *no=(normal*)getElementByType(NORMAL);
			if (!no) {
				no=new normal(this);
				addElement(no,NULL);
			}
			what->setParent(no);
			no->add(what);

		}
		break;
		case DENSITY:
		{
			interior *in=(interior *)getElementByType(INTERIOR);
			if (!in) {
				in=new interior(this);
				addElement(in,NULL);
			}
			what->setParent(in);
			in->add(what);

		}
		break;
		default:
			elements.append(what);
		break;
	}
}

void texture::add(textureBase *tb)
{
	addElement(tb,NULL);
}

bool texture::isEqual(textureBase *tb)
{
textureBase *tb1;
int i;
#ifdef DEBUG
	printf("Comparing Textures...\n");
#endif
	if (tb->getType()!=TEXTURE) return false;

	if (textureBase::isEqual(tb)==false) return false;

	tb1=tb->getFirstElement();	
	if ( ( (getFirstElement()==NULL)&&(tb1==NULL) ) ) return true;

	for (i=0;i<elements.length();i++) {
		
		if (tb1==NULL) return false;
		if (tb1->isEqual(elements[i])==false) return false;
		tb1=tb->getNextElement();		
	}

#ifdef DEBUG
	printf("Compared Textures..found it !\n");
#endif
	return true;
}


texture::~texture()
{
}

int	texture::save(media *m)
{
int i;
	if(!m) return -1;
	#ifdef DEBUG
		printf("saving Texture\n");
	#endif
	setMedia(m);

	writeChunk("TXTR");

	textureBase::save(m);

	for (i=0;i<elements.length();i++) elements[i]->save(m);

	writeChunkLen();

	#ifdef DEBUG
		printf("Texture saved\n");
	#endif

	return 0;
}

int	texture::load(media *m,int l)
{
	pigment  *pi=NULL;
	normal   *no=NULL;
	finish   *fi=NULL;
	interior *in=NULL;
	patternmodifier *mo=NULL;
	Transformations *tr=NULL;

	char	chunk[4];
	int	len,pos = m->tell();
	bool	read=false;

	if(!m)
		return -1;
	#ifdef DEBUG
		printf("loading Texture\n");
	#endif

	setMedia(m);
	
	do {
		m->read(chunk,4);
		m->read(&len,4);
		read=false;

		if(strncmp(chunk,"PGMT",4) == 0) {
			elements.append ( (pi=new pigment(this)) );
			pi->load(m,len);
			read=true;
		}

		if(strncmp(chunk,"NRML",4) == 0) {
			elements.append ( (no=new normal(this)) );
			no->load(m,len);
			read=true;
		}
		if(strncmp(chunk,"TRNS",4) == 0) {
			elements.append ( (tr=new Transformations(this)) );
			tr->load(m,len);
			read=true;
		}

		if(strncmp(chunk,"FNSH",4) == 0) {
			elements.append ( (fi=new finish(this)) );
			fi->load(m,len);
			read=true;
		}
		if(strncmp(chunk,"INTE",4) == 0) {
			elements.append ( (in=new interior(this)) );
			in->load(m,len);
			read=true;
		}
		if(strncmp(chunk,"PATM",4) == 0) {
			elements.append ( (mo=new patternmodifier(this)) );
			mo->load(m,len);
			read=true;
		}
		if(strncmp(chunk,"BASE",4) == 0) {
			textureBase::load(m,len);
			read=true;
		}

		if (read==false) { m->seek(len,SEEK_CUR); }

	} while (m->tell()< (pos+l) );
	m->seek(pos+l,SEEK_SET); // Man weiss ja nie...

	#ifdef DEBUG
		printf("Texture loaded\n");
	#endif

	return 0;
}

int	texture::savePtr(media *m)
{
	if(!m) return -1;

	setMedia(m);
	writeChunk("TXTP");

	writeName(getPath());

	writeChunkLen();

	return 0;
}

int	texture::loadPtr(media *m,int)
{
char *dummy;
	if(!m) return -1;

	setMedia(m);

	dummy = readName();

	return 0;
}

int	texture::exportPOV(FILE *fp,int tab,int tabsize,int anim)
{
	return exportPOV(fp,tab,tabsize,anim,false);
}

int	texture::exportPOV(FILE *fp,int tab,int tabsize,int anim,bool as_macro)
{
	int i;
	pigment  *pi=NULL;
	normal   *no=NULL;
	finish   *fi=NULL;
	interior *in=NULL;
	Transformations *tr=NULL;
//	patternmodifier *mo=NULL;

	#ifdef DEBUG
		printf("exporting Texture (POV)\n");
	#endif	
	if (as_macro==false) {
		printTab(fp,tab);

		fprintf(fp,"// Objecttype = texture\n");
		printTab(fp,tab);

		fprintf(fp,"texture { //name=%s , path=%s\n",name,getPath());	
	} else {
		printTab(fp,tab);
		fprintf(fp,"#declare %s = texture { //name=%s \n",getPath(),name );	
		tab=tab+tabsize;
	}

	//Pigment exportieren
	for (i=0;i<elements.length();i++) {
		if ((elements[i]->getType()==PIGMENT)&&(elements[i]->active()==true)) pi=(pigment *)elements[i];	
	}
 	for (i=0;i<elements.length();i++) {
		if ((elements[i]->getType()==NORMAL)&&(elements[i]->active()==true)) no=(normal *)elements[i];	
	}
	for (i=0;i<elements.length();i++) {
		if ((elements[i]->getType()==FINISH)&&(elements[i]->active()==true)) fi=(finish *)elements[i];	
	}
	for (i=0;i<elements.length();i++) {
		if ((elements[i]->getType()==INTERIOR)&&(elements[i]->active()==true)) in=(interior *)elements[i];	
	}
	for (i=0;i<elements.length();i++) {
		if ((elements[i]->getType()==TRANSFORMATIONS)&&(elements[i]->active()==true)) tr=(Transformations*)elements[i];	
	}
/*	for (i=0;i<elements.length();i++) {
		if (elements[i]->getType()==PATTERNMODIFIER) mo=(patternmodifier *)elements[i];	
	}
  */

	if ((pi)&&(pi->active()==true)) pi->exportPOV(fp,tab+tabsize,tabsize,anim,false);
	if ((no)&&(no->active()==true)) no->exportPOV(fp,tab+tabsize,tabsize,anim,false);
	if ((fi)&&(fi->active()==true)) fi->exportPOV(fp,tab+tabsize,tabsize,anim);
	if ((tr)&&(tr->active()==true)) tr->exportPOV(fp,tab,tabsize,anim,false);
	//if (mo) mo->exportPOV(fp,tab,tabsize,anim,false);

	printTab(fp,tab);
	fprintf(fp,"}\n");	
	if ((in)&&(in->active()==true)) in->exportPOV(fp,tab,tabsize,anim,false);

	#ifdef DEBUG
		printf("Texture exported (POV)\n");
	#endif

	return 0;
}
