/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** dialogs.cpp
*/
#include "dialogs.h"
#include "dialogs.moc"

#include <klocale.h>

#define stloc(a)	((char*)locale->translate((a)))


NameDialog::NameDialog(QWidget *parent=0,const char *name=0) :
	QDialog(parent,name,TRUE)
{
	KLocale	*locale;

	setCaption(name);

	locale = new KLocale();

	line = new QLineEdit(this);

	ok = new QPushButton( this,stloc("Ok"));
	ok->setText(stloc("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,stloc("Cancel") );
	cancel->setText(stloc("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setMaximumSize(150,65);
	QDialog::resize(150,65);
	setMinimumSize(150,65);

	line->setGeometry(5,5,140,25);
	cancel->setGeometry(5,35,67,25);
	ok->setGeometry(77,35,67,25);	

	delete locale;
}

NameDialog::~NameDialog()
{
	delete	ok;
	delete	cancel;
	delete	line;
}

QString	NameDialog::getName()
{
	return line->text();
}

void	NameDialog::setName(QString s)
{
	line->setText(s);
}

// ************************************************************************
InfoDialog::InfoDialog(const char *name=0) :
	QDialog(NULL,"Info",true)
{

	QSize qs;

	setCaption("Info");


	label = new QLabel(name,this);

	ok = new QPushButton("Ok", this);

	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	setMaximumSize(150,65);
	QDialog::resize(150,65);
	setMinimumSize(150,65);

	qs=label->sizeHint();	

	label->setGeometry(qs.width()/4,qs.height()/4,qs.width(),qs.height() );
	ok->setGeometry(width()/2-20,height()-30,40,25);	
}

InfoDialog::~InfoDialog()
{
	delete	ok;
	delete label;
}

// ******************************************************************************************

CameraDialog::CameraDialog(QWidget *parent=0,const char *name=0) :
	QDialog(parent,name,true)
{
	KLocale	*locale;

	setCaption(name);

	locale = new KLocale();

	viewselect = new QComboBox(this);
	viewselect->setInsertionPolicy(QComboBox::NoInsertion);

	ok = new QPushButton( this,stloc("Ok"));
	ok->setText(stloc("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,stloc("Cancel") );
	cancel->setText(stloc("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setMaximumSize(150,65);
	QDialog::resize(150,65);
	setMinimumSize(150,65);

	viewselect->setGeometry(5,5,140,25);
	cancel->setGeometry(5,35,67,25);
	ok->setGeometry(77,35,67,25);	

	delete locale;
}

CameraDialog::~CameraDialog()
{
}

QString	CameraDialog::getView()
{
	QString		str;
	
	str = viewselect->currentText();
	
	return str;
}

void	CameraDialog::setViewList(QStrList s)
{
	viewselect->insertStrList(&s);
}

// ******************************************************************************************

BlobDialog::BlobDialog(QWidget *parent=0,const char *name=0) :
	QDialog(parent,name,TRUE)
{
	KLocale	*locale;
	int	w,w2;

	setCaption(name);

	locale = new KLocale();

	label = new QLabel(this);
	label->setText("Thresholdvalue:");

	line = new QLineEdit(this);

	ok = new QPushButton( this,stloc("Ok"));
	ok->setText(stloc("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,stloc("Cancel") );
	cancel->setText(stloc("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	w2 = label->sizeHint().width();
	w = w2 * 3 / 2 + 15;

	setMaximumSize(w,65);
	QDialog::resize(w,65);
	setMinimumSize(w,65);

	label->setGeometry(5,5,w2,25);
	line->setGeometry(10 + w2,5,w2 / 2,25);

	w2 = w2 * 2 / 3;
	cancel->setGeometry((w - 5) / 2 - w2,35,w2,25);
	ok->setGeometry((w - 5) / 2 + 5,35,w2,25);	

	delete locale;
}

BlobDialog::~BlobDialog()
{
}

double	BlobDialog::thresholdValue()
{
	QString	str;

	str = line->text();

	return str.toDouble();
}

void	BlobDialog::setThresholdValue(double t)
{
	QString	str;

	str.setNum(t);
	line->setText(str);
}


// ******************************************************************************************


DimDialog::DimDialog(QWidget *parent,const char *name) :
	QDialog(parent,name,TRUE)
{
	KLocale	*locale;

	setCaption(name);

	dimw = new DimWidget(this,"");

	locale = new KLocale();

	ok = new QPushButton( this,stloc("Ok"));
	ok->setText(stloc("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,stloc("Cancel") );
	cancel->setText(stloc("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setMaximumSize(300,140);
	QDialog::resize(300,140);
	setMinimumSize(300,140);

	dimw->setGeometry(0,0,300,125);
	cancel->setGeometry(67,110,80,25);
	ok->setGeometry(153,110,80,25);	

	delete locale;

	dimw->layout();
}

DimDialog::~DimDialog()
{
	delete	ok;
	delete	cancel;
	delete	dimw;
}

void	DimDialog::save(Vector3 &s,Vector3 &r,Vector3 &t)
{
	dimw->save(s,r,t);
}

void	DimDialog::load(Vector3 s,Vector3 r,Vector3 t)
{
	dimw->load(s,r,t);
}


// ******************************************************************************************


TextDialog::TextDialog(QWidget *parent,const char *name) :
	QDialog(parent,name,TRUE)
{
	KLocale		*locale;
	QString		path;
	Preferences	prefs;

	path = prefs.getPath(prefs.getFontPath());

	locale = new KLocale();

	setCaption(name);

	lcontext = new QLabel(this);
	lcontext->setText(stloc("Context:"));

	wcontext = new QLineEdit(this);

/*
	lfontname = new QLabel(this);
	lfontname->setText(stloc("Fontname:"));

	wfontname = new QLineEdit(this);
*/

	wfontname = new GetFileWidget(this,"Fontname:",path);

	lthickness = new QLabel(this);
	lthickness->setText(stloc("Thickness:"));

	wthickness = new QLineEdit(this);

	loffset = new QLabel(this);
	loffset->setText(stloc("Offset:"));

	wxoffset = new QLineEdit(this);
	wyoffset = new QLineEdit(this);

	
	ok = new QPushButton( this,stloc("Ok"));
	ok->setText(stloc("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,stloc("Cancel") );
	cancel->setText(stloc("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setMaximumSize(300,170);
	QDialog::resize(300,170);
	setMinimumSize(300,170);

	lcontext->setGeometry(5,5,140,25);
	wcontext->setGeometry(145,5,150,25);

/*
	lfontname->setGeometry(5,30,140,25);
	wfontname->setGeometry(145,30,150,25);
*/
	wfontname->setGeometry(5,30,290,25);

	lthickness->setGeometry(5,60,140,25);
	wthickness->setGeometry(145,60,150,25);

	loffset->setGeometry(5,90,140,25);
	wxoffset->setGeometry(145,90,74,25);
	wyoffset->setGeometry(221,90,74,25);

	cancel->setGeometry(67,130,80,25);
	ok->setGeometry(153,130,80,25);	

	delete locale;
}

TextDialog::~TextDialog()
{
}

void	TextDialog::save(QString &context,QString &fontname,double &thickness,Vector2 &offset)
{
	float		d;
	const char	*str;

	context = wcontext->text();

//	fontname = wfontname->text();
	fontname = wfontname->getFilename();

	str = wthickness->text();
	sscanf(str,"%g",&d);
	thickness = d;

	str = wxoffset->text();
	sscanf(str,"%g",&d);
	offset[0] = d;
	str = wyoffset->text();
	sscanf(str,"%g",&d);
	offset[1] = d;
}

void	TextDialog::load(QString context,QString fontname,double thickness,Vector2 offset)
{
	char		str[128];

	wcontext->setText(context);

	wfontname->setFilename(fontname);

	sprintf(str,"%g",thickness);
	wthickness->setText(str);

	sprintf(str,"%g",offset[0]);
	wxoffset->setText(str);
	sprintf(str,"%g",offset[1]);
	wyoffset->setText(str);
}

// ******************************************************************************************


HeightFieldDialog::HeightFieldDialog(QWidget *parent,const char *name) :
	QDialog(parent,name,TRUE)
{
	KLocale		*locale;
	Preferences	prefs;
        QString		str;

	locale = new KLocale();

	setCaption(name);

	lwaterlevel = new QLabel(this);
	lwaterlevel->setText(stloc("Waterlevel:"));

	wwaterlevel = new QLineEdit(this);

	lfilename = new QLabel(this);
	lfilename->setText(stloc("Filename:"));
	
	str = prefs.getPath(prefs.getHeightfieldPath());
#ifdef DEBUG 	
 	printf("heightfield: %s\n",(const char*)str);
#endif
 	wfilename = new GetFileWidget(this,"",str);

	
	ok = new QPushButton( this,stloc("Ok"));
	ok->setText(stloc("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,stloc("Cancel") );
	cancel->setText(stloc("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setMaximumSize(370,170);
	QDialog::resize(370,170);
	setMinimumSize(370,170);

	lfilename->setGeometry(5,5,140,25);
 	wfilename->setGeometry(140,5,230,25);

	lwaterlevel->setGeometry(5,35,140,25);
	wwaterlevel->setGeometry(145,35,150,25);

	cancel->setGeometry(102,130,80,25);
	ok->setGeometry(188,130,80,25);	

	delete locale;
}

HeightFieldDialog::~HeightFieldDialog()
{
}

void	HeightFieldDialog::save(double *waterlevel,QString &filename)
{
	QString		str;

	str = wwaterlevel->text();
	*waterlevel = str.toDouble();

	filename = wfilename->getFilename();
}

void	HeightFieldDialog::load(double waterlevel,QString filename)
{
	QString		str;

	str.setNum(waterlevel);
	wwaterlevel->setText(str);

	wfilename->setFilename(filename);
}




// ******************************************************************************************




LightPointDialog::LightPointDialog(QWidget *parent,const char *name) :
	QDialog(parent,name,TRUE)
{
	KLocale	*locale;

	locale = new KLocale();

	setCaption(name);

	lfadepower = new QLabel(this);
	lfadepower->setText(stloc("Fadepower:"));

	wfadepower = new QLineEdit(this);

	lfadedistance = new QLabel(this);
	lfadedistance->setText(stloc("Fadedistance:"));

	wfadedistance = new QLineEdit(this);

	lcolor = new QLabel(this);
	lcolor->setText(stloc("Color:"));

	wcolor = new KColorButton(this);

	ok = new QPushButton( this,stloc("Ok"));
	ok->setText(stloc("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,stloc("Cancel") );
	cancel->setText(stloc("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setMaximumSize(300,170);
	QDialog::resize(300,170);
	setMinimumSize(300,170);

	lfadedistance->setGeometry(5,5,140,25);
	wfadedistance->setGeometry(145,5,150,25);

	lfadepower->setGeometry(5,35,140,25);
	wfadepower->setGeometry(145,35,150,25);

	lcolor->setGeometry(5,70,140,25);
	wcolor->setGeometry(145,70,150,25);

	cancel->setGeometry(67,130,80,25);
	ok->setGeometry(153,130,80,25);	

	delete locale;
}

LightPointDialog::~LightPointDialog()
{
}

void	LightPointDialog::save(double *fadepower,double *fadedistance,Vector3 &color)
{
	QString		str;
	QColor		qc;

	str = wfadepower->text();
	*fadepower = str.toDouble();

	str = wfadedistance->text();
	*fadedistance = str.toDouble();

	qc = wcolor->color();
	color[0] = (double)qc.red() / 255.0;
	color[1] = (double)qc.green() / 255.0;
	color[2] = (double)qc.blue() / 255.0;
}

void	LightPointDialog::load(double fadepower,double fadedistance,Vector3 color)
{
	QString		str;
	QColor		qc;

	str.setNum(fadepower);
	wfadepower->setText(str);

	str.setNum(fadedistance);
	wfadedistance->setText(str);

	qc.setRgb(color[0] * 255,color[1] * 255,color[2] * 255);
	wcolor->setColor(qc);
}




// ******************************************************************************************




AnimDialog::AnimDialog(QWidget *parent,const char *name,anim *a) :
	QDialog(parent,name,TRUE)
{
	KLocale		*locale;
	Preferences	prefs;
//	int		n;
	const char	**nl = (const char**)a->getScriptList();

	_anim = a;

	locale = new KLocale();

	setCaption(name);

	fname = new QComboBox(TRUE,this);
	fname->setAutoCompletion(TRUE);
	fname->insertStrList(nl);
	fname->setEditText("");			// provides nonconfusion
	connect(fname,SIGNAL(activated(const char*)),this,SLOT(selectScript(const char*)));
	
	ok = new QPushButton(this,stloc("Ok"));
	ok->setText(stloc("OK"));
	connect(ok,SIGNAL(clicked()),SLOT(saveText()));
	connect(ok,SIGNAL(clicked()),SLOT(accept()));

	add = new QPushButton( this,stloc("Add") );
	add->setText(stloc("Add"));
	connect(add,SIGNAL(clicked()),SLOT(addScript()));

	del = new QPushButton( this,stloc("Delete") );
//	del->setEnabled(FALSE);
	del->setText(stloc("Delete"));
	connect(del,SIGNAL(clicked()),SLOT(delScript()));

	mle = new QMultiLineEdit(this);

	setMaximumSize(1024,960);
	QDialog::resize(prefs.getAnimEditorWidth(),prefs.getAnimEditorHeight());
	setMinimumSize(300,250);

	
	delete locale;

/*
	for(n = 0;nl[n] != 0;n++)
		free(nl[n]);
	free(nl);
*/
	layout();
}

AnimDialog::~AnimDialog()
{
}

void	AnimDialog::paintEvent(QPaintEvent *)
{
	Preferences	prefs;
	
	layout();
	
	prefs.setAnimEditorSize(width(),height());
}

void	AnimDialog::layout()
{
	int	x,y,w,h,w2;
	
	x = y = 5;
	w2 = 60;
	w = width() - w2 * 3 - 25;
	h = 25;
	fname->setGeometry(x,y,w,h);
	x += w + 5;
	add->setGeometry(x,y,w2,h);
	x += w2 + 5;
	del->setGeometry(x,y,w2,h);
	x += w2 + 5;
	ok->setGeometry(x,y,w2,h);
		
	x = 5;
	y += h + 5;
	w = width() - 10;
	h = height() - 15 - h;
	mle->setGeometry(x,y,w,h);
}

void	AnimDialog::addScript()
{
	QString		str;
	int		n;
	char		**nl = _anim->getScriptList();

	saveText();

	str = fname->currentText();

	if(str.isNull() || str.isEmpty())
		return;

	for(n = 0;nl[n] != 0;n++)
	{
		if(str == nl[n])
		{
			for(n = 0;nl[n] != 0;n++)
				free(nl[n]);
			free(nl);

			return;
		}
	}

	_anim->addScript(str.data(),"#!/bin/sh\n");

	fname->insertItem(str);

	for(n = 0;n < fname->count();n++)
		if(fname->text(n) == str)
			fname->setCurrentItem(n);

	for(n = 0;nl[n] != 0;n++)
		free(nl[n]);
	free(nl);
}

void	AnimDialog::delScript()
{
	QString		str;
	int		n,i;
	char		**nl = _anim->getScriptList();

	saveText();

	str = fname->currentText();

	if(str.isNull() || str.isEmpty())
		return;

	for(n = 0;nl[n] != 0;n++)
	{
		if(str == nl[n])
		{
			for(n = 0;nl[n] != 0;n++)
				free(nl[n]);
			free(nl);

			_anim->delScript(str.data());
			
			for(i = 0;i < fname->count();i++)
				if(fname->text(i) == str)
					fname->removeItem(i);
			return;
		}
	}

	for(n = 0;nl[n] != 0;n++)
		free(nl[n]);
	free(nl);
}

void	AnimDialog::selectScript(const char*)
{
	QString		str;
	int		n;
	char		**nl = _anim->getScriptList();

	saveText();

	str = fname->currentText();
	if(str.isNull() || str.isEmpty())
		return;

	for(n = 0;nl[n] != 0;n++)
		if(str == nl[n])
			break;
	if(nl[n] == 0)
	{
		for(n = 0;nl[n] != 0;n++)
			free(nl[n]);
		free(nl);
		return;
	}
	mle->setText(_anim->getText(str.data()));

	for(n = 0;nl[n] != 0;n++)
		free(nl[n]);
	free(nl);
}

void	AnimDialog::saveText()
{
	QString		str;
	int		n;
	char		**nl = _anim->getScriptList();

	str = fname->currentText();

	if(oldstr.isNull() || oldstr.isEmpty())
	{
		oldstr = str;
		return;
	}

	for(n = 0;nl[n] != 0;n++)
		if(oldstr == nl[n])
			break;
	if(nl[n] == 0)
	{
		for(n = 0;nl[n] != 0;n++)
			free(nl[n]);
		free(nl);
		return;
	}

	_anim->setText(oldstr.data(),mle->text().data());

	oldstr = str;

	for(n = 0;nl[n] != 0;n++)
		free(nl[n]);
	free(nl);
}



// ******************************************************************************************


WorldDialog::WorldDialog(QWidget *parent,const char *name) :
	QDialog(parent,name,TRUE)
{
	KLocale	*locale;

	setCaption(name);

	lex = new QLineEdit(this,"");
	lex->setText("0");
	ley = new QLineEdit(this,"");
	ley->setText("0");
	lez = new QLineEdit(this,"");
	lez->setText("0");

	ambient_light = Vector3(0,0,0);

	allabel = new QLabel(this,"");
	allabel->setText("Ambient light:");

	bgc_label = new QLabel(this,"");
	bgc_label->setText("Backgroundcolor:");

	backg_color = new KColorButton(this);

	locale = new KLocale();

	ok = new QPushButton( this,stloc("Ok"));
	ok->setText(stloc("OK"));
	connect( ok, SIGNAL(clicked()), SLOT(accept()) );

	cancel = new QPushButton( this,stloc("Cancel") );
	cancel->setText(stloc("Cancel"));
	connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

	setMaximumSize(300,140);
	QDialog::resize(300,140);
	setMinimumSize(300,140);

	allabel->setGeometry(5,5,155,25);
	lex->setGeometry(165,5,40,25);
	ley->setGeometry(210,5,40,25);
	lez->setGeometry(255,5,40,25);

	bgc_label->setGeometry(5,35,155,25);
	backg_color->setGeometry(165,35,40,25);

	cancel->setGeometry(67,110,80,25);
	ok->setGeometry(153,110,80,25);	

	delete locale;
}

WorldDialog::~WorldDialog()
{
}

void	WorldDialog::save(Vector3 &a,Vector3 &b)
{
	float		d;
	const char	*str;
	QColor		qc;

	str = lex->text();
	sscanf(str,"%g",&d);
	ambient_light[0] = d;

	str = ley->text();
	sscanf(str,"%g",&d);
	ambient_light[1] = d;

	str = lez->text();
	sscanf(str,"%g",&d);
	ambient_light[2] = d;

	a = ambient_light;

	qc = backg_color->color();
	b[0] = (double)qc.red() / 255.0;
	b[1] = (double)qc.green() / 255.0;
	b[2] = (double)qc.blue() / 255.0;
}

void	WorldDialog::load(Vector3 a,Vector3 b)
{
	char		str[128];
	QColor		qc;

	ambient_light = a;

	sprintf(str,"%g",ambient_light[0]);
	lex->setText(str);

	sprintf(str,"%g",ambient_light[1]);
	ley->setText(str);

	sprintf(str,"%g",ambient_light[2]);
	lez->setText(str);

	qc.setRgb(b[0] * 255,b[1] * 255,b[2] * 255);
	backg_color->setColor(qc);
}





