/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** worldtree.cpp
*/

#include <klocale.h>


#define stloc(a)	((char*)klocale->translate((a)))

#include <qpoint.h>

#include "../dialogs/dialogs.h"
#include "../misc/preferences.h"

#include "worldtree.h"
#include "worldtree.moc"

#include "drag3dds.h"

#include "objects.h"

#include "icons/tree/x.xpm"

#include "icons/tree/world.xpm"
#include "icons/tree/obj.xpm"
#include "icons/tree/csg.xpm"
#include "icons/tree/blob.xpm"
#include "icons/tree/box.xpm"
#include "icons/tree/sphere.xpm"
#include "icons/tree/cylinder.xpm"
#include "icons/tree/cone.xpm"
#include "icons/tree/torus.xpm"
#include "icons/tree/mesh.xpm"
#include "icons/tree/triangle.xpm"
#include "icons/tree/text.xpm"
#include "icons/tree/script.xpm"
#include "icons/tree/heightfield.xpm"
#include "icons/tree/plane.xpm"
#include "icons/tree/ref.xpm"
#include "icons/tree/refptr.xpm"
#include "icons/tree/lightpoint.xpm"
#include "icons/tree/spotlight.xpm"
#include "icons/tree/arealight.xpm"
#include "icons/tree/camera2.xpm"

#include "icons/menu/editcopy.xpm"
#include "icons/menu/editpaste.xpm"
#include "icons/menu/editcut.xpm"

#include "icons/menu/edit.xpm"
#include "icons/menu/transform.xpm"

WorldTree::WorldTree(QWidget *p,char *name) :
	TreeList(p,name)
{
	int		t;

	capptr = 0;
	worldptr = 0;

	createWidgets();

	for(t = 0;t < 50;t++) 
	{
		if(t == NUM_WORLD) pixmap[t] = new QPixmap((const char**)world_xpm);
		else if(t == NUM_OBJ) pixmap[t] = new QPixmap((const char**)obj_xpm);
		else if(t == NUM_CSG) pixmap[t] = new QPixmap((const char**)csg_xpm);
		else if(t == NUM_BLOB) pixmap[t] = new QPixmap((const char**)blob_xpm);
		else if(t == NUM_SPHERE) pixmap[t] = new QPixmap((const char**)sphere_xpm);
		else if(t == NUM_CYLINDER) pixmap[t] = new QPixmap((const char**)cylinder_xpm);
		else if(t == NUM_BOX) pixmap[t] = new QPixmap((const char**)box_xpm);
		else if(t == NUM_CONE) pixmap[t] = new QPixmap((const char**)cone_xpm);
		else if(t == NUM_TORUS) pixmap[t] = new QPixmap((const char**)torus_xpm);
		else if(t == NUM_MESH) pixmap[t] = new QPixmap((const char**)mesh_xpm);
		else if(t == NUM_PLANE) pixmap[t] = new QPixmap((const char**)plane_xpm);
		else if(t == NUM_TEXT) pixmap[t] = new QPixmap((const char**)text_xpm);
		else if(t == NUM_SCRIPT) pixmap[t] = new QPixmap((const char**)script_xpm);
		else if(t == NUM_HEIGHTFIELD) pixmap[t] = new QPixmap((const char**)heightfield_xpm);
		else if(t == NUM_TRIANGLE) pixmap[t] = new QPixmap((const char**)triangle_xpm);
		else if(t == NUM_REF) pixmap[t] = new QPixmap((const char**)ref_xpm);
		else if(t == NUM_REFPTR) pixmap[t] = new QPixmap((const char**)refptr_xpm);
		else if(t == NUM_SPOTLIGHT) pixmap[t] = new QPixmap((const char**)spotlight_xpm);
		else if(t == NUM_AREALIGHT) pixmap[t] = new QPixmap((const char**)arealight_xpm);
		else if(t == NUM_LIGHTPOINT) pixmap[t] = new QPixmap((const char**)lightpoint_xpm);
		else if(t == NUM_CAMERA) pixmap[t] = new QPixmap((const char**)camera_xpm);
		else pixmap[t] = new QPixmap((const char**)x_xpm);
	}

	layout();
	empty();
}

WorldTree::~WorldTree()
{
	if(worldptr) delete worldptr;

	if(capptr) delete capptr;
}

void	WorldTree::createWidgets()
{
	addColumn("Object tree",150);
	addColumn("Texture",120);
	addColumn("Sort",1);
	setColumnWidthMode(0,QListView::Maximum);
	setColumnWidthMode(1,QListView::Manual);
	setColumnWidthMode(2,QListView::Manual);
	setMultiSelection(FALSE);
	setSorting(2,TRUE);

	connect(this,SIGNAL(popupMenu(QListViewItem*,int,int)),this,SLOT(popupMenu(QListViewItem*,int,int)));
	connect(this,SIGNAL(selectionChanged(QListViewItem*)),this,SLOT(getSelected(QListViewItem*)));

	layout();
}

void	WorldTree::createTree()
{
	clear();
	
	createTree(worldptr,0,0);

	repaint(TRUE);
}

void	WorldTree::createTree(base *b,QListViewItem *pit,int num)
{
	base		**children = 0;
	int		t;
	TreeListItem	*li;
	char		str[16];

	if(!b) return;

	if(!pit)
		li = new TreeListItem((QListView*)this,b->getName(),20);
	else
		li = new TreeListItem(pit,b->getName(),20);

	if(!li) return;

	if(!pit)
	{
		li->setOpen(TRUE);
	}
	else
	{
		pit->setExpandable(TRUE);
	}
	li->setHeight(20);

	li->setPixmap(0,*pixmap[b->getType()]);

	if(b->getType() != NUM_WORLD && b->getType() != NUM_REF)
	{
		if(((body*)b)->getTexture())
			li->setText(1,((body*)b)->getTexture()->getName());
	}

	sprintf(str,"%i",num);
	li->setText(2,str);

	b->getChildren(&children);
	if(!children)	return;

	t = 0;
	while(children[t])
	{
		createTree(children[t],li,t);
		t++;
	}
	if(children) free(children);
}

void	WorldTree::layout()
{
	int	w,h,cw;

	w = width();
	h = height();

	// should do
	triggerUpdate();

	// but this does better
	cw = w - columnWidth(1) - columnWidth(2);
	setColumnWidth(0,cw);
}

void	WorldTree::draw()
{
	layout();
	repaint();
}

int	WorldTree::exportPOV(const char *filename,view *v,int anim)
{
	FILE	*fp; 
	int	error;

	fp = fopen(filename,"wb"); 
 
	if(fp == 0) return -3;

//	error = v->exportPOV(fp,4);
	error = worldptr->exportPOV(fp,0,4,anim);

	fclose(fp); 

	return error;
}

int	WorldTree::save(const char *filename)
{
	int	error;
	world	x;

	error = x.save3DDS(filename,worldptr);

	return error;
}

int	WorldTree::load(const char *filename)
{
	world	x;
	int	error;
	base	*b;

	error = x.load3DDS(filename,&b); 

	if(!b) return error;

	if(b->getType() == NUM_WORLD)
	{
		empty();
		delete worldptr;
		worldptr = (world*)b;
		worldptr->setParent(0); 
		emit changeWorld(worldptr);
	}
 	else delete b;

	createTree();

	draw();

	return error;
}


int	WorldTree::empty()
{

	if(worldptr)
		delete worldptr;
	worldptr = new world("World");
	createTree();
	
	emit changeWorld(worldptr);

	emit redraw();

	return 0;
}

void	WorldTree::setWorld(world *nw)
{
	worldptr = nw;
	createTree();

	emit setTextureLibrary(worldptr->getTextureLibrary());
	emit redraw();
}

void	WorldTree::paintEvent(QPaintEvent*)
{
	layout();
}

void	WorldTree::resizeEvent(QResizeEvent*)
{
	layout();
}

void	WorldTree::dump()
{
	worldptr->dumpNames(0,4);
}

int	WorldTree::addX(base *child)
{
	QListViewItem	*i;
	QListViewItem	*li;
	QListViewItem	*ci;
	base		*b;
	char		*n;
	int		num;
	char		str[26];
	Preferences	prefs;

	if(!child) return -5;

	n = worldptr->createName((char*)child->getName());
	if(!n)
	{
		delete child;
		return -4;
	}
	child->setName(n);

	if(!(i = currentItem()))
	{
		delete child;
		return -3;
	}

	if(!(b = worldptr->searchName((char*)i->text(0))))
	{
		delete child;
		return -2;
	}

	if(child->addToParent(b) != 0)
	{
		delete child;
		return -1;
	}


	if(prefs.getDrawAxes())
	{
		child->setFlag(base::DRAW_AXIS,1);
	}
	else
	{
		child->setFlag(base::DRAW_AXIS,0);
	}

	if((ci = i->firstChild()))
	{
		while(ci->nextSibling())
			ci = ci->nextSibling();
	}
	if(ci)
		num = atoi(ci->text(2)) + 1;
	else
		num = 0;
	li = new TreeListItem(i,child->getName(),20);
	li->setPixmap(0,*pixmap[child->getType()]);
	sprintf(str,"%i",num);
	li->setText(2,str);
	i->setOpen(TRUE);

	emit redraw();

	return 0;
}

void	WorldTree::addObject()
{
	addX(new obj(0,"Object"));
}

void	WorldTree::addCSG()
{
	addX(new csg(0,"CSGObject"));
}

void	WorldTree::addBlob()
{
	addX(new blob(0,"Blob",0.5));
}

void	WorldTree::addRef()
{
	addX(new ref(0,"ReferenceObject"));
}

void	WorldTree::addSphere()
{
	addX(new sphere(0,"Sphere",1));
}

void	WorldTree::addCylinder()
{
	addX(new cylinder(0,"Cylinder"));
}

void	WorldTree::addBox()
{
	addX(new box(0,"Box"));
}

void	WorldTree::addMesh()
{
	addX(new cmesh(0,"CMesh"));
}

void	WorldTree::addRefPtr()
{
	addX(new refptr(0,"Reference"));
}

void	WorldTree::addPlane()
{
	addX(new plane(0,"Plane"));
}

void	WorldTree::addText()
{
	text		*tp;
	TextDialog	*td;
	Vector2		offset;
	QString		context,fontname;
	double		thickness;
	Preferences	prefs;

	fontname = prefs.getFontPath() + "/times.ttf";
	addX((tp = new text(0,"Text","k4de for PovRay 3.0",prefs.getPath(fontname))));
//	tp->setFontPath("data/fonts");

	offset = tp->getOffset();
	context = tp->getText();
	fontname = tp->getFontName();
	thickness = tp->getThickness();

	td = new TextDialog(this,"Text");

	td->load(context,fontname,thickness,offset);

	td->show();
	if(td->result())
	{
		td->save(context,fontname,thickness,offset);
		tp->setText(context);
		tp->setFontName(fontname);
		tp->setThickness(thickness);
		tp->setOffset(offset);
	}
	delete td;
}

void	WorldTree::addCone()
{
	addX(new cone(0,"Cone"));
}

void	WorldTree::addTorus()
{
	addX(new torus(0,"Torus"));
}

void	WorldTree::addHeightfield()
{
	addX(new heightfield(0,"HeightField"));
}

void	WorldTree::addTriangle()
{
	addX(new triangle(0,"Triangle"));
}

void	WorldTree::addSpotLight()
{
	addX(new spotlight(0,"Spotlight"));
}

void	WorldTree::addAreaLight()
{
	//addX(new arealight(0,"Arealight"));
}

void	WorldTree::addLightPoint()
{
	addX(new lightpoint(0,"Lightpoint"));
}

void	WorldTree::addCamera()
{
	addX(new camera(0,"Camera"));
}

void	WorldTree::popupMenu(QListViewItem *li,int x,int y)
{
	base		*b;
	QPopupMenu	*pm;
	QPoint		mp = mapToGlobal(QPoint(x,y));
	int		id;

	b = worldptr->searchName((char*)li->text(0));

	if(!b)
	{
		printf("Object %s not found\n",li->text(0));
		return;
	}

	pm = new QPopupMenu();
	pm->insertItem("Rename",1);

	switch(b->getType())
	{
		case NUM_WORLD:
			pm->insertItem(QPixmap((const char**)editpaste_xpm),"Paste",21);
			pm->setItemEnabled(21,!(!capptr));
			pm->insertItem(QPixmap((const char**)edit_xpm),"Edit Animscript Main",27);
			pm->insertItem(QPixmap((const char**)world_xpm),"World",29);
		break;
		case NUM_REF:
			pm->insertItem(QPixmap((const char**)editcut_xpm),"Cut",20);
			pm->insertItem(QPixmap((const char**)editcopy_xpm),"Copy",22);
			pm->insertItem(QPixmap((const char**)editpaste_xpm),"Paste",21);
			pm->setItemEnabled(21,!(!capptr));		
			pm->insertSeparator();
			id = pm->insertItem("Hide",25);
			if(b->isFlag(base::HIDE))
				pm->setItemChecked(id,!0);
			id = pm->insertItem("BBox",26);
			if(b->isFlag(base::DRAW_BOUNDINGBOX))
				pm->setItemChecked(id,!0);
			id = pm->insertItem("Axis",30);
			if(b->isFlag(base::DRAW_AXIS))
				pm->setItemChecked(id,!0);
			pm->insertItem(QPixmap((const char**)edit_xpm),"Edit Animscript",28);
		break;
		case NUM_OBJ:
		case NUM_CSG:
		case NUM_BLOB:
			pm->insertItem(QPixmap((const char**)editcut_xpm),"Cut",20);
			pm->insertItem(QPixmap((const char**)editcopy_xpm),"Copy",22);
			pm->insertItem(QPixmap((const char**)editpaste_xpm),"Paste",21);
			pm->setItemEnabled(21,!(!capptr));		
			pm->insertSeparator();
			id = pm->insertItem("Hide",25);
			if(b->isFlag(base::HIDE))
				pm->setItemChecked(id,!0);
			id = pm->insertItem("BBox",26);
			if(b->isFlag(base::DRAW_BOUNDINGBOX))
				pm->setItemChecked(id,!0);
			id = pm->insertItem("Axis",30);
			if(b->isFlag(base::DRAW_AXIS))
				pm->setItemChecked(id,!0);
			pm->insertSeparator();
			id = pm->insertItem("Hollow on",32);
			if(b->isFlag(base::HOLLOW_ON))
				pm->setItemChecked(id,!0);
			id = pm->insertItem("Hollow off",33);
			if(b->isFlag(base::HOLLOW_OFF))
				pm->setItemChecked(id,!0);
			pm->insertSeparator();
			pm->insertItem(QPixmap((const char**)transform_xpm),"Dim",3);
			pm->insertItem(QPixmap((const char**)edit_xpm),"Edit Animscript",28);
		break;
		default:
			pm->insertItem(QPixmap((const char**)editcut_xpm),"Cut",20);
			pm->insertItem(QPixmap((const char**)editcopy_xpm),"Copy",22);
			pm->insertSeparator();
			id = pm->insertItem("Hide",25);
			if(b->isFlag(base::HIDE))
				pm->setItemChecked(id,!0);
			if(b->isFinite())
			{
				id = pm->insertItem("BBox",26);
				if(b->isFlag(base::DRAW_BOUNDINGBOX))
					pm->setItemChecked(id,!0);
			}
			id = pm->insertItem("Axis",30);
			if(b->isFlag(base::DRAW_AXIS))
				pm->setItemChecked(id,!0);
			pm->insertSeparator();
			id = pm->insertItem("Hollow on",32);
			if(b->isFlag(base::HOLLOW_ON))
				pm->setItemChecked(id,!0);
			id = pm->insertItem("Hollow off",33);
			if(b->isFlag(base::HOLLOW_OFF))
				pm->setItemChecked(id,!0);
			pm->insertSeparator();
			pm->insertItem(QPixmap((const char**)transform_xpm),"Dim",3);
			pm->insertItem(QPixmap((const char**)edit_xpm),"Edit Animscript",28);
	}


	switch(b->getType())
	{
		case NUM_CONE:
			pm->insertSeparator();
			pm->insertItem("Cone",4);
		break;
		case NUM_TORUS:
			pm->insertSeparator();
			pm->insertItem("Torus",5);
		break;
		case NUM_REFPTR:
			pm->insertSeparator();
			pm->insertItem("Reference",6);
		break;
		case NUM_CSG:
			pm->insertSeparator();
			id = pm->insertItem("Union",11);
			if(((csg*)b)->getCSGType() == csg::UNION)
				pm->setItemChecked(id,!0);
			id = pm->insertItem("Merge",12);
			if(((csg*)b)->getCSGType() == csg::MERGE)
				pm->setItemChecked(id,!0);
			id = pm->insertItem("Intersection",13);
			if(((csg*)b)->getCSGType() == csg::INTERSECTION)
				pm->setItemChecked(id,!0);
			id = pm->insertItem("Difference",14);
			if(((csg*)b)->getCSGType() == csg::DIFFERENCE)
				pm->setItemChecked(id,!0);
			id = pm->insertItem("Invert",15);
			if(((csg*)b)->getCSGType() == csg::INVERT)
				pm->setItemChecked(id,!0);
		break;
		case NUM_BLOB:
			pm->insertSeparator();
			pm->insertItem(QPixmap((const char**)blob_xpm),"Blob",8);
		break;
		case NUM_TEXT:
			pm->insertSeparator();
			pm->insertItem(QPixmap((const char**)text_xpm),"Text",19);
		break;
		case NUM_TRIANGLE:
			pm->insertSeparator();
			pm->insertItem(QPixmap((const char**)triangle_xpm),"Triangle",9);
		break;
		case NUM_HEIGHTFIELD:
			pm->insertSeparator();
			pm->insertItem(QPixmap((const char**)heightfield_xpm),"Heightfield",10);
		break;
		case NUM_LIGHTPOINT:
			pm->insertSeparator();
			pm->insertItem(QPixmap((const char**)lightpoint_xpm),"Lightpoint",16);
		break;
		case NUM_SPOTLIGHT:
			pm->insertSeparator();
			pm->insertItem(QPixmap((const char**)spotlight_xpm),"Spotlight",17);
		break;
		case NUM_AREALIGHT:
			pm->insertSeparator();
			pm->insertItem(QPixmap((const char**)arealight_xpm),"Arealight",18);
		break;
		case NUM_CAMERA:
			pm->insertSeparator();
			pm->insertItem(QPixmap((const char**)camera_xpm),"Camera",31);
		break;
	}

	pm->popup(mp);
	pm->show();
	switch(pm->exec())
	{
		case 1: // rename
		{
			NameDialog	*nd;
			QString		s;

			s = b->getName();
			nd = new NameDialog(this,"Name");
			nd->setName(s);
			nd->show();
			if(nd->result())
				rename(li,nd->getName());
			delete nd;
		}
		break;
		case 20: // cut
			cut(li);
		break;
		case 21: // paste
			paste(li);
		break;
		case 22: // copy
			copy(li);
		break;
		case 25: // hide
			b->setFlag(base::HIDE,!(b->isFlag(base::HIDE)));
			emit redraw();
		break;
		case 26: // draw boundingbox
			b->setFlag(base::DRAW_BOUNDINGBOX,!(b->isFlag(base::DRAW_BOUNDINGBOX)));
			emit redraw();
		break;
		case 30: // draw axis
			b->setFlag(base::DRAW_AXIS,!(b->isFlag(base::DRAW_AXIS)));
			emit redraw();
		break;
		case 32: // hollow on
			b->setFlag(base::HOLLOW_ON,!(b->isFlag(base::HOLLOW_ON)));
			if(b->isFlag(base::HOLLOW_ON))
				b->setFlag(base::HOLLOW_OFF,0);
		break;
		case 33: // hollow off
			b->setFlag(base::HOLLOW_OFF,!(b->isFlag(base::HOLLOW_OFF)));
			if(b->isFlag(base::HOLLOW_OFF))
				b->setFlag(base::HOLLOW_ON,0);
		break;
		case 3: // dim
		{
			DimDialog	*dd;
			Vector3		v0;
			Vector3		v1;
			Vector3		v2;

			v0 = ((dim*)b)->getVScale();
			v1 = ((dim*)b)->getVRotate();
			v2 = ((dim*)b)->getVTranslate();

			dd = new DimDialog(this,"Dimensions");

			dd->load(v0,v1,v2);

			dd->show();
			if(dd->result())
			{
				dd->save(v0,v1,v2);
				((dim*)b)->getVScale() = v0;
				((dim*)b)->getVRotate() = v1;
				((dim*)b)->getVTranslate() = v2;
			}
			delete dd;
		}
		break;
		case 6: // reference
		{
			NameDialog	*nd;
			base		*bt;
			QString		s;

			if(((refptr*)b)->getPTR())
				s = ((refptr*)b)->getPTR()->getName();

			nd = new NameDialog(this,"Name of Referenceobject");
			nd->setName(s);
			nd->show();
			if(nd->result())
			{
				bt = worldptr->searchName(nd->getName().data());
				if(bt && bt->getType() == NUM_REF)
				{
					if(bt->addToParent(b) == 0)
					{
						QListViewItem *tli;

						while(li->childCount())
						{
							tli = li->firstChild();
							li->removeItem(tli);
							delete tli;
						}
						//createTree(bt,li);
					}
				}
			}
			delete nd;
		}
		break;
		case 8: // blob
		{
			BlobDialog	*bd;

			bd = new BlobDialog(this,"Blob");
			bd->setThresholdValue(((blob*)b)->thresholdValue());
			bd->show();
			if(bd->result())
			{
				((blob*)b)->setThresholdValue(bd->thresholdValue());
			}
			delete bd;
		}
		break;
		case 11: // union
			((csg*)b)->setCSGType(csg::UNION);
		break;
		case 12: // merge
			((csg*)b)->setCSGType(csg::MERGE);
		break;
		case 13: // intersection
			((csg*)b)->setCSGType(csg::INTERSECTION);
		break;
		case 14: // difference
			((csg*)b)->setCSGType(csg::DIFFERENCE);
		break;
		case 15: // invert
			((csg*)b)->setCSGType(csg::INVERT);
		break;
		case 16: // pointlight
		{
			LightPointDialog	*ld;
			Vector3			color;
			double			fadepower,fadedistance;

			fadepower = ((lightpoint*)b)->getFadePower();
			fadedistance = ((lightpoint*)b)->getFadeDistance();
			color = ((lightpoint*)b)->getColor();

			ld = new LightPointDialog(this,"LightPoint");

			ld->load(fadepower,fadedistance,color);

			ld->show();
			if(ld->result())
			{
				ld->save(&fadepower,&fadedistance,color);
				((lightpoint*)b)->setFadePower(fadepower);
				((lightpoint*)b)->setFadeDistance(fadedistance);
				((lightpoint*)b)->setColor(color);
			}
			delete ld;
		}
		break;
		case 17: // spotlight
		break;
		case 18: // arealight
		break;
		case 19: // text
		{
			TextDialog	*td;
			Vector2		offset;
			QString		context,fontname;
			double		thickness;

			offset = ((text*)b)->getOffset();
			context = ((text*)b)->getText();
			fontname = ((text*)b)->getFontName();
			thickness = ((text*)b)->getThickness();

			td = new TextDialog(this,"Text");

			td->load(context,fontname,thickness,offset);

			td->show();
			if(td->result())
			{
				td->save(context,fontname,thickness,offset);
				((text*)b)->setText(context.data());
				((text*)b)->setFontName(fontname.data());
				((text*)b)->setThickness(thickness);
				((text*)b)->setOffset(offset);
			}
			delete td;
		}
		break;
		case 27: // Edit Animscript Main
		case 28: // Edit Animscript
			animScript(li);
		break;
		case 29: // World
		{
			WorldDialog	*wd;
			Vector3		al,bc;

			al = ((world*)b)->getAmbientLight();
			bc = ((world*)b)->getBackgroundColor();

			wd = new WorldDialog(this,"World");

			wd->load(al,bc);

			wd->show();
			if(wd->result())
			{
				wd->save(al,bc);
				((world*)b)->setAmbientLight(al);
				((world*)b)->setBackgroundColor(bc);
			}
			delete wd;
		}
		break;
		case 31: // Camera
		{
			CameraDialog	*cd;
			QString		viewname;
			QStrList	viewlist;
			int		i;
			view		*v;
			
			v = ((camera*)b)->getView();
			if(v)
				viewlist.append(v->getName());
			for(i = 0;i < worldptr->getNumViews();i++)
			{
				v = worldptr->getView(i);
				if(v && v->getCamera() == 0)
				{
					viewlist.append(v->getName());
				}
			}

			if(viewlist.count() < 1)
			{
				return;
			}
			
			cd = new CameraDialog(this,"Select View");

			cd->setViewList(viewlist);

			cd->show();
			if(cd->result())
			{
				viewname = cd->getView();
				for(i = 0;i < worldptr->getNumViews();i++)
				{
					v = worldptr->getView(i);
					if(v && v->getCamera() == 0 && viewname == v->getName())
					{
						((camera*)b)->setView(v);
						break;
					}
				}
			}
			delete cd;
		}
		break;
		case 10: // HeightField
		{
			HeightFieldDialog	*hd;
			QString			str;
			double			wl;

			str = ((heightfield*)b)->getFileName();
			wl = ((heightfield*)b)->getWaterLevel();

			hd = new HeightFieldDialog(this,"HieghtField");

			hd->load(wl,str);

			hd->show();
			if(hd->result())
			{
				QImage		image;

				hd->save(&wl,str);
				((heightfield*)b)->setFileName(str);
				((heightfield*)b)->setWaterLevel(wl);

				if(image.load(str))
				{
					int		w,h;
					float		*field,f;

					w = image.width();
					h = image.height();
					field = (float*)malloc(w * h * sizeof(float));
					if(field)
					{
						int		x,y;

						for(x = 0;x < w;x++)
						{
							for(y = 0;y < h;y++)
							{
								if(image.depth() == 32)
								{
									QRgb	rgb;

									rgb = image.pixel(x,y);
									f = (double)rgb / 16777215;
								}
								else
								{
									f = (double)image.pixelIndex(x,y) / 255;
								}
								field[x + y * w] = f;
							}
						}
						((heightfield*)b)->setField(field,w,h);
						free(field);
					}
				}
			}
			delete hd;
		}
		break;
	}

	delete pm;
}

void	WorldTree::animScript(QListViewItem *li)
{
	AnimDialog	*ad;
	base		*b;

	if(!(b = worldptr->searchName((char*)li->text(0)))) return;

	ad = new AnimDialog(this,"AnimScript",b);
	ad->exec();
}

void	WorldTree::rename(QListViewItem *li,const char *newname)
{
	base		*b;

	if(worldptr->existsName((char*)newname)) return;

	if(!(b = worldptr->searchName((char*)li->text(0)))) return;
	
	b->setName((char*)newname);
	li->setText(0,b->getName());
	draw();
}

void	WorldTree::remove(QListViewItem *li)
{
	base		*b;
	QListViewItem	*i;

	if(!(b = worldptr->searchName((char*)li->text(0)))) return;

	i = li->parent();

	i->removeItem(li);
	delete li;

	delete b;
}

void	WorldTree::cut(QListViewItem *li)
{
	base		*b;
	QListViewItem	*i;

	if(!(b = worldptr->searchName((char*)li->text(0)))) return;

	if(b->removeFromParent() == 0)
	{
		i = li->parent();

		//i->removeItem(li);
		delete li; // reicht aus

		if(capptr) delete capptr;
		capptr = b;
	}	
}

void	WorldTree::copy(QListViewItem *li)
{
	base		*b;

	if(!(b = worldptr->searchName((char*)li->text(0)))) return;

	if(capptr) delete capptr;
	capptr = b->copy(worldptr);
	if(capptr) capptr->removeFromParent();
	capptr->dumpNames(0,4);
}

void	WorldTree::paste(QListViewItem *li)
{
	base		*parent,*child;
	QListViewItem	*i = 0;
	int		num;

	if(!(parent = worldptr->searchName((char*)li->text(0)))) return;

	child = capptr;

	if(child)
	{
		if(child->addToParent(parent) != 0)
		{
			delete child;
			return;
		}
		if((i = li->firstChild()))
		{
			while(i->nextSibling())
				i = i->nextSibling();
		}
		if(i)
			num = atoi(i->text(2)) + 1;
		else
			num = 0;
		createTree(child,li,num);
	}

	capptr = 0;
}

void	WorldTree::getSelected(QListViewItem *li)
{
	base		*b;

	if(!(b = worldptr->searchName((char*)li->text(0)))) return;

	emit setSelected(b);
}

void	WorldTree::setTexture(texture *t)
{
	QListViewItem	*i;
	base		*b;

	if(!(i = currentItem())) return;

	if(!(b = worldptr->searchName((char*)i->text(0)))) return;

	if(b->getType() != NUM_WORLD &&
		b->getType() != NUM_REF)
	{
		((body*)b)->setTexture(t);
		if(t)
			i->setText(1,t->getName());
		else
			i->setText(1,"");
	}
}

void	WorldTree::setTexture(QString object,texture *t)
{
	QListViewItem	*i;
	base		*b;

	if((i = findItem(object)) == 0)
		return;

	if(!(b = worldptr->searchName((const char*)object))) return;

	if(b->getType() != NUM_WORLD &&
		b->getType() != NUM_REF)
	{
		((body*)b)->setTexture(t);
		if(t)
			i->setText(1,t->getName());
		else
			i->setText(1,"");
	}
}
void	WorldTree::startDragCopy(QListViewItem *item,QWidget *dragsource)
{
	if(!item || !dragsource)
		return;

	QPixmap		pixmap;
	Drag3DDS	*drag;
	base		*b;

	if(!(b = worldptr->searchName((char*)item->text(0)))) return;
	if(b->type() == NUM_WORLD) return;
	if(b->type() == NUM_CAMERA) return;
	if(b->type() == NUM_REFPTR) return;

	pixmap = *item->pixmap(0);

	drag = new Drag3DDS("3dds/modell",dragsource);
	drag->encode(b);
	drag->setPixmap(pixmap);
	drag->dragCopy();
}

void	WorldTree::startDragMove(QListViewItem *item,QWidget *dragsource)
{
	if(!item || !dragsource)
		return;

	QPixmap		pixmap;
	Drag3DDS	*drag;
	base		*b;

	if(!(b = worldptr->searchName((char*)item->text(0)))) return;
	if(b->type() == NUM_WORLD) return;
	if(b->type() == NUM_CAMERA) return;
	if(b->type() == NUM_REFPTR) return;

	pixmap = *item->pixmap(0);

	drag = new Drag3DDS("3dds/modell",dragsource);
	drag->encode(b);
	drag->setPixmap(pixmap);
	drag->dragMove();
}

void	WorldTree::dragMoveEvent(QDragMoveEvent *dme)
{
	if(Drag3DDS::canDecode(dme,"3dds/modell"))
	{
		dme->accept();
	}
	else
	{
		dme->ignore();
	}
}

void	WorldTree::dropEvent(QDropEvent *de)
{
	base		*b;
	textureBase		*t;

	if(Drag3DDS::decode(de,&b,"3dds/modell") && b)
	{
		QPoint		p = de->pos();
		QListViewItem	*li = itemAt(p);

		if(li)
		{
			base		*parent,*t;
			QListViewItem	*i;

			if((parent = worldptr->searchName((char*)li->text(0))))
			{
				int	num;

				t = b->copy(worldptr);
				if(t)
				{
					t->removeFromParent();

					if(!t->addToParent(parent))
					{
						if((i = li->firstChild()))
						{
							while(i->nextSibling())
								i = i->nextSibling();
						}
						if(i)
							num = atoi(i->text(2)) + 1;
						else
							num = 0;
						createTree(t,li,num);
					}
					else
					{
						delete t;
					}
				}
			}
		}
		delete b;
	}
	if ((Drag3DDS::decode(de,&t,"3dds/texture") && t)&&(t->getType()==TEXTURE))
	{
		printf("Dropped Texture\n");

		QPoint		p = de->pos();
		QListViewItem	*li = itemAt(p);

		if(li)
		{
			base		*parent;

			if((parent = worldptr->searchName((char*)li->text(0))))
			{
				if(parent->getType() != NUM_WORLD &&
					parent->getType() != NUM_REF)
				{
					((body*)parent)->setTexture((texture *)t);
					t = ((body*)parent)->getTexture();
					if(t)
						li->setText(1,t->getName());
					else
						li->setText(1,"");
				}
			}
		}
	} else printf("Dropped a single TextureObject....\n");
}

void	WorldTree::dragEnterEvent(QDragEnterEvent *de)
{
	if(Drag3DDS::canDecode(de,"3dds/modell"))
	{
		de->accept();
	}
}

void	WorldTree::dragLeaveEvent(QDragLeaveEvent *)
{
    // Give the user some feedback...
}

void	WorldTree::changeTexture(texture *t)
{
	changeTexture(t,firstChild());
}

void	WorldTree::changeTexture(texture *t,QListViewItem *i)
{
	QListViewItem	*li;
        base		*b;
        body		*bo;

	if(t == 0)
		return;
	
	li = i;	
	while(li)
	{
		b = worldptr->searchName(li->text(0));
		bo = 0;
		switch(b->getType())
		{
			case NUM_WORLD:
			case NUM_CAMERA:
			case NUM_DIM:
			case NUM_BASE:
			break;
			default:
				bo = (body*)b;
		}

		if(bo && bo->getTexture() == t)
		{
			li->setText(1,bo->getTexture()->getName());
		}
		changeTexture(t,li->firstChild());
                li = li->nextSibling();
	}
}




