/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** base.cpp
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "base.h"
#include "objects.h"

#include "dragvector.h"

#include <misc.h>


typedef	int (base::*bmethodfn)(param**,list<param*>*);


base::base(base *p,const char *n) :
	chunk(), boundingbox(), anim(), extradata()
{
	parent = 0;
	numtype = NUM_BASE;
	flags=0;
	parent = p;
	name = 0;
	name = getRoot()->createName(n);

	bmethodfn	b;

	b = &asetFlag;
	addMethod("setflag",(methodfn)b);
	b = &aclearFlag;
	addMethod("clearflag",(methodfn)b);
	b = &agetChannelNumber;
	addMethod("getChannelNumber",(methodfn)b);
	addMethod("getchannelnumber",(methodfn)b);
}

base::base(base *p,base *bc) :
	chunk(), boundingbox(), anim(bc), extradata()
{
	parent = 0;
	numtype = bc->numtype;
	flags = bc->flags;
	parent = p;
	name = 0;
	name = getRoot()->createName(bc->name);

	bmethodfn	b;

	b = &asetFlag;
	addMethod("setflag",(methodfn)b);
	b = &aclearFlag;
	addMethod("clearflag",(methodfn)b);
	b = &agetChannelNumber;
	addMethod("getChannelNumber",(methodfn)b);
	addMethod("getchannelnumber",(methodfn)b);
}

base::~base()
{
	int	t;
	base	*b;

#ifdef DEBUG
	if(name)
		printf("destructed %s\n",name);
#endif

	for(t = 0;t < dragvectorlist.length();t++)
	{
		delete dragvectorlist[t];
	}
	
	b = this;
	b->removeFromParent();
			
	if(name)
	{
		free(name);
		name = 0;
	}
}

int	base::removeFromParent()
{
	if(!parent)
	{
		printf("Not connected to parent %s!\n",getName());
		return -1;
	}

	return parent->removeChild(this);
}

base	*base::searchName(const char *n)
{
	if(n == 0)
		return 0;

	if(name != 0 && strcmp(name,n) == 0)
		return this;

	return 0;
}

int	base::existsName(const char *n) 
{
	if(n == 0) return 0;

	if(name != 0 && strcmp(name,n) == 0)
		return 1;

	return 0;
}

char	*base::createName(const char *basename)
{
	int		t;
	char		num[10],*cp;

	cp = strdup(basename);

	for(t = 1;existsName(cp);t++)
	{
		sprintf(num,"%i",t);
		cp = (char*)realloc(cp,strlen(basename) + strlen(num) + 1);
		sprintf(cp,"%s%i",basename,t);
	}

	return cp;
}

const char	*base::getName()
{
	return name;
}
	
int	base::setName(const char *n)
{
	if(name)
		free(name);
		
	name = strdup(n);

	return 0;
}; 

void	base::dumpNames(int tabsize = 2)
{
	dumpNames(0,tabsize);
}

void	base::addDragvector(dragvector *dv)
{
	dragvectorlist += dv;
}

int	base::drawDragvectors(view *v,Matrix44 m,int anim)
{
	int		t;
	dragvector	*dv;

	if(anim)
		return 0;

	if(v->projectionMode() != view::PERSPECTIVE)
		return 0;
	if(this != v->getSelected())
		return 0;

	for(t = 0;t < dragvectorlist.length();t++)
	{
		dv = dragvectorlist[t];

		if(dv->isFlag(dragvector::NO_DIRECTION))
		{
			v->drawDragvector(m,dv->position() + dv->direction(),Vector3(0,0,0),anim);
		}
		else
		{
			v->drawDragvector(m,dv->position(),dv->direction(),anim);
		}
	}

	return 0;
}

int	base::changeDragvector(view *v,Matrix44 m,int fx,int fy,int tx,int ty)
{
	int		t;
	dragvector	*dv;
	Vector2		xy,v2;
	Vector3		v3,t3,f,d,x;
	Vector4		v4;

	if(v->projectionMode() != view::PERSPECTIVE)
		return 0;
	if(this != v->getSelected())
		return 0;

	for(t = 0;t < dragvectorlist.length();t++)
	{
		dv = dragvectorlist[t];

		// find and modify dragvector
		v4 = t3 = dv->position() + dv->direction();
		x = v4 * m;
		if(v->project(t3,xy,m,0) == 0)
		{
			v2 = xy - Vector2(fx,fy);
			if(v2.length() <= 7)
			{
				if(v->reproject(Vector2(fx,fy),t3,m,0) == 0)
				{
//					printf("Movefrom (%g,%g,%g)\n",t3[0],t3[1],t3[2]);
				}
				if(v->reproject(Vector2(tx,ty),v3,m,0) == 0)
				{
//					printf("Moveto   (%g,%g,%g)\n",v3[0],v3[1],v3[2]);
				}

				d = v3 - t3;
			
//				f = d * x.length();
				f = d * (x - v->getPEye()).length();

				dv->add(f);

				return 0;
			}
		}
	}

	return -1;
}

base	*base::getParent()
{
	return parent;
}

base	*base::getRoot()
{
	base	*b = this;

	while(b != 0)
	{
		if(b->getType() == NUM_WORLD) return b;
		if(b->getParent() == 0) return b;
		b = b->getParent();
	}

	return 0;
}

base	*base::parse(media *m,int ver)
{
	base	*b;
	int	l,pos;
	char	chunk[4];
	char	*n = 0;

#ifdef DEBUG
	printf("start parsing ...\n");
#endif

	setMedia(m);

	readChunk(chunk,l);

	pos = m->tell();

	b = 0;

	n = readNameChunk();

	printf("parse %c%c%c%c named %s of len %i\n",chunk[0],chunk[1],chunk[2],chunk[3],n,l);

	if(strncmp(chunk,"WRLD",4) == 0)
	{
		b = new world(n);
	}
	else if(strncmp(chunk,"BOX ",4) == 0) 
	{ 
		b = new box(0,n); 
	} 
	else if(strncmp(chunk,"OBJC",4) == 0)
	{
		b = new obj(0,n);
	}
	else if(strncmp(chunk,"REFO",4) == 0)
	{
		b = new ref(0,n);
	}
	else if(strncmp(chunk,"REFP",4) == 0)
	{
		b = new refptr(0,n);
	}
	else if(strncmp(chunk,"BLOB",4) == 0)
	{
		b = new blob(0,n);
	}
	else if(strncmp(chunk,"CSG ",4) == 0)
	{
		b = new csg(0,n);
	}
	else if(strncmp(chunk,"SPHR",4) == 0)
	{
		b = new sphere(0,n);
	}
	else if(strncmp(chunk,"CYLD",4) == 0)
	{
		b = new cylinder(0,n);
	}
	else if(strncmp(chunk,"TRUS",4) == 0)
	{
		b = new torus(0,n);
	}
	else if(strncmp(chunk,"CONE",4) == 0)
	{
		b = new cone(0,n);
	}
	else if(strncmp(chunk,"PLNE",4) == 0) 
	{ 
		b = new plane(0,n); 
	} 
	else if(strncmp(chunk,"TRIA",4) == 0) 
	{ 
		b = new triangle(0,n); 
	} 
	else if(strncmp(chunk,"TEXT",4) == 0)
	{
		b = new text(0,n);
	}
/*	else if(strncmp(chunk,"SCPT",4) == 0)
	{
		b = new script(0,n);
	}
*/
	else if(strncmp(chunk,"HFLD",4) == 0) 
	{ 
		b = new heightfield(0,n); 
	} 
	else if(strncmp(chunk,"SMSH",4) == 0)
	{
		b = new smesh(0,n);
	}
	else if(strncmp(chunk,"CMSH",4) == 0)
	{
		b = new cmesh(0,n);
	}
/*
	else if(strncmp(chunk,"CUBI",4) == 0) 
	{ 
		b = new cubic(0,n); 
	} 
	else if(strncmp(chunk,"DISC",4) == 0) 
	{ 
		b = new disc(0,n); 
	} 
	else if(strncmp(chunk,"LATH",4) == 0) 
	{ 
		b = new lathe(0,n); 
	} 
	else if(strncmp(chunk,"MESH",4) == 0)
	{
		b = new mesh(0,n);
	}
	else if(strncmp(chunk,"POL ",4) == 0)
	{ 
		b = new poly(0,n); 
	} 
	else if(strncmp(chunk,"PRIS",4) == 0) 
	{ 
		b = new prism(0,n); 
	} 
	else if(strncmp(chunk,"QUAD",4) == 0) 
	{ 
		b = new quadric(0,n); 
	} 
	else if(strncmp(chunk,"QART",4) == 0) 
	{ 
		b = new quartic(0,n); 
	} 
	else if(strncmp(chunk,"SELL",4) == 0) 
	{ 
		b = new superellipsoid(0,n); 
	} 
*/
	else if(strncmp(chunk,"LPNT",4) == 0) 
	{ 
		b = new lightpoint(0,n); 
	} 
	else if(strncmp(chunk,"SPTL",4) == 0) 
	{ 
		b = new spotlight(0,n); 
	} 
/*
*/
	else if(strncmp(chunk,"CMRA",4) == 0)
	{
		b = new camera(0,n,0);
	}
	else if(strncmp(chunk,"EAR ",4) == 0)
	{
		b = new ear(0,n,0);
	}
	// ...
	else m->seek(l,SEEK_CUR);

	if(b)
	{
		b->setMedia(m);
		b->setParent(this);
		if(b->load(m,l - (m->tell() - pos),ver) != 0)
		{
			delete b;
			m->seek(pos + l,SEEK_SET);
			return 0;
		}
	}

	if(n)
		delete n;
		
	m->seek(pos + l,SEEK_SET);

	return b;
}

int	base::save(media *m,int ver)
{
	switch(ver)
	{
		case 0:
		case -1:
		break;
		default:
			return -1;
	}

	return 0;
}

int	base::load(media *m,int l,int ver)
{
	int	pos = m->tell();
	base	*b;

	switch(ver)
	{
		case 0:
		case -1:
			//printf("load len %i\n",l);
			while(m->tell() < pos + l)
			{	
				//printf("  go to parse\n");
				b = parse(m,ver);
				if(b)
				{
					//printf("  go to bind\n");
					b->addToParent(this);
				}
			}
		break;
		default:
			return -1;
	}

	return 0;
}

int	base::save3DDS(const char *fn,base *b,int ver)
{
	FILE	*fp;
	int	error;
	media	*m;
        chunk	c;

	if(!fn) return -3;
	if((fp = fopen(fn,"wb")) == 0) return -4;

	m = new media(fp);
	setMedia(m);

	writeChunk("FORM");
	m->write("3DDS",4);

	c.setMedia(m);
	c.writeChunk("VERS");
	c.writeInt(ver = 0);
	c.writeChunkLen();
	
	error = b->save(m,ver);

	writeChunkLen();

	delete m;

	fclose(fp);

	return error;
}

int	base::save3DDS(media *m,base *b,int ver)
{
	int	error;
        chunk	c;

	setMedia(m);

	writeChunk("FORM");
	m->write("3DDS",4);

	c.setMedia(m);
	c.writeChunk("VERS");
	c.writeInt(ver = 0);
	c.writeChunkLen();
	
	error = b->save(m,ver);

	writeChunkLen();

	return error;
}

int	base::load3DDS(const char *fn,base **b)
{
	FILE	*fp;
	int	len,ver = -1;
	char	buffer[12];
	media	*m;
        chunk	c;

	*b = 0;
	if(!fn) return -3;
	if((fp = fopen(fn,"rb")) == 0) return -4;

	m = new media(fp);
	setMedia(m);

	m->read(buffer,12);

	if(strncmp(buffer,"FORM",4) != 0) return -5;
	len = *(int*)(buffer + 4);
	if(strncmp(buffer + 8,"3DDS",4) != 0) return -6;

	c.setMedia(m);
	c.readChunk(buffer,len);
	if(strncmp(buffer,"VERS",4) == 0)
	{
		ver = c.readInt();
	}
	else
	{
		c.rejectChunk();
		ver = 0;
	}
	
	*b = parse(m,ver);

	delete m;

	fclose(fp);

	return (*b == 0 ? -7:0);
}

int	base::load3DDS(media *m,base **b)
{
	int	len,ver = -1;
	char	buffer[12];
        chunk	c;

	*b = 0;
	setMedia(m);

	m->read(buffer,12);

	if(strncmp(buffer,"FORM",4) != 0) return -5;
	len = *(int*)(buffer + 4);
	if(strncmp(buffer + 8,"3DDS",4) != 0) return -6;

	c.setMedia(m);
	c.readChunk(buffer,len);
	if(strncmp(buffer,"VERS",4) == 0)
	{
		ver = c.readInt();
	}
	else
	{
		c.rejectChunk();
		ver = 0;
	}
	
	*b = parse(m,ver);

	return (*b == 0 ? -7:0);
}

int	base::loadFlags(media *m,int l) 
{
	int	pos = m->tell();

	if(m->tell() < pos + l) 
	{
		m->read(&flags,sizeof(int));	
	} 
 
	return 0; 
} 

int	base::saveFlags(media *m) 
{
	m->write(&flags,sizeof(int));	

	return 0; 
} 

void	base::setParent(base *b)
{
	parent = b;
}

int	base::asetFlag(param**,list<param*> *pl)
{
	char		*v;

	if(pl->isEmpty() || pl->at(0)->type() != param::PM_TEXT) return -1;

	v = ((pmtext*)pl->at(0))->getText();

	if(strncmp(v,"HIDE",4) == 0)
	{
		setFlag(NO_EXPORT,1);
	}

	return 0;
}

int	base::aclearFlag(param**,list<param*> *pl)
{
	char		*v;

	if(pl->isEmpty() || pl->at(0)->type() != param::PM_TEXT) return -1;

	v = ((pmtext*)pl->at(0))->getText();

	if(strncmp(v,"HIDE",4) == 0)
	{
		setFlag(NO_EXPORT,0);
	}

	return 0;
}

int	base::agetChannelNumber(param **p,list<param*> *pl)
{
	char		*n = 0;
	base		*bp;
	ear		*ep;
	int		c;
	
	if(pl->isEmpty() || (pl->at(0)->type() != param::PM_TEXT
		&& pl->at(0)->type() != param::PM_OBJECT)) return -1;

	if(pl->at(0)->type() == param::PM_TEXT)
		n = ((pmtext*)pl->at(0))->getText();
	else if(pl->at(0)->type() == param::PM_OBJECT)
		n = ((pmobject*)pl->at(0))->getName();
	else return -1;

	*p = new pmint(0);
	
	c = -1;
        bp = getRoot();
        if(bp)
        {
        	bp = bp->searchName(n);
        	if(bp)
        	{
#if (USE_RTTI == 1)        		
        		ep = dynamic_cast<ear*>(bp);
#else
			if(bp->getType() == NUM_EAR)
				ep = (ear*)bp;
			else
				ep = 0;
#endif
        		if(ep)
        		{
        			c = ep->getChannel();		
        		}
        	}
        }
 	((pmint*)*p)->setInt(c);

	return 0;
}

void	base::setFlag(FLAGS f,int set)
{
	if(set)
		flags |= (1 << f);
	else
		flags &= ~(1 << f);
}

int	base::isFlag(FLAGS f)
{
	return !(!(flags & (1 << f)));
}

int	base::isFinite()
{
	return !0;
}

Matrix44	base::getMatrix(int anim)
{
	Matrix44	m;

	m.unify();

	if(parent)
	{
		m *= parent->getMatrix(anim);
	}
	
	return m;
} 
 
int	base::lower(base *child)
{
	return !0;
}

int	base::upper(base *child)
{
	return !0;
}



/*
void	*base::operator new(size_t s)
{
	void	*vp;
	
	vp = malloc(s);
	
	printf("called new for size %i = %p\n",s,vp);
	
	return vp;
}

void	base::operator delete(void *vp,size_t s)
{
	printf("called delete for size %i = %p  %s\n",s,vp,((base*)vp)->name);
	if(((base*)vp)->name)
	{
		free(((base*)vp)->name);
		((base*)vp)->name = 0;
	}
	//::delete vp;
	free(vp);
}
 */



