/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** spotlight.cpp
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "spotlight.h"
#include "num.h"
#include "view.h"

spotlight::spotlight(base *p,char *n) :
	lightobj(p,n)
{
	numtype = NUM_SPOTLIGHT;
	
	falloff = 10 * PI / 180;
	tightness = 10;
	radius = 15 * PI / 180;
	
	if(parent) addToParent(parent);
}

spotlight::spotlight(base *p,char *n,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	lightobj(p,n,vs,vr,vt)
{
	numtype = NUM_SPOTLIGHT;
	
	falloff = 10 * PI / 180;
	tightness = 10;
	radius = 15 * PI / 180;
		
	if(parent) addToParent(parent);
}

spotlight::spotlight(base *p,spotlight *sc) :
	lightobj(p,(lightobj*)sc)
{
	numtype = NUM_SPOTLIGHT;
	
	falloff = sc->falloff;
	tightness = sc->tightness;
	radius = sc->radius;

	if(parent) addToParent(parent);
}

spotlight::~spotlight()
{
	// workaround
	removeFromParent();
}

base	*spotlight::copy(base *p)
{
	return new spotlight(p,this);
}

void	spotlight::setTightness(double t)
{
	tightness = t;
}

void	spotlight::setRadius(double r)
{
	radius = r;
}

void	spotlight::setFalloff(double f)
{
	falloff = f;
}

double	spotlight::getTightness()
{
	return tightness;
}

double	spotlight::getRadius()
{
	return radius;
}

double	spotlight::getFalloff()
{
	return falloff;
}

int	spotlight::addToParent(base *p)
{
	if(!p) return -2;

	parent = p;
	return p->addChild(this);
}

int 	spotlight::removeFromParent()
{
	if (!parent) return -2;
	return parent->removeChild(this);
}
 
int     spotlight::exportPOV(FILE *fp,int tab,int tabsize,int anim)
{ 
	if(isFlag(NO_EXPORT)) return 0;

	printTab(fp,tab);
	fprintf(fp,"// Objectname = %s\n",name);
	printTab(fp,tab);
	fprintf(fp,"// Objecttype = light_point\n");

	printTab(fp,tab);
	fprintf(fp,"light_source\n");
	printTab(fp,tab);
	fprintf(fp,"{\n");
	printTab(fp,tab + tabsize);
	fprintf(fp,"<0, 0, 0>\n");
	printTab(fp,tab + tabsize);
	fprintf(fp,"color rgb<%g, %g, %g> \n",red(),green(),blue());
	printTab(fp,tab+tabsize);
	fprintf(fp,"spotlight\n");
	printTab(fp,tab+tabsize);
	fprintf(fp,"radius %g\n",getRadius() * 180 / PI);
	printTab(fp,tab+tabsize);
	fprintf(fp,"falloff %g\n",getFalloff() * 180 / PI);
	printTab(fp,tab+tabsize);
	fprintf(fp,"tightness %g\n",getTightness());
	printTab(fp,tab + tabsize);
	fprintf(fp,"point_at <%g, %g, %g> \n",0.0,0.0,1.0);
	printTab(fp,tab+tabsize);
	fprintf(fp,"fade_distance %g \n",fadeDistance());
	printTab(fp,tab + tabsize);
	fprintf(fp,"fade_power %g \n",fadePower());
	printTab(fp,tab + tabsize);
	fprintf(fp,"media_attenuation %s\n",(mediaAttenuation() ? "on" : "off"));
	printTab(fp,tab + tabsize);
	fprintf(fp,"media_interaction %s\n",(mediaInteraction() ? "on" : "off"));
	
	transform::exportPOV(fp,tab + tabsize,tabsize,anim);
	
	printTab(fp,tab);
	fprintf(fp,"}\n\n");

	return 0;
}

void	spotlight::dumpNames(int tab,int)
{
	printTab(stderr,tab);
	printf("spotlight: %s\n",name);
}


int	spotlight::save(media *m,int ver)
{
	if(!m) return -1;

	switch(ver)
	{
		case 0:
		case -1:
			setMedia(m);
			writeChunk("SPTL");
			writeNameChunk(name);

			saveFlags(m);

			writeDouble(red());
			writeDouble(green());
			writeDouble(blue());
			writeDouble(fadeDistance());
			writeDouble(fadePower());
			writeBool(mediaAttenuation());
			writeBool(mediaInteraction());
 			writeDouble(getRadius());
			writeDouble(getFalloff());
			writeDouble(getTightness());

			anim::save(m);
			transform::save(m,ver);

			writeChunkLen();
		break;
		default:
			return -2;
	}

	return 0;
}

int	spotlight::load(media *m,int l,int ver)
{
	if(!m) return -1;

	pos = m->tell();
	
	switch(ver)
	{
		case 0:
		case -1:
			loadFlags(m,l);

			r = readDouble();
			g = readDouble();
			b = readDouble();
			fade_distance = readDouble();
			fade_power = readDouble();
			media_attenuation = readBool();
			media_interaction = readBool();
			radius = readDouble();
			falloff = readDouble();
			tightness = readDouble();

			anim::load(m,l - (m->tell() - pos));
			transform::load(m,l - (m->tell() - pos),ver);
 		break;
		default:
			return -2;
	}

	return 0;
}

int	spotlight::draw(view *v,Matrix44 m,int anim)
{
	Vector3	p,vx,vy,vz;

	if(isFlag(HIDE))
		return 0;

	p = Vector3(0,0,0);
	vx = Vector3(0.5,0,0);
	vy = Vector3(0,0.5,0);
	vz = Vector3(0,0,0.5);
	
	transformMatrix(m,anim);

	if(this == v->getSelected())
		v->setDrawSelected(1);

	v->drawCross(p,m,anim);
	v->drawSymbol(p,"spotlight",m,anim);
	
	v->drawLine(p,vz * 3,m,anim);
	v->drawCone(p + vz,vx * tan(getRadius()),vy * tan(getRadius()),vz,0,m,anim);
	v->drawCone(p + vz,vx * tan(getFalloff()),vy * tan(getFalloff()),vz,0,m,anim);

	if(this == v->getSelected())
		v->setDrawSelected(0);

	return 0;
}

int	spotlight::calculate(int)
{
	setMin(Vector3(0,0,0));
	setMax(Vector3(0,0,0));

	return 0;
}

int	spotlight::isFinite()
{
	return 0;
}




