/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** interior.cpp
*/
#include "interior.h"


#include <misc.h>

interior::interior(textureBase *p):textureBase(INTERIOR,p)
{
	setName("Interior");
	setInfo("Interior");
	setSubItems(true);

	absorption=new color(this);
	absorption->setRed(0);	
	absorption->setGreen(0);
	absorption->setBlue(0);

	emission=new color(this);
	emission->setRed(1);	
	emission->setGreen(1);
	emission->setBlue(1);

	scatter=new color(this);
	scatter->setRed(0);	
	scatter->setGreen(0);
	scatter->setBlue(0);
	scattertype=1;
	scatterecc=0;
	scatterext=1;	
	samplemin=1;
	samplemax=1;
	intervals=10;
	ratio=0.9;
        confidence=0.9;
	variance=1.0;  //   /128tel !!!!!

	addAddableItem(INTERIOR,false);
	addAddableItem(DENSITY,true);	

}

void	interior::setActive(bool a,textureBase *tb)
{
	if (tb==NULL) {
		textureBase::setActive(a,this);
		return;
	}
        if (tb->getType()!=DENSITY) return;
	
	tb->setActive(a,NULL);
}

void interior::addElement(textureBase* what,textureBase* where)
{
	if (what==NULL) return;
	if (canAdd(what)==false) return;

	what->setParent(this);
	what->setActive(true,NULL);

	switch(what->getType()) {
	default:
		textureBase::addElement(what,where);
	break;
	}
}



bool interior::isEqual(textureBase *tb)
{
//interior *i=(interior *)tb;
textureBase *tb1;

#ifdef DEBUG
	printf("Comparing Interior..\n");
#endif

	if (!tb) return false;
	if (textureBase::isEqual(tb)==false) return false;

	tb1=tb->getFirstElement();

	if (
		//(absorption->isEqual(i->getAbsorption() )==true) &&
		//(emission->isEqual(i->getEmission() )==true) &&
		//(scatter->isEqual(i->getScatter() )==true) &&
		//(scattertype==i->getScatterType() ) &&
		//(scatterecc==i->getScatterECC() ) &&
		//(scatterext==i->getScatterEXT() )&&
		//(samplemin==i->getSampleMin() )&&
		//(samplemax==i->getSampleMax() )&&
		//(intervals==i->getIntervals() )&&
		//(ratio==i->getRatio() )&&
		//(confidence==i->getConfidence() )&&
		//(variance==i->getVariance() ) &&
		(textureBase::isEqual(tb)==true)

	) {

#ifdef DEBUG
		printf("Compared Interior..found it !\n");
#endif
		return true;
	}

	return false;
}


void interior::add(textureBase *tb)
{
interior *i=(interior*)tb;
//Transformations *tr=(Transformations *)tb;
//Turbulence *r=(Turbulence *)tb;
//Density *d=(Density *)tb;

	if (tb->getType()==INTERIOR) {
		absorption->add(i->getAbsorption() );
		emission->add(i->getEmission() );
		scatter->add(i->getScatter() );
		scattertype=i->getScatterType();
		scatterecc=i->getScatterECC();
		scatterext=i->getScatterEXT();
		samplemin=i->getSampleMin();
		samplemax=i->getSampleMax();
		intervals=i->getIntervals();
		ratio=i->getRatio();
		confidence=i->getConfidence();
		variance=i->getVariance();
		return;
	}
	addElement(tb,NULL);
}

color *interior::getScatter()
{
	return scatter;
}



void interior::setScatter(color *c)
{
	if(scatter) delete scatter;
	scatter=c;
}

int interior::getScatterType()
{
	return scattertype;
}

void interior::setRatio(double r)
{
	ratio=r;
}

double interior::getRatio()
{
	return ratio;
}

void interior::setConfidence(double r)
{
	confidence=r;
}

double interior::getConfidence()
{
	return confidence;
}

void interior::setVariance(double r)
{
	variance=r;
}

double interior::getVariance()
{
	return variance;
}


void interior::setIntervals(int s)
{
	intervals=s;
}

int interior::getIntervals()
{
	return intervals;
}

void interior::setScatterType(int s)
{
	scattertype=s;
}

double interior::getScatterECC()
{
	return scatterecc;
}

void interior::setSampleMax(double s)
{
	samplemax=s;
}

double interior::getSampleMax()
{
	return samplemax;
}

void interior::setSampleMin(double s)
{
	samplemin=s;
}

double interior::getSampleMin()
{
	return samplemin;
}

void interior::setScatterECC(double s)
{
	scatterecc=s;
}

double interior::getScatterEXT()
{
	return scatterext;
}

void interior::setScatterEXT(double s)
{
	scatterext=s;
}


void interior::setAbsorption(color *c)
{
	if (absorption) delete absorption;
	absorption=c;
}

color *interior::getAbsorption()
{
	return absorption;
}

void interior::setEmission(color *c)
{
	if (emission) delete emission;
	emission=c;
}

color *interior::getEmission()
{
	return emission;
}

interior::~interior()
{
}

int	interior::load(media *m,int l)
{
	double x;
	char	chunk[4];
	bool 	read;
	int	len,pos = m->tell();
	if(!m)	return -1;

	setMedia(m);

#ifdef DEBUG
	printf("loading interior\n");
#endif
	do {
		m->read(chunk,4);
		m->read(&len,4);
		read=false;
		if(strncmp(chunk,"IND1",4) == 0) {
			m->read(&x,sizeof(double)) ;
			absorption->setRed(x);
			m->read(&x,sizeof(double)) ;
			absorption->setGreen(x);	
			m->read(&x,sizeof(double)) ;
			absorption->setBlue(x);

			m->read(&x,sizeof(double)) ;
			emission->setRed(x);
			m->read(&x,sizeof(double)) ;
			emission->setGreen(x);	
			m->read(&x,sizeof(double)) ;
			emission->setBlue(x);

			m->read(&x,sizeof(double)) ;
			scatter->setRed(x);
			m->read(&x,sizeof(double)) ;
			scatter->setGreen(x);	
			m->read(&x,sizeof(double)) ;
			scatter->setBlue(x);
			m->read(&scattertype,sizeof(int) );
			m->read(&scatterecc,sizeof(double) );
			m->read(&scatterext,sizeof(double) );
			read=true;
		}
		if(strncmp(chunk,"IND2",4) == 0) {
			m->read(&intervals,sizeof(int) );
			m->read(&samplemin,sizeof(double));
			m->read(&samplemax,sizeof(double) );
			m->read(&ratio,sizeof(double) );
			m->read(&confidence,sizeof(double) );
			m->read(&variance,sizeof(double) );
			read=true;
		}
		if(strncmp(chunk,"BASE",4) == 0) {
			textureBase::load(m,len);
			read=true;
		}
		if(strncmp(chunk,"TRNS",4) == 0) {
			Transformations *tr=new Transformations(this);
			tr->load(m,len);
			elements.append(tr);
			read=true;			
		}
		if(strncmp(chunk,"TURB",4) == 0) {
			Turbulence *turb=new Turbulence(this);
			turb->load(m,len);
			elements.append(turb);
			read=true;
		}
		if(strncmp(chunk,"DENS",4) == 0) {
			Density *d=new Density(this);
			d->load(m,len);
			elements.append(d);
			read=true;
		}
		if (read==false) { m->seek(len,SEEK_CUR); }

	} while (m->tell()< (pos+l) );

#ifdef DEBUG
	printf("loaded interior\n");
#endif
	return 0;
}

int	interior::save(media *m)
{
double x;
int i;
chunk ch;
	if(!m) return -1;

	setMedia(m);
	ch.setMedia(m);
#ifdef DEBUG
	printf("saving interior\n");
#endif
	writeChunk("INTE");

	textureBase::save(m);

	ch.writeChunk("IND1");
		x=absorption->red();
		m->write(&x,sizeof(double)) ;
		x=absorption->green();
		m->write(&x,sizeof(double)) ;
		x=absorption->blue();
		m->write(&x,sizeof(double)) ;

		x=emission->red();
		m->write(&x,sizeof(double)) ;
		x=emission->green();
		m->write(&x,sizeof(double)) ;
		x=emission->blue();
		m->write(&x,sizeof(double)) ;

		x=scatter->red();
		m->write(&x,sizeof(double)) ;
		x=scatter->green();
		m->write(&x,sizeof(double)) ;
		x=scatter->blue();
		m->write(&x,sizeof(double)) ;

		m->write(&scattertype,sizeof(int) );
		m->write(&scatterecc,sizeof(double) );
		m->write(&scatterext,sizeof(double) );
	ch.writeChunkLen();

	ch.writeChunk("IND2");

		m->write(&intervals,sizeof(int) );
		m->write(&samplemin,sizeof(double) );
		m->write(&samplemax,sizeof(double) );

		m->write(&ratio,sizeof(double) );
		m->write(&confidence,sizeof(double) );
		m->write(&variance,sizeof(double) );
	ch.writeChunkLen();	

	for (i=0;i<elements.length();i++) {
		elements[i]->save(m);
	}


	writeChunkLen();
#ifdef DEBUG
	printf("saved interior\n");
#endif
	return 0;
}

bool	interior::isExportable()
{
	if ((((absorption->red()==0.0)&&(absorption->green()==0.0)&&(absorption->blue()==0.0)))
	&& (((emission->red()==0.0)&&(emission->green()==0.0)&&(emission->blue()==0.0)))
	&& (((scatter->red()==0.0)&&(scatter->green()==0.0)&&(scatter->blue()==0.0)))
	&& ((samplemax==1)&&(samplemin==1))
	){
		return false;
	}
	return true;
}


int	interior::exportPOV(FILE *fp,int tab,int tabsize,int a,bool dummy)
{
int i;

	if(fp == 0) return -1;
	//if (isExportable()==false) return 0;

#ifdef DEBUG
	printf("exporting interior (POV)\n");
#endif
	printTab(fp,tab);
	fprintf(fp,"interior{ \n");
	printTab(fp,tab+tabsize);
	fprintf(fp,"media {\n");

	if (!((absorption->red()==0.0)&&(absorption->green()==0.0)&&(absorption->blue()==0.0))){
		printTab(fp,tab+2*tabsize);
		fprintf(fp,"absorption rgb<%g,%g,%g>\n",absorption->red(),absorption->green(),absorption->blue());
	}
	if (!((emission->red()==0.0)&&(emission->green()==0.0)&&(emission->blue()==0.0))){
		printTab(fp,tab+2*tabsize);
		fprintf(fp,"emission rgb<%g,%g,%g>\n",emission->red(),emission->green(),emission->blue());
	}
	if (!((scatter->red()==0.0)&&(scatter->green()==0.0)&&(scatter->blue()==0.0))){
		printTab(fp,tab+2*tabsize);
		fprintf(fp,"scattering {\n");
		printTab(fp,tab+3*tabsize);
		fprintf(fp,"%d, rgb<%g,%g,%g>\n",scattertype,scatter->red(),scatter->green(),scatter->blue());
		printTab(fp,tab+3*tabsize);
		fprintf(fp,"%g %g\n",scatterecc,scatterext);
		printTab(fp,tab+2*tabsize);
		fprintf(fp,"}\n");
	}
       if (intervals!=10) {
		printTab(fp,tab+2*tabsize);
		fprintf(fp,"intervals %d \n",intervals);
	}
       	if (ratio!=0.9) {
		printTab(fp,tab+2*tabsize);
		fprintf(fp,"ratio %g \n",ratio);
	}
       	if (confidence!=0.9) {
		printTab(fp,tab+2*tabsize);
		fprintf(fp,"confidence %g \n",confidence);
	}
       	if (variance!=1.0) {
		printTab(fp,tab+2*tabsize);
		fprintf(fp,"variance %g \n",variance);
	}
        if (!( (samplemax==1)&&(samplemin==1) )) {
		printTab(fp,tab+2*tabsize);
		fprintf(fp,"samples %g, %g\n",samplemin,samplemax);
	}
	
	for (i=0;i<elements.length();i++) {
		if ((elements[i]->active()==true)&&(elements[i]->getType()==DENSITY)) {
		elements[i]->exportPOV(fp,tab+tabsize,tabsize,a,false);
		}
	}

	textureBase::exportPOV(fp,tab,tabsize,a,dummy);

	printTab(fp,tab+tabsize); // KLammer fr Media
	fprintf(fp,"}\n");
	printTab(fp,tab);
	fprintf(fp," }\n"); // Klammer zu fr interior
#ifdef DEBUG
	printf("exported interior (POV)\n");
#endif
	return 0;
}

