/*
 *   kwrl - a little VRML 2.0 viewer
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifndef SFIMAGE
#define SFIMAGE

// label
#define SFImageLabel "SFImage"

// classes
#include <SFNode.h>
#include <SFInt32.h>

// class definition
class SFImage : public SFNode{

 public:

  // constructor
  SFImage() : SFNode(SFImageLabel) {
    data           = (unsigned char *) 0;
    width          = 0;
    height         = 0;
    numBands       = 0;
    blendingNeeded = false;
  }
  SFImage(int w, int h) : SFNode(SFImageLabel) {
    data           = new unsigned char[w * h * 4];
    width          = w;
    height         = h;
    numBands       = 4;
    blendingNeeded = false;
  }
  
  // destructor
  ~SFImage() {
    if (data != (unsigned char *) 0) delete data;
    data       = (unsigned char *) 0;
    width      = 0;
    height     = 0;
    numBands   = 0;
  }

  // parse
  void parse (char *, istream      &);

  // prepare
  void prepare (SFVec3f      &, SFVec3f      &);

  // render
  void render(SFRenderInfo &);

  // operators
  SFImage &operator = (SFImage &S2) {
    width           = S2.width;
    height          = S2.height;
    numBands        = S2.numBands;
    alloc(S2.width, S2.height);
    if (S2.width * S2.height * S2.numBands > 0) {
      memcpy(data, S2.data, sizeof(S2.width * S2.height * S2.numBands));
    }
    isValid()       = S2.isValid();
    return(*this);
  }


  // load image from file
  void LoadImage(char *FN);

  // create an image
  void alloc(int nw, int nh) {
    if (data != (unsigned char *) 0) delete data;
    data           = new unsigned char[nw * nh * 4];
    if (data == (unsigned char *) 0) { 
      cerr << "\nError:\n";
      cerr << "\tOccurred in SFImage::alloc(" << nw << ", " << nh << ")\n";
      cerr << "\tCould not allocate memory\n";
      exit(0);
    }
    width          = nw;
    height         = nh;
    numBands       = 4;
    isValid() = true;
  }

  // access
  unsigned char & operator () (int i, int j, int k) {
    if ((i < 0) || (i >= width) ||
	(j < 0) || (j >= height) ||
	(data == (unsigned char *) 0)) {
      cerr << "\nError:\n";
      cerr << "\tOccurred in SFImage::operator()\n";
      cerr << "\tOut of array bounds access (" << i << ", ";
      cerr << j << ", " << k << ")\n";
      cerr << "\tDims (" << width << ", ";
      cerr << height << ", " << numBands << ")\n";
      exit(0);
    }
    return(data[j * width * 4 + i * 4 + k]);
  } 
  unsigned char & pixel(int i, int j, int k) {
    if ((i < 0) || (i >= width) ||
	(j < 0) || (j >= height) ||
	(data == (unsigned char *) 0)) {
      cerr << "\nError:\n";
      cerr << "\tOccurred in SFImage::pixel()\n";
      cerr << "\tOut of array bounds access (" << i << ", ";
      cerr << j << ", " << k << ")\n";
      exit(0);
    }
    return(data[j * width * 4 + i * 4 + k]);
  } 
  SFInt32 w() {
    return(width);
  }
  SFInt32 h() {
    return(height);
  }
  SFInt32 n() {
    return(numBands);
  }

 protected:
  
  /* load an image in a specific format */
  void LoadJPG(char *FN);
  void LoadGIF(char *FN);
  void LoadPGN(char *FN);

  /* resizes non pow two images */
  void Resize();

 private:

  // the data
  unsigned char *data;
  SFInt32        width;
  SFInt32        height;
  SFInt32        numBands;

  // whether not blending is needed
  bool           blendingNeeded;
  
};

#endif // SFImage
