// KreateCD - CD recording software for the K desktop environment
//
// 1999-2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General      
// Public License.  See the file COPYING in the main directory of the       
// KreateCD distribution for more details.                                     

#include "WriteDialog.h"
#include "CDTrack.h"

#include <kapp.h>
#include <klocale.h>
#include <kconfig.h>

#include <qwindowdefs.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include <qmessagebox.h>

#include <strings.h>
#include <stdio.h>
#include <stdlib.h>


extern KLocale *locale;

static int speedtab[]={1,2,4,6,8};

WriteDialog::WriteDialog( QWidget *parent, const char *name,int ntracks,CDTrack **tracks):QDialog(parent,name,TRUE)
 {
  QVBoxLayout *qvlayout;
  QHBoxLayout *endbuttonlay;
  QPushButton *writebutton;
  QPushButton *exitbutton;
  QCheckBox   *simulation;
  QCheckBox   *ejection;
  QButtonGroup*writespeed;
  QButtonGroup*fixating;
  QHBoxLayout *writelayout;
  QVBoxLayout *fixatinglayout;
  QRadioButton *writespeeds[16];
  QRadioButton *fixatings[4];

  int i;

  numtracks=ntracks;
  cdtracks=tracks;

  // set defaults - do prefs later

  Burnspeed=4;
  XFixating=CDWriter::Fix_CD;
  Simmode=0;
  XEject=0;

  qvlayout=new QVBoxLayout(this);
  
  this->setCaption(locale->translate("Write CD"));

  endbuttonlay=new QHBoxLayout();

  qvlayout->addStretch(1);
  simulation=new QCheckBox(locale->translate("Simulation Mode"),this);
  simulation->setFixedSize(simulation->sizeHint());
  qvlayout->addWidget(simulation,0);
  qvlayout->addStretch(1);

  qvlayout->addStretch(1);
  ejection=new QCheckBox(locale->translate("Eject after write"),this);
  ejection->setFixedSize(ejection->sizeHint());
  qvlayout->addWidget(ejection,0);
  qvlayout->addStretch(1);

  writespeed=new QButtonGroup(locale->translate("Write speed"),this);
  writespeed->setAlignment(AlignHCenter);
  writespeed->setMinimumSize(300,60);
  writelayout=new QHBoxLayout(writespeed,20,0);
  qvlayout->addWidget(writespeed,0);
  qvlayout->addStretch(1);

  for (i=0;i<5;++i)
   {
    char tempo[8];
    sprintf(tempo,"%d x",speedtab[i]);
    writespeeds[i]=new QRadioButton(tempo,writespeed);
    writespeeds[i]->setMinimumSize(writespeeds[i]->sizeHint());
    writelayout->addWidget(writespeeds[i]);
   }

  fixating=new QButtonGroup(locale->translate("Fixating"),this);
  fixating->setAlignment(AlignHCenter);
  fixating->setMinimumSize(300,60);
  fixatinglayout=new QVBoxLayout(fixating,20,0);
  qvlayout->addWidget(fixating,0);
  qvlayout->addStretch(1);

  fixatings[0]=new QRadioButton(locale->translate("Don't fixate"),fixating);
  fixatings[0]->setFixedSize(fixatings[0]->sizeHint());
  fixatinglayout->addWidget(fixatings[0]);

  fixatings[1]=new QRadioButton(locale->translate("Fixate session"),fixating);
  fixatings[1]->setFixedSize(fixatings[1]->sizeHint());
  fixatinglayout->addWidget(fixatings[1]);

  fixatings[2]=new QRadioButton(locale->translate("Fixate CD-R"),fixating);
  fixatings[2]->setFixedSize(fixatings[2]->sizeHint());
  fixatinglayout->addWidget(fixatings[2]);

  qvlayout->addLayout(endbuttonlay,0);
  qvlayout->addStretch(1);

  endbuttonlay->addStretch(1);
  writebutton=new QPushButton(locale->translate("Write CD!"),this);
  writebutton->setFixedSize(writebutton->sizeHint());
  endbuttonlay->addWidget(writebutton);
  endbuttonlay->addStretch(1);
  exitbutton=new QPushButton(locale->translate("Exit"),this);
  exitbutton->setFixedSize(writebutton->sizeHint());
  endbuttonlay->addWidget(exitbutton,0);
  endbuttonlay->addStretch(1);

  connect(writebutton,SIGNAL(clicked()),this,SLOT(writeCD()));
  connect(exitbutton,SIGNAL(clicked()),this,SLOT(exitDialog()));
  connect(simulation,SIGNAL(toggled(bool)),this,SLOT(changeSimulation(bool)));
  connect(ejection,SIGNAL(toggled(bool)),this,SLOT(changeEject(bool)));
  connect(fixating,SIGNAL(clicked(int)),this,SLOT(changeFixating(int)));
  connect(writespeed,SIGNAL(clicked(int)),this,SLOT(changeSpeed(int)));


  switch (XFixating)
   {
    case CDWriter::Fix_None:
     fixating->setButton(0);
     break;
    case CDWriter::Fix_Session:
     fixating->setButton(1);
     break;
    case CDWriter::Fix_CD:
     fixating->setButton(2);
     break;
   }

   for (i=0;i<5;++i)
    {
     if (speedtab[i]==Burnspeed) writespeed->setButton(i);
    }

  simulation->setChecked(Simmode);


  writelayout->activate();
  fixatinglayout->activate();
  qvlayout->activate();

 }


void WriteDialog::writeCD(void)
 {
  CDWriter *da;
  da=new CDWriter();
  int ret;
  long int blcks;
  if (da==0) return;
  if ((blcks=da->getCDFree(numtracks,cdtracks,XFixating))<-1)
   {
#if QT_VERSION >= 200
    QString buffer;

    buffer=locale->translate("WARNING: Data does not fit on disk:\n%ld blocks missing").arg(-blcks);
    ret=QMessageBox::information(0,QString::null,buffer,
              locale->translate("Continue"),
              locale->translate("Cancel"));
#else
    char buffer[256];

    sprintf(buffer,locale->translate("WARNING: Data does not fit on disk:\n%ld blocks missing"),-blcks);
    ret=QMessageBox::information(0,0,buffer,
              locale->translate("Continue"),
              locale->translate("Cancel"));
#endif
    if (ret!=0)
     {
      free(da);
      return;
     }
   }
  if (blcks==-1) return;
  da->burnCD(numtracks,cdtracks,Simmode,Burnspeed,XFixating,XEject);
  free(da);
 }

void WriteDialog::exitDialog(void)
 {
  this->done(0);
 }

void WriteDialog::changeSpeed(int id)
 {
  Burnspeed=speedtab[id];
 }

void WriteDialog::changeFixating(int id)
 {
  switch (id)
   {
    case 0:
      XFixating=CDWriter::Fix_None;
      break;
    case 1:
      XFixating=CDWriter::Fix_Session;
      break;
    case 2:
      XFixating=CDWriter::Fix_CD;
      break;
   }
 }

void WriteDialog::changeSimulation(bool sim)
 {
  Simmode=sim;
 }

void WriteDialog::changeEject(bool eject)
 {
  XEject=eject;
 }
