	/*

	Copyright (C) 1998 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#ifndef _SYNTHMODULE_H
#define _SYNTHMODULE_H

#include "config.h"
#include "moduleserver.h"
#include "arts.h"
#include "ioevent.h"
#include <string>
#include <vector>

class Synthesizer_impl;
class SynthModule;

class SynthBuffer
{
public:		// yes, bad style, but needs to be fast
	float *data;
	unsigned long position,size;

	/* every time I write data to a buffer, needread is incremented
	   by the amount of data for every module connected to that buffer,
	   since that (target) module has to read that data before I can write.
	   I can only write at max. (size-needread) items to that buffer
	*/
	unsigned long needread;

	SynthBuffer(float initialvalue, unsigned long size);
	void setValue(float value);
	~SynthBuffer();

	static long instances;
	static long allbytes;
};

class SynthConnection		// nearly "equivalent to a port"
{
protected:
	bool freeBufferOnExit;
public:		// still bad style, but still needs to be fast ;)
	SynthBuffer *buffer;		// pointer to ringbuffered data
	SynthModule *module;		// to which module I belong to

	unsigned long position;
	long ID;   // ID from the Arts::PortDesc

	bool isDynamic;

/*** late connection strategy ***/

	bool needConnect;
	long wantedID;
	
/********************************/

	void allocBuffer(float initialvalue, unsigned long size);

/* connection is either incoming */
	SynthModule *sourcemodule;
	SynthConnection *sourceconn;

/* or outgoing */
	SynthModule **destmodule;
	unsigned long destcount;	/* amount of clients */

//-------
	SynthConnection(SynthModule *module, long ID);
	~SynthConnection();

	void connectToSource(SynthConnection *source);
	void disconnectFromSource(SynthConnection *source);

	inline unsigned long SynthConnection::haveIn()
	{
		if(sourcemodule)
			return(buffer->position - position);
		else
			return(buffer->size);// constant value; no read limitation
	}

	inline unsigned long SynthConnection::outRoom()
	{
		long room;

		room = buffer->size - buffer->needread;
		if(room < 0) return 0;
		return (unsigned long)room;
	}

	static long instances;
};

// IDEA: properties could have a flag "writable" to protect non writable
// properties (could do an assertion to check if violated)
class SynthProperty
{
protected:
	// SynthProperty will probably hold other types as well at a later time
	// (floats, envelopes, etc.)
	string _stringValue;

public:
	SynthProperty(const char *value);
	const char *stringValue();
	void setStringValue(const char *newValue);
};

class SynthPort
{
protected:
	long _ID;
	long _wantedID;
	bool _needConnect;
	unsigned long _index;
	SynthProperty *_property;
	SynthModule *_module;
	SynthPort *_remotePort;
	SynthConnection *_connection;

	list<SynthPort *> clients;

public:
	SynthPort(class SynthModule *parent, unsigned long index, long ID);
	~SynthPort();

	bool needConnect();
	void setNeedConnect(bool newNeedConnect, long wantedID = 0);
	long wantedID();

	SynthModule *module();

	// this also will transparently return the "remote" property if the port
	// is connected
	SynthProperty *property();
	void setProperty(SynthProperty *property);

	// this will happen if the "remote" property changes
	void propertyChanged();

	// this will be called if we should notify the ports that are connected
	// with us that our property changed
	void notifyChanges();

	// conventional "audio" connections
	SynthConnection *connection();
	void setConnection(SynthConnection *connection);

	void connectToPort(SynthPort *port);
	void disconnectFromPort(SynthPort *port);

	void addClient(SynthPort *port);
	void removeClient(SynthPort *port);

	long ID();
	// SynthPort *remotePort(); // don't need that right now
};

#define SYNTHMODULE_DEFAULT_SAMPLINGRATE 44100.0

class SynthModule :public ModuleServerModule
{
	string modulename;

protected:
	// internal tricks for faster access to the connections
	float **in,**out;
	bool haveCalculateBlock;

	long restoreID;	// session management

public:
	SynthConnection **inConn, **outConn;
	vector<SynthPort *> inPorts, outPorts;
	unsigned long inConnCount, outConnCount, doPreProduce;

	float samplingRate;
	unsigned long samplingRateULong;

    long NeedCycles,CanPerform,Busy,BusyHit;
		// scheduling information

	bool enabled; // needed to make the modules inactive while they are created
	vector<SynthProperty *> args;
	long ID,executionID,mID,clients;

	Synthesizer_impl *Synthesizer;

	enum ArgType {arg_in,arg_out};

	void setName(string name) {modulename=name;};
	string getName() {return(modulename);};

	void assignInConn(SynthConnection *newInConn);
	void assignOutConn(SynthConnection *newOutConn);
	void prepareExecution();
	long request(long cycles);
	unsigned long calc(unsigned long cycles);
		// this only looks if calculation can be done, and increments positions
		// after calculation, but doesn't calculate something itself

	string getArgument(int nr, ArgType direction);

	const char *getStringProperty(unsigned long index);
	void setStringProperty(unsigned long index, const char *newValue);

	virtual void propertyChanged(unsigned long index);
	virtual void FirstInitialize();
	virtual void Initialize() = 0;
	virtual void DeInitialize();
	virtual void getfds(int &infd, int &outfd);
	virtual ArtsNotifyIO *ioHandler();
	virtual void CallBack();
	virtual void Calculate() = 0;
	virtual void CalculateBlock(unsigned long size);
	virtual unsigned long PreProduce(unsigned long maxsamples);
	virtual string getParams() = 0;

	virtual void ProcessData(string data);

	SynthModule();
	virtual ~SynthModule();

//-- session management

	virtual Arts::StringSeq *saveSessionParameters(list<long>& IDs);
	virtual void restoreSessionParameters(const Arts::StringSeq& params);
	virtual void setRestoreID(long ID);   // use restoreID member to access it

//-- module server support --
	static void *get_MS();
};

#endif
