/*
  Copyright (C) 1999 Rainer Maximini

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package
*/

#include "appwindow.h"

#include <qkeycode.h>
#include <qlabel.h>
#include <kapp.h>
#include <qfile.h>
#include <kfiledialog.h>
#include <kkeydialog.h>
#include <kiconloader.h>
#include <kmsgbox.h>
#include <iostream.h>
#include <fstream.h>
#include <qfile.h>
#include <kurl.h>
#include <qcursor.h>
#include <icons.h>
#include <qfileinfo.h> 
#include "commandLineParameters.h"
#include "songList.h"


ApplicationWindow::ApplicationWindow()
  : KTMainWindow("kjukebox" )
{
  config = (KApplication::getKApplication())->getConfig();

  config->setGroup(KCONFIG_FILE);
  fileDialogPath = config->readEntry(KCONFIG_FILE_LASTDIRECTORY, "./");

  config->setGroup(KCONFIG_GENERAL);

  shufflePlaying = config->readBoolEntry(KCONFIG_GENERAL_SHUFFLE,true);

  playingSongName = new (QString)[3];

  installEventFilter(this);

  /* Initialize the DataBase */
  debugOutput( cout << "Initialize the DataBase" << endl );
  dataBase = new DataBase(NULL, "database");
  connect(dataBase, SIGNAL(genreSetChanged()), this, SLOT(setGenreMenu()));

  /* Main Widget */
  debugOutput( cout << "InitMainWidget " << endl );
  mainWidget = new MainWidget(this);
  mainWidget->setDataBase(dataBase);
  connect(mainWidget, SIGNAL(playSong(Song *)),
	  this, SLOT(playSelectedSong(Song *)));

  setFocusProxy (mainWidget);
  mainWidget->setFocus();
  mainWidget->setFocusPolicy(QWidget::TabFocus);
  this->setFocusPolicy(QWidget::NoFocus);

  //Initialize the song logger;
  debugOutput( cout << "InitSongLogger " << endl );
  songLogger = new SongLogger(this,"Song Logger");

  // Menu Bar
  debugOutput( cout << "InitMenuBar " << endl );
  initMenuBar();

  //Statusleiste
  debugOutput( cout << "InitStatusBar " << endl );
  enableStatusBar();
  status = statusBar();
  status->message( i18n("Ready"), 2000 );
  config->setGroup(KCONFIG_GENERAL);
  if(config->readBoolEntry(KCONFIG_GENERAL_STATUSBAR,true)){
    controls->setItemChecked(sb, true);
    enableStatusBar(KStatusBar::Show);
  }
  else {
    controls->setItemChecked(sb, false);
    enableStatusBar(KStatusBar::Hide);
  }

  //Initialize the Player  
  debugOutput( cout << "InitPlayer " << endl );
  initializePlayer();

  /*
    Initialize the Toolbars, this must be happen after initializePlayer is called.
    Swap the order and you get a Segmentation Fault.
  */
  debugOutput( cout << "InitMainToolbar " << endl);
  initMainToolbar();
  debugOutput( cout << "Creating the Timer Array" << endl );
  timer  = (TimerDevice **) new (TimerDevice*)[2];
  debugOutput( cout << "InitSoundToolBar for Player One" << endl );
  initSoundToolbar(PLAYER1);

  debugOutput( cout << "InitSoundToolBar for Player Two" << endl );
  initSoundToolbar(PLAYER2);
  

  //something more 
  debugOutput( cout << "Something More " << endl );
  settings = new Configuration();
  connect(settings,SIGNAL(audioDeviceChanged()), this, SLOT(setAudioDevice()));
  connect(settings,SIGNAL(configurationChanged()), songLogger, SLOT(reinitialize()));
  printer = new Printer(this, "printer", statusBar());

  refresh();


  setView(mainWidget);
  setMinimumSize( 200, 50 );
  
  resize(config->readNumEntry( KCONFIG_GENERAL_WIDTH,800),
	 config->readNumEntry( KCONFIG_GENERAL_HEIGHT,500));
  dateiName = "";
  setTitle();
  show();

  initAccelKeys();

  initCommandLineParameter();

  // Now load all command line parameters
}

ApplicationWindow::~ApplicationWindow(){
  removeEventFilter(this);

  audioDevice->close();
  mixerDevice->removeListener(audioDevice);

  removePlayer(PLAYER1);
  removePlayer(PLAYER2);
  delete mixerDevice;
  delete audioDevice;
  delete dataBase;
  debugOutput(cout << "Gui removed" << endl);
}

void ApplicationWindow::removePlayer(int player){
  // Before you delete a player remove the listener
  decoder[player]->removeListener(multicastDevice[player]);
  multicastDevice[player]->removeListener(volumeDevice[player]);
  multicastDevice[player]->removeListener(streamTimeInfoDevice[player]);
  multicastDevice[player]->removeListener(streamAlarmInfoDevice[player]);
  multicastDevice[player]->removeListener(mixerDevice);

  timer[player]->detach();
  delete timer[player];
  fadeDevice[player]->detach();
  delete fadeDevice[player];
  jumpDevice[player]->detach();
  delete jumpDevice[player];
  alarmtimer[player]->detach();
  delete alarmtimer[player];
  delete volumeDevice[player];
  delete decoder[player];
  delete multicastDevice[player];
  delete streamAlarmInfoDevice[player];
  delete streamTimeInfoDevice[player];
  debugOutput(cout << "Player " << player << " is removed" << endl);
}


void ApplicationWindow::initCommandLineParameter(){
  const char * fileName;
  QStrList *files;
  QFileInfo fileInfo;
  QString   file;
  files = CommandLineParameters::getFileList();
  for(fileName=files->first(); fileName!=0; fileName=files->next()){
    fileInfo.setFile(fileName);
    if(!fileInfo.exists()) continue;
    file = fileInfo.absFilePath ();
    if(fileInfo.isDir()){
      dataBase->insert(ioaccess.importDirectory(file));
    }
    else {
      if (file.findRev ( ".mp3", -1, false )!= -1)
	dataBase->insert(ioaccess.importMP3File(file));      
      if (file.findRev ( ".m3u", -1, false )!= -1)
	dataBase->insert(ioaccess.importM3uFileList(file));
      if (file.findRev ( ".jbd", -1, false )!= -1){
	dataBase->insert(ioaccess.load(file));
	if(dateiName.isEmpty()){
	  dateiName = file;
	  setTitle();
	}
      }
    }
  }
}

void ApplicationWindow::initAccelKeys(){
  keys = new KAccel(this, "accel keys");
  keys->insertItem( i18n("Player1 Play"), "play1", SHIFT+Key_Q, true );
  keys->connectItem( "play1", this, SLOT( playNextSong1() ) );
  keys->insertItem( i18n("Player1 Pause"), "pause1", SHIFT+Key_W, true );
  keys->connectItem( "pause1", this, SLOT( player1Pause() ) );
  keys->insertItem( i18n("Player1 Stop"), "stop1", SHIFT+Key_E, true );
  keys->connectItem( "stop1", this, SLOT( player1Stop() ) );
  keys->insertItem( i18n("Player1 Fade In"), "fadeIn1", SHIFT+Key_R, true );
  keys->connectItem( "fadeIn1", fadeDevice[PLAYER1], SLOT( fadeIn() ) );
  keys->insertItem( i18n("Player1 Fade Stop"), "fadeStop1", SHIFT+Key_T, true );
  keys->connectItem( "fadeStop1", fadeDevice[PLAYER1], SLOT( stopFading() ) );
  keys->insertItem( i18n("Player1 Fade Out"), "fadeOut1", SHIFT+Key_Z, true );
  keys->connectItem( "fadeOut1", fadeDevice[PLAYER1], SLOT( fadeOut() ) );
  keys->insertItem( i18n("Player1 Toggle Time Mode"), "toggleTime1", SHIFT+Key_U, true );
  keys->connectItem( "toggleTime1", timer[PLAYER1], SLOT( toggleTimeMode() ) );

  keys->insertItem( i18n("Player2 Play"), "play2", SHIFT+Key_A, true );
  keys->connectItem( "play2", this, SLOT( playNextSong2() ) );
  keys->insertItem( i18n("Player2 Pause"), "pause2", SHIFT+Key_S, true );
  keys->connectItem( "pause2", this, SLOT( player2Pause() ) );
  keys->insertItem( i18n("Player2 Stop"), "stop2", SHIFT+Key_D, true );
  keys->connectItem( "stop2", this, SLOT( player2Stop() ) );
  keys->insertItem( i18n("Player2 Fade In"), "fadeIn2", SHIFT+Key_F, true );
  keys->connectItem( "fadeIn2", fadeDevice[PLAYER2], SLOT( fadeIn() ) );
  keys->insertItem( i18n("Player2 Fade Stop"), "fadeStop2", SHIFT+Key_G, true );
  keys->connectItem( "fadeStop2", fadeDevice[PLAYER2], SLOT( stopFading() ) );
  keys->insertItem( i18n("Player2 Fade Out"), "fadeOut2", SHIFT+Key_H, true );
  keys->connectItem( "fadeOut2", fadeDevice[PLAYER2], SLOT( fadeOut() ) );
  keys->insertItem( i18n("Player2 Toggle Time Mode"), "toggleTime2", SHIFT+Key_J, true );
  keys->connectItem( "toggleTime2", timer[PLAYER2], SLOT( toggleTimeMode() ) );

  keys->insertItem( i18n("Playlist, move up"), "moveUp", SHIFT+Key_P, true );
  keys->connectItem( "moveUp", mainWidget->getPlayList(), SLOT( moveUp() ) );
  keys->insertItem( i18n("Playlist, move down"), "moveDown", SHIFT+Key_L, true );
  keys->connectItem( "moveDown", mainWidget->getPlayList(), SLOT( moveDown() ) );
  keys->insertItem( i18n("Playlist, mark as played"), "markPlayed", SHIFT+Key_M, true );
  keys->connectItem( "markPlayed", mainWidget->getPlayList(), SLOT( markPlayed() ) );
  keys->insertItem( i18n("Playlist, mark as not played"), "markNotPlayed", SHIFT+Key_N, true );
  keys->connectItem( "markNotPlayed", mainWidget->getPlayList(), SLOT( markNotPlayed() ) );

  keys->insertItem( i18n("Playlist, mark all as played"), "markAllPlayed", SHIFT+Key_X, true );
  keys->connectItem( "markAllPlayed", mainWidget->getPlayList(), SLOT( markAllPlayed() ) );
  keys->insertItem( i18n("Playlist, mark all as not played"), "markAllNotPlayed", SHIFT+Key_Y, true );
  keys->connectItem( "markAllNotPlayed", mainWidget->getPlayList(), SLOT( markAllNotPlayed() ) );


  keys->readSettings(config);  
}


void ApplicationWindow::refreshOpenRecentFileDialog(){
  if(openRecent == NULL) {
    openRecent = new QPopupMenu();
    connect(openRecent, SIGNAL(activated(int)), this, SLOT(openRecentFile(int)));
  }
  else {
    openRecent->clear();
  }
  debugOutput( cout << "refresh open recent file dialog" << endl );
  config->setGroup(KCONFIG_OPEN_RECENT_FILE);
  openRecent->insertItem(config->readEntry(KCONFIG_OPEN_RECENT_FILE_1,NULL),1);
  openRecent->insertItem(config->readEntry(KCONFIG_OPEN_RECENT_FILE_2,NULL),2);
  openRecent->insertItem(config->readEntry(KCONFIG_OPEN_RECENT_FILE_3,NULL),3);
  openRecent->insertItem(config->readEntry(KCONFIG_OPEN_RECENT_FILE_4,NULL),4);
}


void ApplicationWindow::initMenuBar(){  
  openRecent = NULL;
  refreshOpenRecentFileDialog();

  config->setGroup(KCONFIG_GENERAL);

  QPopupMenu *import = new QPopupMenu();
  import->insertItem( i18n("MP3 File"), this, SLOT(importMP3Datei()) );
  import->insertItem( i18n("M3u List"), this, SLOT(importM3uListe()) );
  import->insertItem( i18n("MP3 List"), this, SLOT(importMP3Liste()) );
  QPopupMenu *export = new QPopupMenu();
  export->insertItem( i18n("MP3 List"), this, SLOT(exportMP3Liste()));
  export->insertItem( i18n("M3u List"), this, SLOT(exportM3uListe()));

  QPopupMenu *datei = new QPopupMenu();
  datei->insertItem ( ICON_NEWFILE, i18n("&New"), this, SLOT(neu()), CTRL+Key_N);
  datei->insertItem ( ICON_OPENFILE, i18n("&Open"), this, SLOT(laden()), CTRL+Key_O );
  datei->insertItem ( i18n("Open Recen&t..."), openRecent );
  datei->insertSeparator();
  datei->insertItem ( ICON_SAVEFILE, i18n("&Save"), this, SLOT(speichern()), CTRL+Key_S );
  datei->insertItem ( i18n("Save as"), this, SLOT(speichernAls()), CTRL+Key_A );
  datei->insertSeparator();
  datei->insertItem ( i18n("&Import"), import );
  datei->insertItem ( i18n("E&xport"), export );
  datei->insertSeparator();
  datei->insertItem ( ICON_PRINT, i18n("&Print"), this, SLOT(print()), CTRL+Key_P );
  datei->insertItem ( i18n("&Printer Settings"), this, SLOT(printerSettings()) );
  datei->insertItem ( i18n("&Print Preview"), this, SLOT(printerPreview()) );
  datei->insertSeparator();
  datei->insertItem ( i18n("&Quit"), this, SLOT(quitKJukeBox()), CTRL+Key_Q );

  songMenu = new QPopupMenu();
  songMenu->insertItem(ICON_COPYSELECTION, i18n("&Copy Selection"), mainWidget, SLOT(copySongIntoPlayList()), Key_F5, ID_COPYSELECTION);
  songMenu->insertItem(ICON_COPYALL,       i18n("Copy &All"), mainWidget, SLOT(copyAllSongsIntoPlayList()), Key_F6, ID_COPYALL );
  songMenu->insertSeparator();
  songMenu->insertItem(ICON_REMOVESELECTION, i18n("&Remove Selection"), mainWidget, SLOT(removeSong()), Key_F2,  ID_REMOVE );
  songMenu->insertItem(ICON_REMOVEPLAYED,    i18n("Remove P&layed"), mainWidget, SLOT(removePlayedSongs()), Key_F3 );
  songMenu->insertItem(ICON_CLEARPLAYLIST,   i18n("&Clear Playlist"), mainWidget, SLOT(clearPlayList()), Key_F4, ID_CLEARPLAYLIST );
  songMenu->insertSeparator();
  songMenu->insertItem( i18n("&Shuffle Play"), this, SLOT(toggleShufflePlaying()), Key_F9, ID_SHUFFLE );
  songMenu->setCheckable( TRUE );
  songMenu->setItemChecked(ID_SHUFFLE, shufflePlaying);
  songMenu->insertSeparator();
  songMenu->insertItem(ICON_EDITOR, i18n("Show &Info"), mainWidget, SLOT(showMP3Info()), Key_F7);
  //  songMenu->insertLineSeparator();
  songMenu->insertItem(ICON_PLAYIT, i18n("&Play It"), mainWidget, SLOT(playSelectedSong()), Key_F8 );

  genre = new QPopupMenu();
  genre->insertItem ( i18n("All") );
  genre->setCheckable( TRUE );
  genre->setItemChecked(0,true);
  connect(genre,SIGNAL(activated(int)), this , SLOT(genreChanged(int)));

  sort = new QPopupMenu();
  sort->insertItem( i18n("by &Artist, Album, Title"), _SORT_BY_ARTIST );
  sort->insertItem( i18n("by &Title, Artist, Album"), _SORT_BY_TITLE );
  sort->setCheckable( true );
  sort->setItemChecked(_SORT_BY_ARTIST,true);
  //  sort->insertItem ( i18n("by A&lbum, Title, Artist") );
  connect(sort,SIGNAL(activated(int)), this, SLOT(setSortOrder(int)));
  
  QPopupMenu *database = new QPopupMenu();
  database->insertItem( i18n("&Genre"), genre );
  database->insertItem( i18n("&Sort"), sort );

  QPopupMenu *playlist = new QPopupMenu();
  playlist->insertItem (  ICON_LOGFILE, i18n("View song logs"), songLogger, SLOT(toggleVisible()), Key_F12);
  playlist->insertSeparator();
  playlist->insertItem( i18n("Playlist, move up"), mainWidget->getPlayList(), SLOT( moveUp() ));
  playlist->insertItem( i18n("Playlist, move down"), mainWidget->getPlayList(), SLOT( moveDown() ) );
  playlist->insertSeparator();
  playlist->insertItem( i18n("Playlist, mark as played"), mainWidget->getPlayList(), SLOT( markPlayed() ) );
  playlist->insertItem( i18n("Playlist, mark as not played"), mainWidget->getPlayList(), SLOT( markNotPlayed() ) );
  playlist->insertSeparator();
  playlist->insertItem( i18n("Playlist, mark all as played"),  mainWidget->getPlayList(), SLOT( markAllPlayed() ) );
  playlist->insertItem( i18n("Playlist, mark all as not played"),  mainWidget->getPlayList(), SLOT( markAllNotPlayed() ) );
  


  controls = new QPopupMenu();
  controls->insertItem ( i18n("Configure"), this, SLOT(settingsOpen()) );
  controls->insertItem (  i18n("Configure Keys"), this, SLOT(keySettingsOpen()));
  controls->insertSeparator();
  mtb = controls->insertItem(i18n("Main Tool Bar"), this, SLOT(toggleMainToolBar()), CTRL+Key_M);
  controls->setCheckable( TRUE );
  controls->setItemChecked( mtb, TRUE );
  stb = controls->insertItem(i18n("Sound Tool Bar"), this, SLOT(toggleSoundToolBar()));
  controls->setCheckable( TRUE );
  controls->setItemChecked( stb, TRUE );

  sb = controls->insertItem(i18n("Status Bar"), this, SLOT(toggleStatusBar()), CTRL+Key_T);
  controls->setCheckable( TRUE );
  controls->setItemChecked( sb, TRUE );
 
  QPopupMenu *hilfe = new QPopupMenu();
  hilfe->insertItem (ICON_HELP, i18n("&Help"), this, SLOT(hilfe()), Key_F1);
  hilfe->insertSeparator();
  hilfe->insertItem ( i18n("&About KJukeBox"), this, SLOT(ueberkjukebox()) );
  hilfe->insertItem ( i18n("About &KDE"), (KApplication::getKApplication()), SLOT(aboutKDE()) );
    
  menu = menuBar();
  menu->insertItem( i18n("&File"), datei );
  menu->insertItem( i18n("&Song"), songMenu );
  menu->insertItem( i18n("&Database"), database );
  menu->insertItem( i18n("&Playlist"), playlist );
  menu->insertItem( i18n("&Options"), controls );
  menu->insertItem( i18n("&Help"), hilfe );
}


void ApplicationWindow::setSortOrder(int order){
  dataBase->setSortOrder(order);
  mainWidget->setDisplayOrder(order);
  if(order==_SORT_BY_TITLE) {
    sort->setItemChecked(_SORT_BY_TITLE,  true);
    sort->setItemChecked(_SORT_BY_ARTIST, false);
    status->message(i18n("Sort order changed to: Title, Album, Artist"));
  }
  else {
    sort->setItemChecked(_SORT_BY_TITLE,  false);
    sort->setItemChecked(_SORT_BY_ARTIST, true);
    status->message(i18n("Sort order changed to: Artist, Album, Title"));
  }
}

void ApplicationWindow::initSoundToolbar(int player){
  QLabel *tmp;
  int toolBarId;
  QString text;
    
  if (player == PLAYER1) toolBarId = ID_SOUND1_TOOLBAR;
  else toolBarId = ID_SOUND2_TOOLBAR;
  KToolBar *tb = toolBar(toolBarId);

  config->setGroup(KCONFIG_GENERAL);

  tb->insertButton(ICON_PLAY_INACTIVE,  ID_PLAY,  true,i18n("Play") );
  tb->insertButton(ICON_PAUSE_INACTIVE, ID_PAUSE, true,i18n("Pause"));
  tb->insertButton(ICON_STOP_INACTIVE,  ID_STOP,  true,i18n("Stop"));

  FadeDeviceGui *fadeDeviceGui = new FadeDeviceGui(tb,"fadeDeviceGui");

  connect(fadeDeviceGui, SIGNAL(volumeChanged(int)), 
	  fadeDevice[player], SLOT(setVolume(int)));
  connect(fadeDeviceGui, SIGNAL(fadeInClicked()),   
	  fadeDevice[player], SLOT(fadeIn()));
  connect(fadeDeviceGui, SIGNAL(fadeOutClicked()),  
	  fadeDevice[player], SLOT(fadeOut()));
  connect(fadeDeviceGui, SIGNAL(stopFadingClicked()), 
	  fadeDevice[player], SLOT(stopFading()));

  connect(fadeDevice[player], SIGNAL(fadeInReady()), 
	  fadeDeviceGui, SLOT(stopFade()));
  connect(fadeDevice[player], SIGNAL(fadeOutReady()), 
	  fadeDeviceGui, SLOT(stopFade()));
  connect(fadeDevice[player], SIGNAL(fadeInStarted()), 
	  fadeDeviceGui, SLOT(fadeIn()));
  connect(fadeDevice[player], SIGNAL(fadeOutStarted()), 
	  fadeDeviceGui, SLOT(fadeOut()));
  connect(fadeDevice[player], SIGNAL(volumeChanged(int)), 
	  fadeDeviceGui, SLOT(setVolume(int)));

  fadeDeviceGui->setFocusPolicy(QWidget::NoFocus);
  tb->insertWidget(ID_FADE, 80, fadeDeviceGui);

  tb->insertSeparator();

  debugOutput( cout << "  Initialize JumpDevice" << endl );
  jumpDevice[player] = new JumpDevice(tb,"jumpDevice");
  jumpDevice[player]->attachTo(streamTimeInfoDevice[player]);
  jumpDevice[player]->setFocusPolicy(QWidget::NoFocus);
  tb->insertWidget(ID_JUMPDEVICE, 100, jumpDevice[player]);

  debugOutput( cout << "  Initialize Timer " << player << endl );
  timer[player] = new TimerDevice(tb,"timer");
  timer[player]->attachTo(streamTimeInfoDevice[player]);
  tb->insertWidget(ID_COUNTDOWN, timer[player]->sizeHint().width(),
				   timer[player]);
  
  tb->insertButton(ICON_LEFT_INACTIVE, ID_LEFT_ARROW, false);
  tb->insertButton(ICON_HELPSMALL, ID_HELP, false, i18n("Song information"));
  tb->setItemEnabled(ID_HELP,false);

  tmp = new QLabel(tb);
  tmp->setText(i18n("No Song"));
  tmp->setAlignment(AlignLeft|AlignVCenter);
  tb->insertWidget(ID_PLAYER_TEXT,200,tmp);
  tb->setItemAutoSized (ID_PLAYER_TEXT, true);

  if (player == PLAYER1){
    connect(tb, SIGNAL(clicked(int)), 
	    this, SLOT(slotSoundToolbar1Clicked(int)));    
    connect(jumpDevice[player],SIGNAL(jump(int)),this,SLOT(jumpOnPlayer1(int)));
  }
  if (player == PLAYER2){
    connect(tb, SIGNAL(clicked(int)), 
	    this, SLOT(slotSoundToolbar2Clicked(int)));    
    connect(jumpDevice[player],SIGNAL(jump(int)),this,SLOT(jumpOnPlayer2(int)));
  }

  if(config->readBoolEntry(KCONFIG_GENERAL_SOUNDTOOLBAR ,true)){
    controls->setItemChecked(stb, true);
    enableToolBar(KToolBar::Show,toolBarId);
  }
  else{    
    controls->setItemChecked(stb, false);
    enableToolBar(KToolBar::Hide,toolBarId);
  }     
}

void ApplicationWindow::slotSoundToolbar1Clicked(int index){
  slotSoundToolbarClicked(PLAYER1, index);
}
void ApplicationWindow::slotSoundToolbar2Clicked(int index){
  slotSoundToolbarClicked(PLAYER2, index);
}
void ApplicationWindow::slotSoundToolbarClicked(int player, int index){
  switch (index) {
  case ID_PLAY:     playNextSong(player);   break;
  case ID_PAUSE:    playerPause(player); break;
  case ID_STOP:     playerStop(player);  break;
  case ID_HELP:     mainWidget->showMP3Info(currentSong[player]); break;
  }
}

void ApplicationWindow::initMainToolbar(){
  KIconLoader *loader;
  loader = (KApplication::getKApplication())->getIconLoader();
  KToolBar *tb = toolBar(ID_MAIN_TOOLBAR);

  tb->insertButton(ICON_NEWFILE , ID_NEW, true,i18n("New Database"));
  tb->insertButton(ICON_OPENFILE, ID_OPEN, true,i18n("Open new Database"));
  tb->insertButton(ICON_SAVEFILE, ID_SAVE, true,i18n("Save Database"));
  tb->insertButton(ICON_PRINT,    ID_PRINT, true,i18n("Print Database"));

  tb->insertSeparator();

  
  tb->insertButton(ICON_COPYSELECTION, ID_COPYSELECTION, true,i18n("Copy selected Song"));
  tb->insertButton(ICON_COPYALL, ID_COPYALL, true,i18n("Copy all songs"));
  tb->insertButton(ICON_REMOVESELECTION, ID_REMOVE, true,i18n("Remove selected Song"));
  tb->insertButton(ICON_REMOVEPLAYED, ID_REMOVEPLAYED, true,i18n("Remove played Songs"));
  tb->insertButton(ICON_CLEARPLAYLIST, ID_CLEARPLAYLIST, true,i18n("Clear the Playlist"));
  
  tb->insertSeparator();

  if(shufflePlaying)
    tb->insertButton(ICON_SHUFFLE_ACTIVE, ID_SHUFFLE, true,i18n("Toggle shuffle play"));
  else
    tb->insertButton(ICON_SHUFFLE_INACTIVE, ID_SHUFFLE, true,i18n("Toggle shuffle play"));

  tb->insertButton(ICON_EDITOR, ID_SHOWINFO, true, i18n("Show song info and tag editor"));
  tb->insertButton(ICON_PLAYIT, ID_PLAYIT, true, i18n("Play song immediately"));
  tb->insertButton(ICON_LOGFILE, ID_LOGFILE, true,  i18n("View song logs"));
  tb->insertButton(ICON_HELP, ID_HELP, true, i18n("Help"));
  
  connect(tb, SIGNAL(clicked(int)), 
	  this, SLOT(slotMainToolbarClicked(int)));    
  if(config->readBoolEntry(KCONFIG_GENERAL_SOUNDTOOLBAR,true)){
    controls->setItemChecked(mtb, true);
    enableToolBar(KToolBar::Show,ID_MAIN_TOOLBAR);
  }
  else{    
    controls->setItemChecked(mtb, false);
    enableToolBar(KToolBar::Hide,ID_MAIN_TOOLBAR);
  }     

}

void ApplicationWindow::slotMainToolbarClicked(int index){
  switch (index) {  
  case ID_NEW:           neu(); break;
  case ID_OPEN:          laden(); break;
  case ID_SAVE:          speichern(); break;
  case ID_PRINT:         print(); break;
  case ID_COPYSELECTION: mainWidget->copySongIntoPlayList();  break;
  case ID_COPYALL:       mainWidget->copyAllSongsIntoPlayList();  break;
  case ID_REMOVE:        mainWidget->removeSong();  break;
  case ID_REMOVEPLAYED:  mainWidget->removePlayedSongs(); break;
  case ID_CLEARPLAYLIST: mainWidget->clearPlayList(); break;
  case ID_SHUFFLE:       toggleShufflePlaying(); break;
  case ID_SHOWINFO:      mainWidget->showMP3Info(); break;
  case ID_PLAYIT:        mainWidget->playSelectedSong(); break;
  case ID_LOGFILE:       songLogger->toggleVisible(); break;
  case ID_HELP:          hilfe(); break;
  }
}
 
void ApplicationWindow::toggleShufflePlaying(){
  shufflePlaying = !shufflePlaying;
  config->setGroup(KCONFIG_GENERAL);
  config->writeEntry(KCONFIG_GENERAL_SHUFFLE, shufflePlaying);
  config->sync();
  if (shufflePlaying)
    status->message(i18n("Enable shuffle playing"));
  else 
    status->message(i18n("Disable shuffle playing"));
  refresh();
}

void ApplicationWindow::settingsOpen(){
  settings->toggleVisible();
}                    

void ApplicationWindow::keySettingsOpen(){
  KKeyDialog::configureKeys( keys );
}

void ApplicationWindow::settingsDestroyed(){
  debugOutput( cout << "Settings Destroyed" << endl );
  settings = NULL;
  settingsUp=FALSE;
}


void ApplicationWindow::hilfe(){
  kapp->invokeHTMLHelp("kjukebox/index.html","KJukeBox");
}

void ApplicationWindow::ueberkjukebox(){
  QString name;
  name = "KJukeBox v";
  name += VERSION;
  name += " MP3-Player\nBy: Rainer Maximini\nEmail:r_maximi@informatik.uni-kl.de\nhttp://third.informatik.uni-kl.de/~rainer/KJukeBox/index.html";
  KMsgBox::message(0L,"KJukeBox",name,KMsgBox::INFORMATION); 
}

void ApplicationWindow::genreChanged(int i){
  QString message= i18n("Genre changed to ");
  if (i==0) message += GENRE_ALL;
  else message += genre->text(i);
  status->message(message);
  if (i==0) dataBase->setGenre(GENRE_ALL);
  else dataBase->setGenre(genre->text(i));
  setGenreMenu();
}

void ApplicationWindow::setGenreMenu(){
  debugOutput( cout << "Genre Changed" << endl );
  QString *genreName;
  QGenreSet *genreSet;
  int i=0;
  QString selectedGenre = dataBase->getGenre();
  genre->clear();
  genre->insertItem ( i18n("All") );
  if(strcmp(selectedGenre,GENRE_ALL)==0) 
    genre->setItemChecked(0,true);
  i++;
  genreSet = dataBase->getGenres();
  for(genreName=genreSet->first(); genreName!=NULL; genreName=genreSet->next()){
    genre->insertItem ( *genreName );
    if(strcmp(selectedGenre,*genreName)==0)   
      genre->setItemChecked(i,true);
    i++;
  }
  
}

void ApplicationWindow::toggleStatusBar(){
  controls->setItemChecked( sb, (!controls->isItemChecked(sb)) );
  enableStatusBar();

  config->setGroup(KCONFIG_GENERAL);
  config->writeEntry(KCONFIG_GENERAL_STATUSBAR, controls->isItemChecked(sb));
  config->sync();
}

void ApplicationWindow::toggleSoundToolBar(){
  if(controls->isItemChecked(stb)){
    enableToolBar(KToolBar::Hide,ID_SOUND1_TOOLBAR);
    enableToolBar(KToolBar::Hide,ID_SOUND2_TOOLBAR);
  }
  else{
    enableToolBar(KToolBar::Show,ID_SOUND1_TOOLBAR);
    enableToolBar(KToolBar::Show,ID_SOUND2_TOOLBAR);
  }
  controls->setItemChecked( stb, (!controls->isItemChecked(stb)) );

  config->setGroup(KCONFIG_GENERAL);
  config->writeEntry(KCONFIG_GENERAL_SOUNDTOOLBAR, controls->isItemChecked(stb));
  config->sync();
}

void ApplicationWindow::toggleMainToolBar(){
  if(controls->isItemChecked(mtb))
    enableToolBar(KToolBar::Hide,ID_MAIN_TOOLBAR);
  else
    enableToolBar(KToolBar::Show,ID_MAIN_TOOLBAR);
  
  controls->setItemChecked( mtb, (!controls->isItemChecked(mtb)) );

  config->setGroup(KCONFIG_GENERAL);
  config->writeEntry(KCONFIG_GENERAL_MAINTOOLBAR, controls->isItemChecked(mtb));
  config->sync();
}


void ApplicationWindow::setStatus(int status){}
void ApplicationWindow::showError(int error){}


void ApplicationWindow::setTitle(){
  title = "KJukeBox: " + dateiName;
  setCaption(title);
}


void ApplicationWindow::refreshSoundToolBar(int player){  
  QPixmap pix1;
  QPixmap pix2;
  QPixmap pix3;
  QPixmap pix4;
  int toolBarId;
  QString text = i18n("No Song");
  QLabel *label;

  if (player == PLAYER1) toolBarId = ID_SOUND1_TOOLBAR;
  else toolBarId = ID_SOUND2_TOOLBAR;
  KToolBar *tb = toolBar(toolBarId);

  /* refresh the play, pause and stop buttons */
  if (pausing[player]){
    tb->setButtonPixmap(ID_PAUSE,      ICON_PAUSE_ACTIVE);         
    tb->setButtonPixmap(ID_LEFT_ARROW, ICON_LEFT_ACTIVE);
    tb->setButtonPixmap(ID_PLAY,       ICON_PLAY_INACTIVE);         
    tb->setButtonPixmap(ID_STOP,       ICON_STOP_INACTIVE);         
    text = playingSongName[player];
  }
  else if (playing[player]){
    tb->setButtonPixmap(ID_PAUSE,      ICON_PAUSE_INACTIVE);         
    tb->setButtonPixmap(ID_LEFT_ARROW, ICON_LEFT_ACTIVE);
    tb->setButtonPixmap(ID_PLAY,       ICON_PLAY_ACTIVE);         
    tb->setButtonPixmap(ID_STOP,       ICON_STOP_INACTIVE);         
    text = playingSongName[player];
  }
  else {
    tb->setButtonPixmap(ID_PAUSE,      ICON_PAUSE_INACTIVE);         
    tb->setButtonPixmap(ID_LEFT_ARROW, ICON_LEFT_INACTIVE);
    tb->setButtonPixmap(ID_PLAY,       ICON_PLAY_INACTIVE);         
    tb->setButtonPixmap(ID_STOP,       ICON_STOP_ACTIVE);         
  }

  /* refresh the shuffle button and menu*/
  songMenu->setItemChecked(ID_SHUFFLE, shufflePlaying);
  if(shufflePlaying){
    toolBar(ID_MAIN_TOOLBAR)->setButtonPixmap(ID_SHUFFLE, ICON_SHUFFLE_ACTIVE);
  }
  else{
    toolBar(ID_MAIN_TOOLBAR)->setButtonPixmap(ID_SHUFFLE, ICON_SHUFFLE_INACTIVE);
  }

  tb->setItemEnabled(ID_HELP,!(currentSong[player].isNull()));


  /* refresh the textdescription of the playing song */
  label = (QLabel*)tb->getWidget(ID_PLAYER_TEXT);
  label->setText(text);
}


void ApplicationWindow::refresh(){
  refreshSoundToolBar(PLAYER1);
  refreshSoundToolBar(PLAYER2);
}


void ApplicationWindow::resizeEvent( QResizeEvent *tmp ){
  config->setGroup(KCONFIG_GENERAL);
  config->writeEntry(KCONFIG_GENERAL_WIDTH, width());
  config->writeEntry(KCONFIG_GENERAL_HEIGHT, height()); 
  config->sync();

  KTMainWindow::resizeEvent(tmp);
  
  setTitle();
  refresh();
}
