/*
  Copyright (C) 1999 Rainer Maximini

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package
*/

#include "database.h"
#include <qobject.h>
#include <iostream.h>
#include <qprogressdialog.h> 

DataBase::DataBase( QObject *parent=0, const char *name=0 )
  : QObject(parent,name) {
  modifiedFlag = false;
  currentOrder = _START_SORT_ORDER;
  currentGenre = GENRE_ALL;
  playList = 0;

  listByArtist.setAutoDelete ( true ) ;
  listByTitle.setAutoDelete  ( false ) ;
  listForPlayList.setAutoDelete ( true ) ;
  genreSet.setAutoDelete(true);
  connect(&genreSet,SIGNAL(changed()),this,SIGNAL(genreSetChanged()));
}

DataBase::~DataBase(){ 
  listByTitle.clear();
  listByArtist.clear();
  disconnect(&genreSet,0,this,0);
  genreSet.clear();
}

void DataBase::attach(SongList *list){
  if(playList != 0){
    debugOutput( cout << "error: cannot attach a second playlist!" << endl );
  }
  else 
    playList = list;
}

void DataBase::detach(SongList *list){
  if (playList==list) playList = 0;
}

void DataBase::setSortOrder(int order){
  currentOrder = order;
}

void DataBase::insert(QSongList newSongs){
  Song *lauf;
  int i=0;
  if (newSongs.count() >=250){
    QProgressDialog progress(i18n("Loading Songs..."),i18n("Please Wait"),newSongs.count());
    progress.setLabelText(i18n("Loading Songs..."));
    progress.setCancelButtonText("Cancel");
    for(lauf=newSongs.first(); lauf!=0; lauf=newSongs.next()){
      if ( progress.wasCancelled() ) break;
      progress.setProgress( i );
      i++;
      listByArtist.inSort(lauf);
      listByTitle.inSort(lauf);
    }
    progress.setProgress( newSongs.count() );
  }
  else {
    for(lauf=newSongs.first(); lauf!=0; lauf=newSongs.next()){
      listByArtist.inSort(lauf);
      listByTitle.inSort(lauf);
    }
  }
  genreSet.insert(&newSongs);
  modifiedFlag=true;
  emit dBChanged();
}


void DataBase::insert(Song  newSong){
  if(newSong.isNull()) return;
  Song *tmp = new Song;
  *tmp = newSong;
  insert(tmp);
}

void DataBase::insert(Song *newSong){
  if(newSong->isNull()) return;
  listByArtist.inSort(newSong);
  listByTitle.inSort(newSong);
  genreSet.insert(newSong->getGenre());
  modifiedFlag=true;
  //  checkGenreSet();
  emit dBChanged();
}

void DataBase::insertPlayList(QSongList newSongs){
  Song *lauf;
  for(lauf=newSongs.first(); lauf!=0; lauf=newSongs.next()){
    listForPlayList.append(lauf);
  }
}

QSongList DataBase::find(Song *song){ return find(*song); }
QSongList DataBase::find(Song song){
  return find(song.getArtist(), song.getAlbum(),
	      song.getTitle(),  song.getGenre(),
	      song.getSeconds());
}

QSongList DataBase::find(QString artist="", QString album="", 
			 QString title="",  QString genre="",
			 unsigned int seconds=0 ){
  QSongList  results;
  Song      *song;
  if ((artist=="") && (album=="") &&
      (title=="")  && (genre=="") && 
      (seconds==0)) return results;
  for(song=listByArtist.first(); song!=0; song=listByArtist.next()){
    if ( (artist=="") || (strcmp(song->getArtist(),artist)==0) )
      if ( (album=="") || (strcmp(song->getAlbum(),album)==0) )
	if ( (title=="") || (strcmp(song->getTitle(),title)==0) )
	  if ( (genre=="") || (song->getGenre()==genre) )
	    if ( (seconds==0) || (song->getSeconds()==seconds) )
	      results.append(song);
  }
  return results;
}

Song* DataBase::findInPlayList(Song song){
  QString artist=song.getArtist();
  QString album =song.getAlbum();
  QString title =song.getTitle();
  QString genre =song.getGenre();
  unsigned int seconds=song.getSeconds();

  Song      *tmp;
  if ((artist=="") && (album=="") &&
      (title=="")  && (genre=="") && 
      (seconds==0)) return 0;
  for(tmp=listForPlayList.first(); tmp!=0; tmp=listForPlayList.next()){
    if ( (artist=="") || (strcmp(tmp->getArtist(),artist)==0) )
      if ( (album=="") || (strcmp(tmp->getAlbum(),album)==0) )
	if ( (title=="") || (strcmp(tmp->getTitle(),title)==0) )
	  if ( (genre=="") || (tmp->getGenre()==genre) )
	    if ( (seconds==0) || (tmp->getSeconds()==seconds) )
	      return tmp;
  }
  return 0;
}

void DataBase::removeInPlayList(Song *song){
  listForPlayList.remove(song);
}

void DataBase::clearPlayList(){
  listForPlayList.clear();
}

void DataBase::remove(Song song){
  remove(song.getArtist(), song.getAlbum(),
	 song.getTitle(),  song.getGenre(),
	 song.getSeconds());
}

void DataBase::remove(QString artist="", QString album="", 
		      QString title="",  QString genre="",
		      unsigned int seconds=0 ){
  Song *lauf;
  QSongList songList;
  /* First search References to the Songs */
  songList = find(artist,album,title,genre,seconds);
  /* If the list is empty return, else delete them */
  if (songList.isEmpty()) return;
  for(lauf=songList.first(); lauf!=0; lauf=songList.next()) {
    remove(lauf, false, false);
  }
  checkGenreSet();
  emit dBChanged();
  modifiedFlag = true;
}

void DataBase::remove(Song *song){
  remove(song, true);
}

void DataBase::remove(Song *song, bool checkGenres, bool emitSignal=true){
  /* First of all look, if in the playlist is a reference to the song */
  if(playList->hasReference(song)){
    listByTitle.remove(song);
    listByArtist.setAutoDelete ( false ) ;    
    if(listByArtist.remove(song)){
      listForPlayList.append(song);
    }
    listByArtist.setAutoDelete ( true ) ;
  }
  else {
    listByTitle.remove(song);
    listByArtist.remove(song);
  }
  emit songRemovedWithRef(song);
  if(checkGenres) checkGenreSet();
  if(emitSignal) emit dBChanged();
}  

void DataBase::remove(QSongList songs){
  Song *song;
  for(song=songs.first(); song!=0; song=songs.next())
    remove(song, false, false);
  checkGenreSet();
  emit dBChanged();
}

QSongList DataBase::getSongList(){
  QSongList *list;
  QSongList  result;
  Song      *lauf;
  if(strcmp(currentGenre, GENRE_ALL)==0) return getCompleteSongList();
  if (currentOrder==_SORT_BY_TITLE)  list = &listByTitle;
  else list = &listByArtist; /* it could only listByArtist */
  for(lauf=list->first(); lauf!=0; lauf=list->next())
    if(strcmp(lauf->getGenre(), currentGenre)==0) {
      result.append(lauf);
    }
  return result;
}
                    
QSongList DataBase::getCompleteSongList(){
  if (currentOrder==_SORT_BY_TITLE)  return listByTitle;
  return listByArtist; /* it could only listByArtist */
}

QGenreSet* DataBase::getGenres(){
  return &genreSet;
}

void DataBase::setGenre(QString genre){  
  if(strcmp(currentGenre,genre)!=0){
    currentGenre = genre;
    emit genreChanged();
  }
}
QString DataBase::getGenre(){  return currentGenre;}

void DataBase::clear(bool emitSignals=true){
  if(listByTitle.count()==0) return;
  /* first check the playlist */
  Song *tmp;
  QSongList playListRefs = playList->getAllReferencedSongs();
  for(tmp=playListRefs.first(); tmp!=0; tmp=playListRefs.next()){
    if(listByArtist.findRef(tmp)!=-1)
       remove(tmp,false,false);
  }
  /* now all songs of the playlist are saved, the rest can be cleared */
  /* clear the song, be carefull with the order */
  listByTitle.clear();  /* only removed the pointer */
  listByArtist.clear(); /* removed the song */
  genreSet.clear();
  currentGenre = GENRE_ALL;
  if(emitSignals){
    emit genreSetChanged();
    emit dBChanged();  
    emit dBCleared();
  }
}

void DataBase::checkPlayList(){
  Song *tmp;
  for(tmp=listForPlayList.first(); tmp!=0; tmp=listForPlayList.next()){
    while(tmp->getPlayed()){      
      listForPlayList.remove(tmp);
      tmp=listForPlayList.current();
      if(tmp==0) return;
    }
  }
}

void DataBase::checkGenreSet(){
  QString *glauf;
  Song tmp;
  bool changed=false;
  for(glauf=genreSet.first(); glauf!=0; glauf=genreSet.next()){
    tmp.setGenre(*glauf);
    if (find(tmp).count()==0){
      genreSet.remove(glauf);
      changed=true;
    }
  }
  if (changed) emit genreSetChanged();
}

void DataBase::songChanged(Song *song){
  Song *newone = new Song();
  *newone = *song;
  if(findInPlayList(*song)==song){
    listForPlayList.remove(song);
    listForPlayList.append(newone);
  }
  else {
    remove(song);
    insert(newone);
  }
  emit referenceChanged(song, newone);
}

bool DataBase::isModified(){ return modifiedFlag; }
void DataBase::modified(){ modifiedFlag = true; }
void DataBase::modifikationSaved(){ modifiedFlag = false; }


void DataBase::print(){
  cout << "DB: " << listByTitle.count()
       << "\tPLDB: " << listForPlayList.count() 
       << endl;
}
