/*
  Copyright (C) 1999 Rainer Maximini

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package
*/

#include <kapp.h>
#include <qstring.h>
#include <ktablistbox.h> 
#include <time.h>
#include "kjukebox.h"
#include "songList.h"
#include "iostream.h"
#include "qcursor.h"
#include <qkeycode.h>

SongList::SongList(QString bName, QWidget *parent, const char *name) 
  :KTable(parent,name){                                    
  songs.setAutoDelete( false );
  setupSurface(bName);
}

SongList::SongList(QWidget *parent, const char *name) 
  :KTable(parent,name){                                    
  setupSurface("");
}

void SongList::setupSurface(QString bName){
  srand(time(NULL));
  titleWidth = artistWidth=40; // 40% , 40%
  timeWidth = 20;
  setMinimumSize(50,100);

  QString title;
  titleName = bName;
  title.sprintf("%s  (%s %2d:%02d)", titleName.data(),i18n("not played"), 0, 0);
  box = new QGroupBox(title,this,"box");

  table = new KTable(this,"table",3);
  table->setColumn(0, i18n("Artist"));
  table->setColumn(1, i18n("Title"));
  table->setColumn(2, i18n("Time"));
  table->setSeparator('|');
  table->setTableFlags(Tbl_vScrollBar|Tbl_hScrollBar);
  table->setAutoUpdate(true);  

  setFocusPolicy(QWidget::StrongFocus);
  setFocusProxy (table);

  rightButtonMenu = new KPopupMenu(0, "rightbuttonmenu");
  rightButtonMenu->setTitle(i18n("What do you want to do?"));
  rightButtonMenu->insertItem(i18n("Show song info"),1);
  rightButtonMenu->insertItem(i18n("Play song immediately"),2);
  rightButtonMenu->insertSeparator();
  rightButtonMenu->insertItem(i18n("Mark as played"),3);
  rightButtonMenu->insertItem(i18n("Mark as unplayed"),4);
  rightButtonMenu->insertSeparator();
  rightButtonMenu->insertItem(i18n("Mark all as played"),5);
  rightButtonMenu->insertItem(i18n("Mark all as unplayed"),6);
  rightButtonMenu->insertSeparator();
  rightButtonMenu->insertItem(i18n("Move up"),7);
  rightButtonMenu->insertItem(i18n("Move down"),8);


  /* Bei einem Doppelklick auf einen Eintrag */
  connect(table,SIGNAL(selected(int,int)),SLOT(doubleClick(int,int)));
  /* Bei wechseln der Selection */
  connect(table,SIGNAL(highlighted(int,int)),SLOT(selectionChanged(int,int)));

  connect(table,SIGNAL(popupMenu (int ,int )), this, SLOT(rightButtonPressed(int,int)));
}

void SongList::selectionChanged(int , int){
  emit selectionChanged();
}

void SongList::clearSelection(){
  if(table->count() <= 0) return;
  table->unmarkAll();
  table->setCurrentItem(-1);
}

void SongList::doubleClick(int index, int column){
  emit doubleClick();
}


void SongList::rightButtonPressed(int row, int col){
  if(row==-1) return;
  Song song;
  int min, sec, seconds;

  table->setCurrentItem(row);
  
  int result = rightButtonMenu->exec(QCursor::pos());
  if(! ((result>=1) && (result<=6)) ) return ;
  
  song.setArtist(table->text(row,0));
  song.setTitle(table->text(row,1));
  sscanf(table->text(row,2),"%d:%d", &min, &sec);
  seconds = min*60 + sec;
  song.setSeconds(seconds);

  switch(result ) {
  case 1: emit showMP3Info(song); return;
  case 2: emit playSong(song);    return;
  case 3: markPlayed();    return;
  case 4: markNotPlayed(); return;
  case 5: markAllPlayed();    return;
  case 6: markAllNotPlayed(); return;
  case 7: moveUp();   return;
  case 8: moveDown(); return;
  }
}


void SongList::markPlayed(){
  if(songs.count()<=0) return;
  int row = table->currentItem();
  if(row >= 0){
    songs.at(row)->setPlayed(true );
    refreshDisplay();
  }
}

void SongList::markNotPlayed(){
  if(songs.count()<=0) return;
  int row = table->currentItem();
  if(row >= 0){
    songs.at(row)->setPlayed(false);
    refreshDisplay();
  }
}

void SongList::markAllPlayed(){
  Song *tmp;
  if(songs.count()<=0) return;
  for(tmp=songs.first(); tmp!=NULL; tmp=songs.next())
    tmp->setPlayed(true);  
  refreshDisplay();
}

void SongList::markAllNotPlayed(){
  Song *tmp;
  if(songs.count()<=0) return;
  for(tmp=songs.first(); tmp!=NULL; tmp=songs.next())
    tmp->setPlayed(false);  
  refreshDisplay();
}

void SongList::moveUp(){
  unsigned int row = table->currentItem();
  if(row<=0) return;
  Song *tmp;
  tmp = songs.at(row);
  songs.remove(row);
  songs.insert(row-1, tmp);
  table->setCurrentItem(row-1); 
  refreshDisplay();
}

void SongList::moveDown(){
  unsigned int row = table->currentItem();
  if((row<0)||(row>=(table->count()-1))) return;
  Song *tmp;
  tmp = songs.at(row);
  songs.remove(row);
  songs.insert(row+1, tmp);
  table->setCurrentItem(row+1);
  refreshDisplay();
}

void SongList::addSongs(QSongList newSongs){
  Song *lauf;
  if(newSongs.count()<= 0) return;
  for(lauf=newSongs.first(); lauf!=NULL; lauf=newSongs.next()){
    lauf->setPlayed(false);
    songs.append(lauf);
  }
  table->clear();
  refreshDisplay();
}

Song SongList::getSelection(){
  Song song;
  int min, sec, seconds;
  int index = table->currentItem();
  if (index == -1 ) return song;

  song.setArtist(table->text(index,0));
  song.setTitle(table->text(index,1));
  sscanf(table->text(index,2),"%d:%d", &min, &sec);
  seconds = min*60 + sec;
  song.setSeconds(seconds);
  return song;
}

QSongList SongList::getAllReferencedSongs(){
  return songs;
}

bool SongList::hasReference(Song *song){
  if(songs.findRef(song) == -1) return false;
  return true;
}

void SongList::referenceChanged(Song *oldRef, Song *newRef){
  int index;
  index = songs.find(oldRef);
  songs.remove();
  songs.insert(index,newRef);
  refreshDisplay();
}

QSongList SongList::removePlayedSongs(){
  Song *song;
  QSongList tmp;
  QSongList result;

  if(songs.count() == 0) return result;
  for(song=songs.first(); song!=NULL; song=songs.next()){
    if(!song->getPlayed())   tmp.append(song);
    else result.append(song);
  }
  songs = tmp;
  refreshDisplay();
  return result;
}

Song SongList::getRandomSongToPlay(){
  QSongList unplayed;
  Song song;
  Song *tmp = NULL;
  int  index;
  if(songs.count() == 0) return song;

  for(tmp=songs.first(); tmp!=NULL; tmp=songs.next())
    if(!tmp->getPlayed()) unplayed.append(tmp);
  
  if (unplayed.count() == 0){
    KConfig *config = (KApplication::getKApplication())->getConfig();
    config->setGroup(KCONFIG_GENERAL);
    if(config->readBoolEntry(KCONFIG_GENERAL_REPEAT,true)){
      markAllNotPlayed();
      unplayed = songs;
    }
    else{
      return song;
    }
  }
  index = rand() % unplayed.count();

  song.setArtist(unplayed.at(index)->getArtist());
  song.setTitle(unplayed.at(index)->getTitle());
  song.setSeconds(unplayed.at(index)->getSeconds());

  tmp = songs.at(songs.find(unplayed.at(index)));
  if(tmp != NULL) tmp->setPlayed(true);
  refreshDisplay();
  return song;
} 
  

Song SongList::getFirstSongToPlay(){
  Song nullSong;
  Song *song = 0;
  Song *tmp = 0;

  if(songs.count() == 0) return nullSong;

  /* Take all unplayed songs */
  for(tmp=songs.first(); tmp!=NULL; tmp=songs.next()){
    if(!tmp->getPlayed()) {
      song = tmp;
      break;
    }
  }
  
  if (song == 0){ //if all songs played
    /* now check if replay mode */
    KConfig *config = (KApplication::getKApplication())->getConfig();
    config->setGroup(KCONFIG_GENERAL);
    if(config->readBoolEntry(KCONFIG_GENERAL_REPEAT,true)){
      markAllNotPlayed();
    }
    else{ /* if not in replay mode */
      return nullSong;
    }
    song = songs.first();
  }
  
  if(song == 0) return nullSong;
  song->setPlayed(true);
  refreshDisplay();
  return *song;
}


void SongList::removeAllSongs(){
  clearSelection();
  songs.clear();
  table->clear();
  table->repaint();
}


Song* SongList::removeSong(){
  Song *tmp = NULL;
  int index = table->currentItem();
  cout << "Index:" << index << endl;
  if(index == -1) return tmp;
  tmp = songs.at(index);
  songs.removeRef(tmp);
  refreshDisplay();
  if((unsigned)table->count() >= (unsigned)index+1)
    table->setCurrentItem(index);
  else
    table->setCurrentItem(table->count()-1);
  return tmp;
}
 

void SongList::refreshDisplay(){
  QColor color("red");
  Song *song;
  int i=0;
  unsigned int seconds = 0;
  QString title;
  unsigned int selection = table->currentItem();;
  table->setAutoUpdate(false);
  table->clear();
  
  if(songs.count() > 0){
    for(song=songs.first(); song!=NULL; song=songs.next()){
      table->insertItem(songAsString(*song));
      if(song->getPlayed()) table->changeItemColor(color,i);
      else seconds += song->getSeconds();
      i++;
    }
  }
  table->setAutoUpdate(true);
  table->repaint();
  title.sprintf("%s  (%s %2d:%02d)", 
		titleName.data(), 
		i18n("not played"), 
		seconds/60, 
		seconds%60);
  box->setTitle(title);
  
  if( (selection != 0) && (table->count() > selection) ) {
    table->markItem(selection);
    table->setCurrentItem(selection);
  }
}


QString SongList::songAsString(Song song){
  QString helpstring(1000);
  helpstring.sprintf("%s%c%s%c%d:%02d", 
		     song.getArtist().data(), table->separator(),
		     song.getTitle().data(),  table->separator(), 
		     (song.getSeconds() / 60),
		     (song.getSeconds() % 60));
  return helpstring;
}
	  

void SongList::resizeEvent(QResizeEvent *){
  int xOffSet = 20;
  int y;
  box->setGeometry(0,0,width()-2, height());
  if(strcmp(box->title(),"") == 0) xOffSet = 5;
  y = box->width()-2-10;
  table->setColumnWidth(0,((y*artistWidth)/100)-2);
  table->setColumnWidth(1,((y*titleWidth)/100)-2);
  table->setColumnWidth(2,( y-(2.0*(y*titleWidth)/100) - 5));
  table->setGeometry(5,xOffSet,box->width()-10,box->height()-25);
}
