/*
  delays the audio pcm buffer
  Copyright (C) 2000  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */




#include "audioBufferDelayStreamProducer.h"



static void *writerThread(void *arg){
  ((AudioBufferDelayStreamProducer*)arg)->writeloop();
  return NULL;
}   


AudioBufferDelayStreamProducer::AudioBufferDelayStreamProducer(char* name) {
  this->name=name;

  // this is the maximum size a delivered buffer can grow
  maxLen=16384;

  freeTest=new MemChunk(0);
  // on this memory we do the transformation
  insertMem=new MemChunk(0);
  deliverConfig=new DeviceConfig();

  // we store in a ringBufferNT the data
  // we need 4KB of minimum deliver size.
  
  ringBufferNT=new RingBufferNT(1024*250,1024*16);
  producer=NULL;
  lLock=false;
  locked=false;

  pthread_mutex_init(&threadWriteInMut,NULL);
  pthread_mutex_init(&threadChangeMut,NULL);
  pthread_cond_init(&threadChangeCond,NULL);

  registerAsStreamProducer();

  lRun=false;

  pthread_create(&tr,NULL,writerThread,this);


  while (lRun == false) {
    NodeDevice::doSleep(1000);
  }

}


AudioBufferDelayStreamProducer::~AudioBufferDelayStreamProducer() {

  void* ret;
  writeInLock();
  lRun=false;
  while(lThreadEnd==false) {
    ringBufferNT->exitWaitForData();
    NodeDevice::doSleep(100);
  }
  writeInUnlock();
  pthread_join(tr,&ret);
  unregisterAsStreamProducer();
}


char* AudioBufferDelayStreamProducer::getNodeName() {
  return "AudioBufferDelayStreamProducer";
}




/** 
   The method writeIn(...)is entered by a seperate thread.
   <p>
   Here you can directly modify the PCM stream!     
   <pre> 
   NodeDevice* source       the parent Device which calls this method
   DeviceConfig* buf        class which stores the various
                            information asociated with a PCM stream 
                            (eg: stereo,samplesize,name of song, etc...)
   </pre>
*/
void AudioBufferDelayStreamProducer::writeIn(NodeDevice* source,
					     DeviceConfig* config) {



  

  writeOutLock();
  producer=(StreamProducer*)config->getStreamProducer();
  config->copyTo(deliverConfig);
  AudioStream* audioStream=config->getAudioStream();
  AudioBuffer* audioBuffer=audioStream->getAudioBuffer();
  MemChunk* memChunk=audioBuffer->getMemChunk();
  writeOutUnlock();

  char* ptr=NULL;
  int byte=0;

  if (memChunk != NULL) {
    ptr=memChunk->getPtr();
    byte=memChunk->getLen();
  }



  // if we have a status message, we mus deliver it ourself


  // copy data to ringBufferNT


  while(byte > 0){
    if (ringBufferNT->getWriteArea(insertMem,byte) < byte) {
      ringBufferNT->waitForSpace(byte);
    }
    ringBufferNT->getWriteArea(insertMem,byte);
    memcpy(insertMem->getPtr(),ptr,insertMem->getLen());
    ringBufferNT->forwardWritePtr(insertMem->getLen());
    byte-=insertMem->getLen();
    ptr+=insertMem->getLen();
  }

  // does our buffer has an overflowdanger?

  overflowTest();


}


void AudioBufferDelayStreamProducer::wakeUpThread(int lLockInProgress) {

    
}


void AudioBufferDelayStreamProducer::doDeliver() {
  
  MemChunk* memChunk;
  MemChunk* oldChunk;
  writeOutLock();
  
  AudioStream* deliverAudioStream=deliverConfig->getAudioStream();
  AudioBuffer* deliverAudioBuffer=deliverAudioStream->getAudioBuffer();
  int memLen=_AUDIOBUFFER_PREFERRED_FILLGRADE_DEFAULT;
  memChunk=ringBufferNT->requestMemChunk(memLen);


  deliverConfig->setStreamProducer(this);
  deliverAudioBuffer->setMemChunk(memChunk);
  deliverConfig->addLock();

  deliverToListener(deliverConfig);

  deliverConfig->removeLock();
  deliverConfig->forward();
  writeOutUnlock();
  
  

}


void AudioBufferDelayStreamProducer::writeloop() {

  // first copy the workMemory into the ringbuffer

  // ok memory transfered, now we deliver the same packet size
  // we got in
  // this is not possible for the "half" option
  // but the mixer expect fixed sizes for all packets
  // thus we deliver only evey second call


  pthread_mutex_lock(&threadWriteInMut);



  lThreadEnd=false;
  lRun=true;

  
  while(lRun) {
    if (pthread_mutex_trylock(&threadChangeMut) == EBUSY) {
      cThreadPos="change request -s";
      pthread_cond_wait(&threadChangeCond,&threadWriteInMut);
      cThreadPos="change request -e";
      continue;
    }
    pthread_mutex_unlock(&threadChangeMut);
    underflowTest();

    int memLen=_AUDIOBUFFER_PREFERRED_FILLGRADE_DEFAULT;
    MemChunk* memChunk=ringBufferNT->requestMemChunk(memLen);
    if (memChunk->getLen() < memLen) {

      ringBufferNT->waitForData(memLen);
      // nothin to deliver this time
      continue;
    }
    
    int locks=waitForUnblock();

    if (locks) {
      //printf("%s %p had locks\n",name,this);

      continue;
    }

    doDeliver();

    continue;
    
  }
  AudioBuffer* buffer=deliverConfig->getAudioStream()->getAudioBuffer();
  buffer->setMemChunk(NULL);

  lThreadEnd=true;
  pthread_mutex_unlock(&threadWriteInMut);

}


void AudioBufferDelayStreamProducer::lockDeliverThread() {
  pthread_mutex_lock(&threadChangeMut);
  DaisyChain* daisyChain=(DaisyChain*)NodeDevice::getDaisyChain();
  daisyChain->wakeUpThread(this);
  ringBufferNT->setWaitForData(false);
  pthread_mutex_lock(&threadWriteInMut);
  ringBufferNT->setWaitForData(true);
  locked=true;
}


void AudioBufferDelayStreamProducer::unlockDeliverThread() {
  pthread_mutex_unlock(&threadChangeMut);
  pthread_cond_signal(&threadChangeCond);
  pthread_mutex_unlock(&threadWriteInMut);
  locked=false;
}
 


void AudioBufferDelayStreamProducer::overflowTest() {
  if (lLock==false) {
    int free=ringBufferNT->getSize()-ringBufferNT->getFillgrade();
    if (free < maxLen) {
      if (producer == NULL) {
	cout << "fatal Error in AudioBufferDelayStreamProducer::writeIn!"
	     << " StreamProducer is NULL!"<<endl;
	exit(-1);
      }
      DaisyChain* daisyChain=(DaisyChain*)(producer->getDaisyChain());
      daisyChain->addLock(this,producer);
      lLock=true;
    }
  }
}


void AudioBufferDelayStreamProducer::underflowTest() {
  
  if (lLock) {
    int free=ringBufferNT->getSize()-ringBufferNT->getFillgrade();
    if (free > maxLen) {
      if (producer == NULL) {
	cout << "fatal Error in AudioBufferDelayStreamProducer::writeIn!"
	     << " StreamProducer is NULL!"<<endl;
	exit(-1);
      }
      DaisyChain* daisyChain=(DaisyChain*)(producer->getDaisyChain());
      daisyChain->removeLock(this,producer);
      lLock=false;
    }
  }
}
