/*
  Configuration for the InfoDevice
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include <infoDeviceConfig.h>


InfoDeviceConfig::InfoDeviceConfig(const char* configGroup,
				   QWidget * parent=0, const char * name=0 ) 
                 : PrefConfig(parent,name) {

  tmp1=new Buffer(200);
  tmp2=new Buffer(200);

  config = kapp->getConfig();
  this->configGroup=configGroup;
  load();


  useID3Box = new QCheckBox(i18n("use ID3 if avaiable"), this, "useID3Box");
  useID3Box->setGeometry(20,10,125,25);
  useID3Box->setChecked(useID3_cfg);
  connect(useID3Box,SIGNAL(toggled(bool)),this,SLOT(setUseID3(bool)));

  //
  // First the macros for the file
  //

  infoFileBox= new QGroupBox(i18n("File Text"),this, "infoFileBox" );
  infoFileBox->setGeometry(20, 40, 300, 120 );

  lineFileLable = new QLabel(infoFileBox);
  lineFileLable->setGeometry(20,20,60,25);
  lineFileLable->setText(i18n("Macro :"));

  lineFileEdit = new QLineEdit(infoFileBox, "lineEdit");
  lineFileEdit->setGeometry(90,20 ,180,30);
  lineFileEdit->setText(lineFile_cfg);

  legendFile = new QLabel(infoFileBox,"Legend");
  legendFile->setGeometry(20,60,100,80);
  legendFile->setAlignment(AlignLeft|WordBreak);
  legendFile->setText(i18n("Legend:"));

  legendFile = new QLabel(infoFileBox, "Legend");
  legendFile->setGeometry(70,80,230,20);
  legendFile->setAlignment(AlignLeft|WordBreak);
  legendFile->setText(i18n("%a\tfilename         %b\ttime   %s\tspace"));


  //
  // Now the id3 box
  //

  infoID3Box= new QGroupBox(i18n("ID3 Text"),this, "infoID3Box" );
  infoID3Box->setGeometry(20, 200, 300, 170 );

  lineID3Lable = new QLabel(infoID3Box);
  lineID3Lable->setGeometry(20,20,60,25);
  lineID3Lable->setText(i18n("Macro :"));

  lineID3Edit = new QLineEdit(infoID3Box, "lineEdit");
  lineID3Edit->setGeometry(90,20 ,180,25);
  lineID3Edit->setText(lineID3_cfg);

  legendID3 = new QLabel(infoID3Box, "Legend");
  legendID3->setGeometry(20,60,100,90);
  legendID3->setAlignment(AlignLeft|WordBreak);
  legendID3->setText(i18n("Legend: all of file and"));

  legendID3 = new QLabel(infoID3Box, "Legend");
  legendID3->setGeometry(50,80,120,70);
  legendID3->setAlignment(AlignLeft|WordBreak);
  legendID3->setText(i18n("%1\t\tArtist\n%2\t\tSong\n%3\t\tAlbum\n%4\t\tYear\n"));

  legendID3 = new QLabel(infoID3Box, "Legend");
  legendID3->setGeometry(150,80,100,70);
  legendID3->setAlignment(AlignLeft|WordBreak);
  legendID3->setText(i18n("%5\t\tGenre\n%6\t\tComment\n%7\t\tFilename\n%%\t\tsymbol '%'"));

}


InfoDeviceConfig::~InfoDeviceConfig() {
  delete tmp1;
  delete tmp2;
}


void InfoDeviceConfig::setUseID3(bool val) {
  cout <<"setUseID3:"<<val<<endl;
}
   

const char* InfoDeviceConfig::getGroupName() {
  return "Info";
}


void InfoDeviceConfig::load() {
  config->setGroup(configGroup);
  useID3_cfg=config->readBoolEntry( "useID3", true);
  lineID3_cfg=config->readEntry("MacroID3","%1 %2 - [%b]    %s");
  lineFile_cfg=config->readEntry("MacroFile","%a - [%b]    %s");

}


void InfoDeviceConfig::save() {
  useID3_cfg=useID3Box->isChecked();
  lineID3_cfg=lineID3Edit->text();
  lineFile_cfg=lineFileEdit->text();

  config->setGroup(configGroup);
  config->writeEntry("useID3",useID3_cfg);
  config->writeEntry("MacroID3",lineID3_cfg);
  config->writeEntry("MacroFile",lineFile_cfg);
}


void InfoDeviceConfig::apply() {
  save();
  emit(applyEvent());
}


void InfoDeviceConfig::translate(Song* song,Buffer* dest) {
  
  int len1=strlen(lineFile_cfg.data())+250;
  int len2=strlen(lineID3_cfg.data())+250;
  int len;
  int substToken=0;

  len=len2;
  if(len1 > len) {
    len=len1;
  }
  if (len > tmp1->getSize()) {
    tmp1->grow(len-(tmp1->getSize()));
  }
  const char* location=song->getSongLocation();

  if (strncmp("http",location,4) == 0) {
    dest->clear();
    dest->append(location);
    return;
  }

  parseFilename(song,lineFile_cfg.data(),tmp1);
  dest->clear();
  dest->append(tmp1->getData());

  if (useID3_cfg == false) {
    return;
  }
   
    
  substToken=parseID3(song,lineID3_cfg.data(),tmp1);

  if (substToken > 0) {
    dest->clear();
    dest->append(tmp1->getData());
  }
}



int InfoDeviceConfig::parseID3(Song* song,
			       char* macro, Buffer* dest,int lError=true) {
  int i;
  int substToken=0;
  int len;
  MusicInfo* musicInfo=song->getMusicInfo();
  Tag* id3Tag=song->getID3Tag(); 

  parseFilename(song,macro,tmp2,false);
  macro=tmp2->getData();
  len=strlen(macro);
  dest->clear();
  for(i=0;i<len;i++) {
    if (macro[i] != '%') {
      dest->append(&(macro[i]),1);
      continue;
    }
    if (i >= len-1) continue;
    // another char is available
    i++;
    char bst=macro[i];
    if (bst == '1') {
      //
      substToken+=strlen(id3Tag->getArtist());
      dest->append(id3Tag->getArtist());
      continue;
    }
    if (bst == '2') {
      //
      substToken+=strlen(id3Tag->getTitle());
      dest->append(id3Tag->getTitle());
      continue;
    }
    if (bst == '3') {
      //
      substToken+=strlen(id3Tag->getAlbum());
      dest->append(id3Tag->getAlbum());
      continue;
    }
    if (bst == '4') {
      //
      substToken+=strlen(id3Tag->getYear());
      dest->append(id3Tag->getYear());
      continue;
    }
    if (bst == '5') {
      //
      dest->append(id3Tag->getGenre());
      continue;
    }
    if (bst == '6') {
      //
      substToken+=strlen(id3Tag->getComment());
      dest->append(id3Tag->getComment());
      continue;
    }
    if (bst == '6') {
      dest->append(musicInfo->getName());
      continue;
    }
    if (bst == '%') {
      dest->append("%");
      continue;
    }
    if (lError == true) {
      cout<<"unknown escape sequence:%"<<bst<<endl;  
    }
    dest->append("%");
    dest->append(&bst,1);
  }
  
  return substToken;
}



int InfoDeviceConfig::parseFilename(Song* song,
				    char* macro,Buffer* dest,int lError=true) {
  int i;
  int substToken=0;
  int len;
  int min;
  int sec;
  MusicInfo* musicInfo=song->getMusicInfo();
  int time=song->getSongLength();
  min=time/60;
  sec=time%60;
  char buffer[20];
  buffer[19]=0;

  if (sec <= 9) {
    snprintf((char*)&buffer,18,"%d:0%d",min,sec);
  } else {
    snprintf((char*)&buffer,18,"%d:%d",min,sec);
  }
   
  
  dest->clear();
  len=strlen(macro);
  for(i=0;i<len;i++) {
    if (macro[i] != '%') {
      dest->append(&(macro[i]),1);
      continue;
    }
    if (i >= len-1) continue;
    // another char is available
    i++;
    char bst=macro[i];
    if (bst == 'a') {
      //
      substToken+=strlen(musicInfo->getName());
      dest->append(musicInfo->getName());
      continue;
    }
    if (bst == 'b') {
      //
      substToken+=strlen((char*)&buffer);
      dest->append((char*)&buffer);
      continue;
    }
    if (bst == 's') {
      //
      //substToken+=strlen(" ");
      dest->append((char*)" ");
      continue;
    }
    if (bst == '%') {
      dest->append("%");
      continue;
    }
    if (lError == true) {
      cout<<"unknown escape sequence:%"<<bst<<endl;  
    }
    dest->append("%");
    dest->append(&bst,1);

  }
  return 0;
}


 
