/*
  class for reconstruction
  Copyright (C) 1999  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include "recon.h"





/*
 * We use a lookup table to make sure values stay in the 0..255 range.
 * Since this is cropping (ie, x = (x < 0)?0:(x>255)?255:x; ), wee call this
 * table the "crop table".
 * MAX_NEG_CROP is the maximum neg/pos value we can handle.
 */
/*
 * We use a lookup table to make sure values stay in the 0..255 range.
 * Since this is cropping (ie, x = (x < 0)?0:(x>255)?255:x; ), wee call this
 * table the "crop table".
 * MAX_NEG_CROP is the maximum neg/pos value we can handle.
 */

#define MAX_NEG_CROP 2048
#define NUM_CROP_ENTRIES (2048+2*MAX_NEG_CROP)

static unsigned char cropTbl[NUM_CROP_ENTRIES];

#define NDEBUG
#include <assert.h>

#ifndef NDEBUG
/* If people really want to see such things, check 'em */
#define myassert(x,expression)\
  if (!(expression)) {\
  printf ("Bad crop value (%d) at line %d\n", x, __LINE__);\
  (vid_stream->deMux)->next_start_code(vid_stream); return;}
#define assertCrop(x)	myassert(x,((x) >= -MAX_NEG_CROP) && \
				 ((x) <= 2048+MAX_NEG_CROP))
#else
#define assertCrop(x)
#endif


Recon::Recon(VidStream* vid_stream) {
  this->vid_stream=vid_stream;
  int i;


  /* Initialize crop table. */

  for (i = (-MAX_NEG_CROP); i < NUM_CROP_ENTRIES - MAX_NEG_CROP; i++) {
    if (i <= 0) {
      cropTbl[i + MAX_NEG_CROP] = 0;
    } else if (i >= 255) {
      cropTbl[i + MAX_NEG_CROP] = 255;
    } else {
      cropTbl[i + MAX_NEG_CROP] = i;
    }
  }
  
}


Recon::~Recon() {
}


/*
 *--------------------------------------------------------------
 *
 * ReconIMBlock --
 *
 *	Reconstructs intra coded macroblock.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int Recon::ReconIMBlock(int bnum) {
  int mb_row, mb_col, row, col, row_size, rr;
  unsigned char *dest;
  unsigned char *picDest;
  int lumLength=(vid_stream->sequence.pictureArray->getCurrent())->getLumLength();
  int colorLength=(vid_stream->sequence.pictureArray->getCurrent())->getColorLength();
  int endDest=0;

  /* Calculate macroblock row and column from address. */
  if (vid_stream->sequence.mb_width <= 0) {
     cout << "mb_width <= 0"<<endl;
     return false;
  }
  mb_row = vid_stream->mblock.mb_address / vid_stream->sequence.mb_width;
  mb_col = vid_stream->mblock.mb_address % vid_stream->sequence.mb_width;


  /* If block is luminance block... */

  if (bnum < 4) {

    /* Calculate row and col values for upper left pixel of block. */

    row = mb_row * 16;
    col = mb_col * 16;
    if (bnum > 1)
      row += 8;
    if (bnum % 2)
      col += 8;

    /* Set dest to luminance plane of current pict image. */

    picDest = (vid_stream->sequence.pictureArray->getCurrent())->getLuminancePtr();
    endDest=lumLength;

    /* Establish row size. */

    row_size = vid_stream->sequence.mb_width * 16;
  }
  /* Otherwise if block is Cr block... */
  /* Cr first because of the earlier mixup */

  else if (bnum == 5) {

    /* Set dest to Cr plane of current pict image. */

    picDest = (vid_stream->sequence.pictureArray->getCurrent())->getCrPtr();
    endDest=colorLength;

    /* Establish row size. */

    row_size = vid_stream->sequence.mb_width * 8;

    /* Calculate row,col for upper left pixel of block. */

    row = mb_row * 8;
    col = mb_col * 8;
  }
  /* Otherwise block is Cb block, and ... */

  else {

    /* Set dest to Cb plane of current pict image. */

    picDest = (vid_stream->sequence.pictureArray->getCurrent())->getCbPtr();
    endDest=colorLength;

    /* Establish row size. */

    row_size = vid_stream->sequence.mb_width * 8;

    /* Calculate row,col for upper left pixel value of block. */

    row = mb_row * 8;
    col = mb_col * 8;
  }

  
  /*
   * For each pixel in block, set to cropped reconstructed value from inverse
   * dct.
   */
  {
    short *sp = &vid_stream->block.dct_recon[0][0];
    unsigned char *cm = cropTbl + MAX_NEG_CROP;
    dest = picDest+row * row_size + col;


    if ((dest+7*row_size+7 >= picDest+endDest) || (dest < picDest)) {
      cout << "urg! last resort caught before sigsev -4"<<endl;
      return false;
    }


    for (rr = 0; rr < 4; rr++, sp += 16) {
      

      dest[0] = cm[sp[0]];
      dest[1] = cm[sp[1]];
      dest[2] = cm[sp[2]];
      dest[3] = cm[sp[3]];
      dest[4] = cm[sp[4]];
      dest[5] = cm[sp[5]];
      dest[6] = cm[sp[6]];
      dest[7] = cm[sp[7]];


      dest += row_size;

      dest[0] = cm[sp[8]];
      dest[1] = cm[sp[9]];
      dest[2] = cm[sp[10]];
      dest[3] = cm[sp[11]];
      dest[4] = cm[sp[12]];
      dest[5] = cm[sp[13]];
      dest[6] = cm[sp[14]];
      dest[7] = cm[sp[15]];

      dest += row_size;
     
    }



  }
  return true;
}



/*
 *--------------------------------------------------------------
 *
 * ReconPMBlock --
 *
 *	Reconstructs forward predicted macroblocks.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

int Recon::ReconPMBlock(int bnum,
			int recon_right_for,
			int recon_down_for,
			int zflag) {
  int mb_row, mb_col, row, col, row_size, rr;
  unsigned char *picDest, *past;
  unsigned char *rindex1, *rindex2, *rindex3, *rindex4;
  unsigned char *index;
  short int *blockvals;
  int lumLength=(vid_stream->sequence.pictureArray->getCurrent())->getLumLength();
  int colorLength=(vid_stream->sequence.pictureArray->getCurrent())->getColorLength();
  int endPast=0;
  int endDest=0;
  picDest=0;
  /* Calculate macroblock row and column from address. */

  if (vid_stream->sequence.mb_width <= 0) {
     cout << "mb_width <= 0"<<endl;
     return false;
  }

  mb_row = vid_stream->mblock.mb_address / vid_stream->sequence.mb_width;
  mb_col = vid_stream->mblock.mb_address % vid_stream->sequence.mb_width;

  if (bnum < 4) {

    /* Calculate right_for, down_for motion vectors. */

    vid_stream->right_for = recon_right_for >> 1;
    vid_stream->down_for = recon_down_for >> 1;
    vid_stream->right_half_for = recon_right_for & 0x1;
    vid_stream->down_half_for = recon_down_for & 0x1;

    /* Set dest to luminance plane of current pict image. */

    picDest = (vid_stream->sequence.pictureArray->getCurrent())->getLuminancePtr();
    endDest=lumLength;

    if (vid_stream->picture.code_type == B_TYPE) {
       past = (vid_stream->sequence.pictureArray->getPast())->getLuminancePtr();
    } else {

      /* Set predictive frame to current future frame. */

      past = (vid_stream->sequence.pictureArray->getFuture())->getLuminancePtr();
    }
    endPast=lumLength;
    /* Establish row size. */

    row_size = vid_stream->sequence.mb_width << 4;

    /* Calculate row,col of upper left pixel in block. */

    row = mb_row << 4;
    col = mb_col << 4;
    if (bnum > 1)
      row += 8;
    if (bnum % 2)
      col += 8;

  /* Otherwise, block is NOT luminance block, ... */

 } else {

    /* Construct motion vectors. */

    recon_right_for /= 2;
    recon_down_for /= 2;
    vid_stream->right_for = recon_right_for >> 1;
    vid_stream->down_for = recon_down_for >> 1;
    vid_stream->right_half_for = recon_right_for & 0x1;
    vid_stream->down_half_for = recon_down_for & 0x1;

    /* Establish row size. */
    
    row_size = vid_stream->sequence.mb_width << 3;

    /* Calculate row,col of upper left pixel in block. */

    row = mb_row << 3;
    col = mb_col << 3;

    /* If block is Cr block... */
    /* 5 first because order was mixed up in earlier versions */

    if (bnum == 5) {

      /* Set dest to Cr plane of current pict image. */

      picDest = (vid_stream->sequence.pictureArray->getCurrent())->getCrPtr();

      if (vid_stream->picture.code_type == B_TYPE) {

	past = (vid_stream->sequence.pictureArray->getPast())->getCrPtr();
      } else {
	past = (vid_stream->sequence.pictureArray->getFuture())->getCrPtr();
      }
    }
    /* Otherwise, block is Cb block... */

    else {

      /* Set dest to Cb plane of current pict image. */

      picDest = (vid_stream->sequence.pictureArray->getCurrent())->getCbPtr();

      if (vid_stream->picture.code_type == B_TYPE) {
	past = (vid_stream->sequence.pictureArray->getPast())->getCbPtr();
      } else {
	past = (vid_stream->sequence.pictureArray->getFuture())->getCbPtr();
      }
    }
    endPast=colorLength;
    endDest=colorLength;

  }
  
  /* For each pixel in block... */

  index = picDest + (row * row_size) + col;
  rindex1 = past + (row + vid_stream->down_for) * row_size 
    + col + vid_stream->right_for;


  blockvals = &(vid_stream->block.dct_recon[0][0]);

  if ((rindex1+7*row_size+7 >= past+endPast) || (rindex1 < past)) {
    cout << "urg! last resort caught before sigsev -1"<<endl;
    return false;
  }
  if ((index+7*row_size+7 >= picDest+endDest) || (index < picDest)) {
    cout << "urg! last resort caught before sigsev -2"<<endl;
    return false;
  }

 
  /*
   * Calculate predictive pixel value based on motion vectors and copy to
   * dest plane.
   */
  
  if ((!vid_stream->down_half_for) && (!vid_stream->right_half_for)) {
    unsigned char *cm = cropTbl + MAX_NEG_CROP;

    if (!zflag) {

      for (rr = 0; rr < 4; rr++) {
	index[0] = cm[(int) rindex1[0] + (int) blockvals[0]];
	index[1] = cm[(int) rindex1[1] + (int) blockvals[1]];
	index[2] = cm[(int) rindex1[2] + (int) blockvals[2]];
	index[3] = cm[(int) rindex1[3] + (int) blockvals[3]];
	index[4] = cm[(int) rindex1[4] + (int) blockvals[4]];
	index[5] = cm[(int) rindex1[5] + (int) blockvals[5]];
	index[6] = cm[(int) rindex1[6] + (int) blockvals[6]];
	index[7] = cm[(int) rindex1[7] + (int) blockvals[7]];
	index += row_size;
	rindex1 += row_size;
	
	index[0] = cm[(int) rindex1[0] + (int) blockvals[8]];
	index[1] = cm[(int) rindex1[1] + (int) blockvals[9]];
	index[2] = cm[(int) rindex1[2] + (int) blockvals[10]];
	index[3] = cm[(int) rindex1[3] + (int) blockvals[11]];
	index[4] = cm[(int) rindex1[4] + (int) blockvals[12]];
	index[5] = cm[(int) rindex1[5] + (int) blockvals[13]];
	index[6] = cm[(int) rindex1[6] + (int) blockvals[14]];
	index[7] = cm[(int) rindex1[7] + (int) blockvals[15]];
	blockvals += 16;
	index += row_size;
	rindex1 += row_size;
      }
    } else {
      if (vid_stream->right_for & 0x1) {

	/* No alignment, used by copy */
	for (rr = 0; rr < 4; rr++) {
	  
	  memcpy(index,rindex1,sizeof(char)*8);
	    
	    
	  index += row_size;
	  rindex1 += row_size;
	  
	  memcpy(index,rindex1,sizeof(char)*8);
	  
	  index += row_size;
	  rindex1 += row_size;
	}
      } else if (vid_stream->right_for & 0x2) {
	/* Half-word bit aligned, use 16 bit copy */
	short *src = (short *)rindex1;
	short *dest = (short *)index;
	row_size >>= 1;

	for (rr = 0; rr < 4; rr++) {
	  
	  memcpy(dest,src,sizeof(short)*4);
	  
	  dest += row_size;
	  src += row_size;
	  
	  memcpy(dest,src,sizeof(short)*4);
	  
	  dest += row_size;
	  src += row_size;
	}
      } else {
	/* Word aligned, use 32 bit copy */
	int *src = (int*) rindex1;
	int *dest = (int*) index;

	row_size >>= 2;


	for (rr = 0; rr < 4; rr++) {
	  dest[0] = src[0];
	  dest[1] = src[1];
	  dest += row_size;
	  src += row_size;
	  
	  dest[0] = src[0];
	  dest[1] = src[1];
	  dest += row_size;
	  src += row_size;
	}
	
      }
    }
  } else {
    unsigned char *cm = cropTbl + MAX_NEG_CROP;
    rindex2 = rindex1 + vid_stream->right_half_for 
      + (vid_stream->down_half_for * row_size);
    
    /* if one of the two is zero, then quality makes no difference */
    if ((!vid_stream->right_half_for) ||
	(!vid_stream->down_half_for) || (!qualityFlag)) {
      
      if (!zflag) {
	for (rr = 0; rr < 4; rr++) {
	  index[0] = cm[((int) (rindex1[0]+rindex2[0]+1) >> 1)+blockvals[0]];
	  index[1] = cm[((int) (rindex1[1]+rindex2[1]+1) >> 1)+blockvals[1]];
	  index[2] = cm[((int) (rindex1[2]+rindex2[2]+1) >> 1)+blockvals[2]];
	  index[3] = cm[((int) (rindex1[3]+rindex2[3]+1) >> 1)+blockvals[3]];
	  index[4] = cm[((int) (rindex1[4]+rindex2[4]+1) >> 1)+blockvals[4]];
	  index[5] = cm[((int) (rindex1[5]+rindex2[5]+1) >> 1)+blockvals[5]];
	  index[6] = cm[((int) (rindex1[6]+rindex2[6]+1) >> 1)+blockvals[6]];
	  index[7] = cm[((int) (rindex1[7]+rindex2[7]+1) >> 1)+blockvals[7]];
	  index += row_size;
	  rindex1 += row_size;
	  rindex2 += row_size;
	  
	  index[0] = cm[((int) (rindex1[0]+rindex2[0]+1) >> 1)+blockvals[8]];
	  index[1] = cm[((int) (rindex1[1]+rindex2[1]+1) >> 1)+blockvals[9]];
	  index[2] = cm[((int) (rindex1[2]+rindex2[2]+1) >> 1)+blockvals[10]];
	  index[3] = cm[((int) (rindex1[3]+rindex2[3]+1) >> 1)+blockvals[11]];
	  index[4] = cm[((int) (rindex1[4]+rindex2[4]+1) >> 1)+blockvals[12]];
	  index[5] = cm[((int) (rindex1[5]+rindex2[5]+1) >> 1)+blockvals[13]];
	  index[6] = cm[((int) (rindex1[6]+rindex2[6]+1) >> 1)+blockvals[14]];
	  index[7] = cm[((int) (rindex1[7]+rindex2[7]+1) >> 1)+blockvals[15]];
	  blockvals += 16;
	  index += row_size;
	  rindex1 += row_size;
	  rindex2 += row_size;
	}
      } else { /* zflag */

	for (rr = 0; rr < 8; rr++) {
	  index[0] = (int) (rindex1[0] + rindex2[0] + 1) >> 1;
	  index[1] = (int) (rindex1[1] + rindex2[1] + 1) >> 1;
	  index[2] = (int) (rindex1[2] + rindex2[2] + 1) >> 1;
	  index[3] = (int) (rindex1[3] + rindex2[3] + 1) >> 1;
	  index[4] = (int) (rindex1[4] + rindex2[4] + 1) >> 1;
	  index[5] = (int) (rindex1[5] + rindex2[5] + 1) >> 1;
	  index[6] = (int) (rindex1[6] + rindex2[6] + 1) >> 1;
	  index[7] = (int) (rindex1[7] + rindex2[7] + 1) >> 1;
	  index += row_size;
	  rindex1 += row_size;
	  rindex2 += row_size;
	}
      }
    } else { /* qualityFlag on and both vectors are non-zero */
      rindex3 = rindex1 + vid_stream->right_half_for;
      rindex4 = rindex1 + (vid_stream->down_half_for * row_size);
      if (!zflag) {
	for (rr = 0; rr < 4; rr++) {
	  index[0]=cm[((int) (rindex1[0]+rindex2[0]+rindex3[0]+rindex4[0] + 2) >> 2) + blockvals[0]];
	  index[1]=cm[((int) (rindex1[1]+rindex2[1]+rindex3[1]+rindex4[1] + 2) >> 2) + blockvals[1]];
	  index[2]=cm[((int) (rindex1[2]+rindex2[2]+rindex3[2]+rindex4[2] + 2) >> 2) + blockvals[2]];
	  index[3]=cm[((int) (rindex1[3]+rindex2[3]+rindex3[3]+rindex4[3] + 2) >> 2) + blockvals[3]];
	  index[4]=cm[((int) (rindex1[4]+rindex2[4]+rindex3[4]+rindex4[4] + 2) >> 2) + blockvals[4]];
	  index[5]=cm[((int) (rindex1[5]+rindex2[5]+rindex3[5]+rindex4[5] + 2) >> 2) + blockvals[5]];
	  index[6]=cm[((int) (rindex1[6]+rindex2[6]+rindex3[6]+rindex4[6] + 2) >> 2) + blockvals[6]];
	  index[7]=cm[((int) (rindex1[7]+rindex2[7]+rindex3[7]+rindex4[7] + 2) >> 2) + blockvals[7]];
	  index +=row_size;
	  rindex1 += row_size;
	  rindex2 += row_size;
	  rindex3 += row_size;
	  rindex4 += row_size;
	  
	  index[0]=cm[((int) (rindex1[0]+rindex2[0]+rindex3[0]+rindex4[0] + 2) >> 2) + blockvals[8]];
	  index[1]=cm[((int) (rindex1[1]+rindex2[1]+rindex3[1]+rindex4[1] + 2) >> 2) + blockvals[9]];
	  index[2]=cm[((int) (rindex1[2]+rindex2[2]+rindex3[2]+rindex4[2] + 2) >> 2) + blockvals[10]];
	  index[3]=cm[((int) (rindex1[3]+rindex2[3]+rindex3[3]+rindex4[3] + 2) >> 2) + blockvals[11]];
	  index[4]=cm[((int) (rindex1[4]+rindex2[4]+rindex3[4]+rindex4[4] + 2) >> 2) + blockvals[12]];
	  index[5]=cm[((int) (rindex1[5]+rindex2[5]+rindex3[5]+rindex4[5] + 2) >> 2) + blockvals[13]];
	  index[6]=cm[((int) (rindex1[6]+rindex2[6]+rindex3[6]+rindex4[6] + 2) >> 2) + blockvals[14]];
	  index[7]=cm[((int) (rindex1[7]+rindex2[7]+rindex3[7]+rindex4[7] + 2) >> 2) + blockvals[15]];
	  blockvals += 16;
	  index += row_size;
	  rindex1 += row_size;
	  rindex2 += row_size;
	  rindex3 += row_size;
	  rindex4 += row_size;
	}
      } else { /* zflag */
	for (rr = 0; rr < 8; rr++) {
	  index[0] = (int) (rindex1[0] + rindex2[0] + rindex3[0] + rindex4[0] + 2) >> 2;
	  index[1] = (int) (rindex1[1] + rindex2[1] + rindex3[1] + rindex4[1] + 2) >> 2;
	  index[2] = (int) (rindex1[2] + rindex2[2] + rindex3[2] + rindex4[2] + 2) >> 2;
	  index[3] = (int) (rindex1[3] + rindex2[3] + rindex3[3] + rindex4[3] + 2) >> 2;
	  index[4] = (int) (rindex1[4] + rindex2[4] + rindex3[4] + rindex4[4] + 2) >> 2;
	  index[5] = (int) (rindex1[5] + rindex2[5] + rindex3[5] + rindex4[5] + 2) >> 2;
	  index[6] = (int) (rindex1[6] + rindex2[6] + rindex3[6] + rindex4[6] + 2) >> 2;
	  index[7] = (int) (rindex1[7] + rindex2[7] + rindex3[7] + rindex4[7] + 2) >> 2;
	  index += row_size;
	  rindex1 += row_size;
	  rindex2 += row_size;
	  rindex3 += row_size;
	  rindex4 += row_size;
	}
      }
    }
    
  }
  
  return true;
}


/*
 *--------------------------------------------------------------
 *
 * ReconBMBlock --
 *
 *	Reconstructs back predicted macroblocks.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

int Recon::ReconBMBlock(int bnum,
			int recon_right_back,
			int recon_down_back,
			int zflag) {
  int mb_row, mb_col, row, col, row_size, rr;
  unsigned char *dest, *future;
  int right_back, down_back, right_half_back, down_half_back;
  unsigned char *rindex1, *rindex2, *rindex3, *rindex4;
  unsigned char *index;
  short int *blockvals;
  int lumLength=(vid_stream->sequence.pictureArray->getCurrent())->getLumLength();
  int colorLength=(vid_stream->sequence.pictureArray->getCurrent())->getColorLength();
  
  int endFuture=0;
  int endDest=0;



  /* Calculate macroblock row and column from address. */
  if (vid_stream->sequence.mb_width <= 0) {
     cout << "mb_width <= 0"<<endl;
     return false;
  }

  mb_row = vid_stream->mblock.mb_address / vid_stream->sequence.mb_width;
  mb_col = vid_stream->mblock.mb_address % vid_stream->sequence.mb_width;

  /* If block is luminance block... */

  if (bnum < 4) {

    /* Calculate right_back, down_back motion vectors. */

    right_back = recon_right_back >> 1;
    down_back = recon_down_back >> 1;
    right_half_back = recon_right_back & 0x1;
    down_half_back = recon_down_back & 0x1;

    /* Set dest to luminance plane of current pict image. */

    dest = (vid_stream->sequence.pictureArray->getCurrent())->getLuminancePtr();
    endDest=lumLength;
    /*
     * If future frame exists, set future to luminance plane of future frame.
     */

    future = (vid_stream->sequence.pictureArray->getFuture())->getLuminancePtr();
    endFuture=lumLength;
    /* Establish row size. */

    row_size = vid_stream->sequence.mb_width << 4;

    /* Calculate row,col of upper left pixel in block. */

    row = mb_row << 4;
    col = mb_col << 4;
    if (bnum > 1)
      row += 8;
    if (bnum % 2)
      col += 8;


  }
  /* Otherwise, block is NOT luminance block, ... */

  else {

    /* Construct motion vectors. */

    recon_right_back /= 2;
    recon_down_back /= 2;
    right_back = recon_right_back >> 1;
    down_back = recon_down_back >> 1;
    right_half_back = recon_right_back & 0x1;
    down_half_back = recon_down_back & 0x1;

    /* Establish row size. */

    row_size = vid_stream->sequence.mb_width << 3;

    /* Calculate row,col of upper left pixel in block. */

    row = mb_row << 3;
    col = mb_col << 3;


    /* If block is Cr block... */
    /* They were switched earlier, so 5 is first - eyhung */

    if (bnum == 5) {

      /* Set dest to Cr plane of current pict image. */

      dest = (vid_stream->sequence.pictureArray->getCurrent())->getCrPtr();

      /*
       * If future frame exists, set future to Cr plane of future image.
       */

      future = (vid_stream->sequence.pictureArray->getFuture())->getCrPtr();
    }
    /* Otherwise, block is Cb block... */

    else {

      /* Set dest to Cb plane of current pict image. */

      dest = (vid_stream->sequence.pictureArray->getCurrent())->getCbPtr();

      /*
       * If future frame exists, set future to Cb plane of future frame.
       */

      future = (vid_stream->sequence.pictureArray->getFuture())->getCbPtr();
    }
    endDest=colorLength;
    endFuture=colorLength;
  }

  /* For each pixel in block do... */
  
  index = dest + (row * row_size) + col;
  rindex1 = future + (row + down_back) * row_size + col + right_back;

  blockvals = &(vid_stream->block.dct_recon[0][0]);

  if ((index+7*row_size+7 >= dest+endDest) || (index < dest)) {
    cout << "urg! last resort -9"<<endl;
    return false;
  }
  if ((rindex1+7*row_size+7 >= future+endFuture) || (rindex1 < future)) {
    cout << "urg! last resort -8"<<endl;
    return false;
  }

  if ((!right_half_back) && (!down_half_back)) {
    unsigned char *cm = cropTbl + MAX_NEG_CROP;
    if (!zflag)
      for (rr = 0; rr < 4; rr++) {
	index[0] = cm[(int) rindex1[0] + (int) blockvals[0]];
	index[1] = cm[(int) rindex1[1] + (int) blockvals[1]];
	index[2] = cm[(int) rindex1[2] + (int) blockvals[2]];
	index[3] = cm[(int) rindex1[3] + (int) blockvals[3]];
	index[4] = cm[(int) rindex1[4] + (int) blockvals[4]];
	index[5] = cm[(int) rindex1[5] + (int) blockvals[5]];
	index[6] = cm[(int) rindex1[6] + (int) blockvals[6]];
	index[7] = cm[(int) rindex1[7] + (int) blockvals[7]];
	index += row_size;
	rindex1 += row_size;
	
	index[0] = cm[(int) rindex1[0] + (int) blockvals[8]];
	index[1] = cm[(int) rindex1[1] + (int) blockvals[9]];
	index[2] = cm[(int) rindex1[2] + (int) blockvals[10]];
	index[3] = cm[(int) rindex1[3] + (int) blockvals[11]];
	index[4] = cm[(int) rindex1[4] + (int) blockvals[12]];
	index[5] = cm[(int) rindex1[5] + (int) blockvals[13]];
	index[6] = cm[(int) rindex1[6] + (int) blockvals[14]];
	index[7] = cm[(int) rindex1[7] + (int) blockvals[15]];
	blockvals += 16;
	index += row_size;
	rindex1 += row_size;
      }
    else {
      if (right_back & 0x1) {
	/* No alignment, use byte copy */
	for (rr = 0; rr < 4; rr++) {
	  
	  memcpy(index,rindex1,sizeof(char)*8);
	  
	  index += row_size;
	  rindex1 += row_size;
	  
	  memcpy(index,rindex1,sizeof(char)*8);
	  
	  index += row_size;
	  rindex1 += row_size;
	}
	
      } else if (right_back & 0x2) {
	/* Half-word bit aligned, use 16 bit copy */
	short *src = (short *)rindex1;
	short *dest = (short *)index;
	row_size >>= 1;
	for (rr = 0; rr < 4; rr++) {
	  
	  memcpy(dest,src,sizeof(short)*4);
	  
	  
	  dest += row_size;
	  src += row_size;
	  
	  memcpy(dest,src,sizeof(short)*4);
	  
	  dest += row_size;
	  src += row_size;
	}
      } else {
	/* Word aligned, use 32 bit copy */
	int *src = (int *)rindex1;
	int *dest = (int *)index;
	row_size >>= 2;
	for (rr = 0; rr < 4; rr++) {
	  dest[0] = src[0];
	  dest[1] = src[1];
	  dest += row_size;
	  src += row_size;
	  
	  dest[0] = src[0];
	  dest[1] = src[1];
	  dest += row_size;
	  src += row_size;
	}
      }
    }
  } else {
    unsigned char *cm = cropTbl + MAX_NEG_CROP;
    rindex2 = rindex1 + right_half_back + (down_half_back * row_size);
    if (!qualityFlag) {
      
      if (!zflag) {
	for (rr = 0; rr < 4; rr++) {
	  index[0]=cm[((int) (rindex1[0]+rindex2[0]+ 1) >> 1) + blockvals[0]];
	  index[1]=cm[((int) (rindex1[1]+rindex2[1]+ 1) >> 1) + blockvals[1]];
	  index[2]=cm[((int) (rindex1[2]+rindex2[2]+ 1) >> 1) + blockvals[2]];
	  index[3]=cm[((int) (rindex1[3]+rindex2[3]+ 1) >> 1) + blockvals[3]];
	  index[4]=cm[((int) (rindex1[4]+rindex2[4]+ 1) >> 1) + blockvals[4]];
	  index[5]=cm[((int) (rindex1[5]+rindex2[5]+ 1) >> 1) + blockvals[5]];
	  index[6]=cm[((int) (rindex1[6]+rindex2[6]+ 1) >> 1) + blockvals[6]];
	  index[7]=cm[((int) (rindex1[7]+rindex2[7]+ 1) >> 1) + blockvals[7]];
	  index += row_size;
	  rindex1 += row_size;
	  rindex2 += row_size;
	  
	  index[0]=cm[((int) (rindex1[0]+rindex2[0]+ 1) >> 1) + blockvals[8]];
	  index[1]=cm[((int) (rindex1[1]+rindex2[1]+ 1) >> 1) + blockvals[9]];
	  index[2]=cm[((int) (rindex1[2]+rindex2[2]+ 1) >> 1) + blockvals[10]];
	  index[3]=cm[((int) (rindex1[3]+rindex2[3]+ 1) >> 1) + blockvals[11]];
	  index[4]=cm[((int) (rindex1[4]+rindex2[4]+ 1) >> 1) + blockvals[12]];
	  index[5]=cm[((int) (rindex1[5]+rindex2[5]+ 1) >> 1) + blockvals[13]];
	  index[6]=cm[((int) (rindex1[6]+rindex2[6]+ 1) >> 1) + blockvals[14]];
	  index[7]=cm[((int) (rindex1[7]+rindex2[7]+ 1) >> 1) + blockvals[15]];
	  blockvals += 16;
	  index += row_size;
	  rindex1 += row_size;
	  rindex2 += row_size;
	}
      } else { /* zflag */
	for (rr = 0; rr < 8; rr++) {
	  index[0] = (int) (rindex1[0] + rindex2[0] + 1) >> 1;
	  index[1] = (int) (rindex1[1] + rindex2[1] + 1) >> 1;
	  index[2] = (int) (rindex1[2] + rindex2[2] + 1) >> 1;
	  index[3] = (int) (rindex1[3] + rindex2[3] + 1) >> 1;
	  index[4] = (int) (rindex1[4] + rindex2[4] + 1) >> 1;
	  index[5] = (int) (rindex1[5] + rindex2[5] + 1) >> 1;
	  index[6] = (int) (rindex1[6] + rindex2[6] + 1) >> 1;
	  index[7] = (int) (rindex1[7] + rindex2[7] + 1) >> 1;
	  index += row_size;
	  rindex1 += row_size;
	  rindex2 += row_size;
	}
      }
    } else { /* qualityFlag on */
      rindex3 = rindex1 + right_half_back;
      rindex4 = rindex1 + (down_half_back * row_size);
      if (!zflag) {
	for (rr = 0; rr < 4; rr++) {
	  index[0] = cm[((int) (rindex1[0] + rindex2[0] + rindex3[0] + rindex4[0] + 2) >> 2) + blockvals[0]];
	  index[1] = cm[((int) (rindex1[1] + rindex2[1] + rindex3[1] + rindex4[1] + 2) >> 2) + blockvals[1]];
	  index[2] = cm[((int) (rindex1[2] + rindex2[2] + rindex3[2] + rindex4[2] + 2) >> 2) + blockvals[2]];
	  index[3] = cm[((int) (rindex1[3] + rindex2[3] + rindex3[3] + rindex4[3] + 2) >> 2) + blockvals[3]];
	  index[4] = cm[((int) (rindex1[4] + rindex2[4] + rindex3[4] + rindex4[4] + 2) >> 2) + blockvals[4]];
	  index[5] = cm[((int) (rindex1[5] + rindex2[5] + rindex3[5] + rindex4[5] + 2) >> 2) + blockvals[5]];
	  index[6] = cm[((int) (rindex1[6] + rindex2[6] + rindex3[6] + rindex4[6] + 2) >> 2) + blockvals[6]];
	  index[7] = cm[((int) (rindex1[7] + rindex2[7] + rindex3[7] + rindex4[7] + 2) >> 2) + blockvals[7]];
	  index += row_size;
	  rindex1 += row_size;
	  rindex2 += row_size;
	  rindex3 += row_size;
	  rindex4 += row_size;
	  
	  index[0] = cm[((int) (rindex1[0] + rindex2[0] + rindex3[0] + rindex4[0] + 2) >> 2) + blockvals[8]];
	  index[1] = cm[((int) (rindex1[1] + rindex2[1] + rindex3[1] + rindex4[1] + 2) >> 2) + blockvals[9]];
	  index[2] = cm[((int) (rindex1[2] + rindex2[2] + rindex3[2] + rindex4[2] + 2) >> 2) + blockvals[10]];
	  index[3] = cm[((int) (rindex1[3] + rindex2[3] + rindex3[3] + rindex4[3] + 2) >> 2) + blockvals[11]];
	  index[4] = cm[((int) (rindex1[4] + rindex2[4] + rindex3[4] + rindex4[4] + 2) >> 2) + blockvals[12]];
	  index[5] = cm[((int) (rindex1[5] + rindex2[5] + rindex3[5] + rindex4[5] + 2) >> 2) + blockvals[13]];
	  index[6] = cm[((int) (rindex1[6] + rindex2[6] + rindex3[6] + rindex4[6] + 2) >> 2) + blockvals[14]];
	  index[7] = cm[((int) (rindex1[7] + rindex2[7] + rindex3[7] + rindex4[7] + 2) >> 2) + blockvals[15]];
	  blockvals += 16;
	  index += row_size;
	  rindex1 += row_size;
	  rindex2 += row_size;
	  rindex3 += row_size;
	  rindex4 += row_size;
	}
      } else { /* zflag */
	for (rr = 0; rr < 8; rr++) {
	  index[0] = (int) (rindex1[0] + rindex2[0] + rindex3[0] + rindex4[0] + 2) >> 2;
	  index[1] = (int) (rindex1[1] + rindex2[1] + rindex3[1] + rindex4[1] + 2) >> 2;
	  index[2] = (int) (rindex1[2] + rindex2[2] + rindex3[2] + rindex4[2] + 2) >> 2;
	  index[3] = (int) (rindex1[3] + rindex2[3] + rindex3[3] + rindex4[3] + 2) >> 2;
	  index[4] = (int) (rindex1[4] + rindex2[4] + rindex3[4] + rindex4[4] + 2) >> 2;
	  index[5] = (int) (rindex1[5] + rindex2[5] + rindex3[5] + rindex4[5] + 2) >> 2;
	  index[6] = (int) (rindex1[6] + rindex2[6] + rindex3[6] + rindex4[6] + 2) >> 2;
	  index[7] = (int) (rindex1[7] + rindex2[7] + rindex3[7] + rindex4[7] + 2) >> 2;
	  index += row_size;
	  rindex1 += row_size;
	  rindex2 += row_size;
	  rindex3 += row_size;
	  rindex4 += row_size;
	}
      }
    }
    
  }
  return true;
}


/*
 *--------------------------------------------------------------
 *
 * ReconBiMBlock --
 *
 *	Reconstructs bidirectionally predicted macroblocks.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */
int Recon::ReconBiMBlock(int bnum,
			 int recon_right_for,
			 int recon_down_for,
			 int recon_right_back,
			 int recon_down_back,
			 int zflag) {
  int mb_row, mb_col, row, col, row_size, rr;
  unsigned char *dest, *past=NULL, *future=NULL;
  int right_for, down_for, right_half_for, down_half_for;
  int right_back, down_back, right_half_back, down_half_back;
  unsigned char *index, *rindex1, *bindex1;
  short int *blockvals;
  int forw_row_start, back_row_start, forw_col_start, back_col_start;
  int lumLength=(vid_stream->sequence.pictureArray->getCurrent())->getLumLength();
  int colorLength=(vid_stream->sequence.pictureArray->getCurrent())->getColorLength();
  int endPast=0;
  int endFuture=0;



  /* Calculate macroblock row and column from address. */
  if (vid_stream->sequence.mb_width <= 0) {
     cout << "mb_width <= 0"<<endl;
     return false;
  }

  mb_row = vid_stream->mblock.mb_address / vid_stream->sequence.mb_width;
  mb_col = vid_stream->mblock.mb_address % vid_stream->sequence.mb_width;

  /* If block is luminance block... */

  if (bnum < 4) {

    /*
     * Calculate right_for, down_for, right_half_for, down_half_for,
     * right_back, down_bakc, right_half_back, and down_half_back, motion
     * vectors.
     */

    right_for = recon_right_for >> 1;
    down_for = recon_down_for >> 1;
    right_half_for = recon_right_for & 0x1;
    down_half_for = recon_down_for & 0x1;

    right_back = recon_right_back >> 1;
    down_back = recon_down_back >> 1;
    right_half_back = recon_right_back & 0x1;
    down_half_back = recon_down_back & 0x1;

    /* Set dest to luminance plane of current pict image. */

    dest = (vid_stream->sequence.pictureArray->getCurrent())->getLuminancePtr();

    /* If past frame exists, set past to luminance plane of past frame. */

    past = (vid_stream->sequence.pictureArray->getPast())->getLuminancePtr();
    endPast=lumLength;

    /*
     * If future frame exists, set future to luminance plane of future frame.
     */

    future = (vid_stream->sequence.pictureArray->getFuture())->getLuminancePtr();
    endFuture=lumLength;

    /* Establish row size. */

    row_size = (vid_stream->sequence.mb_width << 4);

    /* Calculate row,col of upper left pixel in block. */

    row = (mb_row << 4);
    col = (mb_col << 4);
    if (bnum > 1)
      row += 8;
    if (bnum & 0x01)
      col += 8;

    forw_col_start = col + right_for;
    forw_row_start = row + down_for;

    back_col_start = col + right_back;
    back_row_start = row + down_back;

  }
  /* Otherwise, block is NOT luminance block, ... */

  else {

    /* Construct motion vectors. */
    recon_right_for /= 2;
    recon_down_for /= 2;
    right_for = recon_right_for >> 1;
    down_for = recon_down_for >> 1;
    right_half_for = recon_right_for & 0x1;
    down_half_for = recon_down_for & 0x1;

    recon_right_back /= 2;
    recon_down_back /= 2;
    right_back = recon_right_back >> 1;
    down_back = recon_down_back >> 1;
    right_half_back = recon_right_back & 0x1;
    down_half_back = recon_down_back & 0x1;

    /* Establish row size. */

    row_size = (vid_stream->sequence.mb_width << 3);

    /* Calculate row,col of upper left pixel in block. */

    row = (mb_row << 3);
    col = (mb_col << 3);

    forw_col_start = col + right_for;
    forw_row_start = row + down_for;

    back_col_start = col + right_back;
    back_row_start = row + down_back;

    
    /* If block is Cr block... */
	/* Switched earlier, so we test Cr first - eyhung */

    if (bnum == 5) {
      /* Set dest to Cr plane of current pict image. */

      dest = (vid_stream->sequence.pictureArray->getCurrent())->getCrPtr();

      /* If past frame exists, set past to Cr plane of past image. */

      past = (vid_stream->sequence.pictureArray->getPast())->getCrPtr();
      endPast=colorLength;
      /*
       * If future frame exists, set future to Cr plane of future image.
       */

      future = (vid_stream->sequence.pictureArray->getFuture())->getCrPtr();
      endFuture=colorLength;
    }
    /* Otherwise, block is Cb block... */

    else {
      /* Set dest to Cb plane of current pict image. */
      dest = (vid_stream->sequence.pictureArray->getCurrent())->getCbPtr();

      /* If past frame exists, set past to Cb plane of past frame. */

      past = (vid_stream->sequence.pictureArray->getPast())->getCbPtr();
      endPast=colorLength;
      
      /*
       * If future frame exists, set future to Cb plane of future frame.
       */

      future = (vid_stream->sequence.pictureArray->getFuture())->getCbPtr();
      endFuture=colorLength;
    }
  }

  /* For each pixel in block... */

  index = dest + (row * row_size) + col;

  rindex1 = past + forw_row_start  * row_size + forw_col_start;

  bindex1 = future + back_row_start * row_size + back_col_start;

  blockvals = (short int *) &(vid_stream->block.dct_recon[0][0]);
  if ((rindex1+7*row_size+7 >= past+endPast) || (rindex1 < past))  {
    cout << "urg! last resort -1"<<endl;
    return false;
  }
  if ((bindex1+7*row_size+7 >= future+endFuture) || (bindex1 < future)) {
    cout << "urg! last resort -2"<<endl;
    return false;
  }


  {
  unsigned char *cm = cropTbl + MAX_NEG_CROP;
  if (!zflag) {
    for (rr = 0; rr < 4; rr++) {
      index[0] = cm[((int) (rindex1[0] + bindex1[0]) >> 1) + blockvals[0]];
      index[1] = cm[((int) (rindex1[1] + bindex1[1]) >> 1) + blockvals[1]];
      index[2] = cm[((int) (rindex1[2] + bindex1[2]) >> 1) + blockvals[2]];
      index[3] = cm[((int) (rindex1[3] + bindex1[3]) >> 1) + blockvals[3]];
      index[4] = cm[((int) (rindex1[4] + bindex1[4]) >> 1) + blockvals[4]];
      index[5] = cm[((int) (rindex1[5] + bindex1[5]) >> 1) + blockvals[5]];
      index[6] = cm[((int) (rindex1[6] + bindex1[6]) >> 1) + blockvals[6]];
      index[7] = cm[((int) (rindex1[7] + bindex1[7]) >> 1) + blockvals[7]];
      index += row_size;
      rindex1 += row_size;
      bindex1 += row_size;

      index[0] = cm[((int) (rindex1[0] + bindex1[0]) >> 1) + blockvals[8]];
      index[1] = cm[((int) (rindex1[1] + bindex1[1]) >> 1) + blockvals[9]];
      index[2] = cm[((int) (rindex1[2] + bindex1[2]) >> 1) + blockvals[10]];
      index[3] = cm[((int) (rindex1[3] + bindex1[3]) >> 1) + blockvals[11]];
      index[4] = cm[((int) (rindex1[4] + bindex1[4]) >> 1) + blockvals[12]];
      index[5] = cm[((int) (rindex1[5] + bindex1[5]) >> 1) + blockvals[13]];
      index[6] = cm[((int) (rindex1[6] + bindex1[6]) >> 1) + blockvals[14]];
      index[7] = cm[((int) (rindex1[7] + bindex1[7]) >> 1) + blockvals[15]];
      blockvals += 16;
      index += row_size;
      rindex1 += row_size;
      bindex1 += row_size;
    }
  }
  else {
    for (rr = 0; rr < 4; rr++) {

      index[0] = (int) (rindex1[0] + bindex1[0]) / 2;
      index[1] = (int) (rindex1[1] + bindex1[1]) / 2;
      index[2] = (int) (rindex1[2] + bindex1[2]) / 2;
      index[3] = (int) (rindex1[3] + bindex1[3]) / 2;
      index[4] = (int) (rindex1[4] + bindex1[4]) / 2;
      index[5] = (int) (rindex1[5] + bindex1[5]) / 2;
      index[6] = (int) (rindex1[6] + bindex1[6]) / 2;
      index[7] = (int) (rindex1[7] + bindex1[7]) / 2;
      index += row_size;
      rindex1 += row_size;
      bindex1 += row_size;


      index[0] = (int) (rindex1[0] + bindex1[0]) / 2;
      index[1] = (int) (rindex1[1] + bindex1[1]) / 2;
      index[2] = (int) (rindex1[2] + bindex1[2]) / 2;
      index[3] = (int) (rindex1[3] + bindex1[3]) / 2;
      index[4] = (int) (rindex1[4] + bindex1[4]) / 2;
      index[5] = (int) (rindex1[5] + bindex1[5]) / 2;
      index[6] = (int) (rindex1[6] + bindex1[6]) / 2;
      index[7] = (int) (rindex1[7] + bindex1[7]) / 2;
      index += row_size;
      rindex1 += row_size;
      bindex1 += row_size;
    }
  }
  }
  return true;
}


int Recon::reconstruct(int& recon_right_for,
		       int& recon_down_for,
		       int& recon_right_back,
		       int& recon_down_back,
		       int& mb_motion_forw,
		       int& mb_motion_back) {
  int mask, i;
  int zero_block_flag;
  int back=true;
  static int cnt=0;

  for (mask = 32, i = 0; i < 6; mask >>= 1, i++) {

     
    /* If block exists... */
    if ((vid_stream->mblock.mb_intra) || (vid_stream->mblock.cbp & mask)) {
      zero_block_flag = 0;
      (vid_stream->decoderClass)->ParseReconBlock(i);
    } else {
      zero_block_flag = 1;
    }
    
    // If macroblock is intra coded... 

    if (vid_stream->mblock.mb_intra) {
      back&=ReconIMBlock(i);
    } else if (mb_motion_forw && mb_motion_back) {
      back&=ReconBiMBlock(i,recon_right_for,
			 recon_down_for,recon_right_back,
			 recon_down_back,zero_block_flag);
    } else if (mb_motion_forw || (vid_stream->picture.code_type == P_TYPE)) {
      back&=ReconPMBlock(i,recon_right_for,
			 recon_down_for,zero_block_flag);
    } else if (mb_motion_back) {
      back&=ReconBMBlock(i,recon_right_back,
			 recon_down_back,zero_block_flag);
    } else {
      //cout << "nothing"<<endl;
    }
    
  }

  return true;
}
