/* 
 * video.c --
 *
 *      This file contains C code that implements the video decoder model.
 *
 */


#include "video.h"

#include "deMux.h"
#include "decoderClass.h"
#include "recon.h"
#include "packet.h"
#include "motionVector.h"
#include "slice.h"
#include "proto.h"
#include "util.h"
#include "skipped.h"
#include "../inputPlugin/inputPlugin.h"
#include "pictureArray.h"
#include "../inputPlugin/timeStamp.h"
#include "sequenceParse.h"


/* Declarations of functions. */


static int ParseSeqHead(VidStream* vid_stream);
static int ParseGOP(VidStream* vid_stream);
static int ParsePicture(VidStream* vid_stream);
static int ParseMacroBlock(VidStream*);
static int ParseSlice(VidStream* vid_stream);
static int ResetVidStreamStruc(VidStream* vid_stream);









/*
 *--------------------------------------------------------------
 *
 * NewVidStream --
 *
 *	Allocates and initializes a VidStream structure. Takes
 *      as parameter requested size for buffer length.
 *
 * Results:
 *	A pointer to the new VidStream structure.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

VidStream* NewVidStream(MpegPlayBitWindow* bitwindow,SequenceParse* output) {
  VidStream *newVidStream;

  /* Check for legal buffer length. */
  cout << "init_tables"<<endl;
  init_tables();


  /* Allocate memory for new structure. */

  newVidStream = (VidStream *) malloc(sizeof(VidStream));

  /* Initialize pointers to extension and user data. */

  newVidStream->group.ext_data = newVidStream->group.user_data =
    newVidStream->picture.extra_info = newVidStream->picture.user_data =
    newVidStream->picture.ext_data = NULL;
  
  SequenceParse::init(&(newVidStream->sequence),output->getSequence()->output);
  SequenceParse::copy(&(newVidStream->sequence),output->getSequence());


  /* Initialize fields that used to be global */

  newVidStream->bitwindow=bitwindow;
  newVidStream->decoderClass=new DecoderClass(newVidStream);
  newVidStream->recon=new Recon(newVidStream);
  newVidStream->packet=new Packet();
  newVidStream->motionVector=new MotionVector(newVidStream);
  newVidStream->slice=new Slice(newVidStream);
  newVidStream->startOfPicStamp=new TimeStamp();

  newVidStream->syncState=SYNC_TO_CLOSED_GOP;

  /* Return structure. */
  // fill buffer
  
  return newVidStream;
}



/*
 *--------------------------------------------------------------
 *
 * ResetVidStream --
 *
 *	Re-initializes a VidStream structure. Takes
 *      as parameter a pointer to the VidStream to reset.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

void ResetVidStream(VidStream* vid_stream) {




  (vid_stream->bitwindow)->resetBuffer();


}


/*
 *--------------------------------------------------------------
 *
 * DestroyVidStream --
 *
 *	Deallocates a VidStream structure.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
void DestroyVidStream(VidStream* astream) {

  if (astream->sequence.pictureArray != NULL)
    free(astream->sequence.pictureArray);

  if (astream->sequence.ext_data != NULL)
    free(astream->sequence.ext_data);

  if (astream->sequence.user_data != NULL)
    free(astream->sequence.user_data);

  if (astream->group.ext_data != NULL)
    free(astream->group.ext_data);

  if (astream->group.user_data != NULL)
    free(astream->group.user_data);

  if (astream->picture.extra_info != NULL)
    free(astream->picture.extra_info);

  if (astream->picture.ext_data != NULL)
    free(astream->picture.ext_data);

  if (astream->picture.user_data != NULL)
    free(astream->picture.user_data);

  delete astream->decoderClass;
  delete astream->recon;
  delete astream->packet;
  delete astream->motionVector;
  delete astream->slice;

}







/*
 *--------------------------------------------------------------
 *
 * mpegVidRsrc --
 *
 *      Parses bit stream until MB_QUANTUM number of
 *      macroblocks have been decoded or current slice or
 *      picture ends, whichever comes first. If the start
 *      of a frame is encountered, the frame is time stamped
 *      with the value passed in time_stamp. If the value
 *      passed in buffer is not null, the video stream buffer
 *      is set to buffer and the length of the buffer is
 *      expected in value passed in through length. The current
 *      video stream is set to vid_stream. If vid_stream
 *      is passed as NULL, a new VidStream structure is created
 *      and initialized and used as the current video stream.
 *
 * Results:
 *      A pointer to the video stream structure used.
 *
 * Side effects:
 *      Bit stream is irreversibly parsed. If a picture is completed,
 *      a function is called to display the frame at the correct time.
 *
 *--------------------------------------------------------------
 */

int mpegVidRsrc(VidStream* vid_stream,int first) {
  int back=true;

  unsigned int data;
  int i, status;
  data=0;
  TimeStamp* stamp;

  /* If vid_stream is null, create new VidStream structure. */

  if (vid_stream == NULL) {
    return false;
  }

  /*
   * If called for the first time, find start code, make sure it is a
   * sequence start code.
   */

  if (first) {
    int lfound=false;
    if((vid_stream->bitwindow)->firstSync()==false) {
      cout << "eof"<<endl;
      exit(0);
    }
    cout << "first seek start -s"<<endl;
    while((vid_stream->bitwindow)->isEof()==false) {
      cout << "first seek start -s"<<endl;
      lfound=DeMux::skip_to_start_code(vid_stream->bitwindow,SEQ_START_CODE);
      if (lfound) {
	break;
      }
    }
    if (lfound == false) {
      printf("This is not an MPEG video stream. (%x)\n",data);
      DestroyVidStream(vid_stream);
      exit(0);
    }
    cout << "first seek start -e"<<endl;
  }



  /* Get next 32 bits (size of start codes). */

  data=(vid_stream->bitwindow)->showBits(32);


  /*
   * Process according to start code (or parse macroblock if not a start code
   * at all).
   */
  switch (data) {
    /*
  case PACK_START_CODE:
  case SYSTEM_HEADER_START_CODE:  
    cout << "Packet in Loop **************"<<endl;
    (vid_stream->bitwindow)->flushBits(32);
    (vid_stream->packet)->read_sys(data,vid_stream->bufferReader);
    */
  case SEQ_END_CODE:
  case ISO_11172_END_CODE:   /*  handle ISO_11172_END_CODE too */

    /* Display last frame. */

    // removed!

    /* Sequence done. Do the right thing. For right now, exit. */


    cout << "******** flushin end code"<<endl;
    (vid_stream->bitwindow)->flushBits(32);
    goto done;
    break;
  case EXT_START_CODE:
    cout << "found EXT_START_CODE skipping"<<endl;
    goto error;

  case SEQ_START_CODE:
    
    /* Sequence start code. Parse sequence header. */
    if (ParseSeqHead(vid_stream) != PARSE_OK) {
      printf("SEQ_START_CODE 1-error\n");
      goto error;
    }
    goto done;

  case GOP_START_CODE:
    /* Group of Pictures start code. Parse gop header. */
    if (ParseGOP(vid_stream) != PARSE_OK) {
      printf("GOP_START_CODE 1-error\n");
      goto error;
    }
    goto done;

  case PICTURE_START_CODE:

    /* Picture start code. Parse picture header and first slice header. */

    

    status = ParsePicture(vid_stream);


    if (status == SKIP_PICTURE) {
      DeMux::next_start_code(vid_stream->bitwindow);
      while (!(Util::next_bits(32,PICTURE_START_CODE,vid_stream->bitwindow))) {
        if (Util::next_bits(32, GOP_START_CODE,vid_stream->bitwindow))
          break;
        else if (Util::next_bits(32, SEQ_END_CODE,vid_stream->bitwindow))
          break;
        (vid_stream->bitwindow)->flushBits(24);
	DeMux::next_start_code(vid_stream->bitwindow);
      }
      return true;
    } else if (status != PARSE_OK) {
      printf("PICTURE_START_CODE 1-error\n");
      goto error;
    }
    // parse ok
    if (ParseSlice(vid_stream) != PARSE_OK) {
      printf("PICTURE_START_CODE 2-error\n");
      goto error;
    }
    break;

  case SEQUENCE_ERROR_CODE:
    (vid_stream->bitwindow)->flushBits(32);
    DeMux::next_start_code(vid_stream->bitwindow);

    goto done;

    
  default:

    /* Check for slice start code. */
    if ((data >= SLICE_MIN_START_CODE) && (data <= SLICE_MAX_START_CODE)) {

      /* Slice start code. Parse slice header. */
      if (ParseSlice(vid_stream) != PARSE_OK) {
	printf("default 1-error\n");
        goto error;
      }
    }
    break;
  }

  /* Parse next MB_QUANTUM macroblocks. */
  for (i = 0; i < MB_QUANTUM; i++) {

    /* Check to see if actually a startcode and not a macroblock. */
    data=(vid_stream->bitwindow)->showBits(32);
    if (!(Util::next_bits(23, 0x00000000,vid_stream->bitwindow))) {
      /* Not start code. Parse Macroblock. */
      if (ParseMacroBlock(vid_stream) != PARSE_OK) {
	cout << "ParseMacroBlock error"<<endl;
        goto error;
      }
    } else {
      /* Not macroblock, actually start code. Get start code. */
      DeMux::next_start_code(vid_stream->bitwindow);

      /*
       * If start code is outside range of slice start codes, frame is
       * complete, display frame.
       */
      data=(vid_stream->bitwindow)->showBits(32);

      if (((data < SLICE_MIN_START_CODE) || (data > SLICE_MAX_START_CODE)) &&
	  (data != SEQUENCE_ERROR_CODE)) {
	doPictureDisplay(vid_stream);
      }
      goto done;
    }
  }
  /* Check if we just finished a picture on the MB_QUANTUM macroblock */
  if (Util::next_bits(23, 0x00000000,vid_stream->bitwindow)) {
    DeMux::next_start_code(vid_stream->bitwindow);

    data=(vid_stream->bitwindow)->showBits(32);
    if ((data < SLICE_MIN_START_CODE) || (data > SLICE_MAX_START_CODE)) {
      doPictureDisplay(vid_stream);
    }
  }

  /* Return pointer to video stream structure. */

  goto done;

error:
  printf("Error!!!!\n");
  init_tables();
  vid_stream->syncState=SYNC_HAS_CLOSED_GOP;

  back=false;
  SequenceParse::init_quanttables(&(vid_stream->sequence));
  goto done;

done:
  return back;

}






static int ParseSeqHead(VidStream* vid_stream) {
  int back;

  back=SequenceParse::parseSeq(vid_stream->bitwindow,&(vid_stream->sequence));
  return back;

}



/*
 *--------------------------------------------------------------
 *
 * ParseGOP --
 *
 *      Parses of group of pictures header from bit stream
 *      associated with vid_stream.
 *
 * Results:
 *      Values in gop header placed into video stream structure.
 *
 * Side effects:
 *      Bit stream irreversibly parsed.
 *
 *--------------------------------------------------------------
 */

static int ParseGOP(VidStream* vid_stream) {
  unsigned int data;

  /* Flush group of pictures start code. */

  (vid_stream->bitwindow)->flushBits(32);

  /* Parse off drop frame flag. */

  data=(vid_stream->bitwindow)->getBits(1);

  if (data) {
    vid_stream->group.drop_flag = true;
  } else
    vid_stream->group.drop_flag = false;

  /* Parse off hour component of time code. */

  data=(vid_stream->bitwindow)->getBits(5);

  vid_stream->group.tc_hours = data;

  /* Parse off minute component of time code. */

  data=(vid_stream->bitwindow)->getBits(6);
  vid_stream->group.tc_minutes = data;


  /* Flush marker bit. */

  (vid_stream->bitwindow)->flushBits(1);

  /* Parse off second component of time code. */

  data=(vid_stream->bitwindow)->getBits(6);
  vid_stream->group.tc_seconds = data;

  
  /* Parse off picture count component of time code. */

  data=(vid_stream->bitwindow)->getBits(6);
  vid_stream->group.tc_pictures = data;

  /* Parse off closed gop and broken link flags. */
  data=(vid_stream->bitwindow)->getBits(2);
  if (data > 1) {
    vid_stream->group.closed_gop = true;
    if (data > 2) {
      vid_stream->group.broken_link = true;
    } else
      vid_stream->group.broken_link = false;
  } else {
    vid_stream->group.closed_gop = false;
    if (data) {
      vid_stream->group.broken_link = true;
    } else
      vid_stream->group.broken_link = false;
  }

  if (vid_stream->syncState==SYNC_TO_CLOSED_GOP) {
    vid_stream->syncState=SYNC_HAS_CLOSED_GOP;
  }



  /* Goto next start code. */
  DeMux::next_start_code(vid_stream->bitwindow);

  /*
   * If next start code is extension/user start code, 
   * parse off extension data.
   */

  Util::check_ext_data(vid_stream->bitwindow,vid_stream->group.ext_data);
  Util::check_user_data(vid_stream->bitwindow,vid_stream->group.user_data);

  return PARSE_OK;
}



/*
 *--------------------------------------------------------------
 *
 * ParsePicture --
 *
 *      Parses picture header. Marks picture to be presented
 *      at particular time given a time stamp.
 *
 * Results:
 *      Values from picture header put into video stream structure.
 *
 * Side effects:
 *      Bit stream irreversibly parsed.
 *
 *--------------------------------------------------------------
 */

static int ParsePicture(VidStream* vid_stream) {
  TimeStamp* stamp;

  unsigned int data;

  /* Flush header start code. */
  (vid_stream->bitwindow)->flushBits(32);

  /* Parse off temporal reference. */
  data=(vid_stream->bitwindow)->getBits(10);
  vid_stream->picture.temp_ref = data;

  /* Parse of picture type. */
  data=(vid_stream->bitwindow)->getBits(3);

  vid_stream->picture.code_type = data;
  

  // get timestamp and insert it to picture
  stamp=(vid_stream->bitwindow)->getCurrentTimeStamp();
  stamp->copyTo(vid_stream->startOfPicStamp);

  


  /* Parse off vbv buffer delay value. */
  data=(vid_stream->bitwindow)->getBits(16);

  vid_stream->picture.vbv_delay = data;

  /* If P or B type frame... */

  if ((vid_stream->picture.code_type == P_TYPE) || 
      (vid_stream->picture.code_type == B_TYPE)) {

    /* Parse off forward vector full pixel flag. */
    data=(vid_stream->bitwindow)->getBits(1);

    if (data) {
      vid_stream->picture.full_pel_forw_vector = true;
    } else {
      vid_stream->picture.full_pel_forw_vector = false;
    }

    /* Parse of forw_r_code. */
    data=(vid_stream->bitwindow)->getBits(3);


    /* Decode forw_r_code into forw_r_size and forw_f. */

    vid_stream->picture.forw_r_size = data - 1;
    vid_stream->picture.forw_f = (1 << vid_stream->picture.forw_r_size);
  }
  /* If B type frame... */

  if (vid_stream->picture.code_type == B_TYPE) {

    /* Parse off back vector full pixel flag. */
    data=(vid_stream->bitwindow)->getBits(1);

    if (data)
      vid_stream->picture.full_pel_back_vector = true;
    else
      vid_stream->picture.full_pel_back_vector = false;

    /* Parse off back_r_code. */
    data=(vid_stream->bitwindow)->getBits(3);


    /* Decode back_r_code into back_r_size and back_f. */

    vid_stream->picture.back_r_size = data - 1;
    vid_stream->picture.back_f = (1 << vid_stream->picture.back_r_size);
  }
  /* Get extra bit picture info. */

  if (vid_stream->picture.extra_info != NULL) {
    free(vid_stream->picture.extra_info);
    vid_stream->picture.extra_info = NULL;
  }
  vid_stream->picture.extra_info=
    Util::get_extra_bit_info(vid_stream->bitwindow);

  /* Goto next start code. */
  DeMux::next_start_code(vid_stream->bitwindow);

  /*
   * If next start code is extension/user start code, 
   * parse off extension data.
   */

  Util::check_ext_data(vid_stream->bitwindow,vid_stream->picture.ext_data);
  Util::check_user_data(vid_stream->bitwindow,vid_stream->picture.user_data);

  /* Find a pict image structure in ring buffer not currently locked. */
  /* Set current pict image structure to the one just found in ring. */
  



  /* Reset past macroblock address field. */

  vid_stream->mblock.past_mb_addr = -1;

  return PARSE_OK;
}



/*
 *--------------------------------------------------------------
 *
 * ParseSlice --
 *
 *      Parses off slice header.
 *
 * Results:
 *      Values found in slice header put into video stream structure.
 *
 * Side effects:
 *      Bit stream irreversibly parsed.
 *
 *--------------------------------------------------------------
 */

static int ParseSlice(VidStream* vid_stream) {

  
  (vid_stream->slice)->parseSlice();
  ResetVidStreamStruc(vid_stream);
  return PARSE_OK;
}


void videoPrint(VidStream *vid_stream) {
  cout << "h_size:"<<vid_stream->sequence.h_size<<endl;
  cout << "v_size:"<<vid_stream->sequence.v_size<<endl;
  cout << "mb_height:"<<vid_stream->sequence.mb_height<<endl;
  cout << "mb_width:"<<vid_stream->sequence.mb_width<<endl;
  cout << "mb_size:"<<vid_stream->sequence.mb_size<<endl;
  cout << "aspect_ratio:"<<vid_stream->sequence.aspect_ratio<<endl;
  cout << "bit_rate:"<<vid_stream->sequence.bit_rate<<endl;
  cout << "vbv_buffer_size:"<<vid_stream->sequence.vbv_buffer_size<<endl;
  cout << "const_param_flag:"<<vid_stream->sequence.const_param_flag<<endl;
}

/*
 *--------------------------------------------------------------
 *
 * ParseMacroBlock --
 *
 *      Parseoff macroblock. Reconstructs DCT values. Applies
 *      inverse DCT, reconstructs motion vectors, calculates and
 *      set pixel values for macroblock in current pict image
 *      structure.
 *
 * Results:
 *      Here's where everything really happens. Welcome to the
 *      heart of darkness.
 *
 * Side effects:
 *      Bit stream irreversibly parsed off.
 *
 *--------------------------------------------------------------
 */

static int ParseMacroBlock(VidStream* vid_stream) {
  unsigned int data;
  int recon_right_for, recon_down_for, recon_right_back,
      recon_down_back;
  int mb_quant = 0, mb_motion_forw = 0, mb_motion_back = 0, 
    mb_pattern = 0;

  vid_stream->mblock.mb_address+=(vid_stream->decoderClass)->ParseStuffing();

    
      
  
  if (vid_stream->mblock.mb_address > vid_stream->sequence.mb_size) {

    printf("ParseMacroBlock: SKIP_TO_START_CODE\n");
    printf("vid_stream->mblock.mb_address %d\n",vid_stream->mblock.mb_address);
    printf("mb_height*mb_width-1:%d\n",(vid_stream->sequence.mb_height *
				    vid_stream->sequence.mb_width - 1));
    return SKIP_TO_START_CODE;
  }
  
  /*
   * If macroblocks have been skipped, process skipped macroblocks.
   */

  if (vid_stream->mblock.mb_address - vid_stream->mblock.past_mb_addr > 1) {

    if (vid_stream->picture.code_type == P_TYPE) {

      Skipped::ProcessSkippedPFrameMBlocks(vid_stream->sequence.pictureArray->getCurrent(),
					   vid_stream->sequence.pictureArray->getFuture(),
					   &(vid_stream->mblock),
					   vid_stream->sequence.mb_width);

   } else {
      if (vid_stream->picture.code_type == B_TYPE) {
	Skipped::ProcessSkippedBFrameMBlocks(&(vid_stream->picture),
					     vid_stream->sequence.pictureArray->getPast(),
					     vid_stream->sequence.pictureArray->getCurrent(),
					     vid_stream->sequence.pictureArray->getFuture(),
					     &(vid_stream->mblock),
					     vid_stream->sequence.mb_width);
      }
    }

  }

  /* Set past macroblock address to current macroblock address. */
  vid_stream->mblock.past_mb_addr = vid_stream->mblock.mb_address;
  /* Based on picture type decode macroblock type. */
  switch (vid_stream->picture.code_type) {
  case I_TYPE:
    (vid_stream->decoderClass)->decodeMBTypeI(mb_quant, mb_motion_forw, 
					      mb_motion_back, mb_pattern,
					      vid_stream->mblock.mb_intra);
    break;

  case P_TYPE:
    (vid_stream->decoderClass)->decodeMBTypeP(mb_quant, mb_motion_forw, 
					      mb_motion_back, mb_pattern,
					      vid_stream->mblock.mb_intra);
    break;

  case B_TYPE:
    (vid_stream->decoderClass)->decodeMBTypeB(mb_quant, mb_motion_forw, 
					      mb_motion_back, mb_pattern,
					      vid_stream->mblock.mb_intra);
    break;
  case D_TYPE:
    printf("ERROR:  MPEG-1 Streams with D-frames are not supported\n");
    return SKIP_TO_START_CODE;

  }
  /* If quantization flag set, parse off new quantization scale. */
  if (mb_quant == true) {
    data=(vid_stream->bitwindow)->getBits(5);
    (vid_stream->slice)->setQuantScale(data);
  }
  /* If forward motion vectors exist... */

  if (mb_motion_forw == true) {
    // Parse off and decode horizontal forward motion vector. 
    vid_stream->mblock.motion_h_forw_code=
      (vid_stream->decoderClass)->decodeMotionVectors();

    // If horiz. forward r data exists, parse off. 

    if ((vid_stream->picture.forw_f != 1) &&
	(vid_stream->mblock.motion_h_forw_code != 0)) {
      data=(vid_stream->bitwindow)->getBits(vid_stream->picture.forw_r_size);
      vid_stream->mblock.motion_h_forw_r = data;
    }
    // Parse off and decode vertical forward motion vector. 
    
    vid_stream->mblock.motion_v_forw_code=
      (vid_stream->decoderClass)->decodeMotionVectors();

    // If vert. forw. r data exists, parse off. 

    if ((vid_stream->picture.forw_f != 1) &&
	(vid_stream->mblock.motion_v_forw_code != 0)) {

      data=(vid_stream->bitwindow)->getBits(vid_stream->picture.forw_r_size);
      vid_stream->mblock.motion_v_forw_r = data;
    }
  }

  /* If back motion vectors exist... */

  if (mb_motion_back == true) {
    // Parse off and decode horiz. back motion vector. 
    vid_stream->mblock.motion_h_back_code=
      (vid_stream->decoderClass)->decodeMotionVectors();

    // If horiz. back r data exists, parse off. 

    if ((vid_stream->picture.back_f != 1) &&
	(vid_stream->mblock.motion_h_back_code != 0)) {
      data=(vid_stream->bitwindow)->getBits(vid_stream->picture.back_r_size);
      vid_stream->mblock.motion_h_back_r = data;
    }
    // Parse off and decode vert. back motion vector. 
    vid_stream->mblock.motion_v_back_code=
      (vid_stream->decoderClass)->decodeMotionVectors();

    // If vert. back r data exists, parse off. 

    if ((vid_stream->picture.back_f != 1) &&
	(vid_stream->mblock.motion_v_back_code != 0)) {
      data=(vid_stream->bitwindow)->getBits(vid_stream->picture.back_r_size);
      vid_stream->mblock.motion_v_back_r = data;
    }
  }

  /* If mblock pattern flag set, parse and decode CBP (code block pattern). */
  if (mb_pattern == true) {
    vid_stream->mblock.cbp=(vid_stream->decoderClass)->decodeCBP();
  }
  /* Otherwise, set CBP to zero. */
  else
    vid_stream->mblock.cbp = 0;



  /* Reconstruct motion vectors depending on picture type. */
  if (vid_stream->picture.code_type == P_TYPE) {

    /*
     * If no forw motion vectors, reset previous and current vectors to 0.
     */
    if (!mb_motion_forw) {
      recon_right_for = 0;
      recon_down_for = 0;
      vid_stream->mblock.recon_right_for_prev = 0;
      vid_stream->mblock.recon_down_for_prev = 0;
    }
    /*
     * Otherwise, compute new forw motion vectors. Reset previous vectors to
     * current vectors.
     */

    else {
      (vid_stream->motionVector)->computeForwVector(&recon_right_for, 
						    &recon_down_for);

    }
  }
  if (vid_stream->picture.code_type == B_TYPE) {

    /* Reset prev. and current vectors to zero if mblock is intracoded. */
    if (vid_stream->mblock.mb_intra) {
      vid_stream->mblock.recon_right_for_prev = 0;
      vid_stream->mblock.recon_down_for_prev = 0;
      vid_stream->mblock.recon_right_back_prev = 0;
      vid_stream->mblock.recon_down_back_prev = 0;
    } else {
      
     /* If no forw vectors, current vectors equal prev. vectors. */
      
      if (!mb_motion_forw) {
	recon_right_for = vid_stream->mblock.recon_right_for_prev;
	recon_down_for = vid_stream->mblock.recon_down_for_prev;
      }
      /*
       * Otherwise compute forw. vectors. Reset prev vectors to new values.
       */
      
      else {
	(vid_stream->motionVector)->computeForwVector(&recon_right_for, 
						      &recon_down_for);

     }
      
      /* If no back vectors, set back vectors to prev back vectors. */
      
      if (!mb_motion_back) {
        recon_right_back = vid_stream->mblock.recon_right_back_prev;
        recon_down_back = vid_stream->mblock.recon_down_back_prev;
      }
      /* Otherwise compute new vectors and reset prev. back vectors. */

      else {
        (vid_stream->motionVector)->computeBackVector(&recon_right_back,
						      &recon_down_back);

      }
  
      /*
       * Store vector existence flags in structure for possible skipped
       * macroblocks to follow.
       */

      vid_stream->mblock.bpict_past_forw = mb_motion_forw;
      vid_stream->mblock.bpict_past_back = mb_motion_back;
    }
  }
  int back;
  back=(vid_stream->recon)->reconstruct(recon_right_for,
					recon_down_for,
					recon_right_back,
					recon_down_back,
					mb_motion_forw,
					mb_motion_back);
  

  /* If D Type picture, flush marker bit. */
  if (vid_stream->picture.code_type == D_TYPE) {
    (vid_stream->bitwindow)->flushBits(1);
  }

  /* If macroblock was intracoded, set macroblock past intra address. */
  if (vid_stream->mblock.mb_intra) {
    vid_stream->mblock.past_intra_addr=vid_stream->mblock.mb_address;
  }
  if (back == false) {
    return SKIP_TO_START_CODE;
  }
  return PARSE_OK;
}


int ResetVidStreamStruc(VidStream* vid_stream) {
  /* Reset past intrablock address. */
  Macroblock* mblock=&vid_stream->mblock;
  mblock->past_intra_addr = -2;

  /* Reset previous recon motion vectors. */
  

  mblock->recon_right_for_prev = 0;
  mblock->recon_down_for_prev = 0;
  mblock->recon_right_back_prev = 0;
  mblock->recon_down_back_prev = 0;

  /* Reset macroblock address. */
  mblock->mb_address = (((vid_stream->slice)->getVertPos()-1) *
			 vid_stream->sequence.mb_width) - 1;


  /* Reset past dct dc y, cr, and cb values. */
  Block* block=&vid_stream->block;


  block->dct_dc_y_past = 1024 << 3;
  block->dct_dc_cr_past = 1024 << 3;
  block->dct_dc_cb_past = 1024 << 3;
  
  return true;
}
/**
   After a seek we can only start with an I frame
*/

void resyncToI_Frame(VidStream* vid_stream) {

  cout << "resyncToI_Frame"<<endl;
  vid_stream->syncState=SYNC_TO_CLOSED_GOP;

  vid_stream->picture.temp_ref=0;


}


static void displayPicture(VidStream* vid_stream,Picture* pic) {
  TimeStamp* startStamp=pic->getStartTimeStamp();
  TimeStamp* endStamp=pic->getEndTimeStamp();


  float val=(vid_stream->sequence.pictureArray)->getPicturePerSecond();
  pic->setPicturePerSecond(val);

  (vid_stream->sequence.output)->putImage(startStamp,endStamp,pic);

}

/*
 *--------------------------------------------------------------
 *
 * DoDitherImage --
 *
 *      Called when image needs to be dithered. Selects correct
 *      dither routine based on info in xinfo[0].ditherType.
 *
 * Results:
 *        None.
 *
 * Side effects:
 *        None.
 *
 *--------------------------------------------------------------
 */
void doPictureDisplay(VidStream* vid_stream) {


  // insert end timestamp to current picture
  Picture* pic=(vid_stream->sequence.pictureArray)->getCurrent();

  TimeStamp* endTimeStamp=(vid_stream->bitwindow)->getCurrentTimeStamp();
  pic->setStartTimeStamp(vid_stream->startOfPicStamp);
  pic->setEndTimeStamp(endTimeStamp);
  pic->setFrameType(vid_stream->picture.code_type);
  pic->setVBVDelay(vid_stream->picture.vbv_delay);
  pic->setThreadedDither(false);
  int temp=(vid_stream->picture.temp_ref);

  if (vid_stream->syncState < SYNC_HAS_CLOSED_GOP) {
    return;
  }
  if (vid_stream->syncState < SYNC_HAS_I_FRAME_SYNC) {
    if (vid_stream->picture.code_type != I_TYPE) {
      cout << "********** enforce I frame"<<endl;
      return;
    }
  }
  if (vid_stream->picture.code_type == I_TYPE) {
    Picture* past=(vid_stream->sequence.pictureArray)->getPast();
    Picture* future=(vid_stream->sequence.pictureArray)->getFuture();
    Picture* current=(vid_stream->sequence.pictureArray)->getCurrent();

    Picture* tmp=past;
    past = future;
    future = current;
    current = tmp;
    
    pic=past;

  
    (vid_stream->sequence.pictureArray)->setPast(past);
    (vid_stream->sequence.pictureArray)->setCurrent(current);
    (vid_stream->sequence.pictureArray)->setFuture(future);
    pic->setThreadedDither(true);

    if (vid_stream->syncState < SYNC_HAS_I_FRAME_SYNC) {
      vid_stream->syncState=SYNC_HAS_I_FRAME_SYNC;
      return;
    }
    if (vid_stream->syncState == SYNC_HAS_P_FRAME_SYNC) {
      vid_stream->syncState=SYNC_HAS_FRAME_SYNC;
      return;
    }     
    if (vid_stream->syncState == SYNC_HAS_I_FRAME_SYNC) {
      vid_stream->syncState=SYNC_HAS_P_FRAME_SYNC;
      return;
    }    


  
  }
  
  if (vid_stream->picture.code_type == P_TYPE) {
    Picture* past=(vid_stream->sequence.pictureArray)->getPast();
    Picture* future=(vid_stream->sequence.pictureArray)->getFuture();
    Picture* current=(vid_stream->sequence.pictureArray)->getCurrent();

    Picture* tmp=past;
    past = future;
    future = current;
    current = tmp;
       
    pic = past;
   
    (vid_stream->sequence.pictureArray)->setPast(past);
    (vid_stream->sequence.pictureArray)->setCurrent(current);
    (vid_stream->sequence.pictureArray)->setFuture(future);
    pic->setThreadedDither(true); 

    if (vid_stream->syncState < SYNC_HAS_P_FRAME_SYNC) {
      vid_stream->syncState=SYNC_HAS_P_FRAME_SYNC;
      return;
    }

  }  
  if (vid_stream->picture.code_type == B_TYPE) {
    if (vid_stream->syncState == SYNC_HAS_P_FRAME_SYNC) {
      vid_stream->syncState=SYNC_HAS_FRAME_SYNC;
      Picture* past=(vid_stream->sequence.pictureArray)->getPast();
      displayPicture(vid_stream,past);
      past->setFrameType(PICTURE_NO_TYPE);
    }   
    // we display the current picture
    // (already set)
  }

  
  if (pic == NULL) {
    return;
  }
  if (vid_stream->syncState < SYNC_HAS_FRAME_SYNC) {
    cout << "discarding B frame"<<endl;
    return;
  }
  if (pic->getFrameType() == PICTURE_NO_TYPE) {
    cout << "skipp  PICTURE_NO_TYPE"<<endl;
    return;
  }


  displayPicture(vid_stream,pic);
}





