/*
  here are the main things for dithering
  Copyright (C) 1999  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include "dither.h"

/* 
   Flag for gamma correction 
   Makes images brighter/darker. 
   It's in the source but not activated (for now)
*/
int gammaCorrectFlag = 0;
double gammaCorrect = 1.0;

/* 
   Flag for chroma correction.
   reduce the color intensity..
   It's in the source but not activated (for now)
*/
int chromaCorrectFlag = 0;
double chromaCorrect = 1.0;


/* Range values for lum, cr, cb. */
int LUM_RANGE;
int CR_RANGE;
int CB_RANGE;

/* Array that remaps color numbers to actual pixel values used by X server. */

unsigned char pixel[256];
unsigned long wpixel[256];

/* Arrays holding quantized value ranged for lum, cr, and cb. */

int *lum_values;
int *cr_values;
int *cb_values;

/* Declaration of global variable containing dither type. */


/* Luminance and chrominance lookup tables */
static double *L_tab, *Cr_r_tab, *Cr_g_tab, *Cb_g_tab, *Cb_b_tab;


/*
 *--------------------------------------------------------------
 *
 * InitColor --
 *
 *	Initialize lum, cr, and cb quantized range value arrays.
 *      Also initializes the lookup tables for the possible
 *      values of lum, cr, and cb.
 *    Color values from ITU-R BT.470-2 System B, G and SMPTE 170M
 *    see InitColorDither in 16bits.c for more
 *
 * Results: 
 *      None.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

void initColor() {
  int i;

  L_tab    = new double[LUM_RANGE];
  Cr_r_tab = new double[CR_RANGE];
  Cr_g_tab = new double[CR_RANGE];
  Cb_g_tab = new double[CB_RANGE];
  Cb_b_tab = new double[CB_RANGE];


  if (L_tab == NULL    || Cr_r_tab == NULL ||
      Cr_g_tab == NULL || Cb_g_tab == NULL ||
      Cb_b_tab == NULL) {
    printf("Could not alloc memory in InitColor\n");
    exit(1);
  }

  for (i=0; i<LUM_RANGE; i++) {
    lum_values[i]  = ((i * 256) / (LUM_RANGE)) + (256/(LUM_RANGE*2));
    L_tab[i] = lum_values[i];
    if (gammaCorrectFlag) {
      L_tab[i] = GAMMA_CORRECTION(L_tab[i]);
    }
  }
  
  for (i=0; i<CR_RANGE; i++) {
    register double tmp;
    if (chromaCorrectFlag) {
      tmp = ((i * 256) / (CR_RANGE)) + (256/(CR_RANGE*2));
      Cr_r_tab[i] = (int) (0.419/0.299) * CHROMA_CORRECTION128D(tmp - 128.0);
      Cr_g_tab[i] = (int) -(0.299/0.419) * CHROMA_CORRECTION128D(tmp - 128.0);
      cr_values[i] = CHROMA_CORRECTION256(tmp);
    } else {
      tmp = ((i * 256) / (CR_RANGE)) + (256/(CR_RANGE*2));
      Cr_r_tab[i] = (int)  (0.419/0.299) * (tmp - 128.0);
      Cr_g_tab[i] = (int) -(0.299/0.419) * (tmp - 128.0);
      cr_values[i] = (int) tmp;
    }
  }

  for (i=0; i<CB_RANGE; i++) {
    register double tmp;
    if (chromaCorrectFlag) {
      tmp = ((i * 256) / (CB_RANGE)) + (256/(CB_RANGE*2));
      Cb_g_tab[i] = (int) -(0.114/0.331) * CHROMA_CORRECTION128D(tmp - 128.0);
      Cb_b_tab[i] = (int)  (0.587/0.331) * CHROMA_CORRECTION128D(tmp - 128.0);
      cb_values[i] = CHROMA_CORRECTION256(tmp);
    } else {
      tmp = ((i * 256) / (CB_RANGE)) + (256/(CB_RANGE*2));
      Cb_g_tab[i] = (int) -(0.114/0.331) * (tmp - 128.0);
      Cb_b_tab[i] = (int)  (0.587/0.331) * (tmp - 128.0);
      cb_values[i] = (int) tmp;
    }
  }

}


/*
 *--------------------------------------------------------------
 *
 * ConvertColor --
 *
 *	Given a l, cr, cb tuple, converts it to r,g,b.
 *
 * Results:
 *	r,g,b values returned in pointers passed as parameters.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

void ConvertColor(unsigned int l, 
		  unsigned int cr,
		  unsigned int cb,
		  unsigned char* r,
		  unsigned char* g,
		  unsigned char* b) {
  double fl, fr, fg, fb;

/*
 * Old method w/o lookup table
 *
 * fl = 1.164*(((double) l)-16.0);
 * fcr =  ((double) cr) - 128.0;
 * fcb =  ((double) cb) - 128.0;
 *
 * fr = fl + (1.366 * fcr);
 * fg = fl - (0.700 * fcr) - (0.334 * fcb);
 * fb = fl + (1.732 * fcb);
 */

  fl = L_tab[l];

  fr = fl + Cr_r_tab[cr];
  fg = fl + Cr_g_tab[cr] + Cb_g_tab[cb];
  fb = fl + Cb_b_tab[cb];

  if (fr < 0.0) fr = 0.0;
  else if (fr > 255.0) fr = 255.0;

  if (fg < 0.0) fg = 0.0;
  else if (fg > 255.0) fg = 255.0;

  if (fb < 0.0) fb = 0.0;
  else if (fb > 255.0) fb = 255.0;

  *r = (unsigned char) fr;
  *g = (unsigned char) fg;
  *b = (unsigned char) fb;

}


void initDither() {
  LUM_RANGE= 8;
  CR_RANGE= 4;
  CB_RANGE= 4;

  lum_values = new int[LUM_RANGE];
  cr_values = new int[CR_RANGE];
  cb_values = new int[CB_RANGE];

}




