/*
  stores StreamProducers and takes care that they work step by step
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include <graph/daisyChain.h>

DaisyChain::DaisyChain() {
  int i;

  pthread_mutex_init(&changeMut,NULL);
  pthread_mutex_init(&daisyMut,NULL);

  pthread_cond_init(&daisyCond,NULL);

  for(i=0;i<_MAX_DAISYCHAIN_ELEMENTS;i++) {
    daisyChain[i]=new DaisyChainElement();
    daisyChain[i]->valid=false;
    daisyChain[i]->locks=0;
  }

  for(i=0;i<_MAX_LOCKS;i++) {
    lockTupel[i]=new LockTupel();
    lockTupel[i]->valid=false;
  }
}

DaisyChain::~DaisyChain() {
  int i;

  for(i=0;i<_MAX_DAISYCHAIN_ELEMENTS;i++) {
    delete daisyChain[i];
  }
  for(i=0;i<_MAX_LOCKS;i++) {
    delete lockTupel[i];
  }

}

/**
   ad remove streamproducer operations can only be done
   be the graph controller == GUI
*/
void DaisyChain::addElement(StreamProducer* element) {
  int i;
  lockStreamProducer();
  for(i=0;i<_MAX_DAISYCHAIN_ELEMENTS;i++) {
    if (daisyChain[i]->valid==false) {
      daisyChain[i]->valid=true;
      daisyChain[i]->element=element;
      daisyChain[i]->locks=0;
      pthread_mutex_init(&(daisyChain[i]->blockMut),NULL);
      pthread_cond_init(&(daisyChain[i]->blockCond),NULL);


      break;
    }  
  }
  unlockStreamProducer(); 
}


void DaisyChain::removeElement(StreamProducer* element) {
  int i;
  lockStreamProducer();
  removeAllLocks(this);
  for(i=0;i<_MAX_DAISYCHAIN_ELEMENTS;i++) {
    if (daisyChain[i]->valid == true) {
      if (daisyChain[i]->element==element) {
	daisyChain[i]->valid=false;
	break;
      }    
    }
  }
  if (i == _MAX_DAISYCHAIN_ELEMENTS) {
    cout << "remove element not found in DaisyChain::removeElement"<<endl;
    exit(0);
  }
  unlockStreamProducer(); 
}


void DaisyChain::lockDaisyChain() {
  pthread_mutex_lock(&changeMut);
  pthread_mutex_lock(&daisyMut);  
  lProducerAlreadyLocked=true;
}


void DaisyChain::unlockDaisyChain() {
  lProducerAlreadyLocked=false;
  pthread_cond_signal(&daisyCond);
  pthread_mutex_unlock(&changeMut);
  pthread_mutex_unlock(&daisyMut);
}


/**
   can only be called by the graph controller == GUI
*/

void DaisyChain::lockStreamProducer() {
  int i;
  for(i=0;i<_MAX_DAISYCHAIN_ELEMENTS;i++) {
    if (daisyChain[i]->valid==true) {
      (daisyChain[i]->element)->writeInLock();
      (daisyChain[i]->element)->writeOutLock();
    }  
  }
}


void DaisyChain::unlockStreamProducer() {
  int i;
  for(i=0;i<_MAX_DAISYCHAIN_ELEMENTS;i++) {
    if (daisyChain[i]->valid==true) {
      (daisyChain[i]->element)->writeOutUnlock();
      (daisyChain[i]->element)->writeInUnlock();
    }  
  }
}




void DaisyChain::addLock(void* from,StreamProducer* to) {
  int i;
  for(i=0;i<_MAX_LOCKS;i++) {
    if (lockTupel[i]->valid==false) {
      lockTupel[i]->from=from;
      lockTupel[i]->to=to;
      lockTupel[i]->valid=true;
      incLock(lockTupel[i]->to);
      return;
    }
     
  }
}


void DaisyChain::removeLock(void* from,StreamProducer* to) {
  int i;
  for(i=0;i<_MAX_LOCKS;i++) {
    if (lockTupel[i]->valid==true) {
      if (lockTupel[i]->from==from) {
	if (lockTupel[i]->to==to) {
	  lockTupel[i]->valid=false;
	  decLock(lockTupel[i]->to);
	  return;
	}
      }
    }
  }
}

/**
   can be called by streamproducers
   and by the gui, but only through a removeElement call
   This means if a streamproducer enters he always
   finds a valid daisychain structure.
   So we need no lock.
*/
void DaisyChain::removeAllLocks(void* from) {
  int i;
  for(i=0;i<_MAX_LOCKS;i++) {
    if (lockTupel[i]->valid==true) {
      if (lockTupel[i]->from==from) {
	  lockTupel[i]->valid=false;
	  decLock((lockTupel[i]->to));
      }
    }  
  }
}



void DaisyChain::incLock(StreamProducer* producer) {
  int i;
  for(i=0;i<_MAX_DAISYCHAIN_ELEMENTS;i++) {
    if (daisyChain[i]->valid==true) {
      if (daisyChain[i]->element == producer) {
	pthread_mutex_lock(&(daisyChain[i]->blockMut));
	daisyChain[i]->locks++;
	pthread_mutex_unlock(&(daisyChain[i]->blockMut));
      }
    }  
  }
}

/**
   If we increment a lock, it is possible, that the producer
   is waiting for this decrement operation.
   We must make sure to wake him up, but only if
   the producer is not the one who
   has called this method.
   This is possible, because, the multiInputDevice is currently
   not a streamproducer itsself.
*/

void DaisyChain::decLock(StreamProducer* producer) {
  int i;
  for(i=0;i<_MAX_DAISYCHAIN_ELEMENTS;i++) {
    if (daisyChain[i]->valid==true) {
      if (daisyChain[i]->element == producer) {
	pthread_mutex_lock(&(daisyChain[i]->blockMut));
	daisyChain[i]->locks--;
	if (daisyChain[i]->locks == 0) {
	  pthread_cond_signal(&(daisyChain[i]->blockCond));
	}
	pthread_mutex_unlock(&(daisyChain[i]->blockMut));
      }
    }  
  }
}

/**
   The cool logic of my program is, that a producer only
   can increments locks for itsself, no other can.
   Thus if he checks if there are locks, we can not have a race
   condition.
*/
int DaisyChain::hasLock(StreamProducer* producer) {
  int i;
  int back=false;
  for(i=0;i<_MAX_DAISYCHAIN_ELEMENTS;i++) {
    if (daisyChain[i]->valid==true) {
      if (daisyChain[i]->element == producer) {
	pthread_mutex_lock(&(daisyChain[i]->blockMut));
	if (daisyChain[i]->locks) {
	  back=true;
	  pthread_cond_wait(&(daisyChain[i]->blockCond),
			    &(daisyChain[i]->blockMut));
	}
	pthread_mutex_unlock(&(daisyChain[i]->blockMut));
	return back;
      }  
    } 
  }
  cout << "producer not existing"<<endl;
  return false;
}
 

void DaisyChain::wakeUpThread(StreamProducer* producer) {
  int i;
  int back=false;
  for(i=0;i<_MAX_DAISYCHAIN_ELEMENTS;i++) {
    if (daisyChain[i]->valid==true) {
      if (daisyChain[i]->element == producer) {
	pthread_mutex_lock(&(daisyChain[i]->blockMut));
	pthread_cond_signal(&(daisyChain[i]->blockCond));
	pthread_mutex_unlock(&(daisyChain[i]->blockMut));
      }
    }
  }
}
