/* 
  a thread safe ring buffer
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include <util/ringBuffer.h>


RingBuffer::RingBuffer(int size,int minLinSize) {
  pthread_mutex_init(&mut,NULL);
  pthread_cond_init(&dataCond,NULL);
  pthread_cond_init(&spaceCond,NULL);


  this->size=size;
  startPos=new char[size];
  readPos=startPos;
  writePos=startPos;
  lockPos=startPos;

  lockgrade=0;
  fillgrade=0;
  linAvail=size;
  lastPos=(startPos+size-1);
  eofPos=lastPos+1;

  canWrite=size;
  canRead=0;

  minLinBuf=new char[minLinSize];
  this->minLinBufSize=minLinSize;
  waitMinData=0;
  waitMinSpace=0;
  lWaitForData=false;
  lWaitForSpace=false;

  lAllowWaitForData=true;

  readBytes=0;
  writeBytes=0;
}


RingBuffer::~RingBuffer() {
  // The user of this class must take care that the threads
  // have exited!
  delete startPos;
  delete minLinBuf;
}
  

int RingBuffer::getSize() {
  return size;
}


int RingBuffer::getWriteArea(MemChunk* gmem,int preferredSize) {

  // When we are in this area the following
  // can happen by the other thread
  // fillgrade is decreased (because reader fowards)
  // readPos is changed.

  gmem->setPtr(writePos);
  gmem->setLen(canWrite);
  if (canWrite > preferredSize) {
    gmem->setLen(preferredSize);
  }
  return gmem->getLen();
}


void RingBuffer::updateCanWrite() {
  if (lockPos < writePos) {
    canWrite=eofPos-writePos;
    //printf("1 c:%d l:%p w:%p",canWrite,lockPos,writePos);
  } else if (lockPos > writePos) {
    canWrite=lockPos-writePos;
    //printf("2 c:%d l:%p w:%p",canWrite,lockPos,writePos);
  } else {
    if (fillgrade > 0) {
      canWrite=0;
    } else {
      canWrite=eofPos-writePos;
    }
    //printf("2 c:%d ",canWrite);
    
  }
  if (canWrite < 0) {
    printf("error canWrite:%d fill:%d lock:%p start:%p eof:%p write:%p\n",
           canWrite,fillgrade,lockPos,startPos,eofPos,writePos);
  }

} 


void RingBuffer::updateCanRead() {
  canRead=fillgrade-lockgrade;
  if (size-fillgrade >= waitMinSpace) {
    pthread_cond_broadcast(&spaceCond);
  } 
  if (canRead < 0) {
    printf("error canRead:%d fillgrade:%d lockgrade:%d \n",
           canRead,fillgrade,lockgrade);
  }
 
} 


void RingBuffer::forwardLockPtr(int nBytes) {
  pthread_mutex_lock(&mut);

  if (fillgrade < lockgrade) {
    printf("1:fillgrade:%d < lockgrade:%d\n",fillgrade,lockgrade);
  }
  fillgrade-=nBytes;
  lockgrade-=nBytes;
  if (fillgrade < lockgrade) {
    printf("2:fillgrade:%d < lockgrade:%d nBytes:%d\n",
	   fillgrade,lockgrade,nBytes);
  }
  lockPos=lockPos+nBytes;
  if (lockPos > lastPos) {   // we expects that we had a linAvail part
    // if user forwards more than buffer boundary
    nBytes=lockPos-lastPos;
    lockPos=startPos+nBytes-1;
  }
  updateCanWrite();
  updateCanRead();

  pthread_mutex_unlock(&mut);
  return;
}


void RingBuffer::forwardWritePtr(int nBytes) {
  pthread_mutex_lock(&mut);
  fillgrade=fillgrade+nBytes;
  if (fillgrade < lockgrade) {
    printf("3:fillgrade:%d < lockgrade:%d nBytes:%d\n",
	   fillgrade,lockgrade,nBytes);
  }
  writeBytes+=nBytes;
  writePos=writePos+nBytes;
  if(writePos >= eofPos) {
    if (writePos == eofPos) {
      writePos=startPos;
    } else {
      cout << "writePos > eofPos ! forward error:"<<(eofPos-writePos)
	   <<" bytes"<<endl;
    }
  }

  updateCanWrite();
  updateCanRead();
  if (fillgrade >= waitMinData) {
    pthread_cond_broadcast(&dataCond);
  }
  pthread_mutex_unlock(&mut);
}


int RingBuffer::waitForSpace(int bytes){
  pthread_mutex_lock(&mut);
  int back=0;
  waitMinSpace=bytes;
  if (waitMinSpace > size) {
    waitMinSpace=size;
  }
  if (waitMinSpace < 0) {
    waitMinSpace=0;
  }
  if (canWrite < waitMinSpace) {
    lWaitForSpace=true;
    // it is not possible to wait for data space simultanously
    if (lWaitForData == true) {
      pthread_cond_broadcast(&dataCond);
    }
    pthread_cond_wait(&spaceCond,&mut);
    lWaitForSpace=false;
  }
  if (canWrite >= waitMinSpace) {
    back=1;
  }
  pthread_mutex_unlock(&mut);
  return back;
}


void RingBuffer::exitWaitForSpace(){
  pthread_mutex_lock(&mut);
  pthread_cond_broadcast(&spaceCond);
  pthread_mutex_unlock(&mut);
}

  

void RingBuffer::forwardReadPtr(int nBytes) {
  pthread_mutex_lock(&mut);
  readBytes+=nBytes;
  readPos+=nBytes;
  linAvail=linAvail-nBytes;
  lockgrade+=nBytes;
  if (readPos > lastPos) {   // we expects that we had a linAvail part
    // if user forwards more than buffer boundary
    nBytes=readPos-lastPos;
    readPos=startPos+nBytes-1;
    linAvail=lastPos+1-readPos;
  }
  if (fillgrade < lockgrade) {
    printf("5:fillgrade:%d < lockgrade:%d nBytes:%d\n",
	   fillgrade,lockgrade,nBytes);
  }
  updateCanRead();
  pthread_mutex_unlock(&mut);
}



int RingBuffer::getReadArea(MemChunk* gmem,int pSize) {
  gmem->setPtr(readPos);

  if (canRead == 0) {
    gmem->setLen(0);
    return 0;
  }
  if (pSize < 0) { 
    cout << "Generic Memory Info invalid"<<endl;
    pSize=size/2;
  }
  //
  // Now the part the we deliver a minimum buffer if it is
  // possible
  //

  if ( (pSize > linAvail) &&
       (minLinBufSize >linAvail) &&
       (canRead > linAvail) ) {
    int copySize;
    copySize=canRead;  // we cannot copy more than this
    if (copySize > pSize) { // if it is too much reduce it
      copySize=pSize;
    }
    if (copySize > minLinBufSize) { // if it does not fit in buffer->reduce
      copySize=minLinBufSize;
    }
    memcpy(minLinBuf,readPos,linAvail);
    memcpy(minLinBuf+linAvail,startPos,copySize-linAvail);
    gmem->setLen(copySize);
    gmem->setPtr(minLinBuf);
    return copySize;
  }

  // linAvail part end

  int copyBytes=linAvail;
  if (canRead < copyBytes) {
    copyBytes=canRead;
  }
  if (copyBytes >= pSize) {
    gmem->setLen(pSize);
  } else {
    gmem->setLen(copyBytes);
  }
  return gmem->getLen();
}



void RingBuffer::exitWaitForData(){
  pthread_mutex_lock(&mut);
  pthread_cond_broadcast(&dataCond);
  pthread_mutex_unlock(&mut);
}



int RingBuffer::waitForData(int bytes){
  pthread_mutex_lock(&mut);
  int back=0;
  waitMinData=bytes;
  if (waitMinData > size) {
    waitMinData=size;
  }
  if (waitMinData < 0) {
    waitMinData=0;
  }
  if (canRead < waitMinData) {
    lWaitForData=true;
    // it is not possible to wait for data/space simultanously
    if (lWaitForSpace == true) {
      pthread_cond_broadcast(&spaceCond);
    }
    if (lAllowWaitForData==true) {
      pthread_cond_wait(&dataCond,&mut);
    }
    lWaitForData=false;
  }
  if (canRead >= waitMinData) {
    back=1;
  }
  pthread_mutex_unlock(&mut);
  return back;
}


void RingBuffer::emptyBuffer() {
  pthread_mutex_lock(&mut);
  writePos=readPos;
  if (fillgrade < lockgrade) {
    printf("4:fillgrade:%d < lockgrade:%d\n",fillgrade,lockgrade);
  }
  linAvail=lastPos+1-writePos;
  fillgrade=lockgrade;
  updateCanRead();
  updateCanWrite();
  readBytes=0;
  writeBytes=0;
  if (size-fillgrade >= waitMinSpace) {
    pthread_cond_broadcast(&spaceCond);
  }
  if (fillgrade >= waitMinData) {
    pthread_cond_broadcast(&dataCond);
  }
  pthread_mutex_unlock(&mut);
}


int RingBuffer::getFillgrade() {
  return fillgrade;
}


int RingBuffer::getReadBytes() {
  return readBytes;
}


int RingBuffer::getWriteBytes() {
  return writeBytes;
}


void RingBuffer::setWaitForData(int lallow) {
  pthread_mutex_lock(&mut);
  lAllowWaitForData=lallow;
  pthread_mutex_unlock(&mut);
  
  if (lAllowWaitForData == false) {
    exitWaitForData();
  }
    

}


int RingBuffer::getWaitForData() {
  int back;
  pthread_mutex_lock(&mut);
  back=lAllowWaitForData;
  pthread_mutex_unlock(&mut);
  return back;
}
