/***************************************************************************
                          options.cpp  -  description
                             -------------------
    begin                : Thu Sep 30 1999
    copyright            : (C) 1999 by Ralf Nolden
    email                : Ralf.Nolden@post.rwth-aachen.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
// options.cpp
// ktuner v0.5 , Carl van Schaik
// e-mail carl@leg.uct.ac.za
//
// Radiotrack II implementation : Leendert Meyer (leendert@usa.net).
//
// RadioTrack II Protocol : Ben Pfaff

#ifndef OPTIONS_CPP
#define OPTIONS_CPP

#include "options.h"
#include "aclock.h"
#include <qkeycode.h>
#include <qmsgbox.h>
#include <stdlib.h>
#include <kapp.h>

analogClock *clock;

optionBox::optionBox(configFile* conf, radioCard *card, QWidget *parent, const char *name)
  : QTabDialog( parent, name, FALSE, WStyle_Customize | WStyle_DialogBorder )
{
  cf = conf;

  setCaption(i18n("kTuner - Options"));
  resize(350,250);
  setCancelButton();
  setApplyButton(i18n("Save"));
  setMaximumSize(350,250);

  // General tab ------------------------------------------------------------
  general = new QWidget(this);
  general->resize(300,250);
  ioGroup = new QButtonGroup(general);
  ioGroup->resize(70,50);
  ioGroup->move(5,0);
  ioGroup->setTitle(i18n("I/O Base"));

  io20f = new QRadioButton("0x20f",ioGroup);
  io20f->move(5,15);
  io20f->resize(47,13);
  io30f = new QRadioButton("0x30f",ioGroup);
  io30f->move(5,30);
  io30f->resize(47,13);
  ioGroup->insert(io20f);
  ioGroup->insert(io30f);

  startUp = new QButtonGroup(general);
  startUp->resize(110,80);
  startUp->move(90,0);
  startUp->setTitle(i18n("On Startup"));

  verbose = new QCheckBox(i18n("Verbose output"),startUp);
  verbose->move(5,15);
  verbose->resize(95,13);
  useDef = new QCheckBox(i18n("Set Station"),startUp);
  useDef->move(5,30);
  useDef->resize(75,13);
  setVol = new QCheckBox(i18n("Set Volume"),startUp);
  setVol->move(5,45);
  setVol->resize(75,13);
  setAlarm = new QCheckBox(i18n("Activate Alarm"),startUp);
  setAlarm->move(5,60);
  setAlarm->resize(90,13);
  startUp->insert(verbose);
  startUp->insert(useDef);
  startUp->insert(setVol);
  startUp->insert(setAlarm);
  startUp->setExclusive(FALSE);

  cardType = new QButtonGroup(general);
  cardType->resize(110,80);
  cardType->move(215,0);
  cardType->setTitle(i18n("Card Type"));
  aims = new QRadioButton(i18n("Radio-Track"),cardType);
  aims->move(5,15);
  aims->resize(75,13);
  aimsII = new QRadioButton(i18n("Radio-Track II"),cardType);
  aimsII->move(5,30);
  aimsII->resize(90,13);
  zoltrix = new QRadioButton(i18n("Zoltrix"),cardType);
  zoltrix->move(5,45);
  zoltrix->resize(90,13);
  kerneldev = new QRadioButton("/dev/radio",cardType);
  kerneldev->move(5,60);
  kerneldev->resize(90,13);
  cardType->insert(aims);
  cardType->insert(aimsII);
  cardType->insert(zoltrix);
  cardType->insert(kerneldev);

  //winStyle = new QButtonGroup(general);
  //winStyle->resize(110,50);
  //winStyle->move(215,85);
  //winSty"le->setTitle(Window Style");
  //motif = new QRadioButton("Motif Style",winStyle);
  //motif->move(5,15);
  //motif->resize(75,13);
  //windows = new QRadioButton("Windows Style",winStyle);
  //windows->move(5,30);
  //windows->resize(90,13);
  //winStyle->insert(motif);
  //winStyle->insert(windows);

  sleepGroup = new QGroupBox(general);
  sleepGroup->resize(110,80);
  sleepGroup->move(90,85);
  sleepGroup->setTitle(i18n("Sleep"));
  sleepLabel = new QLabel(sleepGroup);
  sleepLabel->resize(100,10);
  sleepLabel->move(5,15);
  sleepLabel->setText(i18n("Time in Minutes till off"));
  sleepLabel->setAlignment(AlignCenter);
  sleepTimeB = new QScrollBar(0,60,1,5,30,QScrollBar::Horizontal,sleepGroup);
  sleepTimeB->resize(100,15);
  sleepTimeB->move(5,60);
  sleepTimeI = new QLCDNumber(sleepGroup);
  sleepTimeI->resize(50,35);
  sleepTimeI->move(20,25);
  sleepTimeI->setNumDigits(3);
  sleepTimeI->setFrameStyle(QFrame::NoFrame);

  Settings = new QGroupBox(general);
  Settings->resize(70,115);
  Settings->move(5,55);
  Settings->setTitle(i18n("Defaults"));

  defVol = new QScrollBar(0,15,1,5,cf->getVolume(),QScrollBar::Vertical,Settings);
  defVol->move(55,15);
  defVol->resize(10,45);
  defVolI = new QLCDNumber(Settings);
  defVolI->resize(50,30);
  defVolI->move(5,25);
  defVolI->setNumDigits(2);
  defVolI->setFrameStyle(QFrame::NoFrame);
  volText = new QLabel(i18n("Volume"),Settings);
  volText->move(8,15);
  volText->resize(40,10);
  chnText = new QLabel(i18n("Channel"),Settings);
  chnText->move(8,60);
  chnText->resize(40,10);
  defChn = new QScrollBar(1,9+cf->getNumMoreChannels(),1,5,cf->getDefault(),QScrollBar::Vertical,Settings);
  defChn->move(55,65);
  defChn->resize(10,45);
  defChnI = new QLCDNumber(Settings);
  defChnI->resize(50,25);
  defChnI->move(5,72);
  defChnI->setNumDigits(2);
  defChnI->setFrameStyle(QFrame::NoFrame);
  chnName = new QLabel(Settings);
  chnName->setAlignment(AlignRight);
  chnName->move(3,95);
  chnName->resize(48,15);

  // Presets tab ------------------------------------------------------------
  presets = new QWidget(this);
  presets->resize(300,250);

  table = new presetTable(presets);
  table->move(5,25);
  table->resize(226,150);

  presetLabels = new QLabel(i18n("Channel           Frequency        Name"),presets);
  presetLabels->move(10,12);
  presetLabels->setAutoResize(true);

  note = new QLabel(i18n("Note: changes are permanent!"),presets);
  note->move(10,00);
  note->setAutoResize(true);

  tabControls = new QGroupBox(presets);
  tabControls->resize(90,155);
  tabControls->move(235,20);
  tabControls->setTitle(i18n("Controls"));

  rowEdit = new QPushButton(tabControls);
  rowInsert = new QPushButton(tabControls);
  rowAdd = new QPushButton(tabControls);
  rowDel = new QPushButton(tabControls);
  listen = new QPushButton(tabControls);
  rowUp = new QPushButton(tabControls);
  rowDown = new QPushButton(tabControls);

  rowEdit->move(20,20);
  rowEdit->setText(i18n("Edit"));
  rowEdit->resize(50,15);
  rowInsert->move(20,35);
  rowInsert->setText(i18n("Insert"));
  rowInsert->resize(50,15);
  rowAdd->move(20,55);
  rowAdd->setText(i18n("Add"));
  rowAdd->resize(50,15);
  rowDel->move(20,70);
  rowDel->setText(i18n("Delete"));
  rowDel->resize(50,15);
  listen->move(20,90);
  listen->setText(i18n("Listen"));
  listen->resize(50,15);

  rowUp->move(7,120);
  rowUp->setText(i18n("Move Up"));
  rowUp->resize(75,15);
  rowDown->move(7,135);
  rowDown->setText(i18n("Move Down"));
  rowDown->resize(75,15);

  editBox = new presetEdit(cf);
  addBox = new presetAdd(cf);

  // alarm tab ------------------------------------------------------------
  alarmtab = new QWidget(this);
  alarmtab->resize(300,250);

  hrText = new QLabel(i18n("Hours"),alarmtab);
  hrText->move(27,20);
  hrText->setAutoResize(true);
  minText = new QLabel(i18n("Minutes"),alarmtab);
  minText->move(27,40);
  minText->setAutoResize(true);
  credits = new QLabel(i18n("kTuner (c) Copywrite Carl van Schaik (carl@leg.uct.ac.za)"),alarmtab);
  credits->move(5,160);
  credits->setAutoResize(true);

  hrMinus = new QPushButton(alarmtab);
  hrPlus = new QPushButton(alarmtab);
  minMinus = new QPushButton(alarmtab);
  minPlus = new QPushButton(alarmtab);

  hrPlus->move(65,20);
  hrPlus->resize(15,15);
  hrPlus->setText("+");
  hrMinus->move(10,20);
  hrMinus->resize(15,15);
  hrMinus->setText("-");
  minPlus->move(65,40);
  minPlus->resize(15,15);
  minPlus->setText("+");
  minPlus->setAutoRepeat(TRUE);
  minMinus->move(10,40);
  minMinus->resize(15,15);
  minMinus->setText("-");
  minMinus->setAutoRepeat(TRUE);

  clock = new analogClock(alarmtab);
  clock->move(120,40);
  clock->resize(100,100);

  addTab(general,i18n("General"));
  addTab(presets,i18n("Presets"));
  addTab(alarmtab,i18n("Alarm"));
  setMaximumSize(350,250);

  //  t->show();

  connect(sleepTimeB,SIGNAL(valueChanged(int)),SLOT(sleepTimeC(int)));
  connect(defVol,SIGNAL(valueChanged(int)),SLOT(volChg(int)));
  connect(defChn,SIGNAL(valueChanged(int)),SLOT(defChg(int)));
  connect(this, SIGNAL(defaultButtonPressed()), SLOT(optionsDone()));
  connect(this, SIGNAL(applyButtonPressed()), SLOT(optionsSave()));
  connect(this, SIGNAL(cancelButtonPressed()), SLOT(optionsCancel()));
  connect(hrPlus, SIGNAL(clicked()), SLOT(aHrUp()));
  connect(hrMinus, SIGNAL(clicked()), SLOT(aHrDn()));
  connect(minPlus, SIGNAL(clicked()), SLOT(aMinUp()));
  connect(minMinus, SIGNAL(clicked()), SLOT(aMinDn()));
  connect(rowEdit, SIGNAL(clicked()), SLOT(edit()));
  connect(editBox, SIGNAL(edited()), SLOT(edited()));
  connect(listen, SIGNAL(clicked()), SLOT(sendFreq()));
  connect(rowAdd, SIGNAL(clicked()), SLOT(add()));
  connect(rowInsert, SIGNAL(clicked()), SLOT(insert()));
  connect(addBox, SIGNAL(edited()), SLOT(edited()));
  connect(rowDel, SIGNAL(clicked()), SLOT(Delete()));
  connect(rowUp, SIGNAL(clicked()), SLOT(moveUp()));
  connect(rowDown, SIGNAL(clicked()), SLOT(moveDown()));
  connect(aims,SIGNAL(clicked()), SLOT(aimsPushed()));
  connect(aimsII,SIGNAL(clicked()), SLOT(aimsIIPushed()));
  connect(zoltrix,SIGNAL(clicked()), SLOT(zoltrixPushed()));
  connect(kerneldev,SIGNAL(clicked()), SLOT(kerneldevPushed()));
  connect(io20f,SIGNAL(clicked()), SLOT(setWarn()));
  connect(io30f,SIGNAL(clicked()), SLOT(setWarn()));

  configure();
}


void optionBox::sleepTimeC(int T)
{
  int t=T+1;
  if (T >= 30)
    t = 30+(T-29)*5;
  if (T > 45)
    t = 15*5+35 + (T-45)*10;
  sleepTimeI->display(t);
  sleepTime = t;
}

void optionBox::volChg(int T)
{
  defVolI->display(T);
}

void optionBox::defChg(int T)
{
  defChnI->display(T);
  chnName->setText(cf->getName(T-1));
}

void optionBox::optionsSave()
{
  setConf();
  if (cf->save()) QMessageBox::information( 0,i18n( "kTuner Error"),
                                  i18n("Configuration File Error"),
                                  QMessageBox::Ok + QMessageBox::Default );
  if (showWarn) QMessageBox::information( 0, i18n("cRadio Notice"),
                                  i18n("You changed an option that can only take\n"
				  "effect after restarting kTuner"),
                                  QMessageBox::Ok + QMessageBox::Default );
  showWarn = 0;
}

void optionBox::optionsCancel()
{
  configure();
  hide();
  editBox->hide();
}

void optionBox::optionsDone()
{
  hide();
  editBox->hide();
}

void optionBox::configure()
{
  showWarn = 0;
 
  sleepTime = cf->getSleepTime();

  sleepTimeB->setValue(sleepTime-1);
  sleepTimeI->display(sleepTime);

  if ((cf->getBasePort() == 0x20f) || (cf->getBasePort() == 0x20c))
    {
      io20f->setChecked(TRUE);
      io30f->setChecked(FALSE);
    }
  else
    {
      io20f->setChecked(FALSE);
      io30f->setChecked(TRUE);
    }

  if (cf->getCardType() == RADIOTRACK)
    {
      aims->setChecked(TRUE);
      aimsII->setChecked(FALSE);
      zoltrix->setChecked(FALSE);
      kerneldev->setChecked(FALSE);
      io20f->setEnabled(TRUE);
      io30f->setEnabled(TRUE);
   }
  else if (cf->getCardType() == RADIOTRACKII)
    {
      aims->setChecked(FALSE);
      aimsII->setChecked(TRUE);
      zoltrix->setChecked(FALSE);
      kerneldev->setChecked(FALSE);
      io20f->setEnabled(TRUE);
      io30f->setEnabled(TRUE);
   }
  else if (cf->getCardType() == ZOLTRIX)
    {
      aims->setChecked(FALSE);
      aimsII->setChecked(FALSE);
      zoltrix->setChecked(TRUE);
      kerneldev->setChecked(FALSE);
    }
  else if (cf->getCardType() == KERNELDEV)
    {
      aims->setChecked(FALSE);
      aimsII->setChecked(FALSE);
      zoltrix->setChecked(FALSE);
      kerneldev->setChecked(TRUE);
      io20f->setEnabled(FALSE);
      io30f->setEnabled(FALSE);
    }

  /*  if (cf->gui() == MotifStyle)
    {
      motif->setChecked(TRUE);
      windows->setChecked(FALSE);
    }
  else
    {
      motif->setChecked(FALSE);
      windows->setChecked(TRUE);
    }
  */
  if (cf->getVerbose())
    verbose->setChecked(TRUE);
  else
    verbose->setChecked(FALSE);

  if (cf->noVolSet() == 0)
    setVol->setChecked(TRUE);
  else
    setVol->setChecked(FALSE);

  if (cf->noDefSet() == 0)
    useDef->setChecked(TRUE);
  else
    useDef->setChecked(FALSE);

  if (cf->setAlarm())
    setAlarm->setChecked(TRUE);
  else
    setAlarm->setChecked(FALSE);

  defVol->setValue(cf->getVolume());
  defVolI->display(cf->getVolume());
  defChn->setValue(cf->getDefault()+1);
  defChnI->display(cf->getDefault()+1);
  chnName->setText(cf->getName(cf->getDefault()));

  table->setNumberOfRows(9+cf->getNumMoreChannels());
  int i;
  for (i=0;i<(9+cf->getNumMoreChannels());i++)
    {
      table->setRowText(i,cf->getFreq(i),cf->getName(i));
    }

  alarmHr = cf->getAlarmHr();
  alarmMin = cf->getAlarmMin();

  clock->setTime(alarmHr, alarmMin);
}

void optionBox::setConf()
{
  cf->setSleepTime(sleepTime);

  if (aims->isChecked())
    {
      cf->setCardType(RADIOTRACK);
      if (io20f->isChecked())
	cf->setBasePort(0x20f);
      else
	if (io30f->isChecked())
	  cf->setBasePort(0x30f);
    }
  else
  if (aimsII->isChecked())
    {
      cf->setCardType(RADIOTRACKII);
      if (io20f->isChecked())
        cf->setBasePort(0x20f);
      else
      if (io30f->isChecked())
        cf->setBasePort(0x30f);
    }
  else
    if (zoltrix->isChecked())
      {
	cf->setCardType(ZOLTRIX);
	if (io20f->isChecked())
	  cf->setBasePort(0x20c);
	else
	  if (io30f->isChecked())
	    cf->setBasePort(0x30c);
      }
    else
      if (kerneldev->isChecked())
	cf->setCardType(KERNELDEV);

  /*  if (motif->isChecked())
    cf->setGui(MotifStyle);
  else
    cf->setGui(WindowsStyle);
  */
  cf->setVerbose(verbose->isChecked());

  if (setVol->isChecked())
    cf->setNoVolSet(0);
  else
    cf->setNoVolSet(1);

  if (useDef->isChecked())
    cf->setNoDefSet(0);
  else
    cf->setNoDefSet(1);

  cf->setVolume(defVol->value());
  cf->setDefault(defChn->value()-1);

  if (setAlarm->isChecked())
    cf->setAlarmDef(1);
  else
    cf->setAlarmDef(0);

  cf->setAlarmTime(alarmHr,alarmMin);

  emit optionsUpdate();
}


void optionBox::aHrUp()
{
  alarmHr ++;
  if (alarmHr > 23) alarmHr = 0;
  clock->setTime(alarmHr,alarmMin);
}

void optionBox::aHrDn()
{
  alarmHr --;
  if (alarmHr < 0) alarmHr = 23;
  clock->setTime(alarmHr,alarmMin);
}

void optionBox::aMinUp()
{
  alarmMin ++;
  if (alarmMin > 59)
    {
      alarmMin = 0;
      aHrUp();
    } else
  clock->setTime(alarmHr,alarmMin);
}

void optionBox::aMinDn()
{
  alarmMin --;
  if (alarmMin < 0)
    {
      alarmMin = 59;
      aHrDn();
    } else
  clock->setTime(alarmHr,alarmMin);
}

void optionBox::edit()
{
  editBox->setChanNum(table->getSelectedRow());
  editBox->show();
}

void optionBox::add()
{
  addBox->setInsertPos(9+cf->getNumMoreChannels());
  addBox->show();
}

void optionBox::insert()
{
  addBox->setInsertPos(table->getSelectedRow());
  addBox->show();
}

void optionBox::Delete()
{
  int row = table->getSelectedRow();
  cf->delChannel(row);
  
  if (row > 0)
    table->setSelectedRow(row - 1);
  edited();
}

void optionBox::moveUp()
{
  int row = table->getSelectedRow();
  QString temp1;
  int temp2;
  if (row>0)
    {
      temp1 = cf->getName(row);
      temp2 = cf->getFreq(row);
      cf->setName(row,cf->getName(row-1));
      cf->setFreq(row,cf->getFreq(row-1));
      cf->setName(row-1,temp1);
      cf->setFreq(row-1,temp2);
      table->setSelectedRow(row-1);
      edited();
    }
}

void optionBox::moveDown()
{
  int row = table->getSelectedRow();
  QString temp1;
  int temp2;
  if (row<8+cf->getNumMoreChannels())
    {
      temp1 = cf->getName(row+1);
      temp2 = cf->getFreq(row+1);
      cf->setName(row+1,cf->getName(row));
      cf->setFreq(row+1,cf->getFreq(row));
      cf->setName(row,temp1);
      cf->setFreq(row,temp2);
      table->setSelectedRow(row+1);
      edited();
    }
}

void optionBox::sendFreq()
{
  int i = cf->getFreq(table->getSelectedRow());
  emit listenTo(i);
}

void optionBox::edited()
{
  table->setNumberOfRows(9+cf->getNumMoreChannels());
  int i;
  for (i=0;i<(9+cf->getNumMoreChannels());i++)
    {
      table->setRowText(i,cf->getFreq(i),cf->getName(i));
    }
  emit optionsUpdate();
  configure();
}

void optionBox::aimsPushed()
{
  io20f->setEnabled(TRUE);
  io30f->setEnabled(TRUE);
  io20f->setText("0x20f");
  io30f->setText("0x30f");
  showWarn = 1;
}

void optionBox::aimsIIPushed()
{
  io20f->setEnabled(TRUE);
  io30f->setEnabled(TRUE);
  io20f->setText("0x20f");
  io30f->setText("0x30f");
  showWarn = 1;
}

void optionBox::zoltrixPushed()
{
  io20f->setEnabled(TRUE);
  io30f->setEnabled(TRUE);
  io20f->setText("0x20c");
  io30f->setText("0x30c");
  showWarn = 1;
}

void optionBox::setWarn()
{
  showWarn = 1;
}

void optionBox::kerneldevPushed()
{
  io20f->setEnabled(FALSE);
  io30f->setEnabled(FALSE);
  showWarn = 1;
}

presetTable::presetTable( QWidget *parent, const char *name )
    : QTableView(parent,name)
{
    curRow = curCol = 0;                        // set currently selected cell
    setFocusPolicy( StrongFocus );              // we accept keyboard focus
    setBackgroundMode( PaletteBase );           // set widgets background
    setNumCols( 3 );                            // set number of col's in table
    setNumRows( 0 );                            // set number of rows in table
    setCellWidth(70 );                          // set width of cell in pixels
    setCellHeight( 15 );                        // set height of cell in pixels
    setTableFlags( Tbl_vScrollBar |             // always vertical scroll bar
                   Tbl_clipCellPainting |       // avoid drawing outside cell
                   Tbl_smoothScrolling);        // easier to see the scrolling

    contents = new QString[100 * 3];  // make room for contents

    char *temp = new char [3];
    for(int i=0;i<100;i++)
      {
	if (i<9)
	  {
	    temp[0]='1'+i;
	    temp[1]='\0';
	  }
	else
	  {
	    temp[0]='0' + ((i+1) / 10);
	    temp[1]='0'+ ((i+1) % 10);
	    temp[2]='\0';
	  }
	contents[i*3] = temp;
      }
}

presetTable::~presetTable()
{
    delete[] contents;                          // deallocation
}

const char* presetTable::cellContent( int row, int col ) const
{
    return contents[indexOf( row, col )];       // contents array lookup
}

/*
  Set content of cell
*/

void presetTable::setCellContent( int row, int col, const char* c )
{
    contents[indexOf( row, col )] = c;          // contents lookup and assign
    updateCell( row, col );                     // show new content
}

void presetTable::paintCell( QPainter* p, int row, int col )
{
    int w = cellWidth( col );                   // width of cell in pixels
    int h = cellHeight( row );                  // height of cell in pixels
    int x2 = w - 1;
    int y2 = h - 1;

    /*
      Draw our part of cell frame.
    */
    p->setPen( SolidLine );
    p->drawLine( x2, 0, x2, y2 );               // draw vertical line on right
    p->setPen( DotLine );  
    p->drawLine( 0, y2, x2, y2 );               // draw horiz. line at bottom
    p->setPen( SolidLine );

    /*
      Draw extra frame inside if this is the current cell.
    */

    if ( (row == curRow) /*&& (col == 0)*/ ) { // if we are on current cell,
        if ( hasFocus() )
	  {
	    QBrush highlight(yellow);
            p->fillRect( 0, 0, x2, y2, highlight );        // draw rect. along cell edges
            //p->drawRect( 0, 0, x2, y2 );        // draw rect. along cell edges
	  }
        else
	  {                                  // we don't have focus, so
	    QBrush highlight(QColor(220,220,150));
            p->fillRect( 0, 0, x2, y2, highlight );
            p->setPen( DotLine );               // use dashed line to
            p->drawRect( 0, 0, x2, y2 );        // draw rect. along cell edges
            p->setPen( SolidLine );             // restore to normal
	  }
    }

    /*
      Draw cell content (text)
    */
    p->drawText( 0, 0, w, h, AlignCenter, contents[indexOf(row,col)] );
}

void presetTable::mousePressEvent( QMouseEvent* e )
{
    int oldRow = curRow;                        // store previous current cell
    int oldCol = curCol;
    QPoint clickedPos = e->pos();               // extract pointer position
    curRow = findRow( clickedPos.y() );         // map to row; set current cell
    //curCol = findCol( clickedPos.x() );         // map to col; set current cell
    curCol = 0;
    if ( (curRow != oldRow)                     // if current cell has moved,
         || (curCol != oldCol) )
      {
        updateCell( oldRow, 0 );           // erase previous marking
        updateCell( oldRow, 1 );           // erase previous marking
        updateCell( oldRow, 2 );           // erase previous marking
        updateCell( curRow, 0 );           // show new current cell
        updateCell( curRow, 1 );           // show new current cell
        updateCell( curRow, 2 );           // show new current cell
      }
}

void presetTable::keyPressEvent( QKeyEvent* e )
{
    int oldRow = curRow;                        // store previous current cell
    int oldCol = curCol;
    int edge = 0;
    switch( e->key() ) {                        // Look at the key code
        case Key_Up:
            if( curRow > 0 ) {
                curRow--;
                edge = topCell();
                if ( curRow < edge )
                    setTopCell( edge - 1 );
            }
            break;
        case Key_Down:
            if( curRow < numRows()-1 ) {
                curRow++;
                edge = lastRowVisible();
                if ( curRow >= edge )
                    setTopCell( topCell() + 1 );
            }
            break;
        default:                                // If not an interesting key,
            e->ignore();                        // we don't accept the event
            return;
    }

    if ( (curRow != oldRow)                     // if current cell has moved,
         || (curCol != oldCol)  )
      {
	updateCell( oldRow, 0 );
	updateCell( oldRow, 1 );
	updateCell( oldRow, 2 );
	updateCell( curRow, 0 );
	updateCell( curRow, 1 );
	updateCell( curRow, 2 );
      }
}

void presetTable::focusInEvent( QFocusEvent* )
{
    updateCell( curRow, 0 );
    updateCell( curRow, 1 );
    updateCell( curRow, 2 );
}

void presetTable::focusOutEvent( QFocusEvent* )
{
    updateCell( curRow, 0 );
    updateCell( curRow, 1 );
    updateCell( curRow, 2 );
}

int presetTable::indexOf( int row, int col ) const
{
    return (row * numCols()) + col;
}

void presetTable::setNumberOfRows( int row )
{
  setNumRows(row);
  update();
}

void presetTable::setRowText(int row, int freq, QString name)
{
  QString temp;
  if (row < 100)
    {
      temp.sprintf("%3.2f MHz",(freq/20.0 +87.0));
      contents[row*3+1] = temp;
      contents[row*3+2] = name;
    }
}

int presetTable::getSelectedRow()
{
  return curRow;
}

void presetTable::setSelectedRow(int row)
{
  if ((row >= 0) && (row < numRows()))
    {
      curRow=row;
      if ( curRow <= topCell() )
	setTopCell( curRow );
      if ( curRow >  lastRowVisible() )
	setTopCell( curRow - topCell() + lastRowVisible());
    }
}

presetEdit::presetEdit(configFile* conf, QWidget *parent=0, const char *name=0)
  : QWidget( parent, name, WStyle_Customize | WStyle_DialogBorder )
{
  resize(103,103);
  hide();
  cf = conf;

  info = new QLabel(i18n("Channel Name"),this);
  info->move(5,3);
  info->setAutoResize(true);

  freq = new QLabel(i18n("Channel Frequency"),this);
  freq->move(5,40);
  freq->setAutoResize(true);

  chName = new QLineEdit(this);
  chName->move(5,17);
  chName->resize(90,20);
  chName->setMaxLength(15);

  chFreq = new QLineEdit(this);
  chFreq->move(5,55);
  chFreq->resize(45,20);
  chFreq->setMaxLength(6);

  OK = new QPushButton(this);
  OK->setText(i18n("OK"));
  OK->resize(25,20);
  OK->move(40,80);

  Cancel = new QPushButton(this);
  Cancel->setText(i18n("Cancel"));
  Cancel->resize(35,20);
  Cancel->move(65,80);

  connect(OK, SIGNAL(clicked()), SLOT(update()));
  connect(Cancel, SIGNAL(clicked()), SLOT(cancel()));
}

void presetEdit::setChanNum(int num)
{
  chNum=num;
  chName->setText(cf->getName(num));
  QString temp;
  temp.sprintf("%3.2f",(cf->getFreq(num))/20.0 +87.0);
  chFreq->setText(temp);
}

void presetEdit::update()
{
  hide();
  cf->setName(chNum,chName->text());

  double f = atof(chFreq->text());
  int freq = int((f-86.98)*20);

  cf->setFreq(chNum,freq);
  emit edited();
}

void presetEdit::cancel()
{
  hide();
}

presetAdd::presetAdd(configFile* conf, QWidget *parent=0, const char *name=0)
  : QWidget( parent, name, WStyle_Customize | WStyle_DialogBorder )
{
  resize(103,103);
  hide();
  cf = conf;

  info = new QLabel(i18n("Channel Name"),this);
  info->move(5,3);
  info->setAutoResize(true);

  freq = new QLabel(i18n("Channel Frequency"),this);
  freq->move(5,40);
  freq->setAutoResize(true);

  chName = new QLineEdit(this);
  chName->move(5,17);
  chName->resize(90,20);
  chName->setMaxLength(15);

  chFreq = new QLineEdit(this);
  chFreq->move(5,55);
  chFreq->resize(45,20);
  chFreq->setMaxLength(6);

  OK = new QPushButton(this);
  OK->setText(i18n("OK"));
  OK->resize(25,20);
  OK->move(40,80);

  Cancel = new QPushButton(this);
  Cancel->setText(i18n("Cancel"));
  Cancel->resize(35,20);
  Cancel->move(65,80);

  connect(OK, SIGNAL(clicked()), SLOT(update()));
  connect(Cancel, SIGNAL(clicked()), SLOT(cancel()));
}

void presetAdd::setInsertPos(int num)
{
  chNum=num;
  chName->setText(i18n("NoName"));
  QString temp;
  temp.sprintf("%3.2f",87.0);
  chFreq->setText(temp);
}

void presetAdd::update()
{
  hide();
  cf->insChannel(chNum);
  cf->setName(chNum,chName->text());

  double f = atof(chFreq->text());
  int freq = int((f-86.98)*20);

  cf->setFreq(chNum,freq);
  emit edited();
}

void presetAdd::setFreq(int f)
{
  QString temp;
  temp.sprintf("%3.2f",(f/20.0)+87.0);
  chFreq->setText(temp);
}

void presetAdd::cancel()
{
  hide();
}

#endif

