// -*- c++ -*-
// **************************************************************
// $Source: /home/proj/mmm/cvsroot/mmm/base/MacroModule.cc,v $
// $Revision: 1.4 $
// $Date: 1999/05/15 22:48:22 $
// $State: Exp $
// **************************************************************

#include <string.h>
#include <stdlib.h>

#include "MacroModule.h"
#include "Slot.h"
#include "Signal.h"
#include "SignalTypes.h"
#include "InputModuleFile.h"
#include "OutputModuleFile.h"
#include "ModuleGenerator.h"
#include "ModuleErrorReport.h"
#include "toString.h"
#include "DirScan.h"
#include "MGPlugin.h"

string get_parameter(string& parameterlist)
{
    bool correct = false;
    char *scan = strdup(parameterlist.c_str()),
	*start = scan,
	*rest = 0;
    
    if (*scan++ == '{') {
	int brackets_open = 1;
	while (*scan) {
	    if (*scan == '}') {
		brackets_open --;
		if (brackets_open == 0) {
		    *scan = 0; // Make 0 terminated string
		    rest = scan+1;
		    correct = true;
		    break;
		}
	    }
	    else if (*scan == '{') brackets_open ++;
	    scan ++;
	}
    }
    if (correct) {
	string parameter = start+1;
	parameterlist = rest;
	free(start);
	return parameter;
    }
    else {
	free(start);
	return "";
    }
}


MacroModule::MacroModule(EmbeddingType embedding_type, string parameters, InputModuleFile *imf)
    : embedding_type(embedding_type)
{
    last_old_module = (Module *)modules.getLast();
    last_old_wire   = (Wire *)wires.getLast();
    if (imf) {
	filename = imf->getFilename();
	loadFromFile(*imf, parameters, false);
    }

    if (isFileRef()) {
	if (!imf) {
	    filename = get_parameter(parameters);
	    InputModuleFile newimf(filename.c_str());
	    if (newimf.isGood()) loadFromFile(newimf, parameters, false);
	}
	setName("fileref-macro-module");
    }
    else if (isInline())  setName("inline-macro-module");
    else if (!isPlugin() && !imf) setName("macro-module");
}


MacroModule::~MacroModule()
{
    deleteWiresAndModules();
}

void MacroModule::deleteWiresAndModules()
{
    wires.deleteMembers();
    while (!modules.isEmpty()) deleteModule();
}


string MacroModule::toString() const
{
    // Inline embedded Modules don't need parameters, since all submodules
    // are saved with full paramters.

    if (isInline()) return "";

    string parameterlist = "";

    // all visible modules deliver their parameter strings. These
    // are combined to my parameter. FileRef embedded module have as first
    // parameter their filename.

    if (isFileRef()) parameterlist += "{" + filename + "}";

    Module *module = (Module *)modules.getFirst();
    while (!modules.isEndOfList(module)) {
	if (module->isVisible())
	{
	    parameterlist += "{" + module->toString() + "}";
	}
	module = (Module *)(module->getNext());
    }
    return parameterlist;
}


void MacroModule::addModule(Module *module)
{
    modules.addLast(module);

    // Check all connectors of the new module, if the are
    // external. These connectors will become mine!
    Connector * const *a_connector = module->connectorTable();
    for (int c=0; c<module->numberConnectors(); c++)
    {
	if (a_connector[c]->isExternalTo(module)) addConnector(a_connector[c]);
    }
}


void MacroModule::deleteModule(Module *module)
{
    if (!module) module = (Module *)modules.getFirst();

    // Check all connectors of the module, if they are
    // external. These connectors are mine and must be removed!
    Connector * const *a_connector = module->connectorTable();
    for (int c=0; c < module->numberConnectors(); c++)
    {
	if (a_connector[c]->isExternalTo(module)) {
	    removeConnector(a_connector[c]);
	}
    }
    module->remove();
    delete module;
}


// TODO: Error description
bool MacroModule::checkWire(Connector *connector1, Connector *connector2)
{
    if (connector1->getType() != connector2->getType()) return false; // Type don't match
    
    Slot *slot = connector1->isSlot() ? (Slot *)connector1 
	: (connector2->isSlot() ? (Slot *)connector2 : 0);
    if (!slot) return false; // No slot
    else if (slot->isConnected()) return false; // Already wired.
    else if (connector1 == connector2) return true; // Unwired Slot to itself -> Parameter slot
    else if (!connector1->isSignal() && !connector2->isSignal()) return false; // Two different slots
    else return true;
}

    
Wire *MacroModule::addWire(Connector  *connector1, Connector *connector2, 
			   Module *module1, Module *module2)
{
    if (connector1 == connector2) {
	((Slot *)connector1)->makeParameter();
	return 0;
    }
    
    Signal *signal;
    Slot   *slot;
    Module *signal_module;
    Module *slot_module;

    if (connector1->isSignal()) { 
	signal        = (Signal *)connector1;
	signal_module = module1;
	slot          = (Slot   *)connector2;
	slot_module   = module2;
    }
    else {
	signal        = (Signal *)connector2;
	signal_module = module2;
	slot          = (Slot   *)connector1;
	slot_module   = module1;
    }

    slot->wireTo(signal);
    Wire *wire = new Wire(signal, slot, signal_module, slot_module);
    wires.addLast(wire);
    return wire;
}


void MacroModule::deleteWire(Wire *wire)
{
    ((Slot *)(wire->getSlot()))->disconnect();
    delete wire;
}

// ----------------------------------------------------------------------
// MacroModule acting as a Module itself
// ----------------------------------------------------------------------

bool MacroModule::isExecutable() const
{
    DLListNode *module = modules.getFirst();
    while (!modules.isEndOfList(module)) {
	if (((Module *)module)->isExecutable()) return true;
	else module = module->getNext();
    }
    return false;
}


void MacroModule::prepareForExecution(const SamplingConfig *sc)
{
    DLListNode *module = modules.getFirst();
    while (!modules.isEndOfList(module)) {
	if (((Module *)module)->isExecutable()) 
	    ((Module *)module)->prepareForExecution(sc);
	module = module->getNext();
    }
}


bool MacroModule::executeBlock(long sampling_time, long sampling_blocksize)
{
    bool finished = true;
    DLListNode *module = modules.getFirst();
    while (!modules.isEndOfList(module)) {
	if (((Module *)module)->isExecutable())
	    if (((Module *)module)->executeBlock(sampling_time, sampling_blocksize)) finished = false;
	module = module->getNext();
    }
    
    // TODO: remember which Modules to are Executable. This
    // avoids scanning all Modules everytime! And remember which
    // Modules still not have finished executing.
    return !finished;
}


void MacroModule::finishExecution()
{
    DLListNode *module = modules.getFirst();
    while (!modules.isEndOfList(module)) {
	if (((Module *)module)->isExecutable()) ((Module *)module)->finishExecution();
	module = module->getNext();
    }
}

bool MacroModule::reportModuleErrors(string& string)
{
    string = "";
    bool error_occured = false;
    DLListNode *modulenode = modules.getFirst();
    while (!modules.isEndOfList(modulenode)) {
	Module *module = (Module *)modulenode;
	ModuleErrorReport *error_report = module->getErrorReport();
	if (error_report) {
	    error_occured = true;
	    string += "Submodule " + module->getName() + ": " + error_report->getTitle() + ".\n";
	    string += error_report->getText();
	    string += "\n";
	    delete error_report;
	}
	modulenode = modulenode->getNext();
    }
    return error_occured;
}



bool MacroModule::isEmbedded() const
{
    return embedding_type != ET_TOPLEVEL;
}

bool MacroModule::isInline()   const
{
    return embedding_type == ET_INLINE;
}

bool MacroModule::isFileRef()  const
{
    return embedding_type == ET_FILEREF;
}

bool MacroModule::isPlugin()   const
{
    return embedding_type == ET_PLUGIN;
}


// File IO

bool MacroModule::loadFromFile(InputModuleFile& modulefile, string parameterlist, bool insert)
{
    if (!insert) {
	deleteWiresAndModules();
	modulefile.readLookInfo(look_info);
	modulefile.readEditorInfo(editor_info);
    }
    else modulefile.skipHeader();

    last_old_module = (Module *)modules.getLast();
    last_old_wire   = (Wire *)wires.getLast();

    long number_of_modules;
    Module **module_table = loadSubmodules(modulefile, parameterlist, number_of_modules);
    loadWires(modulefile, module_table, number_of_modules);
    delete module_table;
    return modulefile.isGood();
}

Module **MacroModule::loadSubmodules(InputModuleFile& modulefile, 
				     string parameterlist, long& number_of_modules)
{
    number_of_modules = modulefile.beginReadingSubmodules();
    if (!modulefile.isGood()) return 0;
    
    Module **module_table = new Module *[number_of_modules];
    
    if (number_of_modules < 0) {
	modulefile.setError("Invalid number of submodules: " + ::toString(number_of_modules));
	return 0;
    }

    for (int index=0; index<number_of_modules; index++) {
	
	if (!(module_table[index] = loadSubmodule(modulefile, index, parameterlist))) {
	    delete module_table;
	    return 0;
	}
	else addModule(module_table[index]);
    }
    return module_table;
}

Module *MacroModule::loadSubmodule(InputModuleFile& modulefile, long index, string& parameterlist)
{
    // parameterlist is e.g. "{2.0}{alpha,{5.0,6.00}}"

    string name;
    string constructor_string;
    short  xpos, ypos;
    bool   visible;
    long   numparslots;
    string a_parslot[MAX_NUMBER_CONNECTORS];
    modulefile.readModule(index, name, constructor_string, xpos, ypos,
			      visible, numparslots, a_parslot);
    if (modulefile.isGood())
    {
//	printf("Module: %d %s(%s) %d,%d %s %d\n",
//	       index, (const char *)name, (const char *)constructor_string, xpos, ypos,
//	       visible ? "visible" : "invisible", numparslots);

	// Visible modules are provided with parameters, which override the constructor_string!
	if (visible && parameterlist.length() > 0)
	{
	    constructor_string = get_parameter(parameterlist);
	}

	// Next try to find a ModuleGenerator that can generate the module.
	ModuleGenerator *generator
	    = (ModuleGenerator *)ModuleGenerator::findGenerator(name.c_str());
	if (!generator) {
	    modulefile.setError(string("Unknown submodule type '") + name + "'");
	    return 0;
	}
	Module *module = generator->create(constructor_string);
	
	// Define parameter slots
	for (int i=0; i<numparslots; i++) 
	{
	    Connector *connector = module->connectorNamed(a_parslot[i]);
	    if (connector && connector->isSlot()) ((Slot *)connector)->makeParameter();
	   // TODO: error handling.
	}

	// Visibility and Position
	module->setVisibility(visible);
	module->moveTo(xpos, ypos);
	return module;
    }
    return 0;
}


void MacroModule::loadWires(InputModuleFile& modulefile,
			    Module **module_table, int number_of_modules)
{
    long number_of_wires = modulefile.beginReadingWires();
    printf("Loading %ld wires...\n", number_of_wires);
    if (!modulefile.isGood()) return;
    for (int w=0; w<number_of_wires; w++) 
	if (!loadWire(modulefile, module_table, number_of_modules)) break;
}

bool MacroModule::loadWire(InputModuleFile& modulefile,
			   Module **module_table, int number_of_modules)
{
    long from, to;
    string signalname, slotname;
    modulefile.readWire(from, signalname, to, slotname);
    if (modulefile.isGood()) 
    {
	if (from < 0 || from >= number_of_modules) {
	    modulefile.setError("Wire " + ::toString(from) + "." + signalname + " -> " 
				+ ::toString(to) + "." + slotname 
				+ ": Invalid source module number");
	}
	else if (to < 0 || to >= number_of_modules) {
	    modulefile.setError("Wire " + ::toString(from) + "." + signalname + " -> " 
				+ ::toString(to) + "." + slotname 
				+ ": Invalid destination module number");
	}
	else {
	    Module *module1 = module_table[from];
	    Module *module2 = module_table[to];
	    Connector *signal = module1->connectorNamed(signalname);
	    Connector *slot   = module2->connectorNamed(slotname);
	    if (signal && slot && checkWire(signal, slot)) {
		addWire(signal, slot, module1, module2);
		return true;
	    }
	    modulefile.setError("Wire " + ::toString(from) + "." + signalname + " -> " 
				+ ::toString(to) + "." + slotname
				+ ": Signal and Slot don't match");
	}
    }
    printf("Modulefile is not good!\n");
    return false;
}


Module *MacroModule::nextNewModule()
{
    if (modules.isLast(last_old_module)) return 0;
    else {
	last_old_module = (Module *)(last_old_module->getNext());
	return last_old_module;
    }
}


Wire *MacroModule::nextNewWire()
{
    if (wires.isLast(last_old_wire)) return 0;
    else {
	last_old_wire = (Wire *)(last_old_wire->getNext());
	return last_old_wire;
    }
}


bool MacroModule::saveToFile(OutputModuleFile& modulefile, bool selection_only)
{
    if (selection_only) {
	LookInfo li;
	modulefile.writeLookInfo(li);
	EditorInfo ei;
	modulefile.writeEditorInfo(ei);
    }
    else {
	modulefile.writeLookInfo(look_info);
	modulefile.writeEditorInfo(editor_info);
    }
    saveSubmodules(modulefile, selection_only);
    saveWires(modulefile, selection_only);

    return modulefile.isGood();
}


void MacroModule::saveSubmodules(OutputModuleFile& modulefile, bool selection_only)
{
    long number_of_modules = selection_only ? numberOfSelectedModules() : numberOfModules();
    modulefile.beginWritingModules(number_of_modules);
    
    int id=0;
    Module *module;
    for (module = (Module *)modules.getFirst(); !modules.isEndOfList(module);
	 module = (Module *)module->getNext())
    {
	if (!selection_only || module->isSelected())
	{
	    module->setID(id++); // Needed later for wire references
	    modulefile.writeModule(module);
	}
    }
}


void MacroModule::saveWires(OutputModuleFile& modulefile, bool selection_only)
{
    // During the saving of the modules (which must precede the saving
    // of the wires!) are all modules unique numbered with their position
    // in the file, starting with 0. I use this ids as references.

    int number_of_wires = selection_only ? numberOfIntraSelectionWires() : numberOfWires();
    modulefile.beginWritingWires(number_of_wires);
    for (Wire *wire = (Wire *)wires.getFirst(); ! wires.isEndOfList(wire) 
	     ; wire = (Wire *)wire->getNext())
    {
	// If I should save only selected modules, I will only save wires, that connect
	// two selected modules.

	if (!selection_only || wire->bothModulesAreSelected()) modulefile.writeWire(wire);
    }
}    


/**
  * Counts, how many of the layout modules contain modules.
  */
int MacroModule::numberOfModules() const
{
    return modules.count();
}

int MacroModule::numberOfWires() const
{
    return wires.count();
}



int MacroModule::numberOfSelectedModules() const
{
    int count=0;
    for (const Module *module=(Module *)modules.getFirst()
	     ; !modules.isEndOfList(module); module = (Module *)module->getNext())
    {
	if (module->isSelected()) count ++;
    }
    return count;
}


int MacroModule::numberOfIntraSelectionWires() const
{
    int count=0;
    
    for (const Wire *wire = (Wire *)wires.getFirst(); !wires.isEndOfList(wire)
	     ; wire = (Wire *)wire->getNext())
    {
	if (wire->bothModulesAreSelected()) count++;
    }
    return count;
}


void MacroModule::scanPlugins(const char *pluginsdir)
{
    // printf("Scanning for plugins...\n");
    DirScan dirscan(pluginsdir);
    while (dirscan.toNext())
    {
	string menupath = dirscan.relativeDirPath();
	string filename = dirscan.fullFilename();
	MGPlugin *mgp = new MGPlugin(filename, menupath);
	if (!mgp->isGood()) {
	    mgp->remove();
	    printf("Skipped %s: not valid module file.\n", filename.c_str());
	}
    }
    // TODO: Nobody is deleting the MGPlugins at program end.
}
