// -*- c++ -*-
// **************************************************************
// $Source: /home/proj/mmm/cvsroot/mmm/editor/Wirebox.cc,v $
// $Revision: 1.1 $
// $Date: 1999/05/13 08:14:36 $
// $State: Exp $
// **************************************************************

#include <qlist.h>

#include "Wirebox.h"
#include "minmaxabs.h"
#include "ModuleLook.h"
#include "gridspacing.h"

const Cost COST_PLAIN          =   1;
const Cost COST_OCCUPIED       = 100; // Additional if cell locked (by ModuleLook)
const Cost COST_TOUCHING_OTHER =   1; // Touching or crossing other (to corners meet)
const Cost COST_TOUCHING_SAME  =  10; // Touching or crossing same color (is bad!)
const Cost COST_PARALLEL_OTHER =  10; // Add. per cell running parallel to other wire
const Cost COST_PARALLEL_SAME  = 999; // Parallel same color leads to graphical errors!
 
Wirebox::Wirebox(QSize size) 
    : size(size)
{
    aa_wirecell = new Wirecell *[size.width()];
    aa_locked = new unsigned short *[size.width()];
    for (int i=0; i<size.width(); i++) {
	aa_wirecell[i] = new Wirecell[size.height()];
	aa_locked[i] = new unsigned short[size.height()];
	for (int j=0; j<size.height(); j++) aa_locked[i][j] = 0;    
    }
}


Wirebox::~Wirebox()
{
    for (int i=0; i<size.width(); i++) {
	delete aa_wirecell[i];
	delete aa_locked[i];
    }
    delete aa_wirecell;
    delete aa_locked;
}

void Wirebox::lockRegion(const QRect& rect, bool lock_or_release)
{
    for (int i = max(0, rect.left()); i<=rect.right() && i<size.width(); i++)
	for (int j = max(0, rect.top()); j<=rect.bottom() && i<size.height(); j++)
	{
	    if (lock_or_release) aa_locked[i][j] ++;
	    else aa_locked[i][j] --;
	}
}


void Wirebox::paint(QPainter& p, const QRect& updaterect)
{
    short grid_left   =  max(0, updaterect.left() / GRID_SPACING - 1);
    short grid_right  =  min(size.width() - 1, 
			     (updaterect.right() + GRID_SPACING - 1)  / GRID_SPACING);
    short grid_top    =  max(0, updaterect.top() / GRID_SPACING - 1);
    short grid_bottom =  min(size.height() - 1, 
			     (updaterect.bottom() + GRID_SPACING - 1) / GRID_SPACING);

    for (int i = grid_left; i <= grid_right; i++)
	for (int j = grid_top; j <= grid_bottom; j++)
	    aa_wirecell[i][j].paint(p, i * GRID_SPACING, j * GRID_SPACING);
}

QRect Wirebox::harshlyRipoutWire(const QPoint& gridpos_from, QString path, short color)
{
    Colormask colormask = Wirecell::getColormask(color);
    
    QRect grid_updaterect = QRect(gridpos_from, gridpos_from);
    QPoint gridpos_current = gridpos_from;
    const char *readpath = path;
    while (*readpath) {
	switch(*readpath++)
	{
	case LEFT:  
	    gridpos_current.rx() --;
	    aa_wirecell[gridpos_current.x()][gridpos_current.y()]
		.remove(Wirecell::to_the_right, colormask);
	    break;
	case RIGHT:  
	    aa_wirecell[gridpos_current.x()][gridpos_current.y()]
		.remove(Wirecell::to_the_right, colormask);
	    gridpos_current.rx() ++;
	    break;
	case UP: 
	    gridpos_current.ry() --;
	    aa_wirecell[gridpos_current.x()][gridpos_current.y()]
		.remove(Wirecell::to_the_bottom, colormask);
	    break;
	case DOWN:
	    aa_wirecell[gridpos_current.x()][gridpos_current.y()]
		.remove(Wirecell::to_the_bottom, colormask);
	    gridpos_current.ry() ++;
	    break;
	}
	grid_updaterect = grid_updaterect.unite(QRect(gridpos_current, gridpos_current));

    }
    
    return QRect(grid_updaterect.x()      * GRID_SPACING,
		 grid_updaterect.y()      * GRID_SPACING,
		 grid_updaterect.width()  * GRID_SPACING,
		 grid_updaterect.height() * GRID_SPACING);
}

// ----------------------------------------------------------------------
// Find nicest path for wire
// ----------------------------------------------------------------------

struct Cell
{
    const QPoint gridpos;
    const Cell *came_from;
    const Cost cost;
    const Cost optimal_cost;
    Cell(const QPoint& gridpos, const Cell *came_from, Cost cost, Cost optimal_cost) :
	gridpos(gridpos), came_from(came_from), cost(cost), optimal_cost(optimal_cost) {};
};


typedef QListT<Cell>         CellListBase;
typedef QListIteratorT<Cell> CellIterator;


class CellList : public CellListBase
{
public:
    // CellList() {};
    ~CellList() { clear(); };  // Deletes on removal.
private:
    void deleteItem(GCI d) { delete (Cell *)d; };
    int  compareItems(GCI c1, GCI c2) { return ((Cell *)c1)->optimal_cost - ((Cell *)c2)->optimal_cost; };
};



// TEST TEST TEST TEST TEST
#include <stdio.h>
// TEST TEST TEST TEST TEST


QString Wirebox::nicelyLayoutWire(const ModuleLook *from,  const ModuleLook *to, 
				  short color, QPoint& gridpos_from, QPoint& gridpos_to, 
				  QRect& updaterect)
{
    // Search Algorithm A*

    // The search will go backwards. We start at ModuleLook "to" and find a way
    // to "from". For each cell by that we expand the search tree we remember from
    // where we came to that cell. At the end - after finding the "from", we will
    // go backwards this chain and thus get the path from->to in the right order.

    static const QPoint a_direction_horizontal_first[4] = { QPoint(-1,0), QPoint(1,0), 
							    QPoint(0,-1), QPoint(0,1) };
    static const QPoint a_direction_vertical_first[4]   = { QPoint(0,-1), QPoint(0,1),
							    QPoint(-1,0), QPoint(1,0) };

    Colormask colormask = Wirecell::getColormask(color);

    // Setup array for remembering shortest way to a cell so far
    Cost **aa_shortest_way = new Cost *[size.width()];
    for (int i=0; i<size.width(); i++) {
	aa_shortest_way[i] = new Cost[size.height()];
	for (int j=0; j<size.height(); j++) aa_shortest_way[i][j] = MAX_COST;
    }
    
/**/    long number_cells_searched = 0;

    CellList to_search; // cells to be examined, ordered by extimated optimal distance
    CellList searched;  // Already searched cells kept here for later path reconstruction

    // Setup list with starting cells. All cells in the LaytoutElement are possible
    // starting points. To exclude non-border cells would be no real optimization,
    // since all starting cells will be marked with "best path so far costs 0".

    for (int i=0; i<to->gridWidth(); i++)
	for (int j=0; j<to->gridHeight(); j++) {
	    QPoint point(to->gridLeft() + i, to->gridTop()+j);
	    to_search.inSort(new Cell(point, 0, 0, optimalCost(point, from->gridRect())));
	}
    
    const Cell *cell; // currently examined Cell.
    while (0 != (cell = to_search.getFirst()))
    {
/**/	number_cells_searched++; //  TEST TEST TEST TEST TEST
/**/	// printf("?(%d,%d:%u|%u) ", cell->gridpos.x(), cell->gridpos.y(), cell->cost, cell->optimal_cost);
	to_search.take(0);  // Get and remove first cell, i.e. cell with best chances.
	searched.insert(0, cell); // keep old cells for path reconstruction
	
	if (from->gridRect().contains(cell->gridpos)) break; // found target!
	
	// I consider from where I came. Depending on this I will expand
	// the cell in the same direction last (sorting equal thing seems to reverse their order here).
	bool vertical_first = (cell->came_from && cell->came_from->gridpos.x() == cell->gridpos.x());
	for (int d=0; d<4; d++) // Calculate neighbor cells.
	{
	    QPoint gridpos = cell->gridpos + 
		(vertical_first ? a_direction_horizontal_first[d] : a_direction_vertical_first[d]);
	    
	    if (gridpos.x() < 0 || gridpos.y() < 0 
		|| gridpos.x() >= size.width() || gridpos.y() >= size.height()) continue; // outside
	    
	    Cost cost = cell->cost + costFromTo(cell->gridpos, gridpos, colormask);

	    // If to the new cell there is already known a path that is at least not
	    // more expensive, than there is no point in examining the new cell.
	    if (cost >= aa_shortest_way[gridpos.x()][gridpos.y()]) continue; // drop
	    
	    // Remember for new cell the new cost of the entire path sofar.
	    aa_shortest_way[gridpos.x()][gridpos.y()] = cost;

	    // Save cell for later examination. Sort by estimated distance to target
	    to_search.inSort(new Cell(gridpos, cell, cost, 
				      cost + optimalCost(gridpos, from->gridRect())));
	}
    }

    if (!cell) // no way has been found! This is not possible
    {
	printf("No wirepath to target! Segfaulting...\n");
	*((char *)(0x6f840f98)) = 0; // Try to segfault :-|
    }

    gridpos_from = cell->gridpos; // Lies within the ModuleLook "from"

    // No we can reconstruct the path. The path is needed when ripping out the wire.
    QString path = "";

    QRect grid_updaterect = QRect(gridpos_from, gridpos_from);
    do {
	grid_updaterect = grid_updaterect.unite(QRect(cell->gridpos, cell->gridpos));
	if (cell->came_from)
	{
	    Direction direction = directionFromTo(cell->gridpos,cell->came_from->gridpos);
	    layWire(cell->gridpos, direction, colormask);
	    path += direction;
	}
	if (!cell->came_from)  // This is the last one
	    gridpos_to = cell->gridpos; // Lies withing the ModuleLook "to"
	cell = cell->came_from;
    } while (cell);

//    printf("Path: %s (%ld cells searched)\n", (const char *)path, number_cells_searched);
//    printf("Gridupdaterect: (%d,%d) size %d x %d\n",
//	   grid_updaterect.x(),
//	   grid_updaterect.y(),
//	   grid_updaterect.width(),
//	   grid_updaterect.height());

    updaterect = QRect(grid_updaterect.x()      * GRID_SPACING,
		       grid_updaterect.y()      * GRID_SPACING,
		       grid_updaterect.width()  * GRID_SPACING,
		       grid_updaterect.height() * GRID_SPACING);
    
    // Cleanup
    for (int i=0; i<size.width(); i++) delete aa_shortest_way[i];
    delete aa_shortest_way;

    return path;
}


Wirebox::Direction Wirebox::directionFromTo(const QPoint& from, const QPoint& to) const
{
    // I assume, that from and to are one square orthogonally apart.
    if (from.x() < to.x()) return RIGHT;
    else if (from.x() > to.x()) return LEFT;
    else if (from.y() < to.y()) return DOWN;
    else return UP;
}



void Wirebox::layWire(const QPoint& gridpos, Wirebox::Direction direction, Colormask colormask)
{
    switch (direction)
    {
    case LEFT:  
	aa_wirecell[gridpos.x()-1][gridpos.y()  ]
	    .enter(Wirecell::to_the_right, colormask);
	break;
    case RIGHT:  
	aa_wirecell[gridpos.x()]  [gridpos.y()  ]
	    .enter(Wirecell::to_the_right, colormask);
	break;
    case UP: 
	aa_wirecell[gridpos.x()]  [gridpos.y()-1]
	    .enter(Wirecell::to_the_bottom, colormask);
	break;
    case DOWN:
	aa_wirecell[gridpos.x()]  [gridpos.y()  ]
	    .enter(Wirecell::to_the_bottom, colormask);
	break;
    }
}

Wirecell& Wirebox::findWirecell(const QPoint& from, const QPoint& to, 
				Wirecell::Direction& direction) const
{
    switch(directionFromTo(from, to))
    {
    case LEFT:  
	direction = Wirecell::to_the_right;
	return aa_wirecell[to.x()][to.y()];

    case RIGHT:  
	direction = Wirecell::to_the_right;
	return aa_wirecell[from.x()][from.y()];

    case UP: 
	direction = Wirecell::to_the_bottom;
	return aa_wirecell[to.x()][to.y()];

    case DOWN:
	direction = Wirecell::to_the_bottom;
	return aa_wirecell[from.x()][from.y()];
    }
    return aa_wirecell[0][0]; // should not happen, make compiler happy here.
}


Cost Wirebox::costFromTo(const QPoint& from, const QPoint& to, Colormask colormask) const
{
    // I assume, that "from" is orthogonally neighbored to "to"

    Cost cost = COST_PLAIN;

    // parallelly running wires are considered as bad
    Wirecell::Direction wdirection;
    Wirecell& wirecell = findWirecell(from, to, wdirection);
    if (!wirecell.isFree(wdirection))
	cost += (wirecell.isFree(wdirection, colormask)
		 ? COST_PARALLEL_OTHER : COST_PARALLEL_SAME);

    // cells that are locked are considered as bad. But then there
    // will be not modifier for touching other wires in the same
    // cell, since the wires in a locked cell are not visible.
    if (aa_locked[to.x()][to.y()]) return cost += COST_OCCUPIED;
    else  cost += costForTouching(to, colormask);

    // touching other wires (coming from the same cell) is bad 
    // (parallel is a special case that is paid for twice)
    cost += costForTouching(from, colormask);

    return cost;
}

Cost Wirebox::costForTouching(const QPoint& gridpos, Colormask colormask) const
{
    const short x = gridpos.x();
    const short y = gridpos.y();
    Cost cost = 0;
    cost += costForTouching(x, y, Wirecell::to_the_right, colormask);
    cost += costForTouching(x, y, Wirecell::to_the_bottom, colormask);
    if (x > 0) cost += costForTouching(x-1, y, Wirecell::to_the_right, colormask);
    if (y > 0) cost += costForTouching(x, y-1, Wirecell::to_the_bottom, colormask);
    return cost;
}

Cost Wirebox::costForTouching(short x, short y, 
			       Wirecell::Direction wdirection, Colormask colormask) const
{
    if (!aa_wirecell[x][y].isFree(wdirection))
    {
	return (aa_wirecell[x][y].isFree(wdirection, colormask)) 
	    ?  COST_TOUCHING_OTHER : COST_TOUCHING_SAME;
    }
    else return 0;
}



Cost Wirebox::optimalCost(const QPoint& from, const QRect& to_rect) const
{
    // The optimal cost is the Manhattan distance the the nearest cell that
    // lies within to_rect. Since the target cell is always locked (wireing
    // only Layoutelements), the cost is raised by COST_OCCUPIED.

    if (to_rect.contains(from)) return 0;
    else return min(abs((long)(from.x() - to_rect.left())), 
		    abs((long)(from.x() - to_rect.right())))
	     +  min(abs((long)(from.y() - to_rect.top())),
		    abs((long)(from.y() - to_rect.bottom())))
	     + COST_OCCUPIED;
}
