#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <wave.h>
#include <util.h>

// Microsoft PCM format
enum { MICROSOFT_SHIT_PCM = 0x0001 };

typedef __attribute__ ((__packed__)) struct wav_header_t {
	BYTE riff_id[4];
	DWORD riff_data_len;
	BYTE wav_id[4];
	BYTE fmt_tag[4];
	DWORD fmt_len;
	WORD format;
	WORD nchannels;
	DWORD samples_per_second;
	DWORD avg_bytes_per_second;
	WORD block_align;
	WORD fmt_specific;
	BYTE data_tag[4];
	DWORD data_len;
};

wave_t::wave_t()
{
	data = NULL;
	sample_filename = NULL;
	size = 0;
	numchans = 0;
	rate = 0;
	bps = 0;
}

wave_t::~wave_t()
{
	if (sample_filename != NULL) {
		delete[] sample_filename;
	}

	if (data != NULL) {
		delete[] data;
	}
}

bool wave_t::try_wav_format( FILE *f )
{
	struct wav_header_t wav_header;

	memset( &wav_header, 0, sizeof( struct wav_header_t ) );
	fread( &wav_header, sizeof( struct wav_header_t ), 1, f );

	if ( memcmp( &wav_header.riff_id, "RIFF", 4 ) ) {
		DEBUG_PRINTF( ( "Not RIFF\n" ) );
		return false;
	}

	if ( memcmp( &wav_header.wav_id, "WAVE", 4 ) ) {
		DEBUG_PRINTF( ( "No WAVE tag found\n" ) );
		return false;
	}

	if (memcmp( &wav_header.fmt_tag, "fmt", 3 ) ) {
		DEBUG_PRINTF( ( "Missing format header\n" ) );
		return false;
	}

	if ( wav_header.format != MICROSOFT_SHIT_PCM ) {
		DEBUG_PRINTF( ( "Not in MS PCM format\n" ) );
		return false;
	}

	if ( memcmp( &wav_header.data_tag, "data", 4 ) ) {
		DEBUG_PRINTF( ( "Missing data header\n%s\n", wav_header.data_tag) );
		return false;
	}

	numchans = wav_header.nchannels;
	rate = wav_header.samples_per_second;
	bps = wav_header.fmt_specific;
	size = wav_header.data_len;

	if ( bps == 8 ) {
		printf( "Converting to 16 bit...\n" );
		if ( data != NULL ) delete[] data;
		data = new sample_t[size];
		for ( DWORD i = 0; i < size; i++ ) {
			BYTE inframe;
			fread( &inframe, sizeof( BYTE ), 1, f );
			data[ i ] = (sample_t) (inframe << 7);
				// shifting 7 because 8 clips (?)
		}
		bps = 16;
	} else {
		// At this point, we're going to assume it's mono 44100 16 bit
		// and allocate memory for it.  We'll mess with it later
		// if we're wrong.
		size /= 2; // convert size
		if ( data != NULL ) delete[] data;
		data = new sample_t[ size ];
		//fread( data, sizeof(BYTE), size, f );
		for ( DWORD i = 0; i < size; i++ ) {
			SWORD inframe;
			fread( &inframe, sizeof( SWORD ), 1, f);
			data[ i ] = (sample_t) inframe;
		}
	}

	if ( numchans == 2 ) {
		printf( "Converting to mono...\n" );
		monoify();
	}

	if( rate != 44100 ) {
		printf( "Resampling to 44100...\n" );
		resample( 44100 );
	}

	return true;
}

void wave_t::try_raw_format( FILE *f )
{
	fseek( f, 0, SEEK_END );
	size = ( ftell( f ) + 1 ) / 2;
	fseek( f, 0, SEEK_SET );
	numchans = 1;
	rate = 44100;
	bps = 16;
	if ( data != NULL ) delete[] data;
	data = new sample_t[ size ];

#ifdef USE_FLOATS
	for ( DWORD i = 0; i < size; i++ ) {
		SWORD inframe;
		fread( &inframe, sizeof( SWORD ), 1, f );
		data[ i ] = (sample_t) inframe;
	}
#else
	fread( data, sizeof( BYTE ), size, f );
#endif
}

bool wave_t::load_sample( const char *filename )
{
	FILE *f;

	f = fopen( filename, "rb" );
	if ( !f ) {
		printf( "Couldnt open audio file: %s\n", filename );
		return false;
	}

	if ( !try_wav_format( f ) ) {
		DEBUG_PRINTF( ( "Not WAV format, trying RAW mono 16-Bit 44100...\n" ) );
		try_raw_format( f );
	}

	fclose( f );

	if ( sample_filename != NULL ) delete[] sample_filename;
	sample_filename = new char[ strlen( filename ) + 1 ];
	strcpy( sample_filename, filename );

	return true;
}

void wave_t::printInfo()
{
	printf("size: %u\n", size);
	printf("numchans: %d\n", numchans);
	printf("rate: %u\n", rate);
	printf("bps: %d\n", bps);
}

void wave_t::resample( DWORD to_rate )
{
	DWORD delta, f;
	sample_t *new_buf, *frm, *to;

	size = size * to_rate / rate;
	new_buf = new sample_t[ size ];
	frm = data;
	to = new_buf;
	delta = 0;

	for ( f = 0; f < size; f++ ) {
		delta += rate;
		if ( delta >= to_rate ) {
			delta -= to_rate;
			*to++ = *frm++;
		} else {
			*to++ = *frm;
		}
	}

	delete[] data;
	data = new_buf;
	rate = to_rate;
}

void wave_t::monoify()
{
	sample_t *start = data;
	sample_t *end = data + size;

	size = size / 2;
	sample_t *new_buf = new sample_t[ size ];
	sample_t *to = new_buf;

	while ( start < end ) {
		sample_t temp = *start++ / 2;
		temp += *start++ / 2;

		*to++ = temp;
	}

	delete[] data;
	data = new_buf;
	numchans = 1;
}

