/*
    This file is part of ktvision, a front end to frame grabber cards 
	for the KDE-Project
    
    Copyright (C) 1998 Oliver Burtchen <oliver.burtchen@rz.hu-berlin.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include "chanlist.h"

ChanListItem::ChanListItem( const char * name="", int chan=1, 
		int fine=0, int input=0, int format=0) :
		QListBoxItem()
	{ 
		setText(name);
		ch = chan;
		fi = fine;
		in = input;
		fo = format;}

void ChanListItem::paint( QPainter *p )
{
    QFontMetrics fm = p->fontMetrics();
    p->drawText( 3,  fm.ascent() + fm.leading()/2, text() );
}

int ChanListItem::height(const QListBox *lb ) const
{
	return lb->fontMetrics().lineSpacing() + 1;
}

int ChanListItem::width(const QListBox *lb ) const
{
	return lb->fontMetrics().width( text() ) + 6;
}



ChanList::ChanList(KConfig * config) : QListBox()
{
	cfg = config;
	connect(this, SIGNAL(highlighted(int)), SLOT(emitAll(int)));
	firstChanToFreqItem = 0L;
	setMultiSelection( FALSE );
	setDragSelect(false);
	dnd=false;
		
}

ChanList::~ChanList()
{
}


void ChanList::hide()
{
	dnd = false;
	QListBox::hide();
}

void ChanList::mouseReleaseEvent(QMouseEvent *e)
{
	if (!dnd) return;
	changeCurrentPos(findItem(e->y()));
}
		
	
ChanListItem * ChanList::item (int index) const
{
	return (ChanListItem *) QListBox::item(index);
}

void ChanList::setName(const char * name)
{
	if (currentItem()==(-1)) return;
	item(currentItem())->setName(name);
	updateItem(currentItem(), false);
	emit nameChanged(name);
}

void ChanList::setChan(int i)
{
	if (currentItem()==(-1)) return;
	if (i != item(currentItem())->chan())
	{
		item(currentItem())->setChan(i);
		emit chanChanged(i);
		emit freqChanged(chanToFreq(i,item(currentItem())->fine()));
	}
}

void ChanList::setFine(int i)
{
	if (currentItem()==(-1)) return;
	if (i != item(currentItem())->fine())
	{
		item(currentItem())->setFine(i);
		emit freqChanged(chanToFreq(item(currentItem())->chan(), i));
	}
}

void ChanList::setFormat(int i)
{
	if (currentItem()==(-1)) return;
	if (i != item(currentItem())->format())
	{
		item(currentItem())->setFormat(i);
		emit formatChanged(i);
	}
}

void ChanList::setInput(int i)
{
	if (currentItem()==(-1)) return;
	if (i != item(currentItem())->input())
	{
		item(currentItem())->setInput(i);
		emit inputChanged(i);
	}
}

void ChanList::emitAll(int i)
{
	if (currentItem()==(-1)) return;
	saveItemNum = i;
	emit nameChanged(item(i)->name());
	emit chanChanged(item(i)->chan());
	emit fineChanged(item(i)->fine());
	emit inputChanged(item(i)->input());
	emit formatChanged(item(i)->format());
	emit freqChanged(chanToFreq(item(i)->chan(), item(i)->fine()));
	
}

void ChanList::whatHighlighted()
{
	emit highlighted(item(currentItem())->name());
	emit highlighted(currentItem());
}

void ChanList::cloneCurrentItem()
{
	if (currentItem()==(-1))
	{
		insertItem(new ChanListItem(
				"*** NEW ***",1,0,0,0),count());
		 return;}

	ChanListItem * tmp = item(currentItem());
	insertItem(new ChanListItem(
		"*** NEW ***",
		tmp->chan(),
		tmp->fine(),
		tmp->input(),
		tmp->format()),currentItem());
}

void ChanList::changeCurrentPos(int i)
{
	int oldcur;
	if ((oldcur=currentItem())==(-1)) return;
	if (i<0 || i>(int)count()) return;
	
	setAutoUpdate(false);
	ChanListItem * tmp = item(currentItem());
	insertItem(new ChanListItem(
		tmp->name(),
		tmp->chan(),
		tmp->fine(),
		tmp->input(),
		tmp->format()),i);

	if (i<=oldcur) oldcur++;
	removeItem(oldcur);
	setAutoUpdate(true);
	repaint();	
}

void ChanList::removeCurrentItem()
{
	removeItem(currentItem());
}


ChanListItem * ChanList::getItem(int i)
{
	if (i > (-1) && (uint)i < count())
	{
		ChanListItem * tmp = item(i);
		return new ChanListItem(
			tmp->name(),
			tmp->chan(),
			tmp->fine(),
			tmp->input(),
			tmp->format());
	}
	else return new ChanListItem(
				"*** NEW ***",1,0,0,0);
}

void ChanList::chanToFreqClear()
{
	ChanToFreqItem *tmp=firstChanToFreqItem;
	
	while(firstChanToFreqItem)
	{
		tmp=firstChanToFreqItem;
		firstChanToFreqItem=firstChanToFreqItem->next;
		delete tmp;
	}
}
	
void ChanList::chanToFreqAdd(QStrList * l)
{
	if (l->count() < 5)
	{
		delete l;
		return;
	}
	
	ChanToFreqItem *tmp=firstChanToFreqItem;
	QString str;
	if (tmp){
		while (tmp->next)
			tmp = tmp->next;
		tmp->next = new ChanToFreqItem;
		tmp = tmp->next;
		tmp->next = 0L;
	}
	else
	{
		tmp = new ChanToFreqItem;
		tmp->next = 0L;
		firstChanToFreqItem=tmp;
	}
	str = l->at(0);
	tmp->freqStart = str.toFloat();
	str = l->at(1);
	tmp->freqEnd = str.toFloat();
	str = l->at(2);
	tmp->chanStart = str.toInt();
	str = l->at(3);
	tmp->chanEnd = str.toInt();
	str = l->at(4);
	tmp->freqStep = str.toFloat();
	delete l;

}

unsigned long ChanList::chanToFreq( int i, int fine)
{
	float f = fine * 8/50;
	ChanToFreqItem *tmp=firstChanToFreqItem;
	while (tmp){
		if ( i>=tmp->chanStart && i<=tmp->chanEnd)
			return (unsigned long)
					((i-tmp->chanStart)*tmp->freqStep+tmp->freqStart+f)*16;
		tmp = tmp->next;
	}
	return (unsigned long) 0;
}



void ChanList::previous()
{
	int i;
	if ((i=currentItem()-1) > -1)
		setCurrentItem(i);
	else setCurrentItem(count()-1);
}

void ChanList::next()
{
	int i;
	if ((i=currentItem()) == (int)count()-1)
		setCurrentItem(0);
	else setCurrentItem(i+1);
}



#include "chanlist.moc"
