/* 
    sound mixer for Bt848 frame grabber driver

    Copyright (C) 1996,97 Marcus Metzler (mocm@thp.uni-koeln.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * mixer.cpp
 */

#include <stream.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <sys/errno.h>
#include <sys/ioctl.h>
#include <fcntl.h>

#include "mixer.h"
#include "wintv.h"

char* labels[]={  "Volume" , "Bass"  , "Treble"    , "Synth", "Pcm"  , \
		  "Speaker", "Line In"  , "Microphone", "CD"   , "Mix"  , \
		  "Pcm2"   , "RecMon", "IGain", "OGain", "Line1", \
		  "Line2"  , "Line3" };

char *names[] = SOUND_DEVICE_NAMES;
//char *names[] = MixerDevNames;
//char *labels[] = SOUND_DEVICE_LABELS;

#ifdef myDEBUG
#define sDEBUG
#endif

unsigned int mixer::devmask,mixer::recmask,mixer::recsrc,mixer::stereodevs,mixer::caps;
unsigned int mixer::locked_channels, mixer::muted_channels;
int  mixer::devNr;
bool mixer::globaldone= false;
              
mixer::mixer(const char *filename, unsigned int device, bool rec )
{
  int i;

  devNr= -1;
  dev= NO_MDEV;

  svolumes.l=0;
  svolumes.r=0;

  // don't know which mixer device to open
  if ( !filename ) {
    cerr << "Warning: mixer: No device (/dev/mixer) given!" << endl;
    return;
  }
  // try to open /dev/mixer
  if (-1 == (mixer_device = open(filename,O_RDONLY))) {
    cerr << "Warning: mixer: Error opening mixer device " << filename << endl;
    return;
  }
  if ( !globaldone ) {
    devmask=0;
    recmask=0;
    recsrc=0;
    stereodevs=0;
    // read device mask
    if (-1 == ioctl(mixer_device,MIXER_READ(SOUND_MIXER_DEVMASK),&devmask)) {
      cerr << "Warning: mixer: Error mixer read devmask" << endl;
      devmask=0;
      close(mixer_device);
      return;
    }
    // read record mask
    if (-1 == ioctl(mixer_device,MIXER_READ(SOUND_MIXER_RECMASK),&recmask)) {
      cerr << "Warning: mixer: Error mixer read recmask" << endl;
      recmask=0;
      close(mixer_device);
      return;
    }
    // read record sources
    if (-1 == ioctl(mixer_device,MIXER_READ(SOUND_MIXER_RECSRC),&recsrc)) {
      cerr << "Warning: mixer: Error mixer read recscr" << endl;
      recsrc=0;
      close(mixer_device);
      return;
    }
    // read stereo devices
    if (-1 == ioctl(mixer_device,MIXER_READ(SOUND_MIXER_STEREODEVS),&stereodevs)) {
      cerr << "Warning: mixer: Error mixer read stereodevs" << endl;
      stereodevs=0;
      close(mixer_device);
      return;
    }
  } else globaldone=true;

  // check if given device is suitable
  if ( NO_MDEV != device ) {
    if ( ((1<<device) & devmask) ) {
      if ( -1 == ioctl(mixer_device,MIXER_READ(device),&svolumes) ) {
	cerr << "Warning: mixer: Error mixer read values from device" << endl;
	close(mixer_device);
	return;
      } else if ( rec ) {
	// Change status of record source(s)
	unsigned int newrecmask= (1<<device);
	if ( ioctl(mixer_device, SOUND_MIXER_WRITE_RECSRC, &newrecmask) == -1 )
	  cerr << "Warning: mixer: Error mixer write recsource" << endl;
	// Re-read status of record source(s). Just in case, OSS does not like
	// my settings. And with this line mix->recsrc gets its new value. :-)
	if ( ioctl(mixer_device, SOUND_MIXER_READ_RECSRC, &recsrc) == -1 )
	  cerr << "Warning: mixer: Error mixer read recsource" << endl;
	dev = device;
	return;
      } else {
	dev = device;
	return;
      }
    }
  }

  // device is not suitable...
  if ( dev == NO_MDEV && device != NO_MDEV && rec ) {
    cerr << "Warning: mixer: The mixer device <" << labels[device] << "> is not suitable!" << endl;
    cerr << "mixer: Aviable devices are: " << endl;
    for (i = 0; i < SOUND_MIXER_NRDEVICES; i++)
      if ( ((1<<i) & devmask) /*&& ((1<<i) & recsrc)*/ && ((1<<i) & recmask) ) {
	if (-1 == ioctl(mixer_device,MIXER_READ(i),&svolumes)) {
	  cerr << "Warning: mixer: Error mixer read volume" << endl;
	}
	cerr << "mixer: " << labels[i] << " <" << (int)svolumes.l << "," << (int)svolumes.r;
	if ( (1<<i) & stereodevs )  cerr << "> [stereo]";
	else cerr << "> [mono]";
	if ( (1<<i) & recsrc ) cerr << " (active)" << endl;
	else cerr << endl;
	dev=i;
      }
    cerr << endl << "\t Select <Options>|<General Options>|<Mixer> and choose" << endl;
    cerr << "\t a suitable mixer device." << endl;
    cerr << "\t You can also use an external mixer like kmix and" << endl;
    cerr << "\t try to get sound." << endl;
    cerr << "\t Remember: The line out of your tv tuner card has to be" << endl;
    cerr << "\t connected to the line in of your sound card!" << endl << endl;
  } else {
#ifdef sDEBUG
    cerr << "mixer: Aviable devices are: " << endl;
#endif
    unsigned int last;
    for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
      if ( ((1<<i) & devmask) /*&& ((1<<i) & recsrc)*/ && ((1<<i) & recmask) ) {
	if (-1 == ioctl(mixer_device,MIXER_READ(i),&svolumes)) {
	  cerr << "Warning: mixer: Error mixer read volume" << endl;
	}
#ifdef sDEBUG
	cerr << "mixer: " << labels[i] << " <" << (int)svolumes.l << "," << (int)svolumes.r;
	if ( (1<<i) & stereodevs )  cerr << "> [stereo]";
	else cerr << "> [mono]";
	if ( (1<<i) & recsrc ) cerr << " (active)" << endl;
	else cerr << endl;
#endif
	if ( (1<<i) & stereodevs ) dev=i;
	last= i;
      }
    }
    if ( dev == NO_MDEV ) dev= last;
  }

  return;
}


mixer::~mixer()
{ 
  if ( NO_MDEV != dev ) close(mixer_device); 
}

int mixer::name2Device(const char *_devName)
{
  int i;
  for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
    if (strcasecmp(labels[i],_devName) == 0)
      return(i);
  } 
  return(NO_MDEV);
}

char * mixer::getFirstDeviceName( int * nr, bool rec_only )
{
  devNr=-1;
  return getNextDeviceName( nr, rec_only );
}

char * mixer::getNextDeviceName( int * nr, bool rec_only )
{
  devNr++;
  if (devNr >= SOUND_MIXER_NRDEVICES) { *nr=0; return(NULL); }
  if ( rec_only ) {
    while ( !(((1<<devNr) & devmask) /*&& ((1<<devNr) & recsrc)*/ && ((1<<devNr) & recmask)) 
	     && (devNr < SOUND_MIXER_NRDEVICES) ) devNr++;
  } else {
    while ( !(((1<<devNr) & devmask) && (devNr < SOUND_MIXER_NRDEVICES)) ) devNr++;
  }
  if (devNr >= SOUND_MIXER_NRDEVICES) { *nr=0; return(NULL); }
#ifdef sDEBUG
  cerr << "Debug: mixer: getNextDeviceName " << devNr << " " << labels[devNr] << endl;
#endif
  *nr=devNr;
  return(labels[devNr]);
}

bool mixer::setDevice(unsigned int _dev)
{
#ifdef sDEBUG
  cerr << "Debug: mixer: ["<< dev << "] setDevice [" << (int)_dev << "] " << labels[_dev] << endl;
#endif

  if ( dev == NO_MDEV ) return false;

  dev=_dev;
  unsigned int newrecmask= (1<<dev);

  // Change status of record source(s)
  if ( ioctl(mixer_device, SOUND_MIXER_WRITE_RECSRC, &newrecmask) == -1 ) {
    cerr << "Warning: mixer: Error mixer write recsource" << endl;
    return false;
  }
  // Re-read status of record source(s). Just in case, OSS does not like
  // my settings. And with this line mix->recsrc gets its new value. :-)
  if ( ioctl(mixer_device, SOUND_MIXER_READ_RECSRC, &recsrc) == -1 ) {
    cerr << "Warning: mixer: Error mixer read recsource" << endl;
    return false;
  }
  if (-1 == ioctl(mixer_device,MIXER_READ(dev),&svolumes)) {
    cerr << "Warning: mixer: Error mixer read volume" << endl;
    return false;
  }

#ifdef sDEBUG
  cerr << "Debug: mixer: setDevice done [" << (int)dev << "] " << labels[dev] << endl;
  cerr << "Debug: mixer: setDevice: new recsrc: ";
  int i;
  for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
    if (  ((1<<i) & recsrc) ) {
      cerr << labels[i] << "  ";
    }
  }
  cerr << endl;
#endif
  return true;
}

bool mixer::isStereoDevice()
{
  if ( (1<<dev) & stereodevs ) return true;
  return false;
}

void mixer::setVolume(int vol, int pos)
{
  //l &= 0x7f;
  //l = l | (l << 8);;
  switch (pos) {
  case VOL_LEFT:
    svolumes.l=vol;
    break;
  case VOL_RIGHT:
    svolumes.r=vol;
    break;
  case VOL_BOTH:
  default:
    svolumes.l=vol;
    svolumes.r=vol;
    break;
  }

  if ( NO_MDEV != dev ) {
    if ( -1 == ioctl(mixer_device, MIXER_WRITE(dev), &svolumes) ) {
      cerr << "Warning: mixer: Error mixer write volume to dev " << (int)dev << endl;
    }
#ifdef sDEBUG
    cerr << "Debug: mixer: set " << labels[dev] << " to " 
	 << (int) svolumes.l << ", " << (int) svolumes.r << endl;
#endif
  }
}

void mixer::setStereoVolume(int rvol, int lvol)
{
  svolumes.l=lvol;
  svolumes.r=rvol;
  if ( NO_MDEV != dev ) {
    if ( -1 == ioctl(mixer_device, MIXER_WRITE(dev), &svolumes) ) {
      cerr << "Warning: mixer: Error mixer write volume to dev " << (int)dev << endl;
    }
#ifdef sDEBUG
    cerr << "Debug: mixer: set " << labels[dev] << " to " 
	 << (int) svolumes.l << ", " << (int) svolumes.r << endl;
#endif
  }
}

int mixer::getVolume(int pos)
{
#ifdef sDEBUG
  cerr << "Debug: mixer: read " << labels[dev] << " [" << dev << "]:  " 
       << (int) svolumes.l << ", " << (int) svolumes.r << endl;
#endif

  switch (pos) {
  case VOL_LEFT:
    return svolumes.l;
  case VOL_RIGHT:
    return svolumes.r;
  case VOL_BOTH:
  default:
    cerr << "Warning: mixer: Can not get VOL_BOTH!" << endl;
    break;
  }
  return 0;
}




