/* 
    kvtx

    Copyright (C) 1998,1999 Moritz Wenk (wenk@mathematik.uni-kl.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <pagelist.h>

#ifdef myDEBUG
#define pDEBUG
#endif

#define LISTSIZE 1000

pagedir::pagedir( const char * _dir, bool _keepInMem )
  : keepInMem(_keepInMem)
{
  dir= _dir;

  pagelist= (struct onepage **) calloc( LISTSIZE, sizeof(struct onepage *) );

  if ( _dir ) rescan();
}

pagedir::~pagedir()
{
  freeMem();
  
  free( pagelist );
}

void pagedir::freeMem()
{
#ifdef pDEBUG
  fprintf(stderr,"pagelist: freeMem starts\n");
#endif
  int i;
  struct onepage *p,*pp;
  for ( i = 0; i < LISTSIZE; i++ ) {
    p= pagelist[i];
    while ( p ) {
      pp= p->next;
      delete p->pagep;
      delete p;
      p= pp;
    }
    pagelist[i]=0;
  }
#ifdef pDEBUG
  fprintf(stderr,"pagelist: freeMem ends\n");
#endif
}

bool pagedir::setDir( const char * _dir )
{
  dir= _dir;
  return rescan();
}

bool pagedir::rescan()
{
#ifdef pDEBUG
  fprintf(stderr,"pagelist: rescan() starts\n");
#endif
  int cpage,spage;
  onepage * npage;
  onepage * prevp;
  bool firstSPage;

  freeMem();

  cacheDirDir.setPath(dir);
  cacheDirDir.setFilter( QDir::Files | QDir::NoSymLinks );
  cacheDirDir.setSorting( QDir::Name );

  const QFileInfoList *filelist = cacheDirDir.entryInfoList();
  QFileInfoListIterator it( *filelist );      // create list iterator

  while ( (fi=it.current()) ) {           // for each file...
    // current page number
    cpage= fi->baseName().left(3).toInt();
    firstSPage=true;
    // scan for subpages
    while ( (fi=it.current()) && (fi->baseName().left(3).toInt() == cpage) ) {           // for each file...
      spage= fi->baseName().right(2).toInt();
      // insert into list
      npage= new onepage;
      npage->subpage= spage;
      npage->pagep=0;
      npage->next=0;
      npage->prev=0;
      if ( firstSPage ) {
#ifdef pDEBUG
	//fprintf(stderr,"pagelist: insert %d, %d",cpage,spage);
#endif
	pagelist[cpage]=npage;
	firstSPage=false;
	prevp= npage;
      } else {
#ifdef pDEBUG
	//fprintf(stderr,", %d",spage);
#endif
	npage->prev=prevp;
	prevp->next= npage;
	prevp= npage;
      }
      ++it;                               // goto next list element
    }
#ifdef pDEBUG
    //fprintf(stderr,"\n");
#endif
    //++it;                               // goto next list element
  }
  opage=0;
  thepage= 0;
#ifdef pDEBUG
  fprintf(stderr,"pagelist: rescan() ends\n");
#endif
  return true;
}

vtxpage_t * pagedir::setPage( int _page, bool reloadPage )
{
#ifdef pDEBUG
  fprintf(stderr,"pagelist: setPage %d, %s\n",_page,reloadPage?"reload":"no reload");
#endif
  thepage= _page;
  if ( (opage && opage->pagep && !keepInMem) || reloadPage ) {
    // free old page mem
    delete opage->pagep;
    opage->pagep=0;
  }

  opage= pagelist[thepage];
  if ( !opage ) return 0;

  if ( !opage->pagep ) {
    opage->pagep= loadFromFile( (const char *) getPageName( thepage, opage->subpage ) );
  }

  thesubpage= opage->subpage;

  return opage->pagep;
}

bool pagedir::hasSubPages()
{
  return ( (pagelist[thepage])->next != 0 );
}

vtxpage_t * pagedir::cycleSubPages( bool reloadPage )
{
  
  if ( opage->next ) {
    opage= opage->next;
  } else {
    opage= pagelist[thepage];
  }
  thesubpage= opage->subpage;
  
  if ( (opage && opage->pagep && !keepInMem) || reloadPage ) {
    // free old page mem
    delete opage->pagep;
    opage->pagep=0;
  }

  if ( !opage->pagep ) {
    opage->pagep= loadFromFile( (const char *) getPageName( thepage, opage->subpage ) );
  }

  return opage->pagep;
}

vtxpage_t * pagedir::setSubPage( int _spage, bool reloadPage )
{
  thesubpage=-1;

  if ( (opage && opage->pagep && !keepInMem) || reloadPage ) {
    // free old page mem
    delete opage->pagep;
    opage->pagep=0;
  }

  opage= pagelist[thepage];
  if ( !opage ) return 0;

  while ( opage->next && (opage->subpage < _spage) ) {
    opage= opage->next;
  }
  if ( opage->subpage != _spage ) return 0;

  if ( !opage->pagep ) {
    opage->pagep= loadFromFile( (const char *) getPageName( thepage, opage->subpage ) );
  }

  thesubpage= opage->subpage;

  return opage->pagep;
}

vtxpage_t * pagedir::nextPage( bool reloadPage )
{
#ifdef pDEBUG
  fprintf(stderr,"pagelist: nextPage %s\n",reloadPage?"reload":"no reload");
#endif

  thepage++;

  while ( !pagelist[thepage] && (thepage <= LASTPAGE) ) thepage++;

  if ( thepage > LASTPAGE ) {
    return 0;
  }

  if ( (opage && opage->pagep && !keepInMem) || reloadPage ) {
    // free old page mem
    delete opage->pagep;
    opage->pagep=0;
  }

  opage= pagelist[thepage];
  if ( !opage ) return 0;

  if ( !opage->pagep ) {
    opage->pagep= loadFromFile( (const char *) getPageName( thepage, opage->subpage ) );
  }

  thesubpage= opage->subpage;

  return opage->pagep;
}

vtxpage_t * pagedir::prevPage( bool reloadPage )
{
#ifdef pDEBUG
  fprintf(stderr,"pagelist: nextPage %s\n",reloadPage?"reload":"no reload");
#endif

  thepage--;

  while ( !pagelist[thepage] && (thepage >= FIRSTPAGE) ) thepage--;

  if ( thepage < FIRSTPAGE ) {
    return 0;
  }

  if ( (opage && opage->pagep && !keepInMem) || reloadPage ) {
    // free old page mem
    delete opage->pagep;
    opage->pagep=0;
  }

  opage= pagelist[thepage];
  if ( !opage ) return 0;

  if ( !opage->pagep ) {
    opage->pagep= loadFromFile( (const char *) getPageName( thepage, opage->subpage ) );
  }

  thesubpage= opage->subpage;

  return opage->pagep;
}

vtxpage_t * pagedir::nextSubPage( bool reloadPage )
{
  return cycleSubPages( reloadPage );
}

vtxpage_t * pagedir::prevSubPage( bool reloadPage )
{
  if ( opage->prev ) {
    opage= opage->prev;
  } else {
    while ( opage->next )
      opage= opage->next;
  }
  thesubpage= opage->subpage;
  
  if ( (opage && opage->pagep && !keepInMem) || reloadPage ) {
    // free old page mem
    delete opage->pagep;
    opage->pagep=0;
  }

  if ( !opage->pagep ) {
    opage->pagep= loadFromFile( (const char *) getPageName( thepage, opage->subpage ) );
  }

  return opage->pagep;
}

vtxpage_t * pagedir::reload()
{
  if (opage && opage->pagep ) {
    // free old page mem
    delete opage->pagep;
    opage->pagep=0;
  }

  if ( !opage->pagep ) {
    opage->pagep= loadFromFile( (const char *) getPageName( thepage, thesubpage ) );
  }

  return opage->pagep;
}

vtxpage_t * pagedir::loadFromFile( const char * name )
{
#ifdef pDEBUG
  fprintf(stderr,"pagelist: loading file %s\n",name);
#endif
  FILE *fp;

  fp = fopen(name,"r");
  if ( !fp ) {
    warning("vtx: Error loading %s: %s\n",name,strerror(errno));
    return 0;
  }

  vtxpage_t * newpage= new vtxpage_t;

  load_vtx(fp, buffer, &(newpage->info), &virtualpage);
  fclose(fp);

  newpage->info.charset= 4;//vtx_charset; // german 
  decode_page( buffer, newpage, 0, 23);

  return newpage;
}

QString pagedir::getPageName( int p, int sp )
{
  currPageS.setNum(p);
  currSubPageS.setNum(sp);
  if ( sp < 10 ) currSubPageS="0"+currSubPageS;
  QString lpage=dir+"/"+currPageS+"_"+currSubPageS+".vtx";
  return lpage;
}


cachedir::cachedir( const char * _dir )
{
  dir=_dir;
  clist= new QStrList;
  if ( _dir ) rescan();
}

cachedir::~cachedir()
{
  freeMem();
  delete clist;
}

bool cachedir::setDir( const char * _dir )
{
  dir= _dir;
  return rescan();
}

bool cachedir::rescan()
{
  freeMem();

  cacheDirDir.setPath(dir);
  cacheDirDir.setFilter( QDir::Dirs | QDir::NoSymLinks );
  cacheDirDir.setSorting( QDir::Name );

  const QFileInfoList *filelist = cacheDirDir.entryInfoList();
  QFileInfoListIterator it( *filelist );      // create list iterator

  while ( (fi=it.current()) ) {
    if ( (fi->fileName() != ".") && (fi->fileName() != "..") ) {
      clist->append( fi->fileName() );
#ifdef pDEBUG
      fprintf(stderr,"cachedir: rescan: %s\n",(const char *)fi->fileName());
#endif
    }
    ++it;
  }

  return true;
}

QStrList * cachedir::getList()
{
  return clist;
}

void cachedir::freeMem()
{
  clist->clear();
}


