/* -*- C++ -*-
 * This file implements a widget for navigating through sets of data.
 * 
 * the K Desktop Environment
 * copyright:  (C) Mirko Sucker, 1999
 * license:    GNU Public License, Version 2
 * mail to:    Mirko Sucker <mirko.sucker@unibw-hamburg.de>
 * requires:   recent C++-compiler, at least Qt 1.4, STL
 * $Revision: 1.3 $
 */
#include "kdatanavigator.h"
#include <qlineedit.h>
#include <qspinbox.h>
#include <qcombobox.h>
#include <qstrlist.h>
#include <kbutton.h>
#include <kiconloader.h>
#include <kapp.h>
#include "debug.h"

// #############################################################################
// MOC OUTPUT FILES:
#include "kdatanavigator.moc"
// #############################################################################

KDataNavigator::KDataNavigator(QWidget* parent, const char* name)
  : QFrame(parent, name),
    buttonFirst(new KButton(this)),
    buttonPrevious(new KButton(this)),
    buttonNext(new KButton(this)),
    buttonLast(new KButton(this)),
    buttonNew(new KButton(this)),
    leIndex(new QLineEdit(this)),
    sbIndex(new QSpinBox(this)),
    cbIndex(new QComboBox(this)),
    min(0),
    max(0),
    current(0),
    mode(Manual),
    buttonNewVisible(true)
{
  // ###########################################################################
  KButton* buttons[]= {
    buttonFirst, buttonPrevious, buttonNext, buttonLast, buttonNew
  };
  const char* icons[]= {
    "start.xpm", "back.xpm", "forward.xpm", "finish.xpm", "filenew2.xpm"
  };
  int count;
  QPixmap pixmap;
  // -----
  if(buttonFirst==0 || buttonPrevious==0 || buttonNext==0 || buttonLast==0 ||
     buttonNew==0 || leIndex==0 || sbIndex==0 || cbIndex==0)
    {
      // possibly throw exception when they are completely implemented
    }
  leIndex->setEnabled(false);
  for(count=0; count<5; ++count)
    {
      pixmap=kapp->getIconLoader()->loadIcon(icons[count]);
      if(!pixmap.isNull())
	{
	  buttons[count]->setPixmap(pixmap);
	}
    }
  // -----
  connect(buttonFirst, SIGNAL(clicked()), SLOT(buttonFirstClicked()));
  connect(buttonPrevious, SIGNAL(clicked()), SLOT(buttonPreviousClicked()));
  connect(buttonNext, SIGNAL(clicked()), SLOT(buttonNextClicked()));
  connect(buttonLast, SIGNAL(clicked()), SLOT(buttonLastClicked()));
  connect(buttonNew, SIGNAL(clicked()), SLOT(buttonNewClicked()));
  connect(leIndex, SIGNAL(textChanged(const char*)),
	  SLOT(leTextChanged(const char*)));
  connect(cbIndex, SIGNAL(activated(int)), SLOT(cbItemSelected(int)));
  connect(sbIndex, SIGNAL(valueChanged(int)), SLOT(sbValueChanged(int)));
  // -----
  setFrameStyle(QFrame::Box | QFrame::Plain);
  setLineWidth(1);
  enableButtons();
  // ###########################################################################
}

KDataNavigator::~KDataNavigator()
{
  // ###########################################################################
  // ###########################################################################
}

void
KDataNavigator::resizeEvent(QResizeEvent*)
{
  // ###########################################################################
  setLayout();
  // ###########################################################################
}

void
KDataNavigator::setLayout()
{
  // ###########################################################################
  // ----- excuse me the bad construct:
  QWidget *widgets[]= {
    buttonFirst, buttonPrevious,
    mode==Manual ? leIndex : (mode==List ? cbIndex : sbIndex), 
    buttonNext, buttonLast, buttonNew
  };
  const int ButtonHeight=
    buttonFirst->sizeHint().height()>0
    ? buttonFirst->sizeHint().height()
    : height()-2*frameWidth();
  const int NoOfButtons= buttonNewVisible ? 5 : 4;
  const int NoOfElements=sizeof(widgets)/sizeof(widgets[0]);
  const int ButtonWidth=
    width()>(NoOfButtons+2)*ButtonHeight+2*frameWidth()
    ? ButtonHeight
    : (width()-2*frameWidth())/(NoOfElements+1);
  int widths[]= {
    ButtonWidth, ButtonWidth,
    width()-2*frameWidth()-(NoOfButtons)*ButtonWidth,
    ButtonWidth, ButtonWidth, buttonNewVisible ? ButtonWidth : 0
  };
  int count, cx, h;
  // -----
  h=height()-2*frameWidth();
  cx=frameWidth();
  for(count=0; count<NoOfElements; ++count)
    {
      widgets[count]->setGeometry(cx, frameWidth(), widths[count], h);
      cx+=widths[count];
    }  
  // ###########################################################################
}


QSize
KDataNavigator::sizeHint()
{
  // ###########################################################################
  const int ButtonWidth=20; // KButtons do not seem to report it
  const int NoOfButtons= buttonNewVisible ? 5 : 4;
  // -----
  switch(mode)
    {
    case Manual:
      return QSize((NoOfButtons+6)*ButtonWidth+2*frameWidth(),
		   ButtonWidth+2*frameWidth());
    case List:
      return QSize((NoOfButtons+8)*ButtonWidth+2*frameWidth(),
		   ButtonWidth+2*frameWidth());
    default:
      return QSize((NoOfButtons+4)*ButtonWidth+2*frameWidth(),
		   ButtonWidth+2*frameWidth());
    }
  // ###########################################################################
}

void
KDataNavigator::buttonFirstClicked()
{
  // ###########################################################################
  switch(mode)
    {
    case Manual: // the most easy case: do nothing
      break;
    case Index:
      if(min<current)
	{
	  current=min;
	  emit(itemSelected(current));
	  sbIndex->setValue(current);
	  enableButtons();
	} else {
	  kapp->beep(); // cannot happen
	}
      break;
    case List:
      if(current>0 && cbIndex->count()>0)
	{
	  current=0;
	  cbIndex->setCurrentItem(current);
	  emit(itemSelected(current));
	  enableButtons();
	}
      break;
    }
  emit(first());
  // ###########################################################################
}

void
KDataNavigator::buttonPreviousClicked()
{
  // ###########################################################################
  switch(mode)
    {
    case Manual:
      break;
    case Index:
      if(min<current)
	{
	  --current;
	  emit(itemSelected(current));
	  sbIndex->setValue(current);
	  enableButtons();
	} else {
	  kapp->beep();
	}
      break;
    case List:
      if(current>0 && cbIndex->count()>0)
	{
	  --current;
	  cbIndex->setCurrentItem(current);
	  emit(itemSelected(current));
	  enableButtons();
	}  
      break;
    }
  emit(previous());
  // ###########################################################################
}

void
KDataNavigator::buttonNextClicked()
{
  // ###########################################################################
  switch(mode)
    {
    case Manual:
      break;
    case Index:
      if(current<max)
	{
	  ++current;
	  emit(itemSelected(current));
	  sbIndex->setValue(current);
	  enableButtons();
	} else {
	  kapp->beep();
	}
      break;
    case List:
      if(current<cbIndex->count()-1 && cbIndex->count()>0)
	{
	  ++current;
	  cbIndex->setCurrentItem(current);
	  emit(itemSelected(current));
	  enableButtons();
	}      
      break;
    }
  emit(next());
  // ###########################################################################
}

void
KDataNavigator::buttonLastClicked()
{
  // ###########################################################################
  switch(mode)
    {
    case Manual:
      break;
    case Index:
      if(current<max)
	{
	  current=max;
	  emit(itemSelected(current));
	  sbIndex->setValue(current);
	  enableButtons();
	} else {
	  kapp->beep();
	}
      break;
    case List:
      if(current<cbIndex->count()-1 && cbIndex->count()>0)
	{
	  current=cbIndex->count()-1;
	  cbIndex->setCurrentItem(current);
	  emit(itemSelected(current));
	  enableButtons();
	}      
      break;
    }
  emit(last());
  // ###########################################################################
}

void
KDataNavigator::buttonNewClicked()
{
  // ###########################################################################
  emit(newItem());
  // ###########################################################################
}

void
KDataNavigator::setText(const char* c)
{
  // ###########################################################################
  setMode(Manual);
  leIndex->setText(c);
  // ###########################################################################
}

void
KDataNavigator::setIndexMode(int begin, int end, int cur)
{
  // ###########################################################################
  min=begin;
  max=end;
  current=cur;
  sbIndex->setRange(min, max);
  if(min<=current && current<=max)
    {
      sbIndex->setValue(current);
    } else {
      sbIndex->setValue(min);
    }
  leIndex->hide();
  cbIndex->hide();
  sbIndex->show();
  // -----
  mode=Index;
  enableButtons();
  // ###########################################################################
}

void
KDataNavigator::showButtonNew(bool state)
{
  // ###########################################################################
  if(state)
    {
      buttonNew->show();
    } else {
      buttonNew->hide();
    }  
  buttonNewVisible=state;
  // -----
  setLayout();
  // ###########################################################################
}

void
KDataNavigator::enableButtonFirst(bool state)
{
  // ###########################################################################
  buttonFirst->setEnabled(state);
  // ###########################################################################
}
  
void
KDataNavigator::enableButtonNext(bool state)
{
  // ###########################################################################
  buttonNext->setEnabled(state);
  // ###########################################################################
}
  
void
KDataNavigator::enableButtonPrevious(bool state)
{
  // ###########################################################################
  buttonPrevious->setEnabled(state);
  // ###########################################################################
}
  
void
KDataNavigator::enableButtonLast(bool state)
{
  // ###########################################################################
  buttonLast->setEnabled(state);
  // ###########################################################################
}
  
void
KDataNavigator::enableButtonNew(bool state)
{
  // ###########################################################################
  buttonNew->setEnabled(state);
  // ###########################################################################
}

void
KDataNavigator::enableButtons()
{
  // ###########################################################################
  switch(mode)
    {
    case Manual:
      break;
    case Index:
      buttonFirst->setEnabled(current!=min);
      buttonPrevious->setEnabled(current!=min);
      buttonNext->setEnabled(current!=max);
      buttonLast->setEnabled(current!=max);
      break;
    case List:
      buttonFirst->setEnabled(current!=0);
      buttonPrevious->setEnabled(current!=0);
      buttonNext->setEnabled(current!=cbIndex->count()-1 && cbIndex->count()>0);
      buttonLast->setEnabled(current!=cbIndex->count()-1 && cbIndex->count()>0);
      cbIndex->setEnabled(cbIndex->count()>0);
      break;
    }
  // ###########################################################################
}
  
void
KDataNavigator::leTextChanged(const char* c)
{
  // ###########################################################################
  emit(textChanged(c));
  // ###########################################################################
}

void
KDataNavigator::sbValueChanged(int i)
{
  // ###########################################################################
  if(min<=i && i<=max)
    {
      current=i;
      enableButtons();
      emit(itemSelected(i));
    }
  // ###########################################################################
}

void
KDataNavigator::cbItemSelected(int i)
{
  // ###########################################################################
  current=i;
  enableButtons();
  emit(itemSelected(i));
  // ###########################################################################
}

void
KDataNavigator::setMode(Modes m)
{
  // ###########################################################################
  mode=m;
  switch(mode)
    {
    case Manual:
      leIndex->show();
      sbIndex->hide();
      cbIndex->hide();
      buttonFirst->setEnabled(true);
      buttonPrevious->setEnabled(true);
      buttonNext->setEnabled(true);
      buttonLast->setEnabled(true);
      break;
    case Index:
      leIndex->hide();
      sbIndex->show();
      cbIndex->hide();
      break;
    case List:
      leIndex->hide();
      sbIndex->hide();
      cbIndex->show();
      break;
    }
  enableButtons();
  setLayout();
  // ###########################################################################
}

void
KDataNavigator::setList(QStrList* strings)
{
  register bool GUARD; GUARD=true;
  // ###########################################################################
  LG(GUARD, "KDataNavigator::setList: called, %i items.\n", strings->count());
  // -----
  cbIndex->clear();
  CHECK(cbIndex->count()==0);
  cbIndex->insertStrList(strings);
  CHECK((unsigned)cbIndex->count()==strings->count());
  if(cbIndex->count()>0)
    {
      cbIndex->setCurrentItem(0);
      current=0;
    }
  setMode(List);
  // ###########################################################################
}
