/*
 *This file is part of
 * ======================================================
 * 
 *           LyX, the High Level Word Processor
 * 	 
 *	    Copyright (C) 1995 Matthias Ettrich
 *          Copyright (C) 1995-1998 The LyX Team
 *
 *======================================================
 */

#include "klyx.h"
#include <config.h>

#include <kapp.h>
#include <klocale.h>
#include <qmsgbox.h>

#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <ctype.h>
#include "LString.h"
#include "spellchecker.h"
#include "lyx_main.h"
#include "buffer.h"
#include "lyxrc.h"
#include "BufferView.h"
#include "lyx_gui_misc.h"

#include "LyXView.h"
#include "SpellcheckerOptionsDialog.h"
#include "SpellCheckDialog.h"

extern LyXRC *lyxrc;
extern BufferView *current_view;

// 	$Id: spellchecker.C,v 1.9 1999/01/06 13:47:07 kulow Exp $	

#if !defined(lint) && !defined(WITH_WARNINGS)
static char vcid[] = "$Id: spellchecker.C,v 1.9 1999/01/06 13:47:07 kulow Exp $";
#endif /* lint */

// Spellchecker status
enum {
	ISP_OK = 1,
	ISP_ROOT,
	ISP_COMPOUNDWORD,
	ISP_UNKNOWN,
	ISP_MISSED,
	ISP_IGNORE
};

static FILE *in, *out;  /* streams to communicate with ispell */
pid_t isp_pid = -1; // pid for the `ispell' process. Also used (RO) in
                    // lyx_cb.C

static int isp_fd;

void sigchldhandler(int sig);

extern void sigchldchecker(int sig);

struct isp_result {
	int flag;
	int count;
	LString string;
	char **misses;
	isp_result() {
		flag = ISP_UNKNOWN;
		count = 0;
		misses = (char**)NULL;
	}
	~isp_result() {
		if (misses) delete[] misses;
	}
};


/***** Spellchecker options *****/

// Rewritten to use ordinary LyX xforms loop and OK, Apply and Cancel set,
// now also LString. Amazing, eh? (Asger)

// Set (sane) values in form to current spellchecker options
void SpellOptionsUpdate() // Ported th Qt/KDE, Kalle, long time ago :-)
{
  SpellcheckerOptionsDialogData* dlg_data = klyxdialogs->spelloptions->data();

	// Alternate language
	if (lyxrc->isp_alt_lang.empty()) {
		lyxrc->isp_use_alt_lang = false;
	} else {
	  dlg_data->alternate_language = lyxrc->isp_alt_lang.c_str();
	}
	if (lyxrc->isp_use_alt_lang) {
	  dlg_data->dictionary = SpellcheckerOptionsDialogData::Alternate;
	} else {
	  dlg_data->dictionary = SpellcheckerOptionsDialogData::Document;
	}  

	// Personal dictionary
	if (lyxrc->isp_pers_dict.empty()) {
		lyxrc->isp_use_pers_dict = false;
	} else {
	  dlg_data->alternate_personal_dictionary = lyxrc->isp_pers_dict.c_str();
	}
	dlg_data->use_alternate_personal_dictionary = (lyxrc->isp_use_pers_dict ? 1: 0 );

	// Escape chars
	if (lyxrc->isp_esc_chars.empty()) {
		lyxrc->isp_use_esc_chars = false;
	} else {
	  dlg_data->extra_special_chars_allowed = lyxrc->isp_esc_chars.c_str();
	}
	dlg_data->allow_extra_special_chars = (lyxrc->isp_use_esc_chars ? 1 : 0 );

	// Options
	dlg_data->runtogether_words_legal = (lyxrc->isp_accept_compound ? 1 : 0);
	dlg_data->input_encoding_ispell = (lyxrc->isp_use_input_encoding ? 1 : 0);
	klyxdialogs->spelloptions->setData( dlg_data );
}

// Update spellchecker options
void SpellOptionsApplyCB()
{
	// Build new status from form data
  SpellcheckerOptionsDialogData* dlg_data = klyxdialogs->spelloptions->data();
  if( dlg_data->dictionary == SpellcheckerOptionsDialogData::Alternate )
	lyxrc->isp_use_alt_lang = false;
  else 
	lyxrc->isp_use_alt_lang = true;
  lyxrc->isp_use_pers_dict = dlg_data->use_alternate_personal_dictionary;
  lyxrc->isp_accept_compound = dlg_data->runtogether_words_legal;
  lyxrc->isp_use_esc_chars = dlg_data->allow_extra_special_chars;
  lyxrc->isp_use_input_encoding = dlg_data->input_encoding_ispell;

  // Update strings with data from input fields
  lyxrc->isp_alt_lang = dlg_data->alternate_language;
  lyxrc->isp_pers_dict = dlg_data->alternate_personal_dictionary;
  lyxrc->isp_esc_chars = dlg_data->extra_special_chars_allowed;

  // Update form
  SpellOptionsUpdate();
}


// Show spellchecker options form
void SpellCheckerOptions()
{
  if( klyxdialogs->spelloptions == 0 )
	klyxdialogs->spelloptions = new SpellcheckerOptionsDialog( k_tlw );
  
  // Update form to current options
  SpellOptionsUpdate();
  
  // Focus in alternate language field
  klyxdialogs->spelloptions->setFocusAlternateLanguage();

  int ret = klyxdialogs->spelloptions->exec();
  if( ret == QDialog::Accepted )
	SpellOptionsApplyCB();
}


/***** Spellchecker *****/

// Could also use a clean up. (Asger Alstrup)

static
void create_ispell_pipe(LString const & lang)
{
	static char o_buf[BUFSIZ];  // jc: it could be smaller
	int pipein[2], pipeout[2];
	char * argv[14];
	int argc;

	isp_pid = -1;

	if(pipe(pipein)==-1 || pipe(pipeout)==-1) {
		fprintf(stderr,"LyX: Can't create pipe for spellchecker!");
		return;
	}

	if ((out = fdopen(pipein[1], "w"))==NULL) {
		fprintf(stderr,"LyX: Can't create stream for pipe for spellchecker!");
		return;
	}

	if ((in = fdopen(pipeout[0], "r"))==NULL) {
		fprintf(stderr,"LyX: Can't create stream for pipe for spellchecker!");
		return;
	}

	setvbuf(out, o_buf, _IOLBF, BUFSIZ);

	isp_fd = pipeout[0];

	isp_pid = fork();

	if(isp_pid==-1) {
		fprintf(stderr,"LyX: Can't create child process for spellchecker!");
		return;
	}

	if(isp_pid==0) {        
		/* child process */
		dup2(pipein[0], STDIN_FILENO);
		dup2(pipeout[1], STDOUT_FILENO);
		close(pipein[0]);
		close(pipein[1]);
		close(pipeout[0]);
		close(pipeout[1]);

		argc = 0;
		argv[argc++] = LString("ispell").copy();
		argv[argc++] = LString("-a").copy(); // "Pipe" mode

		if (lang != "default") {
			argv[argc++] = LString("-d").copy(); // Dictionary file
			argv[argc++] = lang.copy();
		}

		if (lyxrc->isp_accept_compound)
			// Consider run-together words as legal compounds
			argv[argc++] = LString("-C").copy(); 
		else
			// Report run-together words with
			// missing blanks as errors
			argv[argc++] = LString("-B").copy(); 

		if (lyxrc->isp_use_esc_chars) {
			// Specify additional characters that
			// can be part of a word
			argv[argc++] = LString("-w").copy(); 
			// Put the escape chars in ""s
			argv[argc++] = (LString("\"") + lyxrc->isp_esc_chars +
					LString("\"")).copy();
		}
		if (lyxrc->isp_use_pers_dict) {
			// Specify an alternate personal dictionary
			argv[argc++] = LString("-p").copy(); 
			argv[argc++] = lyxrc->isp_pers_dict.copy();
		}
		if (lyxrc->isp_use_input_encoding &&
        	    current_view->currentBuffer()->params.inputenc != "default") {
			argv[argc++] = LString("-T").copy(); // Input encoding
			argv[argc++] = current_view->currentBuffer()->params.inputenc.copy();
		}

		argv[argc++] = NULL;

		execvp("ispell", (char * const *) argv);

		// free the memory used by LString::copy in the
		// setup of argv
		for (int i=0; i < argc -1; i++)
			delete[] argv[i];
		
		fprintf(stderr, "LyX: Failed to start ispell!\n");
		_exit(0);
	}

	/* Parent process: Read ispells identification message */
	// Hmm...what are we using this id msg for? Nothing? (Lgb)
	char buf[2048];
	fgets(buf, 2048, in);
	// This hangs if ispell exists without spitting anything out on
	// stdout.  This happens when it does not understand the parameters.
	// This needs to be rewritten to use "select". (Asger)
#ifdef WITH_WARNINGS
#warning Asger! If you have the time, look at this. (Lgb)
#endif
	// I don't have ispell installed so I can't check this
	// myself. I think it should be correct.
#if 0
	// Like this ?  (Lgb)
	fd_set infds;
	struct timeval tv;
	int retval = 0;
	FD_ZERO(&infds];
	FD_SET(pipeout[0], &infds);
	tv.tv_sec = 5; // five second timeout.
	tv.tv_usec =0;

	retval = select(pipeout[0]+1, &infds, 0, 0, &tv);

	if (retval > 0) {
		// Ok, do the reading. We don't have to FD_ISSET since
		// there is only one fd in infds.
		fgets(buf, 2048, in);
	} else if (retval == 0) {
		// timeout. Give nice message to user.
	} else {
		// Select returned error
	}
	
#endif
}


// Send word to ispell and get reply
static
isp_result *ispell_check_word(char *word)
{
	//Please rewrite to use LString.
	isp_result *result;
	char buf[1024], *p;
	int count, i;

	fputs(word, out); 
	fputc('\n', out);
  
	fgets(buf, 1024, in); 
  
	/* I think we have to check if ispell is still alive here because
	   the signal-handler could have disabled blocking on the fd */
	if (isp_pid == -1) return (isp_result *) NULL;

	result = new isp_result;
  
	switch (*buf) {
	case '*': // Word found
		result->flag = ISP_OK;
		break;
	case '+': // Word found through affix removal
		result->flag = ISP_ROOT;
		break;
	case '-': // Word found through compound formation
		result->flag = ISP_COMPOUNDWORD;
		break;
	case '\n': // Number or when in terse mode: no problems
		result->flag = ISP_IGNORE;
		break;
	case '#': // Not found, no near misses and guesses
		result->flag = ISP_UNKNOWN;
		break;
	case '?': // Not found, and no near misses, but guesses (guesses are ignored)
	case '&': // Not found, but we have near misses
	{
		result->flag = ISP_MISSED;
		result->string = buf;
		char * nb = result->string.copy(); // This leaks.
		p = strpbrk(nb+2, " ");
		sscanf(p, "%d", &count); // Get near misses count
		result->count = count;
		if (count) result->misses = new char*[count];
		p = strpbrk(nb, ":");
		p +=2;
		for (i = 0; i<count; i++) {
			result->misses[i] = p;
			p = strpbrk(p, ",\n");
			*p = 0;
			p+=2;
		}
		break;
	}
	default: // This shouldn't happend, but you know Murphy
		result->flag = ISP_UNKNOWN;
	}

	*buf = 0;
	if (result->flag!=ISP_IGNORE) {
		while (*buf!='\n') fgets(buf, 255, in); /* wait for ispell to finish */
	}
	return result;
}


static
inline void ispell_terminate()
{
	fputs("#\n", out); // Save personal dictionary

	fflush(out);
	fclose(out);
}


static
inline void ispell_terse_mode()
{
	fputs("!\n", out); // Set terse mode (silently accept correct words)
}


static
inline void ispell_insert_word(char const *word)
{
	fputc('*', out); // Insert word in personal dictionary
	fputs(word, out);
	fputc('\n', out);
}


static
inline void ispell_accept_word(char const *word) 
{
	fputc('@', out); // Accept in this session
	fputs(word, out);
	fputc('\n', out);
}


void ShowSpellChecker()
{
  // Exit if we don't have a document open
  if (!current_view->getScreen())
	return;
  
  if( !klyxdialogs->spellchecker )
	klyxdialogs->spellchecker = new SpellCheckDialog( current_view->getOwner() );
  
  klyxdialogs->spellchecker->reset();
  
  (void)klyxdialogs->spellchecker->exec();
  
  return;
}


// Perform an ispell session
bool RunSpellChecker(LString const & lang) // Ported to Qt/KDE, Kalle,
										   // 98-01-18
{
  isp_result *result;
  char *word;
  int i, oldval, clickline, newvalue;
  float newval;
  unsigned int word_count = 0;
  
  LString tmp = (lyxrc->isp_use_alt_lang) ? lyxrc->isp_alt_lang:lang;
  
  oldval = 0;  /* used for updating slider only when needed */
  newval = 0.0;
  
  /* create ispell process */
  signal(SIGCHLD, sigchldhandler);
  
  create_ispell_pipe(tmp);
  
  if (isp_pid == -1) {
	QMessageBox::warning( NULL, i18n( "ispell problem" ),
						  i18n("The ispell-process has died for some reason. *One* possible reason\n"
											 "could be that you do not have a dictionary file\n"
											 "for the language of this document installed.\n"
											 "Check /usr/lib/ispell or set another\n"
											 "dictionary in the Spellchecker Options menu."),
						  i18n( "OK"), 0, 0, 0, 0 );
	fclose(out);
	return true;
  }
  
  // Put ispell in terse mode to improve speed
  ispell_terse_mode();
  
  while (true) {
	word = NextWord(newval);
	if (word==NULL) break;
	word_count++;
	
	// Update slider if and only if value has changed
	newvalue = int(100.0*newval);
	if(newvalue!=oldval) {
	  oldval = newvalue;
	  klyxdialogs->spellchecker->setProgressValue( oldval );
	}
	
	result = ispell_check_word(word);
	if (isp_pid==-1) {
	  delete[] word;
	  break;
	}
	
	/* IMPORTANT NOTE: I have copied the old XForms style of calling
	   the event loop and checking afterwards if a button had been
	   pressed. This is ugly in a Qt-based program and should better
	   be integrated with the dialog itself. But when we switch to
	   KSpell, tbis will probably not be needed any longer anyway.
	*/
	kapp->processEvents();
	if( klyxdialogs->spellchecker->isStop() ) {
	  delete result;
	  delete[] word;
	  ispell_terminate();
	  return true;
	}
	if( klyxdialogs->spellchecker->isClose() ) {
	  delete result;
	  delete[] word;
	  ispell_terminate(); 
	  return false;
	}
    
	switch (result->flag) {
	case ISP_UNKNOWN:
	case ISP_MISSED:
	  SelectLastWord();
	  klyxdialogs->spellchecker->setUnknownText( word );
	  klyxdialogs->spellchecker->setReplaceText( word );
	  klyxdialogs->spellchecker->clearAlternatives();
	  for (i=0; i<result->count; i++) {
		klyxdialogs->spellchecker->addAlternative( result->misses[i] );
	  }
	  
	  clickline = -1;
	  while (true) {
		kapp->processEvents();
		if( klyxdialogs->spellchecker->isInsert() ) {
		  ispell_insert_word(word);
		  klyxdialogs->spellchecker->clearFlags();
		  break;
		}
		if( klyxdialogs->spellchecker->isAccept() ) {
		  ispell_accept_word(word);
		  klyxdialogs->spellchecker->clearFlags();
		  break;
		}
		if( klyxdialogs->spellchecker->isIgnore() ) {
		  klyxdialogs->spellchecker->clearFlags();
		  break;
		}
		if( klyxdialogs->spellchecker->isReplace() ) {
		  ReplaceWord( klyxdialogs->spellchecker->replaceText().data() );
		  klyxdialogs->spellchecker->clearFlags();
		  break;
		}

		if( klyxdialogs->spellchecker->isStop() ) {
		  delete result;
		  delete[] word;
		  ispell_terminate();
		  return true;
		}

	    if( klyxdialogs->spellchecker->isClose() ) {
		  delete result;
		  delete[] word;
		  ispell_terminate();
		  return false;
		}
	  }
	default:
	  delete result;
	  delete[] word;
	}
  }
  
  if(isp_pid!=-1) {
	ispell_terminate();
	LString word_msg;
	word_msg += int(word_count);
	if (word_count != 1) {
	  word_msg += i18n(" words checked.");
	} else {
	  word_msg += i18n(" word checked.");
	}
	QString text;
	text.sprintf( i18n( "Spellchecking completed!\n\n%s" ), 
				  word_msg.c_str() );
	QMessageBox::information( 0, i18n( "Ispell process" ),
							  text,
							  i18n( "OK" ), 0, 0, 0, 0 );
	return false;
  } else {
	QMessageBox::warning( 0, i18n("The ispell-process has died for some reason.\n" 
												"Maybe it has been killed."),
						  i18n( "OK" ), 0, 0, 0, 0 );
	fclose(out);
	return true;
  }
}


void sigchldhandler(int sig)
{ 
	int status ;
	// sigset_t oldmask, mask = 1 << (SIGCHLD-1);

	// we cannot afford any other child signal while handling this one
	// so we have to block it and unblock at the end of signal servicing
	// routine
	// causes problems, so we will disable it temporarily
	//sigprocmask(SIG_BLOCK, &mask, &oldmask);

	if (isp_pid>0)
		if (waitpid(isp_pid,&status,WNOHANG)==isp_pid) {
			isp_pid=-1;
			fcntl(isp_fd, F_SETFL, O_NONBLOCK); /* set the file descriptor
							       to nonblocking so we can 
							       continue */
		}
	sigchldchecker(sig);

	// set old signal block mask
	//sigprocmask(SIG_SETMASK, &oldmask, NULL);
}
