/*
   Siag, Scheme In A Grid
   Copyright (C) 1996, 1997  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * siodi.c
 */

#include <stdio.h>
#include <time.h>
#include <ctype.h>
#include <stdlib.h>

#include "../common/cmalloc.h"

#include "types.h"

#include "calc.h"
#include "../siod/siod.h"

/* various siod modules */
extern void init_ss(void);
#if defined(NDBM) || defined(GDBM)
extern void init_ndbm(void);
#endif
#if defined(REGEX)
extern void init_regex(void);
#endif
#if defined(TAR)
extern void init_tar(void);
#endif

/* from slib.c */
extern char *repl_c_string_arg;
extern long repl_c_string_flag;
extern LISP repl_c_string_read();

static int siag_row, siag_col;
static cval siag_result;
static buffer *siag_buffer;

static LISP lput_property(LISP bname, LISP key, LISP value)
{
	buffer *b;
	char *retval;

	if (NULLP(bname)) b = buffer_of_window(w_list);
	else b = find_buffer_by_name(get_c_string(bname));
	if (!b) {
		llpr("No such buffer");
		return NIL;
	}

	retval = put_property(b, get_c_string(key), get_c_string(value));
	if (retval) return strcons(strlen(retval), retval);
	return NIL;
}

static LISP lget_property(LISP bname, LISP key)
{
	buffer *b;
	char *retval;

	if (NULLP(bname)) b = buffer_of_window(w_list);
	else b = find_buffer_by_name(get_c_string(bname));
	if (!b) {
		llpr("No such buffer");
		return NIL;
	}

	retval = get_property(b, get_c_string(key));

	if (retval) return strcons(strlen(retval), retval);
	return NIL;
}

static LISP get_row()
{
	double drow = siag_row;
	return flocons(drow);
}

static LISP get_col()
{
	double dcol = siag_col;
	return flocons(dcol);
}

static LISP x_get_cell(LISP row, LISP col, LISP bname)
{
	int r, c;
	char *p;
	buffer *buf;
	spread **matrix;

	r = get_c_long(row);
	c = get_c_long(col);
	if (r < 1 || r > BUFFER_ROWS || c < 1 || c > BUFFER_COLS)
		return NIL;
	if (NULLP(bname))
		buf = siag_buffer;
	else if (TYPEP(bname, tc_string)) {
		buf = find_buffer_by_name(bname->storage_as.string.data);
		if (buf == NULL) return NIL;
		matrix = buf->matrix;
	}
	else return NIL;

	switch (ret_type(buf, r, c)) {
	case STRING:
		p = ret_string(buf, r, c);
		return strcons(strlen(p), p);
	default:
		return flocons(ret_val(buf, r, c).number);
	}
}

static LISP get_cell(LISP row, LISP col)
{
	return x_get_cell(row, col, NIL);
}

static LISP x_get_string(LISP row, LISP col, LISP bname)
{
	int r, c;
	char *p;
	buffer *buf;
	spread **matrix;

	r = get_c_long(row);
	c = get_c_long(col);
	if (r < 1 || r > BUFFER_ROWS || c < 1 || c > BUFFER_COLS)
		return NIL;
	if (NULLP(bname))
		buf = siag_buffer;
	else if (TYPEP(bname, tc_string)) {
		buf = find_buffer_by_name(bname->storage_as.string.data);
		if (buf == NULL) return NIL;
		matrix = buf->matrix;
	}
	else return NIL;

	if (ret_type(buf, r, c) == ERROR) p = "";
	else p = ret_pvalue(NULL, buf, r, c, -1);
	return strcons(strlen(p), p);
}

static LISP get_string(LISP row, LISP col)
{
	return x_get_string(row, col, NIL);
}

static LISP get_type(LISP bname, LISP row, LISP col)
{
	buffer *buf;

	if (NULLP (bname)) buf = buffer_of_window(w_list);
	else buf = find_buffer_by_name(get_c_string(bname));

	return flocons(ret_type(buf, FLONM(row), FLONM(col)));
}

static LISP get_text(LISP row, LISP col)
{
	char *text = ret_text(buffer_of_window(w_list), FLONM(row), FLONM(col));
	if (!text) text = "";

	return strcons(strlen(text), text);
}

static LISP siag_sum(LISP start, LISP end)
{
	double sum;
	int r, c, startr, startc, endr, endc;
	startr = get_c_long(car(start));
	startc = get_c_long(cdr(start));
	endr = get_c_long(car(end));
	endc = get_c_long(cdr(end));
	if (startr > endr) {
		r = startr;
		startr = endr;
		endr = r;
	}
	if (startc > endc) {
		c = startc;
		startc = endc;
		endc = c;
	}
	sum = 0;
	for (r = startr; r <= endr; r++)
		for (c = startc; c <= endc; c++)
			sum += ret_val(siag_buffer, r, c).number;
	return flocons(sum);
}

static LISP siag_time()
{
	double t = time(NULL);
	return flocons(t);
}

static LISP lexec_expr(LISP intp, LISP expr)
{
	exec_expr(name2interpreter(get_c_string(intp)), get_c_string(expr));
	return NIL;
}

static void siag_puts(char *p)
{
}

static void siag_print(LISP p)
{
	if (FLONUMP(p)) {
		siag_type = EXPRESSION;
		siag_result.number = FLONM(p);
	} else if (TYPEP(p, tc_string)) {
		siag_type = STRING;
		siag_result.text = p->storage_as.string.data;
	} else {
		siag_type = ERROR;
		siag_result.number = 0, errorflag = 1;
	}
}


/* expand Visicalc references */

#define BREAKCHARS "() \t\r\n"
#define TEMPLATE "(get-cell %ld %ld)"
#define RANGE "'RANGE %ld %ld %ld %ld"

static char *expand_references(char *orig)
{
	char *new;

	int rc = ref_counter(orig);
	if (!rc) return orig;

	/* strlen("(get-cell 1 1)")-strlen("r1c1") = 10 */
	new = cmalloc(strlen(orig)+10*rc+1);

	ref_expander(orig, new, BREAKCHARS, TEMPLATE, RANGE);
	return new;
}


/* move block between (r1,c1) and (r2,c2), direction (rd,cd) */
static char *update_SIOD_references(char *expr, int r1, int c1,
				int r2, int c2, int rd, int cd)
{
	char *p;
	int rc = ref_counter(expr);

	if (!rc) return expr;

	/* rough upper bound on new length. A few bytes extra don't matter */
	/* strlen("r100000c100000")-strlen("r1c1") = 10 */
	p = cmalloc(strlen(expr)+10*rc+1);
	rc = ref_updater(expr, p, BREAKCHARS, r1, c1, r2, c2, rd, cd);
	if (rc) {
		expr = cstrdup(p);
	}
	cfree(p);
	return expr;
}


static long wrapper(buffer *b, char *expr, int row, int col)
{
	/* mostly stolen from repl_c_string in slib.c */
	struct repl_hooks h;
	long retval;
	int want_sigint = 1;
	static int want_init = 0;
	char *expansion = expand_references(expr);

TRACEME((f,"wrapper(%s)",expr));

	h.repl_puts = siag_puts;
	h.repl_read = repl_c_string_read;
	h.repl_eval = NULL;
	h.repl_print = siag_print;
	repl_c_string_arg = expansion;
	repl_c_string_flag = 0;
	siag_row = row;
	siag_col = col;
	siag_buffer = b;
	setvar(cintern("R"), flocons((double)siag_row), NIL);
	setvar(cintern("C"), flocons((double)siag_col), NIL);

	retval = repl_driver(want_sigint, want_init, &h);
TRACEME((f,"repl_driver returns %ld",retval));
	if (expansion != expr) cfree(expansion);
	want_init = 0;			/* only once... */
	return retval;
}

static cval parse_SIOD_expr(buffer *b, char *expr, int row, int col)
{
	long retval;

	retval = wrapper(b, expr, row, col);
	return siag_result;
}

/*
   This function was originally in cmds.c, but has been
   moved here because of its similarity to parse_expr.
 */
static void execute_siod(char *s)
{
	long retval;

TRACEME((f,"execute_siod(%s)",s));

	if (ok2print)
		hide_cur(w_list);

	retval = wrapper(buffer_of_window(w_list), s,
			get_point(w_list).row, get_point(w_list).col);

	if (ok2print)
		show_cur(w_list);
}


/* from position.c */

extern void prot_fixup(window *);

/* These functions allow implementation of commands in Scheme
	rather than in C with Scheme wrappers */

static LISP lget_point()
{
	return MAKE_POSITION(get_point(w_list));
}

static LISP lget_mark()
{
	return MAKE_POSITION(get_mark(w_list));
}

static LISP lget_top()
{
	return MAKE_POSITION(get_top(w_list));
}

static LISP lget_blku()
{
	return MAKE_POSITION(block_upper(w_list));
}

static LISP lget_blkl()
{
	return MAKE_POSITION(block_lower(w_list));
}

static LISP lset_top(LISP top)
{
	position newtop;
	newtop.row = POSITION_ROW(top);
	newtop.col = POSITION_COL(top);
	set_top(w_list, newtop);
	prot_fixup(w_list);
	return NIL;
}

static LISP lset_prot(LISP prot)
{
	position newprot;
	newprot.row = POSITION_ROW(prot);
	newprot.col = POSITION_COL(prot);
	set_prot(w_list, newprot);
	prot_fixup(w_list);
	return NIL;
}

static LISP lget_prot()
{
	return MAKE_POSITION(get_prot(w_list));
}

static LISP lset_point(LISP p)
{
	position newpoint;
	newpoint.row = POSITION_ROW(p);
	newpoint.col = POSITION_COL(p);
TRACEME((f,"(set-point (%d,%d))",newpoint.row,newpoint.col));
	set_point(w_list, newpoint);
	prot_fixup(w_list);
	return NIL;
}

static LISP lset_mark(LISP p)
{
	position newmark;
	newmark.row = POSITION_ROW(p);
	newmark.col = POSITION_COL(p);
	set_mark(w_list, newmark);
	return NIL;
}

static LISP line_last_used_fun()
{
	int n = line_last_used(buffer_of_window(w_list));
	return flocons(n);
}

static LISP col_last_used_fun(LISP row)
{
	int n = get_c_long(row);
	return flocons(col_last_used(buffer_of_window(w_list), n));
}

static LISP max_lines_fun()
{
	return flocons(BUFFER_ROWS);
}

static LISP max_columns_fun()
{
	return flocons(BUFFER_COLS);
}

static LISP set_pr_scr()
{
	pr_scr_flag = TRUE;
	return NIL;
}

void init_position()
{
TRACEME((f,"init_position()"));
	init_subr_0("get-point", lget_point);
	init_subr_0("get-mark", lget_mark);
	init_subr_0("get-top", lget_top);
	init_subr_0("get-blku", lget_blku);
	init_subr_0("get-blkl", lget_blkl);
	init_subr_1("set-top", lset_top);
	init_subr_1("set-prot", lset_prot);
	init_subr_0("get-prot", lget_prot);
	init_subr_1("set-point", lset_point);
	init_subr_1("set-mark", lset_mark);
	init_subr_0("line-last-used", line_last_used_fun);
	init_subr_1("col-last-used", col_last_used_fun);
	init_subr_0("max-lines", max_lines_fun);
	init_subr_0("max-columns", max_columns_fun);
	init_subr_0("set-pr-scr", set_pr_scr);
	init_subr_3("put-property", lput_property);
	init_subr_2("get-property", lget_property);
}

/* dummy args for SIOD */
static char *siod_argv[] = {
  "siod",
  "-h100000:10",
  "-g0",
  "-o1000",
  "-s200000",
  "-n2048"};
static int siod_argc = sizeof siod_argv / sizeof siod_argv[0];

int init_parser(int argc, char **argv)
{
	print_welcome();
	process_cla(siod_argc, siod_argv, 1);
	process_cla(argc, argv, 1);
	print_hs_1();
	init_storage();
	init_subrs();
	init_trace();
#if defined(SLIBU)
	init_slibu();
#endif
#if defined(SS)
	init_ss();
#endif
#if defined(NDBM) || defined(GDBM)
	init_ndbm();
#endif
#if defined(REGEX)
	init_regex();
#endif
#if defined(TAR)
	init_tar();
#endif
	init_subr_0("row", get_row);
	init_subr_0("col", get_col);
	init_subr_2("get-cell", get_cell);
	init_subr_2("get-string", get_string);
	init_subr_3("get-type", get_type);
	init_subr_2("get-text", get_text);
	init_subr_2("sum", siag_sum);
	init_subr_3("x-get-cell", x_get_cell);
	init_subr_3("x-get-string", x_get_string);
	init_subr_0("time", siag_time);
	init_subr_2("exec-expr", lexec_expr);

	return register_interpreter("SIOD", parse_SIOD_expr, execute_siod,
					update_SIOD_references);
}

