#include <qwt_dimap.h>
#include <qwt_math.h>

const double QwtDiMap::LogMin = 1.0e-150;
const double QwtDiMap::LogMax = 1.0e150;

QwtDiMap::QwtDiMap()
{
    d_x1 = 0.0;
    d_x2 = 1.0;
    d_y1 = 0;
    d_y2 = 1;
    d_cnv = 1.0;
}

QwtDiMap::QwtDiMap(int i1, int i2, double d1, double d2)
{
    setIntRange(i1,i2);
    setDblRange(d1, d2);
}

QwtDiMap::~QwtDiMap()
{
}

//------------------------------------------------------------
//
//.S	QwtDiMap::contains
//	Returns trus if a value x lies inside or at the border of the
//	map's double range
//
//.s	Parameters
//.p	double x -- value under test
//
//.s	Return Type
//		bool
//
//.s	Syntax
//.f	bool QwtDiMap::contains(double x)
//
//------------------------------------------------------------
bool QwtDiMap::contains(double x) const
{
    return ( (x >= qwtMin(d_x1, d_x2)) && (x <= qwtMax(d_x1, d_x2)));
}

//------------------------------------------------------------
//
//.S	QwtDiMap::contains
//	Returns true if a value x lies inside or at the border of the
//	map's integer range
//
//.s	Parameters
//.p	int x -- value under test
//
//.s	Return Type
//		bool
//
//.s	Syntax
//.f	bool QwtDiMap::contains(int x)
//
//------------------------------------------------------------
bool QwtDiMap::contains(int x) const
{
   return ( (x >= qwtMin(d_y1, d_y2)) && (x <= qwtMax(d_y1, d_y2)));
}

void QwtDiMap::setIntRange(int i1, int i2)
{
    d_y1 = i1;
    d_y2 = i2;
    newFactor();
}

void QwtDiMap::setDblRange(double d1, double d2,int lg)
{
    if (lg)
    {
	d_log = true;
	if (d1 < LogMin) 
	   d1 = LogMin;
	else if (d1 > LogMax) 
	   d1 = LogMax;
	
	if (d2 < LogMin) 
	   d2 = LogMin;
	else if (d2 > LogMax) 
	   d2 = LogMax;
	
	d_x1 = log(d1);
	d_x2 = log(d2);
    }
    else
    {
	d_log = false;
	d_x1 = d1;
	d_x2 = d2;
    }
    newFactor();

}
//------------------------------------------------------------
//
//.S	QwtDiMap::limTransform
//	transform and limit
//
//.s	Parameters
//.p	double x
//
//.s	Return Type
//		int
//
//.s	Return Value
//		transformed value
//.s	Description
//	The value is limited to the map's range. If it lies
//	outside, the function returns the transformed value
//	of the nearest border.
//
//.s	Syntax
//.f	int QwtDiMap::limTransform(double x)
//
//------------------------------------------------------------
int QwtDiMap::limTransform(double x) const
{
    if ( x > qwtMax(d_x1, d_x2) )
       x = qwtMax(d_x1, d_x2);
    else if ( x < qwtMin(d_x1, d_x2))
       x = qwtMin(d_x1, d_x2);
    return transform(x);
}


int QwtDiMap::transform(double x) const
{
    if (d_log)
       return (d_y1 + int(rint( (log(x) - d_x1) * d_cnv )));    
    else
       return (d_y1 + int(rint( (x - d_x1) * d_cnv )));
}

//------------------------------------------------------------
//
//.S	QDiMap::xTransform
//	exact inverse transformation
//
//.s	Parameters
//.p	double x
//
//ss	Return Type
//		double
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	double QDiMap::dTransform(double x)
//
//------------------------------------------------------------
double QwtDiMap::xTransform(double x) const
{
    double rv;
    
    if (d_log)
       rv = double(d_y1) + (log(x) - d_x1) * d_cnv;    
    else
       rv = double(d_y1) + (x - d_x1) * d_cnv;

    return rv;
}

double QwtDiMap::invTransform(int y) const 
{
    if (d_cnv == 0.0)
       return 0.0;
    else
    {
	if(d_log) 
	   return exp(d_x1 + double(y - d_y1) / d_cnv );
	else
	   return ( d_x1 + double(y - d_y1) / d_cnv );	
    }
}


void QwtDiMap::newFactor()
{
    if (d_x2 != d_x1)
       d_cnv = double(d_y2 - d_y1) / (d_x2 - d_x1); 
    else 
       d_cnv = 0.0;
}






