/**************************************************************************
 * $Id: script_dev.cpp 1.1 Thu, 03 Dec 1998 12:49:42 +0100 samo $
 * $ReleaseVersion: 1.3.1 $
 *
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include "script.h"          /* all prototypes and structures */
#include <kapp.h>
#include <qmessagebox.h>


int convert_read(char *src, int len, double *dst, int type, int endian)
{
   int bytes;
   int itemsize;
   int i,ii;
   unsigned long tmp_u;
   long tmp;
   int swapendian=FALSE;
   unsigned char tmp_t;
   
   if(endian==ENDIAN_BIG && BYTE_ORDER==LITTLE_ENDIAN)swapendian=TRUE;
   if(endian==ENDIAN_LITTLE && BYTE_ORDER==BIG_ENDIAN)swapendian=TRUE;   
   
   if(type>0){
      bytes=len*type+1;
      itemsize=type;
   }

   if(type<0){
      bytes=-len*type+1;
      itemsize=-type;
   }
   if(type==0){
      bytes=(len+7)/8;
      itemsize=0;
   }
   
   if(type>0){
      for(i=0;i<len;++i){
	 tmp_u=0;
	 for(ii=0;ii<itemsize;++ii){
	    if(ii<sizeof(long)){
	       if(swapendian==FALSE)
		 tmp_u|=((unsigned char)*(src+i*itemsize+ii))<<(ii*8);
	       else
		 tmp_u|=((unsigned char)*(src+i*itemsize+(itemsize-1-ii)))<<(ii*8);
	    }
	 }
	 *(dst+i)=(double)tmp_u;
      }
   }
   
   if(type<0){
      for(i=0;i<len;++i){
	 tmp=0;
	 for(ii=0;ii<sizeof(long);++ii){
	    if(ii<itemsize){
	       if(swapendian==FALSE)
		 tmp_t=((unsigned char)*(src+i*itemsize+ii));
	       else
		 tmp_t=((unsigned char)*(src+i*itemsize+(itemsize-1-ii)));	       

//	       debug("tmp_t=%li\n",tmp_t);
	       tmp|=tmp_t<<(ii*8);
	    }
	    else tmp|=(0xff*((tmp_t&0x80)!=0))<<(ii*8);
//	    debug("->tmp=%li,gg=%li\n",tmp,(0xff*(tmp_t&0x80!=0))<<(ii*8));
	 }
	 *(dst+i)=(double)tmp;
      }
   }
   
   if(type==0){
      for(i=0;i<bytes;++i){
	 for(ii=0;ii<8;++ii){
	    if(i*8+ii<len)
	      *(dst+i*8+ii)=(*(src+i)&(1<<ii))!=0;
	 }
      }
   }

   return(bytes);
}



int convert_write(double *src, int len, char **dst, int type, int endian)
{
   int bytes;
   char *buf;
   long tmp;
   unsigned long tmp_u;
   int i,ii,itemsize, swapendian=FALSE;

   if(endian==ENDIAN_BIG && BYTE_ORDER==LITTLE_ENDIAN)swapendian=TRUE;
   if(endian==ENDIAN_LITTLE && BYTE_ORDER==BIG_ENDIAN)swapendian=TRUE;   
   
   if(type>0){
      bytes=len*type;
      itemsize=type;
   }
      
   if(type<0){
      bytes=-len*type;
      itemsize=-type;
   }
   if(type==0){
      bytes=(len+7)/8;
      itemsize=0;
   }
   
   buf=(char *)calloc(bytes,sizeof(char));
   if(buf!=NULL){
      *dst=buf;
      
      if(type>0)
	for(i=0;i<len;++i){
	   tmp_u=(unsigned long)(*(src+i));	       
	   for(ii=0;ii<itemsize;++ii){
	      if(ii>=sizeof(long))tmp_u=0;
	      
	      if(swapendian==FALSE)
		*(buf+i*itemsize+ii)=tmp_u&0xff;
	      else
		*(buf+i*itemsize+(itemsize-1-ii))=tmp_u&0xff;
	      
	     tmp_u=tmp_u>>8;
	   }
	}
      
      if(type<0)
	for(i=0;i<len;++i){
	   tmp=(long)(*(src+i));
	   for(ii=0;ii<itemsize;++ii){

	      if(ii>=sizeof(long)){
		if(*(src+i)>=0)tmp=0;
		 else tmp=0xff;
	      }

	      if(swapendian==FALSE)
		*(buf+i*itemsize+ii)=tmp&0xff;
	      else
		*(buf+i*itemsize+(itemsize-1-ii))=tmp&0xff;
	      
	   tmp=tmp>>8;
	   }
	}
      if(type==0)
	for(i=0;i<bytes;++i){
	   tmp=0;
	   for(ii=0;ii<8;++ii){
	      if(i*8+ii<len)
		tmp|=(*(src+i*8+ii)!=0)<<ii;
	   }
	   *(buf+i)=tmp;
	}
      
   }
   else bytes=0;
   
   return(bytes);
}

int SamplinScript::setDevicesFile(const char *str)
{
   devices_file=str;
   
   QFile f(devices_file);
   if(f.open( IO_ReadOnly)==TRUE){
      QDataStream s(&f);
      devices->clear();
      s >> *devices;
      //printf("got %i devices\n",Devices.count());
      f.close();
      return 0;
   }
   else {
//      QMessageBox::warning(NULL,"Warning","Can't open device list file "+devices_file);
      error(WARNING,"Can't open device configuration file %s",devices_file.data());
      return -1;
   }
   
}

SamplinDevice *SamplinScript::getDevice(char *str)
{
   SamplinDevice *dev;
   
   if(devices!=NULL)
     for(dev=devices->first();dev!=NULL;dev=devices->next())
	if(dev!=NULL)if(strcmp(dev->getName(),str)==0)return(dev);

   return(NULL);
}

void SamplinScript::closeDevices(void)
{
   SamplinDevice *dev;

   error(NOTE,"Closing all devices");
   
   if(devices!=NULL)
     for(dev=devices->first();dev!=NULL;dev=devices->next())
	if(dev!=NULL)dev->Close();

}

void SamplinScript::create_devwr(char *dev, int args)
{
   struct command *cmd;

   error(DIAGNOSTIC,"Creating command 'devwr'");
   cmd=add_command(DEVWR);
   cmd->pointer=my_strdup(dev);
   cmd->args=args;
}

void SamplinScript::create_devctl(char *dev)
{
  struct command *cmd;

   error(DIAGNOSTIC,"Creating command 'devctl'");
  cmd=add_command(DEVCTL);
  cmd->pointer=my_strdup(dev);

}

void SamplinScript::create_devflush(char *dev)
{
   struct command *cmd;
   
   error(DIAGNOSTIC,"Creating command 'devflush'");
   cmd=add_command(DEVFLUSH);
   cmd->pointer=my_strdup(dev);

}

void SamplinScript::create_devseek(char *dev,int from)
{
   struct command *cmd;

   error(DIAGNOSTIC,"Creating command 'devseek'");
   cmd=add_command(DEVSEEK);
   cmd->args=from;
   cmd->pointer=my_strdup(dev);

}


void SamplinScript::create_devopen(char *dev)
{
  struct command *cmd;

   error(DIAGNOSTIC,"Creating command 'devopen'");
  cmd=add_command(DEVOPEN);
  cmd->pointer=my_strdup(dev);

}

void SamplinScript::create_devclose(char *dev)
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'devclose'");
  cmd=add_command(DEVCLOSE);
  cmd->pointer=my_strdup(dev);

}

void SamplinScript::create_devrd(char *dev, int args) /* create command myswitch */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'devrd'");
  cmd=add_command(DEVRD);
  cmd->pointer=my_strdup(dev);
  cmd->args=args;
}

void SamplinScript::create_devrdb(char *dev, char *symbol) /* create command myswitch */
{
  struct command *cmd;
  struct symbol *a;
  struct array *ar;
  char *c;

   a=get_sym(symbol,ARRAY,FALSE);
 
   if (a==NULL) {
      error(ERROR,"array '%s' has not been dimed",symbol);
      return;
   }
   
   ar=(struct array *)(a->pointer);
   if (ar->dimension!=1) {
      error(ERROR,"array dimension must be 1 for '%s'",symbol);
      return;
   }
 
//   printf("Array name=%s\n",symbol);
   error(DIAGNOSTIC,"Creating command 'devrdb'");
   cmd=add_command(DEVRDB);
   cmd->spointer=ar;
   cmd->pointer=my_strdup(dev);
}

void SamplinScript::create_devwrb(char *dev, char *symbol) /* create command myswitch */
{
  struct command *cmd;
  struct symbol *a;
  struct array *ar;
  char *c;

   a=get_sym(symbol,ARRAY,FALSE);
 
   if (a==NULL) {
      error(ERROR,"array '%s' has not been dimed",symbol);
      return;
   }
   
   ar=(struct array *)(a->pointer);
   if (ar->dimension!=1) {
      error(ERROR,"array dimension must be 1 for '%s'",symbol);
      return;
   }
 
//   printf("Array name=%s\n",symbol);
   error(DIAGNOSTIC,"Creating command 'devwrb'");
   cmd=add_command(DEVWRB);
   cmd->spointer=ar;
   cmd->pointer=my_strdup(dev);
}

void SamplinScript::devopen(struct command *cmd) /* switch to specified stream */
{
   char *device;
   struct stackentry *p;
   SamplinDevice *dev;
   int status;
   
   device=(char *)(cmd->pointer);
   
   ERROR(NOTE,(NOTE,"Opening device %s",device));

   dev = getDevice(device);
   
   if(dev==NULL){
      error(WARNING,"Device %s not available.",device);      
      return;
   }
   
   status=dev->Open();

   if(status<=0){
      error(WARNING,"Cannot open device %s.",device);
   }
   if(status==-3){
      error(WARNING,dev->getErrorLog());
   }   
   return;
}

void SamplinScript::devclose(struct command *cmd) /* switch to specified stream */
{
   char *device;
   struct stackentry *p;
   SamplinDevice *dev;
   int status;
   
   device=(char *)(cmd->pointer);
   
   ERROR(NOTE,(NOTE,"Closing device %s",device));

   dev = getDevice(device);
   
   if(dev==NULL){
      error(WARNING,"Device %s not available.",device);      
      return;
   }

   status=dev->Close();

   if(status!=0){
      error(WARNING,"Cannot close device %s.",device);
   }
   if(status==-3){
      error(WARNING,dev->getErrorLog());
   }   
   return;
}

void SamplinScript::devwr(struct command *cmd) /* switch to specified stream */
{
   char *device;
   char *data;
   struct stackentry *p;
   SamplinDevice *dev;
   int status;
   int lenght;
   
   if(cmd->args==1){
      p=pop();
      lenght=(int)p->value;
      lenght=MAX(0,lenght);
   }
   
   device=(char *)(cmd->pointer);
   p=pop();
   data=(char *)(p->pointer);

   if(cmd->args==1)lenght=MIN(lenght,strlen(data));
   else lenght=strlen(data);
   
   ERROR(NOTE,(NOTE,"Writing device %s: %s",device,data));

   dev = getDevice(device);
   
   if(dev==NULL){
      error(WARNING,"Device %s not available.",device);      
      return;
   }

   status=dev->Write(data,lenght,FALSE);
   
   if(status==-1){
      error(WARNING,"Device %s not open.",device);
   }
   if(status==-2){
      error(WARNING,"Error writing device %s.",device);         
      }
   if(status==-3){
      error(WARNING,dev->getErrorLog());
   }      
   return;
}

void SamplinScript::devctl(struct command *cmd) /* switch to specified stream */
{
   char *device;
   struct stackentry *p;
   SamplinDevice *dev;
   int status;
   int request;
   long arg,*argp;

   device=(char *)(cmd->pointer);

   p=pop();
   if(p->type==NUMBER){
      arg=(long)p->value;
      argp=&arg;
   }
   if(p->type==PTR){
      argp=(long *)p->pointer;
   }
   p=pop();
   
   request=(int)p->value;
   
   
   ERROR(NOTE,(NOTE,"Ioctl device %s, request %i, argument %li",device,request, arg));

   dev = getDevice(device);
   
   if(dev==NULL){
      error(WARNING,"Device %s not available.",device);      
      return;
   }
   
   status=dev->Ioctl(request,(char*)argp);

   if(status==-1){
      error(WARNING,"Device %s not open.",device);
   }
   if(status==-2){
      error(WARNING,"Error on ioctl device %s.",device);
      }
   if(status==-3){
      error(WARNING,dev->getErrorLog());
   }         
   return;
}

void SamplinScript::devseek(struct command *cmd) /* switch to specified stream */
{
   char *device;
   struct stackentry *p;
   SamplinDevice *dev;
   int status;
   int from;
   long arg;

   device=(char *)(cmd->pointer);

   p=pop();
   arg=(long)p->value;
   from=cmd->args;
   
   ERROR(NOTE,(NOTE,"Seek device %s, mode %i, position %li",device, from, arg));

   dev = getDevice(device);
   
   if(dev==NULL){
      error(WARNING,"Device %s not available.",device);      
      return;
   }
   
   status=dev->Seek(from,arg);

   if(status==-1){
      error(WARNING,"Device %s not open.",device);
   }
   if(status==-2){
      error(WARNING,"Error on seek device %s.",device);         
      }
   if(status==-3){
      error(WARNING,dev->getErrorLog());
   }         
   return;
}

void SamplinScript::devflush(struct command *cmd) /* switch to specified stream */
{
   char *device;
   struct stackentry *p;
   SamplinDevice *dev;
   int status;
   int request;
   long arg;

   device=(char *)(cmd->pointer);

   ERROR(NOTE,(NOTE,"Flush device %s",device));

   dev = getDevice(device);
   
   if(dev==NULL){
      error(WARNING,"Device %s not available.",device);      
      return;
   }
   
   status=dev->Flush();

   if(status==-1){
      error(WARNING,"Device %s not open.",device);
   }
   if(status==-2){
      error(WARNING,"Error on flush device %s.",device);         
      }
   if(status==-3){
      error(WARNING,dev->getErrorLog());
   }         
   return;
}

void SamplinScript::devrd(struct command *cmd) /* switch to specified stream */
{
   char *device;
   char *data;
   struct stackentry *p;
   struct stackentry *s;
   SamplinDevice *dev;
   char *buff=NULL;
   int status;
   int lenght;
   
   if(cmd->args==1){
      p=pop();
      lenght=(int)p->value;
      lenght=MAX(0,lenght);
   }
   else lenght=1024;
   
   buff=(char *)calloc(lenght+1,sizeof(char));
   strcpy(buff,"");
   device=(char *)(cmd->pointer);
//   data=(int *)(p->pointer);
//   p=pop();
   
   ERROR(NOTE,(NOTE,"Reading %g bytes from device %s",lenght,device));

   dev=getDevice(device);

   if(dev==NULL){
      error(WARNING,"Device %s not available.",device);
   }
   else{
      status=dev->Read(buff,lenght,0);
      if(status==-1){
	 error(WARNING,"Device %s not open.",device);
      }
      else if(status==-2){
	 error(WARNING,"Error reading device %s.",device);   
      }
      if(status==-3){
	 error(WARNING,dev->getErrorLog());
      }      
      //      else  printf("Result: %s\n", buff);
   }

   s=push();
   s->type=STRING;
   s->pointer=buff;
   
   return;
}

void SamplinScript::devrdb(struct command *cmd) /* switch to specified stream */
{
   char *device;
   char *data;
   struct stackentry *p;
   struct stackentry *s;
   SamplinDevice *dev;
   int status;   
   int itemsize,lenght,i,ii,y,bytes;
   unsigned int tmp_sign;
   unsigned long tmp;
   struct array *ar;
   char *buff=NULL;
   double *adata;

   p=pop();
   itemsize=(int)p->value;
   p=pop();
   lenght=(int)p->value;

   device=(char *)(cmd->pointer);

   ar=(struct array *)current->spointer;
   if(ar->type!='d'){
      error(ERROR,"Invalid array type (double required)");
      return;
   }
   if(lenght>ar->total || lenght<0){
      error(ERROR,"Array too short (%i) for reading %i values",ar->total,lenght);
      return;
   }

   adata=(double *)(ar->pointer);

   error(NOTE,"Reading %i %i-byte values from device %s",lenght,itemsize,device);

   dev=getDevice(device);

   if(dev==NULL){
      error(WARNING,"Device %s not available.",device);
      return;
   }
   else{
      if(itemsize>0)
	bytes=lenght*itemsize;
      if(itemsize<0)
	bytes=-lenght*itemsize;
      if(itemsize==0)
	bytes=(lenght+7)/8;
	
      buff=(char *)calloc(bytes+1,1);
      if(buff!=NULL)
	status=dev->Read((char *)buff,bytes,1);      
      else{
	 error(WARNING,"Cannot allocate %i bytes buffer.",bytes);
	 return;
      }
      
      if(status==-1){
	 error(WARNING,"Device %s not open.",device);
	 free(buff);
	 return;
      }
      if(status==-2){
	 error(WARNING,"Error reading device %s.",device);   
	 free(buff);
	 return;
      }
      if(status==-3){
	 error(WARNING,dev->getErrorLog());
	 free(buff);
	 return;
      }            

      convert_read(buff,lenght,adata,itemsize,dev->getEndian());
      
      /*
      if(itemsize>0)
	for(i=0;i<lenght;++i){
	   adata[i]=0;
	   for(ii=0;ii<itemsize;++ii){
	      tmp=(unsigned char)buff[i*itemsize+ii];
	      if(dev->getEndian()==ENDIAN_BIG){
		 if(ii==itemsize-1 && (tmp & 128 ) && itemsize>1 ){
		    adata[i]+=(tmp<<(ii*8));
		    tmp_sign=(unsigned int)adata[i];
		    tmp_sign=(-tmp_sign)&0xffff;
		    adata[i]=-(int)tmp_sign;
		 }
		 else
		   adata[i]+=(tmp<<(ii*8));
	      }
	      else
		adata[i]+=(tmp<<((itemsize-ii)*8));
	   }
	}
      else
	for(i=0;i<bytes;++i){	 
	   for(ii=0;ii<8;++ii){
	      if(i*8+ii>=lenght)break;
	      adata[i*8+ii]=(buff[i] & (1<<ii))!=0;
	   }
	   
	}
 */ 
/*      
      for(i=0;i<lenght*itemsize;++i){
	 //if(dev->getEndian()==LITTLE_ENDIAN)SWAP_32_ENDIAN(buff[i]);
	 //adata[i]=buff[i];
	 adata[y]+=buff[i]<<(ii*8);
	 ++ii;
	 if(ii==itemsize){
	    adate[]=0;
	    ii=0;  
	    ++y;
	 }
      }
*/
      free(buff);
   }
   return;
}


void SamplinScript::devwrb(struct command *cmd) /* switch to specified stream */
{
   char *device;
   char *data;
   struct stackentry *p;
   struct stackentry *s;
   SamplinDevice *dev;
   int status;   
   int itemsize,lenght;
   struct array *ar;
   char *buff=NULL;
   double *adata;
   int write_len;
   
   p=pop();
   itemsize=(int)p->value;
   p=pop();
   lenght=(int)p->value;

   device=(char *)(cmd->pointer);
   
   ar=(struct array *)current->spointer;
   if(ar->type!='d'){
      error(ERROR,"Invalid array type (double required)");
      return;
   }
   if(lenght>ar->total || lenght < 0){
      error(ERROR,"Array too short (%i) for writing %i values",ar->total,lenght);
      return;
   }
   
   adata=(double *)(ar->pointer);
   
   error(NOTE,"Writing %i %i-byte values to device %s",lenght,itemsize,device);
   
   dev=getDevice(device);
   
   if(dev==NULL){
      error(WARNING,"Device %s not available.",device);
   }
   else{
      write_len=convert_write(adata,lenght,&buff,itemsize,dev->getEndian());
      if(buff!=NULL){
	 status=dev->Write((char *)buff, write_len,1);	
	 free(buff);
      }
      
      if(status==-1){
	 error(WARNING,"Device %s not open.",device);
      }
      if(status==-2){
	 error(WARNING,"Error writing device %s.",device);
      }      
      if(status==-3){
	 error(WARNING,dev->getErrorLog());
      }      
   }
   return;
}
