#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <qapp.h>
#include <qlabel.h>
#include <qpopmenu.h>
#include <qkeycode.h>
#include <qpushbt.h>
#include <qmsgbox.h>
#include <qlist.h>
#include <qfiledlg.h>
#include <qregexp.h>
#include <qprinter.h>

#include "gui.hpp"
#include "gui.moc"

#define cp(a) printf("cp%d\n", a);

//extern unsigned char *swatch[];
//extern unsigned char *mswatch[];
extern unsigned char *gclock[];
extern unsigned char *mgclock[];

KTurtleGUI::KTurtleGUI
(
	QWidget *p
)
: QWidget(p)
{
	QPopupMenu *file = new QPopupMenu;
	file->insertItem("New", this, SLOT(menuNew()), CTRL+Key_N);
	file->insertItem("Load", this, SLOT(menuLoad()), CTRL+Key_L);
	file->insertItem("Save", this, SLOT(menuSave()), CTRL+Key_S);
	file->insertSeparator();
	file->insertItem("Print", this, SLOT(menuPrint()), CTRL+Key_P);
	file->insertItem("Bitmap", this, SLOT(menuBitmap()), CTRL+Key_B);
	file->insertSeparator();
	file->insertItem("Quit",  qApp, SLOT(quit()), CTRL+Key_Q);

	QPopupMenu *edit = new QPopupMenu;
	edit->insertItem("Edit Fractal", this, SLOT(menuEdit()), CTRL+Key_E);

	QPopupMenu *view = new QPopupMenu;
	view->insertItem("Next Level", this, SLOT(menuNext()), CTRL+Key_T);
	view->insertSeparator();
	levels[0]  = view->insertItem("Level 0", this, SLOT(menuLevel()));
	levels[1]  = view->insertItem("Level 1", this, SLOT(menuLevel()));
	levels[2]  = view->insertItem("Level 2", this, SLOT(menuLevel()));
	levels[3]  = view->insertItem("Level 3", this, SLOT(menuLevel()));
	levels[4]  = view->insertItem("Level 4", this, SLOT(menuLevel()));
	levels[5]  = view->insertItem("Level 5", this, SLOT(menuLevel()));
	levels[6]  = view->insertItem("Level 6", this, SLOT(menuLevel()));
	levels[7]  = view->insertItem("Level 7", this, SLOT(menuLevel()));
	levels[8]  = view->insertItem("Level 8", this, SLOT(menuLevel()));
	levels[9]  = view->insertItem("Level 9", this, SLOT(menuLevel()));
	levels[10] = view->insertItem("Level 10", this, SLOT(menuLevel()));
	levels[11] = view->insertItem("Level 11", this, SLOT(menuLevel()));
	levels[12] = view->insertItem("Level 12", this, SLOT(menuLevel()));
	levels[13] = view->insertItem("Level 13", this, SLOT(menuLevel()));
	levels[14] = view->insertItem("Level 14", this, SLOT(menuLevel()));
	levels[15] = view->insertItem("Level 15", this, SLOT(menuLevel()));
	view->insertSeparator();
	view->insertItem("Last Level", this, SLOT(menuLast()), CTRL+Key_Z);

	QPopupMenu *help = new QPopupMenu;
	help->insertItem("Help", this, SLOT(menuHelp()));
	help->insertSeparator();
	help->insertItem("About", this, SLOT(menuAbout()));

	menu = new QMenuBar(this);
	menu->insertItem("&File", file);
	menu->insertItem("&Edit", edit);
	menu->insertItem("&View", view);
	menu->insertSeparator();
	menu->insertItem("&Help", help);
	connect(menu, SIGNAL(highlighted(int)), this, SLOT(menuSetLevel(int)));

	// the turtle widget is the drawing machine
	turtle = new KTurtle(this);

	// dreaded misc
	level = 0;
	axiom = 0;
	rules = 0;
	dirs = 8;

	// center on screen
	setGeometry(QApplication::desktop()->width()/2-width()/2,
		QApplication::desktop()->height()/2-height()/2,
		width(), height());

	// animated cursor with stopwatch for long waits
	cur = new KAnimCursor(gclock, mgclock, 10);
}

KTurtleGUI::~KTurtleGUI()
{
	delete turtle;
	clear();
	delete cur;
}

void KTurtleGUI::clear()
{
	delete axiom;
	axiom = 0;
	for (int i = 0; rules && rules[i]; delete rules[i++]);
	delete rules;
	rules = 0;
	dirs = 9;
	level = 0;
	turtle->reset();
}

void KTurtleGUI::resizeEvent
(
	QResizeEvent *
)
{
	turtle->setGeometry(0, menu->height(), width(), height()-menu->height());
}

void KTurtleGUI::menuNew()
{
	clear();
}

void KTurtleGUI::menuLoad()
{
	QList<char> lst;
	int len, i;
	char line[1024], *ptr;
	FILE *f;
	QString s(QFileDialog::getOpenFileName(0, "*.tur"));
	if (s.isNull())
		return;

	// load fractal from file
	if (!(f = fopen(s, "r")))
	{
		QMessageBox m(this);
		m.setText("Failed to access fractal description file");
		m.show();
		return;
	}
	for (fgets(line, 1024, f); !feof(f) && !ferror(f); fgets(line, 1024, f))
	{
		// strip new-line
		if (line[(len = strlen(line))-1] == '\n')
			line[--len] = 0;

		// if anything left and not a comment
		if (len && line[0] != '#')
		{
			ptr = new char[len+1];
			strcpy(ptr, line);
			lst.append(ptr);
		}
	}

	// traverse loaded fractal description and store in internal containers
	// must be at least axiom, direction and 1 rule
	QListIterator<char> it(lst);
	if (it.count() >= 3)
	{
		// get rid of previous axiom and rules
		clear();

		// direction first
		if (!(dirs = atoi(ptr = it.toFirst())))
		{
			QMessageBox m(this);
			m.setText("WARNING : invalid direction count\n"
				"Default to 8 directions");
			m.show();
		}
		delete ptr;

		// axiom is second
		axiom = ++it;

		// rules are rest
		rules = new char*[it.count()-2+1];
		for (i = 0, ++it; it.current(); ++it, ++i)
			rules[i] = it.current();
		rules[i] = 0;
	}
	else
	{
		QMessageBox m(this);
		m.setText("Could not derive axiom and rule(s) from file");
		m.show();
	}
	fclose(f);
}

void KTurtleGUI::menuSave()
{
	FILE *f;

	// make sure there is anything to save
	if (!axiom || !rules)
	{
		QMessageBox m(this);
		m.setText("Nothing to save. Define axiom and rules first");
		m.show();
		return;
	}

	QString s(QFileDialog::getSaveFileName(0, "*.tur"));
	if (s.isNull())
		return;
	QRegExp r = QRegExp(".tur$", TRUE, FALSE);
	if (!s.contains(r))
		s.append(".tur");
	if (!(f = fopen(s, "w")))
	{
		QMessageBox m(this);
		m.setText("Failed to access target fractal file");
		m.show();
		return;
	}

	// save direction count
	fprintf(f, "%d\n", dirs);

	// save axiom
	fprintf(f, "%s\n", axiom);

	// traverse rules and save to file
	for (char **ptr = rules; *ptr; fprintf(f, "%s\n", *ptr++));

	fclose(f);
}

void KTurtleGUI::menuEdit()
{
	char *rstr = 0;
	int rlen, i, rcnt;

	// build string from rules array
	if (rules)
	{
		for (rlen = 0, i = 0; rules[i]; rlen += strlen(rules[i])+1, i++);
		rstr = new char[rlen+1];
		for (*rstr = 0, i = 0; rules[i];
			strcat(rstr, rules[i]), strcat(rstr, "\n"), i++);
	}

	// dialog to setup axiom and rules
	KTurtleSetup *ts = new KTurtleSetup(this);
	ts->setAxiom(axiom ? axiom : "");
	ts->setRules(rstr ? rstr : "");
	ts->setDirs(dirs);
	delete rstr;
	if (ts->exec() == QDialog::Accepted)
	{
		// entered values accepted so make this the new current ruleset

		// get rid of previous vals
		clear();

		// retrieve entered axiom ( assumes setup validation not accepted
		// if no valid axiom entered )
		axiom = new char[strlen(ts->getAxiom())+1];
		strcpy(axiom, ts->getAxiom());

		// retrieve rules from multiline edit
		if ((rcnt = ts->getRuleCount()))
		{
			rules = new char*[rcnt+1];
			for (i = 0; i < rcnt; i++)
			{
				rules[i] = new char[strlen(ts->getRule(i))+1];
				strcpy(rules[i], ts->getRule(i));
			}
			rules[i] = 0;
		}

		// get entered number of direction
		dirs = ts->getDirs();
	}
}

void KTurtleGUI::menuPrint()
{
	int w = (int)(8.26 * 72.0);
	int h = (int)(11.7 * 72.0);

	// make sure valid rule-set
	if (!rules || !axiom)
	{
		QMessageBox m(this);
		m.setText("Please create an axiom and ruleset first");
		m.show();
		return;
	}

	QPrinter p;
	if (p.setup(this))
	{
		// manually derive device dims ( only way i know of doing it )
		if (p.pageSize() == QPrinter::B5)
		{
			w = (int)(7.17 * 72.0);
			h = (int)(10.13 * 72.0);
		}
		if (p.pageSize() == QPrinter::Letter)
		{
			w = (int)(8.5 * 72.0);
			h = (int)(11 * 72.0);
		}
		if (p.pageSize() == QPrinter::Legal)
		{
			w = (int)(8.5 * 72.0);
			h = (int)(14 * 72.0);
		}
		if (p.pageSize() == QPrinter::Executive)
		{
			w = (int)(7.5 * 72.0);
			h = (int)(10 * 72.0);
		}

		// render scene with string replacement mechanism
		turtle->setDevice(&p, (int)(8.26*72.0), (int)(11.7*72.0));
		turtle->go(axiom, rules, dirs, level);
		turtle->setDevice(0, 0, 0);
	}
}

void KTurtleGUI::menuBitmap()
{
	QString s(QFileDialog::getSaveFileName(0, "*.bmp"));
	if (s.isNull())
		return;
	QRegExp r = QRegExp(".bmp$", TRUE, FALSE);
	if (!s.contains(r))
		s.append(".bmp");

	// ask turtle to save its canvas to file
	turtle->save(s);
}

void KTurtleGUI::menuNext()
{
	level++;
	menuLevel();
}

void KTurtleGUI::menuLast()
{
	level = -1;
	menuLevel();
}

void KTurtleGUI::menuSetLevel
(
	int l
)
{
	// if it is one of the level menu items, set the new requested level
	for (int i = 0; i <= 15; i++)
	{
		if (levels[i] == l)
		{
			level = i;
			return;
		}
	}
}

void KTurtleGUI::menuLevel()
{
	// make sure valid rule-set
	if (!rules || !axiom)
	{
		QMessageBox m(this);
		m.setText("Please create an axiom and ruleset first");
		m.show();
		return;
	}

	// might take long
	cur->start(this);

	// render scene with string replacement mechanism
	if (turtle->go(axiom, rules, dirs, level) == -1)
		level--;

	// finito
	cur->stop();
	setCursor(arrowCursor);
}

void KTurtleGUI::menuHelp()
{
#ifndef NOKDE
	if (fork() == 0)
	{
		char *kdedir;
		QString path = "";
		if ((kdedir = getenv("KDEDIR")))
			path.append(kdedir);
		else
			path.append("/usr/local/kde");
		path.append("/doc/HTML/kturtle");
		path.append("/kturtle.html");

		execlp("kdehelp", "kdehelp", path.data(), 0);
		exit( 1 );
	}
#else
	QMessageBox m(this);
	m.setText("Please use your browser to read kturtle.html");
	m.show();
#endif
}

void KTurtleGUI::menuAbout()
{
	QMessageBox m(this);
	m.setText("KTurtle v0.1\n\r\n\rwritten by Tiaan Wessels 1997\n\rtiaan@netsys.co.za");
	m.show();
}

KTurtleSetup::KTurtleSetup
(
	QWidget *p
)
: QDialog(p, "turtles", TRUE)
{
	(new QLabel("Axiom", this))->setGeometry(10, 10, 50, 10);
	(axiom = new QLineEdit(this))->setGeometry(10, 20, 200, 20);

	(new QLabel("Rules", this))->setGeometry(10, 45, 50, 10);
	(rules = new QMultiLineEdit(this))->setGeometry(10, 55, 200, 20*5);

	(new QLabel("Directions", this))->setGeometry(10, 160, 70, 10);
	(dirs = new QLineEdit(this))->setGeometry(10, 170, 200, 20);

	QPushButton *cancel = new QPushButton("Cancel", this);
	cancel->setGeometry(10, 200, 70, 20);
	cancel->setDefault(TRUE);
	connect(cancel, SIGNAL(clicked()), this, SLOT(reject()));

	QPushButton *ok = new QPushButton("OK", this);
	ok->setGeometry(180, 200, 30, 20);
	connect(ok, SIGNAL(clicked()), this, SLOT(okPressed()));

	adjustSize();
}

void KTurtleSetup::okPressed()
{
	int match = 0;

	// validate axiom
	if (!strlen(axiom->text()))
	{
		QMessageBox m(this);
		m.setText("An axiom is compulsory");
		m.show();
		return;
	}

	// validate rules
	for (int i = 0, c = rules->numLines(); i < c; i++)
	{
		if (strchr(axiom->text(), *(rules->textLine(i))))
		{
			match = 1;
			break;
		}
	}

	// if not rule that match at least one char in the axiom then error
	if (!match)
	{
		QMessageBox m(this);
		m.setText("You should have at least one rule\n"
			"affecting the axiom during expansion");
		m.show();
		return;
	}

	// validate directions to be number only
	if (atoi(dirs->text()) == 0)
	{
		QMessageBox m(this);
		m.setText("Invalid number of directions");
		m.show();
		return;
	}

	accept();
}
