/* -*- C++ -*-
 * This file declares the main view of the kardinfo utility.
 * kardinfo is used to manage PCMCIA slots from the KDE desktop.
 * 
 * Process this source files with doxygen by running 
 *    "doxygen -c doxygen-config" 
 * from the command line. HTML output will be generated in the "html" 
 * subdirectory.
 *
 * copyright:  (C) Mirko Sucker, 1999
 * license:    GNU Public License, Version 2
 * mail to:    Mirko Sucker <mirko.sucker@unibw-hamburg.de>
 * requires:   recent C++-compiler, at least Qt 1.4
 * $Revision: 1.3 $
 */

#ifndef KARDINFO_KARDINFO_H
#define KARDINFO_KARDINFO_H

#include <ktabctl.h>
#include <qframe.h>
#include <qsize.h>
#include <list>

class QLabel;
class KLedLamp;
class QPushButton;
class QTimer;
class QPopupMenu;

extern "C" {
#include <sys/types.h>
#include <stdio.h>
}

// ----- typedefs taken from cardinfo.c:

typedef enum s_state {
    S_EMPTY, S_PRESENT, S_READY, S_BUSY, S_SUSPEND
} s_state;


// ----- the maximum number of PCMCIA slots:

#define MAX_SOCKETS 8

/** KITab is used to implement the tabs in the tabcontrol. There is a tab for 
 *  every slot found. 
 *  To read the sources, it will be handy to have a dump of the cardinfo sources
 *  (see Linux PCMCIA package) available, since most stuff is derived from it.
 *  The class encapsulates the information provided by the \t socket_info_t 
 *  struct used by cardinfo. */

class KITab : public QWidget
{
  // ############################################################################
  Q_OBJECT
  // ----------------------------------------------------------------------------
public:
  /** QT's standard constructor. */
  KITab(QWidget* parent=0, const char* name=0);
  /** The destructor. */
  ~KITab();
  /** Report a size as the preferred minimum size. */
  QSize sizeHint();
  /** Set the cards device. */ 
  void setCardDevice(const char* dev);
  /** Set the cards description. */
  void setCardDescription(const char* desc);
  /** Set the cards io ports. */
  void setCardPorts(const char* ports);
  /** Set the cards irq lines. */
  void setCardIRQ(const char* lines);
  /* Set the cards flags. */
  void setCardFlags(bool cd, bool power, bool pp, bool wp);
  /** Set the cards state. */
  void setState(s_state state);
  /** Get the cards state. */
  s_state getState();
  // ----------------------------------------------------------------------------
protected:
  // methods:
  /** The resize event. */
  void resizeEvent(QResizeEvent*);
  // members:
  /** Displays the description of the card. */
  QLabel *content;
  /** A horizontal line below the description. */
  QFrame *line;
  /** A label showing the state of the slot at the bottom. */
  QLabel *labelState;
  /** A LED showing if a card has been detected. */
  KLedLamp *cd;
  /** A label explaining the cd lamp. */
  QLabel *labelCd;
  /** A LED showing the power state of the card. */
  KLedLamp *power;
  /** A label explaining the power lamp. */
  QLabel* labelPower;
  /** A LED showing the programming power state of the card. */
  KLedLamp *pp;
  /** A label explaining the programming power lamp. */
  QLabel* labelPp;
  /** A LED showing if the card is write protected. */
  KLedLamp *wp;
  /** A label explaining the wp lamp. */
  QLabel *labelWp;
  /** A button for showing some details about the card hardware. */
  QPushButton *details;
  /** The button showing the menu for the card management actions. */
  QPushButton *actions;
  /** The popup menu. */
  QPopupMenu *menu;
  // members that deal with card settings:
  /** The card description (literal). */
  QString card;
  /** The card device. */
  QString dev;
  /** The io ports (literal). */
  QString io;
  /** The irq (literal). */
  QString irq;
  /** The state of the slot. */
  s_state state;
  // ----------------------------------------------------------------------------
public slots:
  /** Catch clicks on the "Details" button. */
  void detailsClickedSlot();
  /** Catch clicks on the "Actions" button. */
  void actionsClickedSlot();
  /** Catch menu operations. */
  void menuActivatedSlot(int);
signals:
  /** Emit to notify KardInfo about the menu actions. 
   *  The tabs do not handle card operations itselfes. */
  void menuActivated(KITab *emitter, int item);
  // ############################################################################
};

/** KardInfo implements the programs main view. The program uses the KTMainWindow
 *  class to handle menus and toolbars easily, and adds an object of this class 
 *  to the toplevel window as the main widget. */

class KardInfo : public QFrame
{
  // ############################################################################
  Q_OBJECT
  // ----------------------------------------------------------------------------
public:
  /** QT's standard constructor. */
  KardInfo(QWidget* parent=0, const char* name=0);
  /** The destructor. */
  ~KardInfo();
  /** The size hint. */
  QSize sizeHint();
  // ----------------------------------------------------------------------------
protected:
  // methods (mostly derived from cardinfo):
  /** The resize event. */
  void resizeEvent(QResizeEvent*);
  /** Lookup the device by its name. */
  int lookupDevice(const char*);
  /** Open the device. */
  int openDevice(dev_t dev);
  /** Show an alert. */
  void alert(const char* headline, const char* text);
  // members:
  /** The file descriptors of the socket device. */
  int fd[MAX_SOCKETS];
  /** The pointers to the tabs according to the sockets. */
  KITab *tabs[MAX_SOCKETS];
  /** The number of slots found in the system. */
  int ns;
  /** The tab control that contains one tab per slot. */
  KTabCtl *tabctl;
  /** A timer to trigger to update polls. */
  QTimer *timer;
  // ----------------------------------------------------------------------------
protected slots:
  /** Setup the widget geometry. This sets a minimum size. */
  void initializeGeometry();
  /** Update the card information. */
  void updateCardData();
  /** Catch the clicks  on the actions buttons shown by the tabs. */
  void menuActivatedSlot(KITab *emitter, int item);
  // ############################################################################
};

#endif // KARDINFO_KARDINFO_H
