#include <qpainter.h>

#include <kconfig.h>
#include <kapp.h>

#include <stdlib.h>

#include "ColorSet.h"
#include "../EditorSectionName.h"
#include "../moc/ColorSet.moc"
#include "../moc/ColorSetData.moc"

#define Inherited ColorSetData



//////////////////////////////////////////////
static QFont sFont;		// Font used in syntax element listbox
//////////////////////////////////////////////



//////////////////////////////////////////////
class SyntaxStateItem : public QListBoxText
{
public:
	SyntaxStateItem(SyntaxState* state)
		: QListBoxText(state->name())
	{ State = state; }
	
	virtual void paint(QPainter* p)
	{
		QFont orig(p->font());
		QFont f(sFont);
		QFontMetrics fm(sFont);
		State->setDrawingAttrs(*p, f);
		p->drawText(3, (fm.leading()>>1) + fm.ascent(), State->name());
		p->setFont(orig);
	}
	
	virtual int height(const QListBox*) const
	{
		QFontMetrics fm(sFont);
		return fm.lineSpacing() + 2;
	}

	virtual int width(const QListBox*) const
	{
		QFontMetrics fm(sFont);
		return fm.width(State->name());
	}
	
	SyntaxState* state() const
	{ return State; }

	
protected:
	SyntaxState* State;
};
//////////////////////////////////////////////



/////////////////////////////////////////////////////////////////////
class QUsableListBox : public QListBox
{
    Q_OBJECT
public:
	QUsableListBox(QWidget* parent = 0, const char* name = 0, WFlags f = 0)
		: QListBox(parent, name, f)
	{}
		
	QListBoxItem* item(int index) const;
   
};
/////////////////////////////////////////////////////////////////////



/////////////////////////////////////////////////////////////////////
QListBoxItem* QUsableListBox::item(int index) const
{ 
	return QListBox::item(index); 
}
/////////////////////////////////////////////////////////////////////



//////////////////////////////////////////////
ColorSet::ColorSet(Colorizer& col, const QFont& font, QWidget* parent)
	: Inherited(parent),
	  colorizer(col)
{
	sFont = font;

	setCaption( "KColorEditor Color Settings" );
	btCancel->setEnabled(false);

	LocalSStates.setAutoDelete(true);
	LocalSStates.resize(colorizer.stateArraySize());
	for(int i = 0; i < colorizer.stateArraySize(); i++)
	{
		SyntaxState* ss = new SyntaxState(colorizer.getStateInfo(i));
		LocalSStates.insert(i, ss);
		seListbox->inSort(new SyntaxStateItem(ss));
	}

	QString grp = kapp->getConfig()->group();
	kapp->getConfig()->setGroup(EditorSectionName);
	QStrList colors;
	kapp->getConfig()->readListEntry("ColorDlgColors", colors);
	kapp->getConfig()->setGroup(grp);
	
	QStrListIterator it(colors);
	while(it.current())
	{
		QString s = it.current();
		s = s.stripWhiteSpace();
		insertColorItem(s, QColor(s));
		++it;
	}
}
//////////////////////////////////////////////



//////////////////////////////////////////////
ColorSet::~ColorSet()
{}
//////////////////////////////////////////////



//////////////////////////////////////////////
void ColorSet::insertColorItem(const char* name, const QColor& color)
{
	ColorItem* ci = new ColorItem(name, color);
	colorsListbox->inSort(ci);
	ColorTable.insert(color.rgb(), ci);
}
//////////////////////////////////////////////



//////////////////////////////////////////////
void ColorSet::elemChanged()
{
	int currentItem = seListbox->currentItem();
	if(currentItem == -1)
		return;
		
	QListBoxItem* i = ((QUsableListBox*)seListbox)->item(currentItem);
	if(!i)
		return;
		
	SyntaxState* state = ((SyntaxStateItem*)i)->state();
	if(!state)
		return;

	state->setWeight(cbBold->isChecked() ? QFont::Bold : QFont::Normal);
	state->setItalic(cbItalic->isChecked());
	
	int currentColorItem = colorsListbox->currentItem();
	if(currentColorItem != -1)
	{	
		ColorItem* ci = ((ColorItem*)((QUsableListBox*)colorsListbox)->item(currentColorItem));
		if(ci)
			state->setColor(ci->color());
	}
		
	seListbox->repaint();
}
//////////////////////////////////////////////



//////////////////////////////////////////////
void ColorSet::colorChanged(int)
{
	elemChanged();
}
//////////////////////////////////////////////



//////////////////////////////////////////////
void ColorSet::applyClicked()
{
	for(int i = 0; i < colorizer.stateArraySize(); i++)
	{
		SyntaxState* dlgState = LocalSStates[i];
		colorizer.setStateInfo(*dlgState, i);
	}

	emit applied();
}
//////////////////////////////////////////////



//////////////////////////////////////////////
void ColorSet::syntaxElementHighlighted(int index)
{
	QListBoxItem* i = ((QUsableListBox*)seListbox)->item(index);
	if(!i)
		return;

	SyntaxState* state = ((SyntaxStateItem*)i)->state();
	if(!state)
		return;
	
	cbBold->setChecked(state->weight() == QFont::Bold);
	cbItalic->setChecked(state->italic());
	
	ColorItem* ci = ColorTable.find(state->color().rgb());
	if(ci)
	{
		unsigned int index;
		for(index = 0; index < colorsListbox->count(); index++)
		{
			if(((ColorItem*)((QUsableListBox*)colorsListbox)->item(index)) == ci)
				break;
		}
		if(index < colorsListbox->count()) 
		{
			colorsListbox->setCurrentItem(index);
			colorsListbox->centerCurrentItem();
		}
		else
			colorsListbox->clearSelection();
	}
}
//////////////////////////////////////////////

